'use strict';

/**
 * @name sort-values
 * @desc Sort Values directive used to sort the directive
 */
export default angular.module('app.sort-values.directive', [])
    .directive('sortValues', sortValuesDirective);

sortValuesDirective.$inject = [];

function sortValuesDirective() {
    sortValuesCtrl.$inject = [];
    sortValuesLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        scope: {},
        restrict: 'EA',
        require: ['^widget'],
        controllerAs: 'sortValues',
        bindToController: {},
        template: require('./sort-values.directive.html'),
        controller: sortValuesCtrl,
        link: sortValuesLink
    };

    function sortValuesCtrl() {}

    function sortValuesLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];

        // variables
        scope.sortValues.dataTableOptions = {};
        scope.sortValues.sortOptions = [];
        scope.sortValues.alias = '';

        // functions
        scope.sortValues.newSort = newSort;
        scope.sortValues.updateSort = updateSort;
        scope.sortValues.removeSort = removeSort;

        /** Logic **/
        /**
         * @name resetPanel
         * @desc function that is resets the panel when the data changes
         * @returns {void}
         */
        function resetPanel() {
            // set sort options
            let usedHeaders = {},
                active = scope.widgetCtrl.getWidget('active'),
                layout = scope.widgetCtrl.getWidget('view.' + active + '.layout'),
                keys = scope.widgetCtrl.getWidget('view.' + active + '.keys.' + layout),
                availableHeaders = scope.widgetCtrl.getFrame('headers') || [],
                layerIndex = 0;

            scope.sortValues.sortOptions = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.sortInfo') || [];

            scope.sortValues.alias = '';

            for (let i = 0, len = scope.sortValues.sortOptions.length; i < len; i++) {
                usedHeaders[scope.sortValues.sortOptions[i].alias] = true;
            }

            scope.sortValues.headers = [];

            for (let i = 0, len = keys.length; i < len; i++) {
                if (keys[i].derived) {
                    scope.sortValues.headers.push({
                        alias: keys[i].alias,
                        dir: '',
                        derived: true
                    });
                }
            }

            for (let i = 0, len = availableHeaders.length; i < len; i++) {
                if (!usedHeaders.hasOwnProperty(availableHeaders[i].alias)) {
                    scope.sortValues.headers.push({
                        alias: availableHeaders[i].alias,
                        dir: '',
                        derived: false
                    });
                }
            }
        }

        /**
         * @name newSort
         * @param {string} dir - that sorting should occur in
         * @desc create an new sort
         * @returns {void}
         */
        function newSort(dir) {
            scope.sortValues.sortOptions.push({
                alias: scope.sortValues.alias,
                dir: dir
            });
            executeSort();
        }


        /**
         * @name updateSort
         * @param {string} dir - direction that sorting should occur in
         * @param {string} index - sort to update
         * @desc update an existing sort
         * @returns {void}
         */
        function updateSort(dir, index) {
            scope.sortValues.sortOptions[index].dir = dir;
            executeSort();
        }


        /**
         * @name removeSort
         * @param {string} index - sort to remove
         * @desc remove an existing sort
         * @returns {void}
         */
        function removeSort(index) {
            scope.sortValues.sortOptions.splice(index, 1);
            executeSort();
        }

        /**
         * @name executeSort
         * @desc executes the sort
         * @returns {void}
         */
        function executeSort() {
            let sortOptions = {
                    task: [],
                    panel: []
                },
                layerIndex = 0,
                layout = scope.widgetCtrl.getWidget('view.visualization.layout'),
                keys = scope.widgetCtrl.getWidget('view.visualization.keys.' + layout),
                groupByInfo = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.groupByInfo'),
                taskOptionsComponent,
                selectComponent,
                groupComponent,
                panelSortType,
                limit = scope.widgetCtrl.getOptions('limit');


            // create the sortOptions
            for (let i = 0, len = scope.sortValues.sortOptions.length; i < len; i++) {
                let derived = false;
                for (let j = 0, len2 = scope.sortValues.headers.length; j < len2; j++) {
                    if (scope.sortValues.headers[j].alias === scope.sortValues.sortOptions[i].alias) {
                        if (scope.sortValues.headers[j].derived) {
                            derived = true;
                        }
                        break;
                    }
                }

                if (derived) {
                    sortOptions.task.push(scope.sortValues.sortOptions[i]);
                } else {
                    sortOptions.panel.push(scope.sortValues.sortOptions[i]);
                }
            }

            // create components
            taskOptionsComponent = {};
            taskOptionsComponent[scope.widgetCtrl.panelId] = {
                'layout': layout,
                'alignment': {}
            };

            selectComponent = [];
            groupComponent = [];

            for (let i = 0, len = keys.length; i < len; i++) {
                // add in the model
                if (!taskOptionsComponent[scope.widgetCtrl.panelId].alignment[keys[i].model]) {
                    taskOptionsComponent[scope.widgetCtrl.panelId].alignment[keys[i].model] = [];
                }

                if (keys[i].math) {
                    // add in the group component
                    if (groupComponent.length === 0) {
                        groupComponent = keys[i].groupBy;
                    }
                }


                // add in the select component
                if (keys[i].calculatedBy) {
                    selectComponent.push({
                        calculatedBy: keys[i].calculatedBy,
                        math: keys[i].math,
                        alias: keys[i].alias
                    });
                } else {
                    selectComponent.push({
                        alias: keys[i].alias
                    });
                }

                // add to the view component
                taskOptionsComponent[scope.widgetCtrl.panelId].alignment[keys[i].model].push(keys[i].alias);
            }

            panelSortType = 'unsortPanel';
            if (sortOptions.panel.length > 0) {
                panelSortType = 'setPanelSort';
            }

            // If Facet exists, update taskOptions and add Filter component to pixel
            if (groupByInfo && groupByInfo.viewType) {
                let instanceIdx;

                taskOptionsComponent[scope.widgetCtrl.panelId].groupByInfo = groupByInfo;

                if (typeof groupByInfo.instanceIndex === 'string') {
                    instanceIdx = parseInt(groupByInfo.instanceIndex, 10);
                }

                if (groupByInfo.viewType === 'Individual Instance') {
                    let filterComponent = {};

                    filterComponent[groupByInfo.selectedDim] = {
                        'comparator': '==',
                        'value': groupByInfo.uniqueInstances[instanceIdx]
                    };

                    scope.widgetCtrl.execute([{
                        'type': 'panel',
                        'components': [
                            scope.widgetCtrl.panelId
                        ]
                    },
                    {
                        'type': panelSortType,
                        'components': [
                            sortOptions.panel
                        ],
                        'terminal': true
                    },
                    {
                        'type': 'frame',
                        'components': [
                            scope.widgetCtrl.getFrame('name')
                        ]
                    },
                    {
                        'type': 'select2',
                        'components': [
                            selectComponent
                        ]
                    },
                    {
                        'type': 'group',
                        'components': [
                            groupComponent
                        ]
                    },
                    {
                        'type': 'filter',
                        'components': [
                            filterComponent
                        ]
                    },
                    {
                        'type': 'sortOptions',
                        'components': [
                            sortOptions.task
                        ]
                    },
                    {
                        'type': 'with',
                        'components': [
                            scope.widgetCtrl.panelId
                        ]
                    },
                    {
                        'type': 'format',
                        'components': ['table']
                    },
                    {
                        'type': 'taskOptions',
                        'components': [
                            taskOptionsComponent
                        ]
                    },
                    {
                        'type': 'collect',
                        'components': [limit],
                        'terminal': true
                    }
                    ]);
                } else {
                    scope.widgetCtrl.execute([{
                        'type': 'panel',
                        'components': [
                            scope.widgetCtrl.panelId
                        ]
                    },
                    {
                        'type': panelSortType,
                        'components': [
                            sortOptions.panel
                        ],
                        'terminal': true
                    },
                    {
                        'type': 'frame',
                        'components': [
                            scope.widgetCtrl.getFrame('name')
                        ]
                    },
                    {
                        'type': 'select2',
                        'components': [
                            selectComponent
                        ]
                    },
                    {
                        'type': 'group',
                        'components': [
                            groupComponent
                        ]
                    },
                    {
                        'type': 'sortOptions',
                        'components': [
                            sortOptions.task
                        ]
                    },
                    {
                        'type': 'with',
                        'components': [
                            scope.widgetCtrl.panelId
                        ]
                    },
                    {
                        'type': 'format',
                        'components': ['table']
                    },
                    {
                        'type': 'taskOptions',
                        'components': [
                            taskOptionsComponent
                        ]
                    },
                    {
                        'type': 'collect',
                        'components': [limit],
                        'terminal': true
                    }
                    ]);
                }
            } else {
                scope.widgetCtrl.execute([{
                    'type': 'panel',
                    'components': [
                        scope.widgetCtrl.panelId
                    ]
                },
                {
                    'type': panelSortType,
                    'components': [
                        sortOptions.panel
                    ],
                    'terminal': true
                },
                {
                    'type': 'frame',
                    'components': [
                        scope.widgetCtrl.getFrame('name')
                    ]
                },
                {
                    'type': 'select2',
                    'components': [
                        selectComponent
                    ]
                },
                {
                    'type': 'group',
                    'components': [
                        groupComponent
                    ]
                },
                {
                    'type': 'sortOptions',
                    'components': [
                        sortOptions.task
                    ]
                },
                {
                    'type': 'with',
                    'components': [
                        scope.widgetCtrl.panelId
                    ]
                },
                {
                    'type': 'format',
                    'components': ['table']
                },
                {
                    'type': 'taskOptions',
                    'components': [
                        taskOptionsComponent
                    ]
                },
                {
                    'type': 'collect',
                    'components': [limit],
                    'terminal': true
                }
                ]);
            }
        }

        /**
         * @name initialize
         * @desc function that is called on directive load
         * @returns {void}
         */
        function initialize() {
            let sortUpdateFrameListener,
                sortupdateTaskListener;


            // listeners
            sortUpdateFrameListener = scope.widgetCtrl.on('update-frame', resetPanel);
            sortupdateTaskListener = scope.widgetCtrl.on('update-task', resetPanel);

            // cleanup
            scope.$on('$destroy', function () {
                sortUpdateFrameListener();
                sortupdateTaskListener();
            });

            resetPanel();
        }

        initialize();
    }
}
