import angular from 'angular';
import jsPlumb from 'jsplumb';
import { CONNECTORS, PREVIEW_LIMIT } from '../../core/constants.js';
import Movable from '../../core/utility/movable';
import Resizable from '../../core/utility/resizable';
import PixelASTLeaf from './PixelASTLeaf';

import './pipeline.scss';

import './pipeline-component/pipeline-component.directive.js';
import './pipeline-compiler/pipeline-compiler.directive.js';
import './pipeline-landing/pipeline-landing.directive';
import '../federate/federate.directive';

export default angular.module('app.pipeline.directive', [
    'app.pipeline.component',
    'app.pipeline.compiler',
    'app.pipeline.landing',
    'app.federate.directive'
]).directive('pipeline', pipelineDirective);

pipelineDirective.$inject = ['$compile', '$timeout', 'ENDPOINT', 'semossCoreService', 'monolithService', 'CONFIG', '$q'];

/**
 * @name pipelineDirective
 * @desc This directive is responsible for connecting components in the pipeline, pipeline playback,
 * and reconstructing the pipeline based on pixel steps.
 *
 * 1. Connecting Components
 * Pipeline components are all stored in pipeline.data. This is an object where the keys are the
 * component ID with a value of component meta data (this such as input, output, pixel created by component, etc...)
 * The meta data comes from a combination of the widget's config.js as well as data added to the component
 * as pipeline processes it. When the user adds a component, addComponent is the starting point.
 *
 * 2. Pipeline Playback
 * Pipeline has a secondary state to determine if it is rerunning itself. pipeline.paused is a boolean which determines
 * that. When the pipeline is paused it can be stepped through, or reran automatically. togglePause() is how
 * the user changes states via the ui. Users will have to rerun the pipeline if they go back and edit
 * a component.
 *
 * 3. Pipeline Reconstruction
 * pipeline.data is persisted even when user leaves the pipeline in order to load it instantly from
 * memory. When the user leaves pipeline and interacts with Semoss, the pipeline will reconstruct
 * itself based on the persisted pipeline.data as well as building components based on the return from 
 * monolithService.getPipeline. The kick off to recreate the pipeline is the renderPipeline function
 */
function pipelineDirective(
    $compile: ng.ICompileService,
    $timeout: ng.ITimeoutService,
    ENDPOINT: EndPoint,
    semossCoreService: SemossCoreService,
    monolithService: MonolithService,
    CONFIG: any,
    $q: any
) {
    pipelineCtrl.$inject = [];
    pipelineLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        template: require('./pipeline.directive.html'),
        scope: {},
        require: ['^widget'],
        controller: pipelineCtrl,
        controllerAs: 'pipeline',
        bindToController: {},
        replace: true,
        link: pipelineLink
    };

    function pipelineCtrl() { }

    function pipelineLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];

        let pipelineEle: HTMLElement,
            pipelineLeftEle: HTMLElement,
            pipelineContentEle: HTMLElement,
            pipelineWorkspaceContentEle: HTMLElement,
            pipelinePreviewEle: HTMLElement,
            plumbing,
            timeout;

        const COMPONENT_STYLES = { // TODO: convert to em instead of pixel
            gutter: 24,
            endpoint: 24,
            height: 124,
            width: 160
        },
            DUMMY = 'DUMMY',
            REPLAY_TIMER = 750;

        // Each type has a very specific value. THIS IS IMPORTANT TO KEEP IF YOU WANT TO CASCADE
        // QUERY_STRUCT -  {qsType, engineName, isDistinct, limit, offset, relations, ordes, selectors, explicitFilters, havingFilters, query}
        // FRAME  - {name, type, create}
        // JOINS - {from, join, to}

        scope.pipeline.positionPipeline = positionPipeline;
        scope.pipeline.searchMenu = searchMenu;
        scope.pipeline.addComponent = addComponent;
        scope.pipeline.getComponent = getComponent;
        scope.pipeline.routeRemoveComponent = routeRemoveComponent;
        scope.pipeline.removeComponent = removeComponent;
        scope.pipeline.showComponent = showComponent;
        scope.pipeline.closeComponent = closeComponent;
        scope.pipeline.validateComponent = validateComponent;
        scope.pipeline.buildComponent = buildComponent;
        scope.pipeline.previewComponent = previewComponent;
        scope.pipeline.executeComponent = executeComponent;
        scope.pipeline.togglePause = togglePause;
        scope.pipeline.step = step;
        scope.pipeline.turnOffPause = turnOffPause;
        scope.pipeline.viewComponent = viewComponent;
        scope.pipeline.visualizeComponent = visualizeComponent;
        scope.pipeline.openPreview = openPreview;
        scope.pipeline.closePreview = closePreview;
        scope.pipeline.exportCSV = exportCSV;
        scope.pipeline.showFederate = showFederate;
        scope.pipeline.createFrameName = createFrameName;
        scope.pipeline.initializePipeline = initializePipeline;
        scope.pipeline.finishPipelineRender = finishPipelineRender;

        scope.pipeline.previewLimit = PREVIEW_LIMIT;
        scope.pipeline.landing = {
            open: false
        };

        scope.pipeline.menu = {
            searched: '',
            accordion: [
                {
                    'name': 'Source',
                    'height': 40,
                    'items': []
                },
                {
                    'name': 'Transform',
                    'height': 40,
                    'items': []
                },
                {
                    'name': 'Destination',
                    'height': 20,
                    'items': []
                }
            ]
        };

        scope.pipeline.nodeDeleted = false;
        scope.pipeline.paused = false;
        scope.pipeline.pausedAt = null;
        scope.pipeline.numComponents = 0;
        scope.pipeline.data = {};
        scope.pipeline.eles = {};
        scope.pipeline.movable = {};
        scope.pipeline.preview = {
            open: false,
            selected: undefined
        };

        scope.pipeline.STATE = {
            INITIAL: 'initial',
            EXECUTED: 'executed'
        };
        scope.pipeline.componentToPixelMap = {};
        scope.pipeline.edit = {
            open: false,
            selected: undefined,
            height: 80,
            preview: 20
        };
        scope.pipeline.federate = {
            open: false,
            selectedFrame: ''
        };

        /**
         * @name checkErrors
         * @param response pixel response
         * @desc checks for errors in pixel response
         * @return true if has errors, false otherwise
         */
        function checkErrors(response: PixelReturnPayload) {
            for (let outputIdx = 0, outputLen = response.pixelReturn.length; outputIdx < outputLen; outputIdx++) {
                if (response.pixelReturn[outputIdx].operationType.indexOf('ERROR') > -1 || response.pixelReturn[outputIdx].operationType.indexOf('INVALID_SYNTAX') > -1) {
                    scope.widgetCtrl.alert('error', response.pixelReturn[outputIdx].output);
                    return true;
                }
            }

            return false;
        }

        /**
         * @name initializePipeline
         * @desc initialize the pipeline
         */
        function initializePipeline(): void {
            const components = semossCoreService.getWidgetState('all');

            // get all of the pipeline widgets
            scope.pipeline.components = [];

            for (let componentIdx = 0, componentLen = components.length; componentIdx < componentLen; componentIdx++) {
                if (components[componentIdx].hasOwnProperty('pipeline')) {
                    scope.pipeline.components.push(components[componentIdx]);
                }
            }

            // load the pipeline
            loadPipeline();

            // render the menu
            searchMenu();
        }


        /**
         * @name loadPipelxine
         * @desc render the pipeline's UI
         */

        function loadPipeline(): void {
            let pipeline = semossCoreService.getPipeline('shared', scope.widgetCtrl.insightID),
                callback: (res: PixelReturnPayload) => void;

            if (pipeline) {
                renderPipeline(pipeline.components);
            } else if (scope.widgetCtrl.getShared('insight.app_insight_id')) {
                callback = response => {
                    const output = response.pixelReturn[0].output,
                        type = response.pixelReturn[0].operationType;

                    if (type.indexOf('ERROR') > -1) {
                        return;
                    }

                    renderPipeline(output.components);
                };

                scope.widgetCtrl.meta([{
                    type: 'Pixel',
                    components: ['RetrieveInsightPipeline(app=["' + scope.widgetCtrl.getShared('insight.app_id') + '"], id=["' + scope.widgetCtrl.getShared('insight.app_insight_id') + '"])'],
                    terminal: true,
                    meta: true
                }], callback);
            } else {
                renderPipeline({});
            }
        }

        /**
         * @name configureExternalComponent
         * @param node - pixel ast node
         * @param idx - node position in pixel ast
         * @param self - entirepixel ast
         * @desc creates pipeline component based on the node data
         */
        function configureExternalComponent(node: PixelASTLeaf[], idx: number, self: PixelASTLeaf[][]): void {
            let removedFrames: any = [];
            let selfIdx: number;
            let removedIdx: number;
            let frameRemoved: boolean;
            
            // check for removed frames
            // we will not paint these because the FE will not have any information on this frame to populate node
            for (selfIdx = 0; selfIdx < self.length; selfIdx++) {
                if (self[selfIdx][0].opName === 'RemoveFrame') {
                    removedFrames.push(self[selfIdx][0].rowInputs[0].value);
                }
            }
            for (let i = 0; i < node.length; i++) {
                const leaf = node[i];
                frameRemoved = false;
                if (leaf.widgetId) {
                    // extra configuration to handle external component edge cases
                    const externalComponentConfig = {
                        dependsOn: '',
                        createFrame: ''
                    },
                        prevComponent = scope.pipeline.lastAddedKey;
                    let key: string;
                    if (leaf.opName === 'Import') {
                        // create frame pixel
                        externalComponentConfig.createFrame = self[idx - 1][0].opString + ';';
                        removeOverwrittenComponents(self, idx, scope.pipeline.data);
                    }
                    // always comes right after the component it spawned from
                    // need to link them as they will always be removed together
                    if (leaf.opName === DUMMY) {
                        externalComponentConfig.dependsOn = prevComponent;
                    }

                    for (removedIdx = 0; removedIdx < removedFrames.length; removedIdx++) {
                        if (externalComponentConfig.createFrame.indexOf(removedFrames[removedIdx]) > -1) {
                            frameRemoved = true;
                            break;
                        }
                    }

                    // don't process it because we dont have the frame info for frames that were removed.
                    if (frameRemoved) {
                        continue;
                    }

                    key = addComponent(leaf.widgetId, false, externalComponentConfig, true);
                    // add component will set auto for false but this is wrong for external
                    // components
                    scope.pipeline.data[key].position.auto = true;
                    renderComponentOutput(key);
                    executeExternalComponent(key, prevComponent, node);
                }
            }
        }

        /**
         * @name findNewPipelineNodes
         * @param ast - pixel ast from recipe
         * @param currentNodes - nodes already in pipeline
         * @desc finds the last placed node then determines where that node is in the pixel ast
         *      used to connect recipe steps done in semoss to those already in pipeline
         * @return ast of just the latest pixels
         */
        function findNewPipelineNodes(ast: PixelASTLeaf[][], currentNodes: any): PixelASTLeaf[][] {
            let overlapPosition: number = 0,
                astNodeCount: number = 0,
                finalNode: string = '',
                currentNodeMap = {};

            if (!currentNodes) {
                overlapPosition = 0;
            } else {
                for (let key in currentNodes) {
                    if (currentNodes.hasOwnProperty(key)) {
                        if (currentNodes[key].output[0] && Object.keys(currentNodes[key].output[0].downstream).length === 0) {
                            finalNode = currentNodes[key].id;
                        }
                        if (!currentNodeMap[currentNodes[key].id]) {
                            currentNodeMap[currentNodes[key].id] = 1;
                        } else {
                            currentNodeMap[currentNodes[key].id]++;
                        }
                    }
                }

                for (let i = 0; i < ast.length; i++) {
                    let widgetId = ast[i][ast[i].length - 1].widgetId;

                    if (widgetId && widgetId === finalNode) {
                        astNodeCount++;

                        if (astNodeCount === currentNodeMap[widgetId]) {
                            overlapPosition = i + 1;

                            break;
                        }
                    }
                }

                // determine if need to overwrite current pipeline components
                // this is the case when the overlap position comes at an import frame,
                // which creates the same frame as an internal component
                removeOverwrittenComponents(ast, overlapPosition, currentNodes);
            }

            return ast.slice(overlapPosition);
        }

        /**
         * @name removeOverwrittenComponents
         * @param {array} ast - ast node
         * @param {number} overlapIdx - where the overlap between internal and external components occurs
         * @param {object} components - internal (current) pipeline components
         * @desc removes pipeline components that are being updated by the external components
         */
        function removeOverwrittenComponents(ast: PixelASTLeaf[][], overlapIdx: number, components: any): void {
            const createFrameComponents = {}, frames: Map<string, string> = new Map();
            for (let c in components) {
                if (components.hasOwnProperty(c)) {
                    let params = components[c].parameters;
                    for (let param in params) {
                        if (
                            params.hasOwnProperty(param) &&
                            (params[param].type === 'CREATE_FRAME' || components[c].createFrame) &&
                            params[param].value
                        ) {
                            // set frame name to key and value is the component to remove
                            if (params[param].value) {
                                createFrameComponents[params[param].value.name] = c;
                            }
                        }
                    }
                }
            }

            if (Object.keys(createFrameComponents).length > 0) { // otherwise theres nothing to remove
                for (let i = overlapIdx; i < ast.length; i++) {
                    if (ast[i][ast[i].length - 1].opName === 'Import' && determineIfComponentShouldBeMerged(ast, i)) {
                        getExternalFrames(ast[i], frames);
                    }
                }

                for (let frame of frames.keys()) {
                    if (createFrameComponents.hasOwnProperty(frame)) {
                        removeComponent(createFrameComponents[frame], false);
                    }
                }
            }
        }

        /**
         * @name determinIfComponentShouldBeMerged
         * @param ast all leaves
         * @param  i which leaf
         * @desc determine if import component is same as another
         * @return true if should be merged
         */
        function determineIfComponentShouldBeMerged(ast: PixelASTLeaf[][], i: number): boolean {
            const curNode = ast[i], prevNode = ast[i - 1];
            let frame;

            if (curNode[curNode.length - 1].nounInputs) {
                frame = curNode[curNode.length - 1].nounInputs.frame;
            }

            return (
                (
                    frame &&
                    frame[0] &&
                    frame[0].value &&
                    frame[0].value.opName === 'CreateFrame'
                ) ||
                prevNode && prevNode[prevNode.length - 1].opName === 'CreateFrame'
            );
        }

        /**
         * @name renderTransformedComponents
         * @desc function serves to loop through the data pulled from the getPipeline() call and create the components
         * @param {array} nodesFromLastClear the sablecc raw data from getPipeline()
         * @returns {void}
         */
        function renderTransformedComponents(nodesFromLastClear: any): void {
            // TODO need to fully build this out to handle all of the components before exposing
            transformPipelineData(nodesFromLastClear).then(function(response: any) {
                let comp: string;
                let frames: Map<string, string> = new Map();
                let componentIdx: number;
                let key: string;
                // remove all existing components and add the updated ones from pixel
                for (comp in scope.pipeline.data) {
                    if (scope.pipeline.data.hasOwnProperty(comp)) {
                        removeComponent(comp, true);
                    }
                }

                for (componentIdx = 0; componentIdx < response.length; componentIdx++) {
                    frames = new Map();
                    key = addComponent(response[componentIdx].widget, {}, {}, true);

                    if (response[componentIdx].name === 'Import') {
                        frames.set(response[componentIdx].frame, '');
                        renderProcessedComponent(frames.keys(), key);

                        scope.pipeline.data[key].name = response[componentIdx].appName;
                        scope.pipeline.data[key].icon = semossCoreService.app.generateAppImageURL(response[componentIdx].appId) + '?time=' + new Date().getTime();
                        scope.pipeline.data[key].disableEdit = false;
                        scope.pipeline.data[key].parameters['IMPORT_FRAME'].value = {
                            name: response[componentIdx].frame,
                            type: response[componentIdx].frameType,
                            override: true
                        };
                        scope.pipeline.data[key].parameters['QUERY_STRUCT'].value = {
                            qsType: 'ENGINE',
                            engineName: response[componentIdx].appId,
                            isDistinct: true,
                            limit: -1,
                            offset: -1,
                            relations: false,
                            orders: false,
                            selectors: response[componentIdx].selectors,
                            // CHECK FILTERS STRUCTURE
                            explicitFilters: response[componentIdx].filters,
                            havingFilters: []
                        };
                        scope.pipeline.data[key].parameters['SELECTED_APP'].value = {
                            display: response[componentIdx].appName,
                            value: response[componentIdx].appId,
                            image: semossCoreService.app.generateAppImageURL(response[componentIdx].appId)
                        };
                        scope.pipeline.data[key].parameters['SELECTED'].value = {};
                        // scope.pipeline.data[key].output = [{
                        //     output: 0,
                        //     downstream: {
                        //         key: '',
                        //         parameter: ''
                        //     },
                        //     frame: true,
                        //     value: {
                        //         name: response[componentIdx].frame,
                        //         type: response[componentIdx].frameType,
                        //         headers: [],
                        //         joins: []
                        //     }
                        // }];
                        scope.pipeline.data[key].state = scope.pipeline.STATE.EXECUTED;
                        // TODO need to do a downstream check in merge
                        // scope.pipeline.data[key].hasDownstream = true;
                    } else if (response[componentIdx].name === 'Merge') {
                        let compKey: string;
                        let joinIdx: number;
                        scope.pipeline.data[key].disableEdit = false;
                        scope.pipeline.data[key].parameters['DESTINATION'].value = {};
                        scope.pipeline.data[key].parameters['QUERY_STRUCT'].value = {};
                        scope.pipeline.data[key].parameters['JOINS'].value = [];

                        for (joinIdx = 0; joinIdx < response[componentIdx].joins.length; joinIdx++) {
                            scope.pipeline.data[key].parameters['JOINS'].value.push({
                                from: response[componentIdx].joins[joinIdx].sourceColumn,
                                join: response[componentIdx].joins[joinIdx].joinType,
                                to: response[componentIdx].joins[joinIdx].targetColumn
                            });
                        }
                        scope.pipeline.data[key].parameters['REMOVE'].value = 'RemoveFrame(\"' + response[componentIdx].sourceFrame + '\");';
                        scope.pipeline.data[key].validinput = true;
                        scope.pipeline.data[key].prevConnection = true;
                        // loop through scope.pipeline.data and check the frame to set its downstream
                        for (compKey in scope.pipeline.data) {
                            if (scope.pipeline.data.hasOwnProperty(compKey) &&
                                scope.pipeline.data[compKey].parameters.hasOwnProperty('IMPORT_FRAME')) {
                                    if (scope.pipeline.data[compKey].parameters['IMPORT_FRAME'].value.name === response[componentIdx].sourceFrame) {
                                        scope.pipeline.data[compKey].output[0].downstream.key = key;
                                        scope.pipeline.data[compKey].output[0].downstream.parameter = 'SOURCE';
                                        scope.pipeline.data[key].parameters['SOURCE'].upstream = {
                                            key: compKey,
                                            output: 0
                                        };
                                        scope.pipeline.data[key].parameters['SOURCE'].value = {
                                            name: scope.pipeline.data[compKey].parameters['IMPORT_FRAME'].value.name,
                                            type: scope.pipeline.data[compKey].parameters['IMPORT_FRAME'].value.type,
                                            headers: [],
                                            joins: [] // NEED TO POPULATE THIS with table joins
                                        };
                                    } else if (scope.pipeline.data[compKey].parameters['IMPORT_FRAME'].value.name === response[componentIdx].targetFrame) {
                                        scope.pipeline.data[compKey].output[0].downstream.key = key;
                                        scope.pipeline.data[compKey].output[0].downstream.parameter = 'DESTINATION';
                                        scope.pipeline.data[key].parameters['DESTINATION'].value = {
                                            name: scope.pipeline.data[compKey].parameters['IMPORT_FRAME'].value.name,
                                            type: scope.pipeline.data[compKey].parameters['IMPORT_FRAME'].value.type,
                                            headers: [],
                                            joins: [] // NEED TO POPULATE THIS with table joins
                                        };
                                        scope.pipeline.data[key].output = [{
                                            output: 0,
                                            downstream: {}, // TODO this should be populated with whatever that goes after that's the 'downstream' e.g. clean routines or other joins..
                                            frame: true,
                                            headers: [],
                                            joins: [],
                                            value: scope.pipeline.data[compKey].output[0].value
                                        }];
                                        renderComponentOutput(key);
                                    }
                            }
                        }
                    }

                    // renderComponentOutput(key);
                }
                finishPipelineRender();
                positionPipeline();
            });
        }

        /**
         * @name renderPipeline
         * @desc construct a pipeline from the json
         * @param components - components to construct form the pipeline
         */
        function renderPipeline(components: any): void {
            let recipe: string;

            recipe =
                scope.widgetCtrl.getShared('steps')
                    .filter(s => {
                        for (let i = 0; i < s.type.length; i++) {
                            if (s.type[i] === 'ERROR' || s.type[i] === 'INVALID_SYNTAX') {
                                return false;
                            }
                        }

                        return true;
                    })
                    .map(s => s.expression)
                    .join('');

            // after any json components have rendered check for user interaction outside of pipeline
            monolithService
                .getPipeline(scope.widgetCtrl.insightID, recipe)
                .then(pixelASTNodes => {
                    let nodesFromLastClear: PixelASTLeaf[][],
                        lastClearInsightIdx = 0,
                        astWithMergeNodes: any = [],
                        newNodes: any;
                    // add components
                    if (components) {
                        for (let key in components) {
                            if (components.hasOwnProperty(key)) {
                                if (components[key].componentToPixelMap) {
                                    scope.pipeline.componentToPixelMap[key] = components[key].componentToPixelMap;
                                }
                                components[key].pipeline = {
                                    parameters: components[key].parameters,
                                    preview: components[key].preview,
                                    input: components[key].input,
                                    output: components[key].output,
                                    state: components[key].state
                                };

                                addComponent(components[key].id, components[key].position, components[key], true);

                                renderComponentOutput(key);
                            }
                        }
                    }

                    // get last time we cleared insight to shrink recipe to parse
                    for (let i = 0; i < pixelASTNodes.length; i++) {
                        if (pixelASTNodes[i] && pixelASTNodes[i][0] && pixelASTNodes[i][0].opName === 'ClearInsight') {
                            lastClearInsightIdx = i;
                        }
                    }

                    nodesFromLastClear = pixelASTNodes.slice(lastClearInsightIdx);
                    
                    for (let i = 0; i < nodesFromLastClear.length; i++) {
                        const node = nodesFromLastClear[i], lastLeaf = node[node.length - 1];
                        let pushMerge = false, pushFuzzy = false;
                        if (node.length === 0) {
                            console.warn('You have an empty node. Check your Pixel.');

                            continue;
                        }

                        if (lastLeaf.opName === 'Merge') {
                            pushMerge = true;
                            lastLeaf.opName = 'Import';
                        } else if (lastLeaf.opName === 'FuzzyMerge') {
                            pushFuzzy = true;
                            lastLeaf.opName = 'Import';
                        }
                        astWithMergeNodes.push(node);
                        if (pushMerge) {
                            astWithMergeNodes.push(
                                [{
                                    widgetId: 'pipeline-merge',
                                    opName: DUMMY,
                                    opString: '',
                                    nounInputs: {},
                                    rowInputs: []
                                }]
                            );
                        }

                        if (pushFuzzy) {
                            astWithMergeNodes.push(
                                [{
                                    widgetId: 'pipeline-fuzzy-blend',
                                    opName: DUMMY,
                                    opString: '',
                                    nounInputs: {},
                                    rowInputs: []
                                }]
                            );
                        }
                    }

                    newNodes = findNewPipelineNodes(astWithMergeNodes, components);
                    // kicks off any external component generation
                    // part two if the render is complete after the Pixel AST
                    // has been fully consumed

                    newNodes.forEach(configureExternalComponent);
                    finishPipelineRender();
                });
        }

        /**
         * @name finishPipelineRender
         * @desc once previous nodes and nodes from outside of pipeline have been rendered, run the rest of pipeline
         *      render code
         */
        function finishPipelineRender(): void {
            Object.keys(scope.pipeline.data).forEach(renderComponentConnection);

            // nothing, open the landing
            if (Object.keys(scope.pipeline.data).length === 0) {
                scope.pipeline.landing.open = true;
            } else {
                setDownstream();
                positionComponents();
            }
        }

        /**
         * @name transformPipelineData
         * @param {array} pipelineComponents the pipeline components passed from the BE
         * @desc loop through the BE output and transform it to be more readable and to look for the components we want to paint
         * @returns {array} the array of transformed components
         */
        function transformPipelineData(pipelineComponents: any): any {
            let transformedComponents: any = [];
            let componentInfo: any = {};
            let reactorInputs: any;
            let componentIdx: number;
            let lastReactor: any = {};
            let filterReactors: any = [];
            let reactorIdx: number;
            let selectors: any = [];
            let joinIdx: number;
            let selectIdx: number;
            let filterIdx: number;
            let importMap: any = {};
            let importKey: string;
            let appInfoComponents: any = [];
            let tComponentIdx: number;
            let deferred = $q.defer();

            /**
             * @name _getLastReactor
             * @param {array} reactors the reactors to look through
             * @desc looks for the last reactor and return
             * @returns {object} the last reactor info
             */
            function _getLastReactor(reactors: any): any {
                const lastIdx = reactors.length - 1;

                return reactors[lastIdx];
            }

            /**
             * @name _getLambdaValue
             * @desc traverse through the nested structure and get the value
             * @returns {string} the value
             */
            function _getLambdaValue(component: any, baseCase: string): string {
                // rowInputs = value that is passed into reactor without a key e.g. CreateFrame(FRAME12345);
                // nounInputs = value that is passed into reactor with a key e.g. CreateFrame(frame=[FRAME12345])
                let componentIdx: number;

                for (componentIdx = 0; componentIdx < component.length; componentIdx++) {
                    if (component[componentIdx].type === baseCase) {
                        return component[componentIdx].value;
                    } else if (component[componentIdx].type === 'LAMBDA') {
                        return _getLambdaValue(component[componentIdx].value.rowInputs, baseCase);
                    }
                }

                return '';
            }

            for (componentIdx = 0; componentIdx < pipelineComponents.length; componentIdx++) {
                lastReactor = _getLastReactor(pipelineComponents[componentIdx]);
                componentInfo = {};
                reactorInputs = {};
                selectors = [];
                filterReactors = [];

                if (lastReactor.opName === 'Import') {
                    componentInfo = {
                        name: 'Import',
                        frame: '',
                        frameType: '',
                        appId: '',
                        appName: '',
                        selectors: [],
                        filters: [],
                        widget: 'pipeline-app'
                    };

                    reactorInputs = lastReactor.nounInputs;

                    // TODO see if there's a generic way to pull these information out instead of traversing deep into the nested data
                    componentInfo.frame = _getLambdaValue(reactorInputs.frame, 'CONST_STRING');
                    componentInfo.frameType = semossCoreService.utility.getter(reactorInputs.frame[0], 'value.nounInputs.frameType.0.value') || _getLambdaValue(reactorInputs.frame, 'COLUMN') ;
                    componentInfo.appId = reactorInputs.qs[0].value.engineName;
                    importMap[componentInfo.appId] = '';

                    for (selectIdx = 0; selectIdx < reactorInputs.qs[0].value.selectors.length; selectIdx++) {
                        selectors.push(reactorInputs.qs[0].value.selectors[selectIdx]);
                    } 
                    componentInfo.selectors = selectors;

                    // look to see if any filters exist
                    for (reactorIdx = 0; reactorIdx < pipelineComponents[componentIdx].length; reactorIdx++) {
                        if (pipelineComponents[componentIdx][reactorIdx].opName === 'Filter') {
                            filterReactors.push(pipelineComponents[componentIdx][reactorIdx]);
                        }
                    }

                    if (filterReactors.length > 0) {
                        // we have filters so lets process
                        let column: any = {};
                        let comparator: string = '';
                        let values: any = [];
                        let rowInputIdx: number;
                        for (filterIdx = 0; filterIdx < filterReactors.length; filterIdx++) {
                            column = '';
                            comparator = '';
                            values = [];
                            for (rowInputIdx = 0; rowInputIdx < filterReactors[filterIdx].rowInputs[0].value.rowInputs.length; rowInputIdx++) {
                                if (filterReactors[filterIdx].rowInputs[0].value.rowInputs[rowInputIdx].type === 'COLUMN') {
                                    column = filterReactors[filterIdx].rowInputs[0].value.rowInputs[rowInputIdx];
                                    column.value = [filterReactors[filterIdx].rowInputs[0].value.rowInputs[rowInputIdx].value];
                                    column.pixelType = column.type;
                                } else if (filterReactors[filterIdx].rowInputs[0].value.rowInputs[rowInputIdx].type === 'COMPARATOR') {
                                    comparator = filterReactors[filterIdx].rowInputs[0].value.rowInputs[rowInputIdx].value;
                                } else if (filterReactors[filterIdx].rowInputs[0].value.rowInputs[rowInputIdx].type === 'CONST_STRING') {
                                    values.push(filterReactors[filterIdx].rowInputs[0].value.rowInputs[rowInputIdx].value);
                                }
                            }

                            if (column && comparator && values.length > 0) {
                                // assumming a simple join
                                componentInfo.filters.push('SIMPLE');
                                debugger;
                                componentInfo.filters.push({
                                    left: column,
                                    comparator: comparator,
                                    right: {
                                        pixelType: 'CONSTANT',
                                        value: values
                                    }
                                });
                                // componentInfo.filters.push({
                                //     column: column,
                                //     comparator: comparator,
                                //     values: values
                                // });
                            }
                        }
                    }

                    transformedComponents.push(componentInfo);
                } else if (lastReactor.opName === 'Merge' && pipelineComponents[componentIdx][0].opName === 'Frame') {
                    // We are only processing the merge that happens between two frames.
                    // other types of merges will not be displayed in pipeline
                    componentInfo = {
                        name: 'Merge',
                        widget: 'pipeline-merge',
                        joins: [],
                        sourceFrame: '',
                        targetFrame: ''
                    };

                    reactorInputs = lastReactor.nounInputs;
                    for (joinIdx = 0; joinIdx < reactorInputs.joins.length; joinIdx++) {
                        if (reactorInputs.joins[joinIdx].type === 'JOIN') {
                            componentInfo.joins.push({
                                sourceColumn: reactorInputs.joins[joinIdx].value.lColumn,
                                targetColumn: reactorInputs.joins[joinIdx].value.rColumn,
                                joinType: reactorInputs.joins[joinIdx].value.joinType
                            });
                        }
                    }

                    componentInfo.targetFrame = _getLambdaValue(reactorInputs.frame, 'CONST_STRING');
                    componentInfo.sourceFrame = semossCoreService.utility.getter(pipelineComponents[componentIdx][0], 'rowInputs.0.value') || _getLambdaValue(pipelineComponents[componentIdx][0].nounInputs.frame, 'CONST_STRING');
                    transformedComponents.push(componentInfo);
                }
            }

            // we need to get the app name...so going to make the AppInfo pixel to get the app info
            // then we will return the completed transformedComonents array.
            if (Object.keys(importMap).length > 0) {
                for (importKey in importMap) {
                    if (importMap.hasOwnProperty(importKey)) {
                        appInfoComponents.push({
                            type: 'appInfo',
                            components: [importKey],
                            terminal: true
                        });
                    }
                }

                scope.widgetCtrl.meta(appInfoComponents, function (response: any) {
                    const pixelReturn: any = response.pixelReturn;
                    let output: any;
                    let returnIdx: number;

                    for (returnIdx = 0; returnIdx < pixelReturn.length; returnIdx++) {
                        output = pixelReturn[returnIdx].output;
                        importMap[output.app_id] = output.app_name;
                    }

                    for (tComponentIdx = 0; tComponentIdx < transformedComponents.length; tComponentIdx++) {
                        if (transformedComponents[tComponentIdx].name === 'Import') {
                            transformedComponents[tComponentIdx].appName = importMap[transformedComponents[tComponentIdx].appId];
                        }
                    }

                    deferred.resolve(transformedComponents);
                });
            } else {
                deferred.resolve(transformedComponents);
            }

            return deferred.promise;
        }

        /**
         * @name setDownstream
         * @desc once pipeline is set, confirm downstream and upstream for UI
         */
        function setDownstream(): void {
            Object.keys(scope.pipeline.data).forEach((key: string) => {
                const component = scope.pipeline.data[key];
                if (
                    component.output &&
                    component.output[0].downstream &&
                    Object.keys(component.output[0].downstream).length > 0
                ) {
                    component.hasDownstream = true;
                }

                for (let i = 0; i < component.input.length; i++) {
                    let param = component.input[i];

                    if (
                        !(
                            component.parameters[param] &&
                            component.parameters[param].value &&
                            component.parameters[param].required
                        )
                    ) {
                        component.validInput = false;
                        return;
                    }
                }

                component.validInput = true;
            });
        }

        /**
         * @name positionPipeline
         * @desc reposition the pipeline form the UI
         */
        function positionPipeline(): void {
            for (let key in scope.pipeline.data) {
                if (scope.pipeline.data.hasOwnProperty(key)) {
                    if (!scope.pipeline.data[key].position) {
                        scope.pipeline.data[key].position = {};
                    }

                    scope.pipeline.data[key].position.auto = true;
                }
            }

            positionComponents();
        }

        /**
         * @name positionComponents
         * @desc position components (and set the arrow) if is auto positioned
         */
        function positionComponents(): void {
            let keys = Object.keys(scope.pipeline.data),
                startTop = COMPONENT_STYLES.gutter;


            // TODO: Fix positioning to take children into account.
            const keyLen = keys.length;

            // get the starting position map
            for (let keyIdx = 0; keyIdx < keyLen; keyIdx++) {
                const key = keys[keyIdx];
                // we want the position of the ones that we are not touching (they will stay the same)
                if (scope.pipeline.data[key].position.auto) {
                    continue;
                }

                if (scope.pipeline.data[key].position.left === COMPONENT_STYLES.gutter) {
                    let currentTop = scope.pipeline.data[key].position.top + COMPONENT_STYLES.height + COMPONENT_STYLES.gutter;
                    if (startTop < currentTop) {
                        startTop = currentTop;
                    }


                }
            }

            for (let keyIdx = 0; keyIdx < keyLen; keyIdx++) {
                const key = keys[keyIdx];

                // if it was already positioned don't do it again
                if (!scope.pipeline.data[key].position.auto) {
                    continue;
                }

                const upstreamKeys: [string, string] = ['', ''];

                for (let parameter in scope.pipeline.data[key].parameters) {
                    if (scope.pipeline.data[key].parameters.hasOwnProperty(parameter)) {
                        const param = scope.pipeline.data[key].parameters[parameter];

                        if (param.upstream && param.upstream.key) {
                            if (!upstreamKeys[0]) {
                                upstreamKeys[0] = param.upstream.key;
                            } else {
                                upstreamKeys[1] = param.upstream.key;
                            }
                        }
                    }
                }

                let parentA = scope.pipeline.data[upstreamKeys[0]],
                    parentB = scope.pipeline.data[upstreamKeys[1]];

                // there are two parents so put middle of component in middle of the parents
                let alignmentTop: number = 0,
                    alignmentLeft: number = 0;

                if (parentA && parentB) {
                    let topA: number = parentA.position.top,
                        topB: number = parentB.position.top,
                        leftA: number = parentA.position.left,
                        leftB: number = parentB.position.left;

                    if (topA > topB) {
                        alignmentTop = topA - topB - (COMPONENT_STYLES.height);
                    } else {
                        alignmentTop = topB - topA - (COMPONENT_STYLES.height);
                    }

                    if (leftA > leftB) {
                        alignmentLeft = leftA + (COMPONENT_STYLES.width * 1.25);
                    } else {
                        alignmentLeft = leftB + (COMPONENT_STYLES.width * 1.25);
                    }
                } else if (parentA) {
                    alignmentTop = parentA.position.top;
                    alignmentLeft = parentA.position.left + (COMPONENT_STYLES.width * 1.25);
                } else {
                    alignmentTop = startTop;
                    alignmentLeft = COMPONENT_STYLES.gutter
                }

                // set auto to false, so we don't reset it
                scope.pipeline.data[key].position.auto = false;

                scope.pipeline.data[key].position.top = alignmentTop;
                scope.pipeline.data[key].position.left = alignmentLeft;

                // actually update the view
                scope.pipeline.eles[key].style.top = alignmentTop + 'px';
                scope.pipeline.eles[key].style.left = alignmentLeft + 'px';

                // increment to a new one
                if (alignmentLeft === COMPONENT_STYLES.gutter) {
                    let currentTop = alignmentTop + COMPONENT_STYLES.height + COMPONENT_STYLES.gutter;
                    if (startTop < currentTop) {
                        startTop = currentTop;
                    }
                }
            }

            // position the arrow
            if (keyLen > 0) {
                const key = keys[keyLen - 1];

                if (key) {
                    const elId = `pipeline__component__side__output--${key}__0`;

                    let outputEle = ele[0].querySelector(`#${elId}`);
                    if (outputEle) {
                        plumbing.manage(elId, outputEle);

                        plumbing.updateOffset({
                            elId: elId,
                            offset: {
                                top: scope.pipeline.data[key].position.top + 30,
                                left: scope.pipeline.data[key].position.left + 134,
                            }
                        });
                    }
                }
            }


            // repaint the connections
            plumbing.repaintEverything();

            savePipeline();
        }

        /**
         * @name savePipeline
         * @desc when a pipeline component executes, update the pipeline config
         */
        function savePipeline(): void {
            const components = JSON.parse(JSON.stringify(scope.pipeline.data));

            Object.keys(components).forEach(key => {
                // do not save if it is initial
                if (components[key].state === scope.pipeline.STATE.INITIAL) {
                    // look through all of the parameters, find the upstream and remove the reference
                    Object.keys(components[key].parameters).forEach(parameter => {
                        if (
                            components[key].parameters[parameter].upstream &&
                            components[key].parameters[parameter].upstream.key
                        ) {
                            if (components.hasOwnProperty(components[key].parameters[parameter].upstream.key)) {
                                // this is the upstream parent, we are removing the linking
                                components[components[key].parameters[parameter].upstream.key].output.forEach(output => {
                                    if (output.downstream && output.downstream.key === key) {
                                        output.downstream = {};
                                    }
                                });
                            }
                        }
                    });

                    delete components[key];
                } else if (scope.pipeline.componentToPixelMap[key]) {
                    // need to cache this data for when we come back
                    components[key].componentToPixelMap = scope.pipeline.componentToPixelMap[key];
                }
            });

            semossCoreService.emit('set-pipeline', {
                insightID: scope.widgetCtrl.insightID,
                pipeline: {
                    components: components
                }
            });
        }

        /** Pipeline Actions */
       /**
         * @name createFrameName
         * @desc will create a frame name using the app that was selected
         * @param {string} base - the base name for the frame
         * @param {array} frames - optional parameter, list of existing frames
         * @returns {string} - the name created for the frame
         */
        function createFrameName(base: string): string {
            let name: string,
                rootName: string,
                counter: number = 0,
                frameObj: any = scope.widgetCtrl.getShared('frames'),
                frames: any = [],
                frameName: string;

            if (!base) {
                // if nothing passed, return empty string
                return '';
            }

            name = base.replace(/[^a-zA-Z0-9._ ]/g, '').replace(/[ .]/g, '_') + '_FRAME' + semossCoreService.utility.random();
            // double underscores not allowed in frame name due to FRAME__TABLE structure
            name = name.replace(/__/g, '_');
            if (!isNaN(Number(name[0]))) {
                // first letter is a number, we need to clean so BE SQL doesn't error out
                // add an underscore to the front.
                name = '_' + name;
            }
            rootName = name;
            // Need to create an array of all the frame names uppercased because it must be case insensitive
            for (frameName in frameObj) {
                if (frameObj.hasOwnProperty(frameName)) {
                    frames.push(frameName.toUpperCase());
                }
            }

            while (frames.indexOf(name.toUpperCase()) > -1) {
                counter++;
                name = rootName + '_' + counter;
            }
            return name;
        }

        /**
         * @name searchMenu
         * @desc search the menu for the pipeline
         */
        function searchMenu(): void {
            const searchTerm: string = String(scope.pipeline.menu.searched).toUpperCase(),
                accordion = scope.pipeline.menu.accordion;

            for (let idx = 0, len = accordion.length; idx < len; idx++) {
                accordion[idx].items = [];

                for (let componentIdx = 0, componentLen = scope.pipeline.components.length; componentIdx < componentLen; componentIdx++) {
                    // it has to have a menu
                    // Nothing should be searched or it has to be part of the name or description
                    if (
                        scope.pipeline.components[componentIdx].pipeline.group === accordion[idx].name &&
                        scope.pipeline.components[componentIdx].id !== 'pipeline-existing-insight' &&
                        (!searchTerm || scope.pipeline.components[componentIdx].name.toUpperCase().indexOf(searchTerm) > -1 || scope.pipeline.components[componentIdx].description.toUpperCase().indexOf(searchTerm) > -1)
                    ) {
                        accordion[idx].items.push(scope.pipeline.components[componentIdx]);
                        // open up accordion if it has item
                        if (accordion[idx].height === 0) {
                            accordion[idx].height = 20;
                            for (let i = 0; i < len; i++) {
                                if (i === idx) continue;
                                if (accordion[i].height >= 80) {
                                    accordion[i].height -= 20;
                                    break;
                                } else if (accordion[i].height > 10) {
                                    accordion[i].height -= 10;
                                }
                            }
                        }
                    }
                }
            }
        }

        /** Component Functions */
        /**
         * @name addComponent
         * @desc add a new widget
         * @param id - id of the widget to add
         * @param pos - position information
         * @param updated - updated config
         * @param suppressEdit - true to not open component, false otherwise
         * @returns key - added key
         */
        function addComponent(id: string, pos: any, updated: any, suppressEdit: boolean): string {
            let config,
                key,
                options,
                parameter,
                parameters,
                json,
                required;


            config = semossCoreService.getSpecificConfig(id);
            if (!config) {
                config = updated;
            } else {
                config = angular.merge(config, updated);
            }

            if (config.key) {
                key = config.key;
            } else {
                key = 'pipeline--' + Date.now();
            }

            parameters = config.pipeline.parameters;

            // convert to an options string
            if (config.hasOwnProperty('content')) {
                if (config.content.hasOwnProperty('template')) {
                    options = config.content.template.options;
                } else if (config.content.hasOwnProperty('json')) {
                    options = {
                        json: config.content.json
                    };

                    // not defined, set the default
                    if (typeof parameters === 'undefined') {
                        parameters = {
                            'SMSS_FRAME': {
                                'frame': true
                            }
                        };

                        config.pipeline.input = ['SMSS_FRAME'];
                    }
                }
            }

            scope.pipeline.data[key] = {
                id: config.id,
                icon: config.icon,
                options: options,
                position: {
                    auto: true, // auto position
                    top: 0,
                    left: 0
                },
                dependsOn: config.dependsOn,
                createFrame: config.createFrame,
                hidePreview: config.pipeline.hidePreview,
                name: config.name,
                disableEdit: true,
                description: config.description,
                key: key,
                pixel: config.pipeline.pixel ? config.pipeline.pixel : '',
                parameters: {},
                required: config.required,
                preview: config.pipeline.preview ? config.pipeline.preview : '',
                input: config.pipeline.input ? config.pipeline.input : [],
                output: config.pipeline.output ? config.pipeline.output : [],
                state: config.pipeline.state ? config.pipeline.state : scope.pipeline.STATE.INITIAL // three states initial, executed, final
            };

            if (scope.pipeline.data[key].input.length === 0) {
                scope.pipeline.data[key].validInput = true;
            }

            if (typeof pos !== 'undefined' && typeof pos.top !== 'undefined') {
                scope.pipeline.data[key].position.auto = false; // turn auto position off because it was set

                scope.pipeline.data[key].position.top = pos.top;
            }

            if (scope.pipeline.data[key].position.top < COMPONENT_STYLES.gutter) {
                scope.pipeline.data[key].position.top = COMPONENT_STYLES.gutter;
            }

            if (typeof pos !== 'undefined' && typeof pos.left !== 'undefined') {
                scope.pipeline.data[key].position.auto = false; // turn auto position off because it was set

                scope.pipeline.data[key].position.left = pos.left;
            }

            if (scope.pipeline.data[key].position.left < COMPONENT_STYLES.gutter) {
                scope.pipeline.data[key].position.left = COMPONENT_STYLES.gutter;
            }

            // construct the data backing the value, it is bound by the key
            // important note: We link the the upstream and downstream and use it to validate the options as we add more
            // set up the defaults
            for (let parameter in parameters) {
                if (parameters.hasOwnProperty(parameter)) {
                    scope.pipeline.data[key].parameters[parameter] = {
                        parameter: parameter,
                        upstream: parameters[parameter].upstream ? parameters[parameter].upstream : {},
                        frame: parameters[parameter].frame ? parameters[parameter].frame : true,
                        type: parameters[parameter].type ? parameters[parameter].type : '',
                        value: parameters[parameter].value ? parameters[parameter].value : undefined,
                        required: undefined // set later
                    };
                }
            }

            // process the parameters and json to match
            if (config.hasOwnProperty('content') && config.content.hasOwnProperty('json')) {
                parameters = scope.pipeline.data[key].parameters;
                json = scope.pipeline.data[key].options.json;

                scope.pipeline.data[key].pixel = '';

                // loop through the JSON and update it
                for (let queryIdx = 0, queryLen = json.length; queryIdx < queryLen; queryIdx++) {
                    // remove the extra
                    // separate checks for <SMSS_REFRESH_INSIGHT> and <SMSS_REFRESH> to be added as components
                    if (json[queryIdx].query.indexOf('<SMSS_REFRESH_INSIGHT>') > -1) {
                        // remove this internal param
                        json[queryIdx].query = json[queryIdx].query.replace('<SMSS_REFRESH_INSIGHT>', '');
                    } else if (json[queryIdx].query.indexOf('<SMSS_REFRESH>') > -1) {
                        // remove this internal param
                        json[queryIdx].query = json[queryIdx].query.replace('<SMSS_REFRESH>', '');
                    } else if (json[queryIdx].query.indexOf('<SMSS_AUTO>') > -1) {
                        // remove this internal param
                        json[queryIdx].query = json[queryIdx].query.replace('<SMSS_AUTO>', '');
                    }

                    // make the pixel
                    scope.pipeline.data[key].pixel += json[queryIdx].query;

                    // remove the execute as I take control
                    if (json[queryIdx].hasOwnProperty('execute')) {
                        delete json[queryIdx].execute;
                    }

                    // all of the params in the pipeline, must be in the json...
                    pipelineLoop: for (let parameter in parameters) {
                        if (
                            parameters.hasOwnProperty(parameter)
                        ) {
                            for (let paramIdx = 0, paramLen = json[queryIdx].params.length; paramIdx < paramLen; paramIdx++) {
                                // it is already there, continue
                                if (json[queryIdx].params[paramIdx].paramName === parameters[parameter].parameter) {
                                    continue pipelineLoop;
                                }
                            }

                            json[queryIdx].params.push({
                                paramName: parameters[parameter].parameter
                            });
                        }
                    }

                    // all of the params in the json, must be in the pipeline...
                    for (let paramIdx = 0, paramLen = json[queryIdx].params.length; paramIdx < paramLen; paramIdx++) {
                        if (
                            !parameters.hasOwnProperty(json[queryIdx].params[paramIdx].paramName)
                        ) {
                            parameters[json[queryIdx].params[paramIdx].paramName] = {
                                key: key,
                                parameter: parameter,
                                upstream: {},
                                type: 'PIXEL',
                                value: json[queryIdx].params[paramIdx] && json[queryIdx].params[paramIdx].hasOwnProperty('model') && json[queryIdx].params[paramIdx].model.hasOwnProperty('defaultValue') ? json[queryIdx].params[paramIdx].model.defaultValue : undefined // this doesn't really matter since we will update
                            };
                        }
                    }
                }
            }

            //  mark what is required after processing all
            required = semossCoreService.pixel.parameterize(semossCoreService.pixel.tokenize(scope.pipeline.data[key].pixel));

            // preview is not valid clean it reset it
            if (!scope.pipeline.data[key].parameters.hasOwnProperty(scope.pipeline.data[key].preview)) {
                scope.pipeline.data[key].preview = '';
            }

            for (let parameter in parameters) {
                if (parameters.hasOwnProperty(parameter)) {
                    scope.pipeline.data[key].parameters[parameter].required = required.indexOf(parameter) > -1;

                    // figure out what is preview
                    if (
                        !scope.pipeline.data[key].preview &&
                        (
                            scope.pipeline.data[key].parameters[parameter].frame
                        )
                    ) {
                        scope.pipeline.data[key].preview = parameter;
                    }
                }
            }

            scope.pipeline.lastAddedKey = key;

            // render in the UI
            renderComponent(key);

            // position if we need to
            if (scope.pipeline.data[key].position.auto) {
                positionComponents();
            }

            // show it to edit the data
            // supressed when we build from saved components
            if (!suppressEdit && scope.pipeline.data[key].input.length === 0) {
                showComponent(key);
            }
            // save that something has changed
            savePipeline();

            scope.pipeline.numComponents++;

            return key;
        }

        /**
         * @name renderComponent
         * @desc renders the initial component
         * @param key - key of the component
         */
        function renderComponent(key: string): void {
            let componentHTML: string;
            // create the HTML and append to the DOM. We do this manually so we can control the applied events
            componentHTML = `
                        <div class="pipeline__component smss-clear"
                            id='${key}'
                            title="{{pipeline.data['${key}'].description}}">
                            <div class="pipeline__component__side smss-left">
                                <div id="pipeline__component__side__input" class="pipeline__component__side__holder">
                                </div>
                            </div>
                            <div class="pipeline__component__content smss-left smss-clear">
                                <div id="pipeline__component__box" class="pipeline__component__box smss-left"
                                     ng-class="{'pipeline__component__box--replay': pipeline.data['${key}'].replay}">
                                    <div id="pipeline__component__box__action" class="pipeline__component__box__action smss-clear">
                                        <smss-btn title="Move Component"
                                            class="smss-btn--flat smss-btn--icon pipeline__component__box__action__icon pipeline__component__box__action__icon--drag smss-left">
                                            <i class="fa fa-arrows-alt"></i>
                                        </smss-btn>
                                        <smss-btn title="Remove Component"
                                            ng-click="$event.stopPropagation();pipeline.routeRemoveComponent('${key}')"
                                            ng-disabled="pipeline.replaying"
                                            class="smss-btn--flat smss-btn--icon pipeline__component__box__action__icon smss-right">
                                            <i class="fa fa-times smss-color--error"></i>
                                        </smss-btn>
                                    </div>
                                    <div class="smss-center pipeline__component__box__img">
                                        <img ng-src="{{pipeline.data['${key}'].icon}}">
                                    </div>
                                    <div class="smss-center smss-text pipeline__component__box__text">
                                        {{pipeline.data['${key}'].name}}
                                    </div>
                                </div>
                                <div class="pipeline__component__action smss-action smss-left smss-clear">
                                    <smss-btn class="smss-btn--icon smss-btn--white"
                                        ng-if="!pipeline.paused && !pipeline.data['${key}'].hasDownstream && pipeline.data['${key}'].validInput"
                                        title="View the underlying data"
                                        ng-click="pipeline.viewComponent('${key}')">
                                        <i class="fa fa-table"></i>
                                    </smss-btn>
                                    <smss-btn class="smss-btn--icon smss-btn--white"
                                        ng-disabled="pipeline.replaying || pipeline.data['${key}'].disableEdit"
                                        title="Update this data"
                                        ng-click="pipeline.showComponent('${key}')">
                                        <i class="fa fa-edit"></i>
                                    </smss-btn>
                                    <smss-btn class="smss-btn--icon smss-btn--white"
                                        ng-if="!pipeline.paused && !pipeline.data['${key}'].hasDownstream && pipeline.data['${key}'].validInput"
                                        title="Federate this data"
                                        ng-click="pipeline.showFederate('${key}')">
                                        <i class="fa fa-link"></i>
                                    </smss-btn>
                                    <smss-btn class="smss-btn--icon smss-btn--white"
                                        ng-if="!pipeline.paused && !pipeline.data['${key}'].hasDownstream && pipeline.data['${key}'].validInput"
                                        title="Visualize this data"
                                        ng-click="pipeline.visualizeComponent('${key}')">
                                        <i class="fa fa-bar-chart"></i>
                                    </smss-btn>
                                </div>
                            </div>
                            <div class="pipeline__component__side smss-left">
                                <div id="pipeline__component__side__output" class="pipeline__component__side__holder">
                                </div>
                            </div>
                        </div>`;

            // create, mount, and compile the element
            scope.pipeline.eles[key] = angular.element(componentHTML)[0];
            pipelineWorkspaceContentEle.appendChild(scope.pipeline.eles[key]);
            $compile(scope.pipeline.eles[key])(scope);

            // position
            scope.pipeline.eles[key].style.top = scope.pipeline.data[key].position.top + 'px';
            scope.pipeline.eles[key].style.left = scope.pipeline.data[key].position.left + 'px';

            // add the input fields
            renderComponentInput(key);

            scope.pipeline.movable[key] = Movable({
                handle: scope.pipeline.eles[key].querySelector('#pipeline__component__box'),
                content: scope.pipeline.eles[key],
                on: () => {
                    // TODO: do we want to repaint everything, or just what was moved?
                    plumbing.repaintEverything();
                },
                stop: (top: string, left: string) => {
                    const component = scope.pipeline.data[key];

                    component.position.top = top;
                    component.position.left = left;

                    if (component.output[0] && Object.keys(component.output[0].downstream).length === 0) {
                        plumbing.updateOffset({
                            elId: `pipeline__component__side__output--${key}__0`,
                            offset: {
                                left: left + 134,
                                top: top + 30
                            }
                        });
                    }

                    savePipeline();
                }
            });
        }

        /**
         * @name renderComponentInput
         * @desc render the input of the component
         * @param key - key of the component
         */
        function renderComponentInput(key: string): void {
            let sideEle: HTMLElement = scope.pipeline.eles[key].querySelector('#pipeline__component__side__input');

            for (let inputIdx = 0, inputLen = scope.pipeline.data[key].input.length; inputIdx < inputLen; inputIdx++) {
                // html
                let inputHTML: string,
                    inputEle: HTMLElement;

                inputHTML = `
                <smss-btn class="smss-btn--primary smss-btn--block pipeline__component__side__btn"
                    id="pipeline__component__side__input--${key}__${scope.pipeline.data[key].input[inputIdx]}"
                    ng-class="{'pipeline__component__side__btn--disabled': pipeline.data['${key}'].state !== pipeline.STATE.INITIAL}"
                    ng-disabled="pipeline.data['${key}'].state !== pipeline.STATE.INITIAL">
                    <i class="fa fa-arrow-circle-right"></i>
                </smss-btn>`;

                inputEle = $compile(inputHTML)(scope)[0];

                sideEle.appendChild(inputEle);

                plumbing.makeTarget(inputEle, {
                    isTarget: true,
                    anchor: 'LeftMiddle',
                    endpoint: 'Blank',
                    connectionsDetachable: false,
                    maxConnections: 1,
                    beforeDrop: linkComponent
                });
            }

            // position it
            sideEle.style.top = `calc(50% - ${sideEle.clientHeight / 2}px)`;
        }

        /**
         * @name renderComponentOutput
         * @desc render the output of the component
         * @param {string} key - key of the component
         * @returns {void}
         */
        function renderComponentOutput(key: string): void {
            let sideEle: HTMLElement = scope.pipeline.eles[key].querySelector('#pipeline__component__side__output'),
                outputEles: HTMLElement[] = [];

            for (let outputIdx = 0, outputLen = scope.pipeline.data[key].output.length; outputIdx < outputLen; outputIdx++) {
                // html
                let outputHTML: string,
                    outputEle: HTMLElement;

                outputHTML = `<smss-btn class="smss-btn--primary smss-btn--block pipeline__component__side__btn"
                    id="pipeline__component__side__output--${key}__${outputIdx}"
                    ng-disabled="pipeline.data['${key}'].hasDownstream">
                    <i class="fa fa-arrow-circle-right"></i>
                </smss-btn>`;

                // re-executing component will generate multiple side arrows if we dont check children
                if (sideEle.children.length === 0) {
                    outputEle = $compile(outputHTML)(scope)[0];
                    sideEle.appendChild(outputEle);

                    makePlumbingSource(outputEle);
                    outputEles.push(outputEle);
                }

                if (scope.pipeline.data[key].disableEdit) {
                    scope.pipeline.data[key].disableEdit = false;
                }
            }

            // position it
            sideEle.style.top = `calc(50% - ${sideEle.clientHeight / 2}px)`;
            // add it to managed elements
            outputEles.forEach(ele => plumbing.manage(ele.id, ele));
        }
        /**
         * @name makePlumbingSource
         * @param ele - ele to make jsplumb source
         * @desc makes the ele a jsplumb source
         */
        function makePlumbingSource(ele: HTMLElement): void {
            plumbing.makeSource(ele, {
                isSource: true,
                anchor: 'RightMiddle',
                endpoint: 'Blank',
                connectionsDetachable: false,
                maxConnections: 1,
                connector: [
                    'Flowchart',
                    {
                        cssClass: 'pipeline__component__connector'
                    }
                ],
                connectorOverlays: [
                    [
                        'Arrow',
                        {
                            location: 1,
                            length: 8,
                            width: 8,
                            cssClass: 'pipeline__component__overlay'
                        }
                    ]
                ]
            });
        }

        /**
         * @name renderComponentConnection
         * @desc render all of the downstream connections
         * @param key - key of the component
         */
        function renderComponentConnection(key: string): void {
            for (let outputIdx = 0, outputLen = scope.pipeline.data[key].output.length; outputIdx < outputLen; outputIdx++) {
                if (
                    Object.keys(scope.pipeline.data[key].output[outputIdx].downstream).length > 0
                ) {
                    plumbing.connect({
                        source: scope.pipeline.eles[key].querySelector(`#pipeline__component__side__output--${key}__${scope.pipeline.data[key].output[outputIdx].output}`),
                        target: scope.pipeline.eles[scope.pipeline.data[key].output[outputIdx].downstream.key].querySelector(`#pipeline__component__side__input--${scope.pipeline.data[key].output[outputIdx].downstream.key}__${scope.pipeline.data[key].output[outputIdx].downstream.parameter}`),
                        detachable: false,
                        anchors: ['RightMiddle', 'LeftMiddle'],
                        endpoint: 'Blank',
                        connectionsDetachable: false,
                        maxConnections: 1,
                        connector: [
                            'Flowchart',
                            {
                                cssClass: 'pipeline__component__connector'
                            }
                        ],
                        connectorOverlays: [
                            [
                                'Arrow',
                                {
                                    location: 1,
                                    length: 8,
                                    width: 8,
                                    cssClass: 'pipeline__component__overlay'
                                }
                            ]
                        ]
                    });
                }
            }
        }

        /**
         * @name getComponent
         * @param key - key of the component
         * @desc called to get the value of the current parameter
         * @param accessor - accessor what data to grab
         * @returns current value
         */
        function getComponent(key: string, accessor: string): any {
            return semossCoreService.utility.getter(scope.pipeline.data[key], accessor);
        }

        /**
         * @name routeRemoveComponent
         * @param key component key
         * @desc determine what to do when user clicks x on component
         */
        function routeRemoveComponent(key: string): void {
            const state = scope.pipeline.data[key].state;

            switch (state) {
                case scope.pipeline.STATE.INITIAL:
                    removeComponent(key, false);
                    return;
                default:
                    scope.pipeline.componentToRemove = key;
                    scope.pipeline.confirmRemove = true;
            }
        }

        /**
         * @name removeComponent
         * @param key - key of the component
         * @param rerun - true if need to rerun the pipeline
         * @desc close the widget
         */
        function removeComponent(key: string, rerun: boolean): void {
            let componentKey = key;

            if (scope.pipeline.data[componentKey].output.length > 0) {
                scope.pipeline.data[componentKey].output.forEach(out => {
                    if (out.downstream && out.downstream.key) {
                        removeComponent(out.downstream.key, false);
                    }
                });
            }

            scope.pipeline.componentToPixelMap[componentKey] = null;
            // make it not movable (remove the reference)
            if (scope.pipeline.movable[componentKey]) {
                scope.pipeline.movable[componentKey].destroy();
            }

            for (let inputIdx = 0, inputLen = scope.pipeline.data[componentKey].input.length; inputIdx < inputLen; inputIdx++) {
                plumbing.remove(`pipeline__component__side__input--${componentKey}__${scope.pipeline.data[componentKey].input[inputIdx]}`);
            }

            for (let outputIdx = 0, outputLen = scope.pipeline.data[componentKey].output.length; outputIdx < outputLen; outputIdx++) {
                plumbing.remove(`pipeline__component__side__output--${componentKey}__${outputIdx}`);
            }

            if (scope.pipeline.eles[componentKey] && scope.pipeline.eles[componentKey].parentNode !== null) {
                scope.pipeline.eles[componentKey].parentNode.removeChild(scope.pipeline.eles[componentKey]);
            }

            // delete the reference
            delete scope.pipeline.data[componentKey];

            scope.pipeline.numComponents--;

            // look through the parent components and remove references to deleted components
            Object.keys(scope.pipeline.data).forEach(k => {
                scope.pipeline.data[k].output.forEach(output => {
                    if (output.downstream && output.downstream.key === componentKey) {
                        output.downstream = {};
                        scope.pipeline.data[k].hasDownstream = false;
                    }
                });
            });

            if (rerun) {
                scope.pipeline.confirmRemove = false;
                scope.pipeline.nodeDeleted = true;
                closePreview();
            }

            // save that something has changed
            savePipeline();
        }

        // TODO: dont go by index go by downstream nodes
        /**
         * @name rerunPipeline
         * @param idx - which component to run
         * @desc reruns component pixels in order
         */
        function rerunPipeline(idx: number, cb?: () => any): void {
            let pixel: string = '';
            // need to sort in case keys come out of order
            const key: string = Object.keys(scope.pipeline.data).sort()[idx],
                callback = (response: PixelReturnPayload) => {
                    if (checkErrors(response)) {
                        turnOffPause();
                        scope.pipeline.data[key].replay = false;
                        return;
                    }
                    const next = idx + 1;
                    if (!scope.pipeline.paused) {
                        if (key) {
                            if (!scope.pipeline.paused) {
                                scope.pipeline.data[key].replay = false;
                            }
                        } else { // no components close preview
                            closePreview();
                        }

                        rerunPipeline(next, cb);
                    } else {
                        scope.pipeline.pausedAt = idx;
                    }
                    if (scope.pipeline.preview.open && scope.pipeline.pausedAt <= idx) {
                        previewFrame(key);
                    }
                };

            if (idx === 0) {
                pixel += 'ClearInsight();';
            }

            if (scope.pipeline.data[key] && scope.pipeline.data[key].createFrame && typeof scope.pipeline.data[key].createFrame === 'string') {
                pixel += scope.pipeline.data[key].createFrame;
            }

            if (scope.pipeline.componentToPixelMap[key]) {
                pixel += scope.pipeline.componentToPixelMap[key];
            }

            if (!pixel || !key) {
                const dataKeys = Object.keys(scope.pipeline.data),
                    lastDataKey = dataKeys[dataKeys.length - 1];
                scope.pipeline.replaying = false;

                turnOffPause();
                // not there if delete 0th component
                if (lastDataKey) {
                    // if there is a deleted node, then the last one will be the new node,
                    // we do not want to set this to executed
                    if (!scope.pipeline.nodeDeleted) {
                        scope.pipeline.data[lastDataKey].state = scope.pipeline.STATE.EXECUTED;
                    }
                    if (scope.pipeline.preview.open) {
                        previewFrame(lastDataKey);
                    }
                    if (scope.pipeline.nodeDeleted) {
                        // when deleting node, need to make sure we preivew a component with output
                        let lastComponentWithOutput = lastDataKey;
                        if (scope.pipeline.data[lastComponentWithOutput].output.length === 0) {
                            for (let param in scope.pipeline.data[lastComponentWithOutput].parameters) {
                                if (
                                    scope.pipeline.data[lastComponentWithOutput].parameters.hasOwnProperty(param) &&
                                    param === scope.pipeline.data[lastComponentWithOutput].input[0]
                                ) {
                                    lastComponentWithOutput = scope.pipeline.data[lastComponentWithOutput].parameters[param].upstream.key;
                                    break;
                                }
                            }
                        }
                        if (!scope.pipeline.edit.open) {
                            openPreview(lastComponentWithOutput);
                        }
                        scope.pipeline.nodeDeleted = false;
                        previewFrame(lastComponentWithOutput);
                    }
                }

                if (cb) {
                    // right now only comes into play when user goes back into a component
                    // and visualizes. need to update the frame, then visualize
                    cb();
                }
            }
            // need to clear insight always (remove all components for example)
            if (pixel) {
                scope.pipeline.replaying = true;
                if (key) {
                    if (scope.pipeline.pausedAt === null || scope.pipeline.pausedAt < idx) {
                        scope.pipeline.data[key].replay = true;
                    }
                }
                $timeout(() => {
                    scope.widgetCtrl.execute([{
                        type: 'Pixel',
                        components: [pixel],
                        terminal: true
                    }], callback, [/* no listeneers so we dont get loading screen */]);
                }, REPLAY_TIMER);
            }
        }

        /**
         * @name togglePause
         * @desc stops the replay
         */
        function togglePause(): void {
            const keys = Object.keys(scope.pipeline.data);
            let wherePreview = scope.pipeline.pausedAt ? scope.pipeline.pausedAt : 0;
            scope.pipeline.paused = !scope.pipeline.paused;
            if (!scope.pipeline.replaying) {
                scope.pipeline.paused = false;
            }
            if (!scope.pipeline.paused) {
                if (!scope.pipeline.preview.open) {
                    openPreview(keys[wherePreview]);
                }
                rerunPipeline(0);
            }
        }

        /**
         * @name turnOffPause
         * @desc turns off pause variables
         */
        function turnOffPause(): void {
            const keys = Object.keys(scope.pipeline.data);
            scope.pipeline.paused = false;
            if (scope.pipeline.data[keys[scope.pipeline.pausedAt]]) {
                scope.pipeline.data[keys[scope.pipeline.pausedAt]].replay = false;
            }
            scope.pipeline.pausedAt = null;
            scope.pipeline.replaying = false;
        }

        /**
         * @name step
         * @param direction - 1 forward, -1 backward
         * @desc goes forward or backward in replay
         */
        function step(direction: number): void {
            const keys = Object.keys(scope.pipeline.data);
            let key: string,
                callback: (res: PixelReturnPayload) => void,
                pixel = '',
                i = 0;

            if (!scope.pipeline.paused) {
                if (direction < 0) {
                    return;
                }
                scope.pipeline.paused = true;
                key = keys[0];
                scope.pipeline.pausedAt = -1;
            } else if (
                direction > 0 &&
                scope.pipeline.pausedAt === keys.length - 1
            ) {
                turnOffPause();

                return;
            } else if (direction < 0 && scope.pipeline.pausedAt === 0) {
                turnOffPause();

                return;
            }
            if (scope.pipeline.pausedAt > -1) {
                scope.pipeline.data[keys[scope.pipeline.pausedAt]].replay = false;
            }

            scope.pipeline.pausedAt += direction;
            scope.pipeline.data[keys[scope.pipeline.pausedAt]].replay = true;

            if (!scope.pipeline.preview.open) {
                openPreview(scope.pipeline.data[keys[scope.pipeline.pausedAt]]);
            }
            callback = (response) => {
                if (checkErrors(response)) {
                    turnOffPause();

                    return;
                }

                if (scope.pipeline.preview.open) {
                    previewFrame(keys[scope.pipeline.pausedAt]);
                }
            };

            if (direction > 0) {
                i = scope.pipeline.pausedAt;
            }

            if (direction < 0 || scope.pipeline.pausedAt === 0) {
                pixel += 'ClearInsight();';
            }

            for (; i <= scope.pipeline.pausedAt; i++) {
                key = keys[i];

                if (scope.pipeline.data[key] && scope.pipeline.data[key].createFrame) {
                    pixel += scope.pipeline.data[key].createFrame;
                }

                if (scope.pipeline.componentToPixelMap[key]) {
                    pixel += scope.pipeline.componentToPixelMap[key];
                }
            }

            scope.widgetCtrl.execute([{
                type: 'Pixel',
                components: [pixel],
                terminal: true
            }], callback, [/* no listeneers so we dont get loading screen */]);
        }

        /**
         * @name previewFrame
         * @param key - last component in pipeline frame
         * @desc called when user reruns pipeline and preview was already opened, just need to query the frame
         */
        function previewFrame(key: string): void {
            if (scope.pipeline.data[key].output.length === 0) {
                return;
            }
            const pixel = semossCoreService.pixel.build([{
                type: 'frame',
                components: [
                    scope.pipeline.data[key].output[0].value.name
                ],
                meta: true
            }, {
                type: 'queryAll',
                components: []
            }, {
                type: 'limit',
                components: [
                    PREVIEW_LIMIT
                ]
            }, {
                type: 'collect',
                components: [
                    500
                ],
                terminal: true
            }]);

            scope.widgetCtrl.emit('load-preview', {
                pixelComponents: [{
                    type: 'Pixel',
                    components: [
                        pixel
                    ],
                    terminal: true
                }]
            });
        }

        /**
         * @name showFederate
         * @param key - key of the component
         * @desc show the federate overlay
         * @returns {void}
         */
        function showFederate(key: string): void {
            let frame: string = '',
                type: string = '',
                outputIdx: number,
                headers: any = [];

            const output = getComponent(key, 'output');

            // find the frame name in the output
            for (outputIdx = 0; outputIdx < output.length; outputIdx++) {
                // if its a frame output, we will set the output
                if (output[outputIdx].frame) {
                    frame = output[outputIdx].value.name;
                    headers = output[outputIdx].value.headers;
                    type = output[outputIdx].value.type;
                }
            }

            closePreview();
            scope.pipeline.federate = {
                name: 'federate',
                open: true,
                key: key,
                frame: frame,
                type: type,
                importHeight: 100,
                previewHeight: 0,
                headers: headers
            };
        }

        /**
         * @name showComponent
         * @param key - key of the component
         * @desc close the component detail for the selected component and stop editing
         */
        function showComponent(key: string): void {
            let componentHeight = 80;
            closePreview();

            if (scope.pipeline.data[key].hidePreview) {
                componentHeight = 100;
            }

            scope.pipeline.edit = {
                name: scope.pipeline.data[key].name,
                open: true,
                selected: key,
                height: componentHeight,
                preview: 20,
                hidePreview: scope.pipeline.data[key].hidePreview
            };
        }

        /**
         * @name closeComponent
         * @param key - key of the component
         * @desc close the component detail for the selected component and stop editing, if key is empty
         *       simply closes the edit overlay
         */
        function closeComponent(key: string): void {
            // if there is only one component and it is not set, remove it.
            if (scope.pipeline.data.hasOwnProperty(key) &&
                scope.pipeline.data[key].state === scope.pipeline.STATE.INITIAL &&
                Object.keys(scope.pipeline.data).length === 1) {
                // remove it
                removeComponent(key, false);

                // render the landing
                scope.pipeline.landing.open = true;
            }

            scope.pipeline.edit = {
                open: false,
                selected: undefined,
                height: 80,
                preview: 20
            };
        }

        /**
         * @name validateComponent
         * @desc validate that the component is valid
         * @param key - key of the component
         * @param values - map containing parameter to updated values
         * @returns valid or invalid
         */
        function validateComponent(key: string, values: string): boolean {
            // check if it is there or not
            for (let parameter in scope.pipeline.data[key].parameters) {
                if (
                    scope.pipeline.data[key].parameters.hasOwnProperty(parameter) &&
                    scope.pipeline.data[key].parameters[parameter].required
                ) {
                    // if it is part of the pixel script, then it must be valid
                    if (
                        typeof values[parameter] === 'undefined'
                    ) {
                        return false;
                    }
                }
            }

            return true;
        }

        /**
         * @name buildComponent
         * @desc preview the data of this component
         * @param key - key of the component
         * @param values - map containing parameter to updated values
         */
        function buildComponent(key: string, values: any): string {
            let tokens = semossCoreService.pixel.tokenize(scope.pipeline.data[key].pixel),
                updated: any = {};

            for (let parameter in values) {
                if (values.hasOwnProperty(parameter) &&
                    scope.pipeline.data[key].parameters.hasOwnProperty(parameter)
                ) {
                    updated[parameter] = generateComponentValue(scope.pipeline.data[key].parameters[parameter].type, values[parameter]);
                }
            }

            return semossCoreService.pixel.construct(tokens, updated);
        }

        /**
         * @name buildCountComponent
         * @desc get the total count for this component
         * @param key - key of the component
         * @param values - map containing parameter to updated values
         */
        function buildCountComponent(key: string, values: any): string {
            let tokens = semossCoreService.pixel.tokenize('<QUERY_STRUCT> | CollectAll();'),
                updated: any = {},
                tempValues = semossCoreService.utility.freeze(values);

            for (let parameter in tempValues) {
                // we only care about the QUERY_STRUCT
                if (tempValues.hasOwnProperty(parameter) &&
                    scope.pipeline.data[key].parameters.hasOwnProperty(parameter)
                ) {
                    if (parameter === 'QUERY_STRUCT') {
                        delete tempValues[parameter].limit;
                        delete tempValues[parameter].offset;
                        delete tempValues[parameter].orders;
                        // we only need one selector wrapped in a count
                        tempValues[parameter].selectors = [tempValues[parameter].selectors[0]];
                        tempValues[parameter].selectors[0].content.math = 'Count';
                        tempValues[parameter].selectors[0].content.calculatedBy = tempValues[parameter].selectors[0].content.table + '__' + tempValues[parameter].selectors[0].content.column;
                    }
                    updated[parameter] = generateComponentValue(scope.pipeline.data[key].parameters[parameter].type, tempValues[parameter]);
                }
            }

            return semossCoreService.pixel.construct(tokens, updated);
        }

        /**
         * @name generateComponentValue
         * @desc preview the data of this component
         * @param type - type to generate value for
         * @param value - value to generate for
         */
        function generateComponentValue(type: string, value: any): any {
            let val;

            if (type === 'CREATE_FRAME' && value && value.type === 'default') {
                if (CONFIG.defaultFrameType) {
                    value.type = CONFIG.defaultFrameType;
                } else {
                    value.type = 'GRID';
                }
            }

            if (typeof value !== 'object') {
                val = value;
            } else {
                val = JSON.parse(JSON.stringify(value))
            }

            if (!type || type === 'PIXEL') {
                return val;
            }

            // this is the value (we just use another pixel string as the value)
            val = semossCoreService.pixel.build([{
                type: type,
                components: [
                    val
                ]
            }]);
            val = val.slice(0, -2); // remove the last pipe and semicolon

            return val;
        }


        /**
         * @name previewComponent
         * @desc preview the data of this component
         * @param key - key of the component
         * @param values - map containing parameter to updated values
         */
        function previewComponent(key: string, values: any): void {
            let previewData: any = {};

            if (timeout) {
                $timeout.cancel(timeout);
            }

            timeout = $timeout(() => {
                let pixel = '';

                // validate, if it isn't valid load the 'base' (nothing or the table)
                if (!validateComponent(key, values)) {
                    pixel = '';
                    // preview on of the frames
                    if (
                        scope.pipeline.data[key].parameters[scope.pipeline.data[key].preview] &&
                        scope.pipeline.data[key].parameters[scope.pipeline.data[key].preview].value &&
                        scope.pipeline.data[key].parameters[scope.pipeline.data[key].preview].value.name
                    ) {
                        pixel += semossCoreService.pixel.build([{
                            type: 'frame',
                            components: [
                                scope.pipeline.data[key].parameters[scope.pipeline.data[key].preview].value.name
                            ],
                            meta: true
                        }, {
                            type: 'queryAll',
                            components: []
                        }, {
                            type: 'limit',
                            components: [
                                scope.pipeline.previewLimit
                            ]
                        }, {
                            type: 'collect',
                            components: [
                                500
                            ],
                            terminal: true
                        }]);
                    }

                    if (pixel.length === 0) {
                        scope.widgetCtrl.emit('load-preview', {
                            pixelComponents: []
                        });
                        return;
                    }


                    scope.widgetCtrl.emit('load-preview', {
                        pixelComponents: [{
                            type: 'Pixel',
                            components: [
                                pixel
                            ],
                            terminal: true
                        }]
                    });

                    return;
                }

                // we are editing an already set component.
                // to preview, we need to dump frame and rerun the pipeline
                // with the new values in this component;
                if (scope.pipeline.data[key].state !== scope.pipeline.STATE.INITIAL) {
                    pixel += 'ClearInsight();';

                    Object.keys(scope.pipeline.data).forEach(component => {
                        if (component === key) {
                            pixel += buildComponent(key, values);
                        } else {
                            if (scope.pipeline.data[component].createFrame) {
                                pixel += scope.pipeline.data[component].createFrame;
                            }
                            pixel += scope.pipeline.componentToPixelMap[component] || '';
                        }
                    });
                } else {
                    pixel += buildComponent(key, values);
                }

                if (pixel.length === 0) {
                    scope.widgetCtrl.emit('load-preview', {
                        pixelComponents: []
                    });
                    return;
                }

                // preview on of the frames
                if (scope.pipeline.data[key].preview) {
                    pixel += semossCoreService.pixel.build([{
                        type: 'frame',
                        components: [
                            values[scope.pipeline.data[key].preview].name
                        ],
                        meta: true
                    }, {
                        type: 'queryAll',
                        components: []
                    }, {
                        type: 'limit',
                        components: [
                            scope.pipeline.previewLimit
                        ]
                    }, {
                        type: 'collect',
                        components: [
                            500
                        ],
                        terminal: true
                    }]);
                }

                previewData = {
                    pixelComponents: [{
                        type: 'Pixel',
                        components: [
                            `CopyInsight(limit=[${scope.pipeline.previewLimit}], recipe=["<encode>${pixel}</encode>"])`
                        ],
                        terminal: true
                    }]
                }

                // if we're querying from the database, we will append the count pixel
                if (values.QUERY_STRUCT && values.QUERY_STRUCT.qsType === 'ENGINE') {
                    previewData.totalCountPixelComponents = [
                        {
                            type: 'Pixel',
                            components: [
                                buildCountComponent(key, values)
                            ],
                            terminal: true
                        }
                    ]
                }

                scope.widgetCtrl.emit('load-preview', previewData);
            }, 300);
        }

        /**
         * @name getExternalFrames
         * @param node - pixel ast node
         * @param frames - frame names
         * @desc looks for imports and grabs the frames
         */
        function getExternalFrames(node: PixelASTLeaf[], frames: Map<string, string>): void {
            let sib = node[node.length - 1], frameName: string, frameType: string;

            // traverse for this because it changes
            if (sib.nounInputs.frame) {
                for (let i = 0; i < sib.nounInputs.frame.length; i++) {
                    if (sib.nounInputs.frame[i].type === 'LAMBDA') {
                        let rowInputs = sib.nounInputs.frame[i].value.rowInputs;
                        for (let j = 0; j < rowInputs.length; j++) {
                            if (rowInputs[j].type === 'LAMBDA') {
                                for (let k = 0; k < rowInputs[j].value.rowInputs.length; k++) {
                                    if (rowInputs[j].value.rowInputs[k].type === 'CONST_STRING') {
                                        frameName = rowInputs[j].value.rowInputs[k].value;
                                        frameType = scope.widgetCtrl.getShared('frames.' + frameName + '.type');
                                        frames.set(frameName, frameType);

                                        break;
                                    }
                                }
                            }
                        }
                    }
                }
            } else {
                console.error('go a node leaf without a frame on the noun input')
            }
        }

        /**
         * @name configureSourceAndTarget
         * @param target target component
         * @param frames available frames
         * @desc when connecting two components, need to run thru this configuration before actually making the connection
         */
        function configureSourceAndTarget(target: any, frames: Map<string, string>): void {
            let targetParam, source;
            for (let key in scope.pipeline.data) {
                if (
                    scope.pipeline.data.hasOwnProperty(key) &&
                    scope.pipeline.data[key].output &&
                    scope.pipeline.data[key].output[0] &&
                    Object.keys(scope.pipeline.data[key].output[0].downstream).length === 0
                ) {
                    source = scope.pipeline.data[key];
                }
            }

            source.output.forEach(out => {
                if (out.frame) {
                    frames.set(out.value.name, out.value.name);
                }
            });
            renderComponentOutput(source.key)
            for (let param in target.parameters) {
                if (
                    target.parameters.hasOwnProperty(param) &&
                    target.parameters[param].frame &&
                    Object.keys(target.parameters[param].upstream).length === 0
                ) {
                    targetParam = target.parameters[param];
                    break;
                }
            }
            if (targetParam) {
                setComponentsAndShow(
                    targetParam,
                    target,
                    source.output[0], // only have one output right now
                    source
                );

                if (!scope.pipeline.data[source.key].hasDownstream) {
                    scope.pipeline.data[source.key].hasDownstream = true;
                }
            }
        }

        /**
         * @name executeExternalComponent
         * @param key component key
         * @param prevComponent - previous component in pipeline
         * @param node - pixel ast node to become a component
         * @desc connects latest node to previous component, otherwise grabs frames
         */
        function executeExternalComponent(key: string, prevComponent: any, node: PixelASTLeaf[]): void {
            const frames: Map<string, string> = new Map(),
                lastSib = node[node.length - 1],
                opName = lastSib.opName;

            scope.pipeline.componentToPixelMap[key] = node.map(sib => sib.opString).join(' | ');
            // nodes like dummy merges do not have pixel
            if (scope.pipeline.componentToPixelMap[key]) {
                scope.pipeline.componentToPixelMap[key] += ';';
            }
            if (prevComponent && opName !== 'Import') {
                let target = scope.pipeline.data[scope.pipeline.lastAddedKey];
                if (opName === DUMMY) {
                    target.disableEdit = true;
                }
                // code block totally useless externally if we do not fill params,
                // but we do need a way to do this generically
                if (lastSib.widgetId === 'pipeline-codeblock') {
                    const code = lastSib.rowInputs[0].value,
                        language = lastSib.opName;

                    scope.pipeline.data[key].parameters.CODE.value = code;
                    scope.pipeline.data[key].parameters.LANGUAGE.value = language;
                }
                for (let i = 0; i < target.input.length; i++) {
                    configureSourceAndTarget(target, frames);
                }
            } else if (opName === 'Import') {
                let component = scope.pipeline.data[key];
                getExternalFrames(node, frames);
                setOptions(lastSib, key);
                for (let frame in frames) {
                    if (frames.hasOwnProperty(frame)) {
                        for (let param in component.parameters) {
                            if (component.parameters.hasOwnProperty(param)) {
                                if (component.parameters[param].frame && !component.parameters[param].value) {
                                    component.parameters[param].value = { name: frame, type: frames[frame] };
                                }
                            }
                        }
                    }
                }
            }

            renderProcessedComponent(frames.keys(), key);
        }

        /**
         * @name setOptions
         * @param importNode - node where data is imported
         * @param key - key for pipeline component to update
         * @desc sets options on component to reflect imported data
         */
        function setOptions(importNode: PixelASTLeaf, key: string): void {
            const opts: { name: string, icon: string } = {
                name: '',
                icon: ''
            };
            let qs: any;

            if (importNode.nounInputs && importNode.nounInputs.qs) {
                qs = importNode.nounInputs.qs[0];
                if (importNode.widgetId === 'pipeline-file') {
                    let fileName = qs.value.filePath, param = 'INSIGHT_FOLDER\\',
                        fileExt = qs.value.filePath.substring(qs.value.filePath.lastIndexOf('.'));

                    fileName = fileName.substring(param.length, fileName.lastIndexOf('.'));
                    opts.name = fileName + fileExt;
                } else if (importNode.widgetId === 'pipeline-external') {
                    let connector = CONNECTORS[qs.value.config.dbDriver];
                    opts.name = connector.name;
                    opts.icon = connector.image;
                } else if (qs && qs.value && qs.value.engineName) {
                    scope.widgetCtrl.meta([{
                        type: 'appInfo',
                        components: [qs.value.engineName],
                        terminal: true,
                        meta: true
                    }], response => {
                        scope.pipeline.data[key].name = response.pixelReturn[0].output.app_name.replace(/_/g, ' ');
                    });
                    opts.name = '';
                    opts.icon = semossCoreService.app.generateAppImageURL(qs.value.engineName) + '?time=' + new Date().getTime();
                }
            }

            if (opts.name) {
                scope.pipeline.data[key].name = opts.name;
            }

            if (opts.icon) {
                scope.pipeline.data[key].icon = opts.icon;
            }
        }

        /**
         * @name updateOptions
         * @param options options
         * @param key component key
         * @desc updates name and icon if available
         */
        function updateOptions(options: { name: string, icon: string }, key: string): void {
            // update the name
            if (options.hasOwnProperty('name')) {
                scope.pipeline.data[key].name = options.name;
            }

            // update the icon
            if (options.hasOwnProperty('icon')) {
                scope.pipeline.data[key].icon = options.icon;
            }
        }

        /**
         * @name updateParameters
         * @param values - update parameter values
         * @param key - key for component to update
         * @desc updates a components params after execution
         */
        function updateParameters(values: any, key: string): void {
            for (let parameter in values) {
                if (
                    values.hasOwnProperty(parameter) &&
                    scope.pipeline.data[key].parameters.hasOwnProperty(parameter)
                ) {
                    scope.pipeline.data[key].parameters[parameter].value = values[parameter];
                }
            }
        }

        /**
         * @name executeComponent
         * @desc close the component detail for the selected component
         * @param key - key of the component
         * @param values - map containing parameter to updated values
         * @param options - updated options
         */
        function executeComponent(key: string, values: any, options: any, cb?: (...args: any) => any): void {
            let pixel: string,
                callback: (response: PixelReturnPayload) => void;


            if (!validateComponent(key, values)) {
                return;
            }

            pixel = buildComponent(key, values);

            if (!pixel) {
                return;
            }

            callback = (response) => {
                let frames: Map<string, boolean>;

                // bad
                if (checkErrors(response)) {
                    scope.pipeline.componentToPixelMap[key] = false;
                    return;
                }
                scope.pipeline.componentToPixelMap[key] = pixel;
                updateOptions(options, key);
                updateParameters(values, key);
                // grab all of the frames
                frames = new Map();
                processComponent(frames, response.pixelReturn, key);

                renderProcessedComponent(frames.keys(), key);

                if (cb) {
                    cb();
                }
            };
            if (scope.pipeline.data[key].state !== scope.pipeline.STATE.INITIAL) {
                // lets test to confirm this is a valid pixel
                scope.widgetCtrl.meta([{
                    type: 'Pixel',
                    components: [pixel],
                    terminal: true
                }], response => {
                    if (checkErrors(response)) {
                        return;
                    }
                    scope.pipeline.componentToPixelMap[key] = pixel;
                    updateOptions(options, key);
                    updateParameters(values, key);
                    closeComponent('');

                    if (!scope.pipeline.paused) {
                        rerunPipeline(0, cb);
                    }
                });
            } else {
                scope.widgetCtrl.execute([{
                    type: 'Pixel',
                    components: [pixel],
                    terminal: true
                }], callback);
            }
        }

        /**
         * @name renderProcessedComponent
         * @param frames - frames in the pipeline
         * @param key - component key
         * @desc after component has been processed, set values to connect it to other component
         */
        function renderProcessedComponent(frames: IterableIterator<string>, key: string): void {
            scope.pipeline.data[key].output = [];
            for (let frame of frames) {
                let stored = scope.widgetCtrl.getShared('frames.' + frame) || {
                    name: '',
                    type: '',
                    headers: [],
                    joins: []
                };
                scope.pipeline.eles[key].title = stored.name;
                scope.pipeline.data[key].output.push({
                    output: scope.pipeline.data[key].output.length, // right now the only thing that is unique is the index
                    downstream: {},
                    frame: true,
                    value: {
                        name: stored.name,
                        type: stored.type,
                        headers: stored.headers,
                        joins: stored.joins
                    }
                });
            }

            // add the output fields
            renderComponentOutput(key);

            // update the designation, if there is a frame it can be joined, otherwise you can't do anything
            if (scope.pipeline.data[key].output.length > 0) {
                scope.pipeline.data[key].state = scope.pipeline.STATE.EXECUTED;
            }

            // close the modal
            closeComponent('');

            // save that something has changed
            savePipeline();

            if (scope.pipeline.preview.open) {
                previewFrame(key);
            }
        }

        /**
         * @name processComponent
         * @param frames - updated frames
         * @param response - current response
         * @param key - component key
         * @desc function that process the component and finds out all of the changed frames
         */
        function processComponent(frames: Map<string, boolean>, response: PixelReturn[], key: string): void {
            let output,
                type: string[];

            // Need this to reassign the param
            for (let i = 0, len = response.length; i < len; i++) {
                // easy mapping to the output object
                output = response[i].output;
                type = response[i].operationType;

                for (let typeIdx = 0; type && typeIdx < type.length; typeIdx++) {
                    let opType = type[typeIdx];
                    if (opType === 'CODE_EXECUTION' || opType === 'VECTOR' || opType === 'SUB_SCRIPT') {
                        // this is nested
                        if (output !== 'no output') {
                            processComponent(frames, output, key);
                        } else {
                            console.error('Error: Unknown response type: ' + opType + ' with response: ' + output);
                        }
                    }
                    if (
                        opType === 'FRAME' ||
                        opType === 'FRAME_FILTER' ||
                        opType === 'FRAME_HEADERS' ||
                        opType === 'FRAME_HEADERS_CHANGE' ||
                        opType === 'FRAME_DATA_CHANGE'
                    ) {
                        let frameName = output.name;
                        // output does not have frame name
                        if (!frameName) {
                            frameName = getComponentInputFrame(key, 'name');
                        }
                        frames.set(frameName, true);
                    }
                }

                // This was run 'after' the current step
                if (response[i].additionalOutput) {
                    processComponent(frames, response[i].additionalOutput, key);
                }
            }

            // just output the source frame, need this for code block where the user can just execute code without
            // returning a frame. ASSUMPTION: i am fairly certain there is no component where we want a hard 
            // stop on the pipeline
            if (frames.size === 0) {
                const component = scope.pipeline.data[key];
                component.input.forEach((input: string) => {
                    const param = component.parameters[input];
                    if (param.frame) {
                        frames.set(param.value.name, true);
                    }
                });
            }
        }

        /**
         * @name getComponentInputFrame
         * @param key - component key
         * @param accessor - what key to access in component frame parameter
         * @desc gets data about components input frame
         * @return data about input frame
         */
        function getComponentInputFrame(key: string, accessor?: string): any {
            const component = scope.pipeline.data[key],
                frameParamName = component.input[0],
                frameParam = component.parameters[frameParamName].value,
                accessors = accessor ? accessor.split('.') : [];
            let value = frameParam;

            for (let i = 0; i < accessors.length; i++) {
                value = value[accessors[i]];
            }

            return value;
        }

        /**
         * @name viewComponent
         * @param key - key of the component
         * @desc view data of the component
         */
        function viewComponent(key: string): void {
            closeComponent('');
            openPreview(key);

            if (timeout) {
                $timeout.cancel(timeout);
            }

            // TODO: Can we view multiple? What happens with multiple?
            timeout = $timeout(() => {
                let output: any;

                if (scope.pipeline.data[key]) {
                    for (let outputIdx = 0, outputLen = scope.pipeline.data[key].output.length; outputIdx < outputLen; outputIdx++) {
                        output = scope.pipeline.data[key].output[outputIdx];
                        if (output.frame) {
                            scope.widgetCtrl.emit('load-preview', {
                                pixelComponents: [{
                                    type: 'frame',
                                    components: [
                                        output.value.name
                                    ],
                                    meta: true
                                }, {
                                    type: 'queryAll',
                                    components: []
                                }, {
                                    type: 'limit',
                                    components: [
                                        PREVIEW_LIMIT
                                    ]
                                }, {
                                    type: 'collect',
                                    components: [
                                        500
                                    ],
                                    terminal: true
                                }]
                            });

                            break;
                        }
                    }
                }
            }, 300);
        }

        /**
         * @name linkComponent
         * @desc called when before a connection can occur
         * @param params - params
         * @returns is the connection valid?
         */
        function linkComponent(params: any): boolean {
            let source,
                target,
                sourceComponent,
                targetComponent,
                sourceOutput,
                targetParameter,
                error;

            source = params.sourceId.replace('pipeline__component__side__output--', '').split('__');

            if (!error) {
                sourceComponent = scope.pipeline.data[source[0]];
                if (typeof sourceComponent === 'undefined') {
                    error = 'Source is not defined';
                }
            }

            if (!error) {
                sourceOutput = sourceComponent.output[+source[1]];
                if (typeof sourceOutput === 'undefined') {
                    error = 'Source output is not defined';
                }
            }

            // source is already linked
            if (!error) {
                if (Object.keys(sourceOutput.downstream).length !== 0) {
                    error = 'Source is already linked';
                }
            }

            target = params.targetId.replace('pipeline__component__side__input--', '').split('__');

            if (!error) {
                targetComponent = scope.pipeline.data[target[0]];
                if (typeof targetComponent === 'undefined') {
                    error = 'Target is not defined';
                }
            }


            if (!error) {
                targetParameter = targetComponent.parameters[target[1]];
                if (typeof targetParameter === 'undefined') {
                    error = 'Target parameter is not defined';
                }
            }

            // parameter it nos an input
            if (!error) {
                if (targetComponent.input.indexOf(targetParameter.parameter) === -1) {
                    error = 'Target is not an input';
                }
            }

            // parameter already linked
            if (!error) {
                if (Object.keys(targetParameter.upstream).length !== 0) {
                    error = 'Target is already linked';
                }
            }

            // they both must be frames to link.
            // TODO : expand to have other types
            if (!error) {
                if (!sourceOutput.frame || !targetParameter.frame) {
                    error = 'Source or target are not frames';
                }
            }

            if (error) {
                $timeout(function () {
                    scope.widgetCtrl.alert('error', error);
                });

                return false;
            }

            targetComponent.disableEdit = false;
            targetComponent.validInput = true;
            if (targetComponent.required && targetComponent.required.Frame) {
                for (let i = 0; i < sourceComponent.output.length; i++) {
                    let output = sourceComponent.output[i],
                        inputFrameType;

                    inputFrameType = targetComponent.required.Frame[0];
                    if (
                        output.frame &&
                        output.value &&
                        output.value.type !== inputFrameType
                    ) {
                        // make sure we remember to do this if user reruns
                        if (scope.pipeline.componentToPixelMap[source[0]]) {
                            scope.pipeline.componentToPixelMap[source[0]] += `${output.value.name} | Convert(frameType=[${inputFrameType}]);`;
                        } else {
                            scope.pipeline.componentToPixelMap[source[0]] = `${output.value.name} | Convert(frameType=[${inputFrameType}]);`;
                        }
                        scope.pipeline.data[source[0]].pixel += `${output.value.name} | Convert(frameType=[${inputFrameType}]);`;
                        scope.widgetCtrl.execute(
                            [
                                {
                                    type: 'variable',
                                    components: [
                                        output.value.name
                                    ]
                                },
                                {
                                    type: 'convert',
                                    components: [
                                        inputFrameType
                                    ],
                                    terminal: true
                                }
                            ],
                            setComponentsAndShow.bind(
                                null,
                                targetParameter,
                                targetComponent,
                                sourceOutput,
                                sourceComponent
                            )
                        );

                        return true;
                    }
                }
            }

            $timeout(function() {
                setComponentsAndShow(targetParameter, targetComponent, sourceOutput, sourceComponent);
            });

            return true;
        }

        /**
         * @name allInputsLinked
         * @param component - pipeline component
         * @desc determines if component has all inputs linked
         * @return true if all inputs are linked
         */
        function allInputsLinked(component: any): boolean {
            if (component.input.length === 1) return true;
            if (component.input.length === 2) {
                if (component.prevConnection) {
                    return true;
                }

                component.prevConnection = true;
            }

            return false;
        }

        /**
         * @name setComponentsAndShow
         * @param targetParameter - target component parameters
         * @param targetComponent - target component data
         * @param sourceOutput - source component output data
         * @param sourceComponent - source component
         * @desc sets the necessary component values and reveals it
         */
        function setComponentsAndShow(targetParameter: any, targetComponent: any, sourceOutput: any, sourceComponent: any): void {
            targetParameter.upstream = {
                key: sourceComponent.key,
                output: sourceOutput.output
            };

            // save the value;
            targetParameter.value = JSON.parse(JSON.stringify(sourceOutput.value));

            // record the downstream
            sourceOutput.downstream = {
                key: targetComponent.key,
                parameter: targetParameter.parameter
            };

            sourceComponent.hasDownstream = true;
            if (allInputsLinked(targetComponent)) {
                showComponent(targetComponent.key);
            }
        }

        /**
         * @name visualizeComponent
         * @param key - key of the component
         * @desc called to jump and switch to the visualization menu
         */
        function visualizeComponent(key: string): void {
            closeComponent('');

            if (timeout) {
                $timeout.cancel(timeout);
            }

            // TODO: Can we view multiple? What happens with multiple?
            timeout = $timeout(() => {
                if (hasSameHeaders(scope.pipeline.data[key].output[0].value.name)) {
                    scope.widgetCtrl.execute([
                        {
                            type: 'panel',
                            components: [
                                scope.widgetCtrl.panelId
                            ]
                        },
                        {
                            type: 'setPanelView',
                            components: ['visualization'],
                            terminal: true
                        }
                    ], () => scope.widgetCtrl.open('widget-tab', 'view'));
                } else {
                    buildAndOpenGrid(key);
                }
            }, 300);
        }

        /**
         * @name buildAndOpenGrid
         * @param key - component key
         * @desc builds and open a grid based on frame
         */
        function buildAndOpenGrid(key: string): void {
            for (let outputIdx = 0, outputLen = scope.pipeline.data[key].output.length; outputIdx < outputLen; outputIdx++) {
                let output = scope.pipeline.data[key].output[outputIdx];
                if (output.frame) {
                    const callback = () => {
                        scope.widgetCtrl.open('widget-tab', 'view');
                    };

                    scope.widgetCtrl.execute([{
                        type: 'panel',
                        components: [
                            scope.widgetCtrl.panelId
                        ]
                    },
                    {
                        type: 'setPanelView',
                        components: ['visualization'],
                        terminal: true
                    },
                    {
                        type: 'frame',
                        components: [
                            output.value.name
                        ]
                    },
                    {
                        type: 'queryAll',
                        components: []
                    },
                    {
                        type: 'autoTaskOptions',
                        components: [
                            scope.widgetCtrl.panelId,
                            'Grid'
                        ]
                    },
                    {
                        type: 'collect',
                        components: [scope.widgetCtrl.getOptions('limit')],
                        terminal: true
                    }
                    ], callback);

                    break;
                }
            }
        }

        /**
         * @name hasSameHeaders
         * @param frame {string} the frame to check headers against 
         * @desc determines if user has not altered headers from their last visualization
         * @return true if headers have not changed and we just want to change the panel view;
         *          false if headers have changed and we want to rebuild task for a Grid
         */
        function hasSameHeaders(frame: string): boolean {
            let headers: any = {}, newHeaders: any = [];
            const frameHeadersCurrent = scope.widgetCtrl.getFrame('headers') || [],
                layout = semossCoreService.getWidget(scope.widgetCtrl.widgetId, 'view.visualization.layout'),
                frameHeadersNew = semossCoreService.getShared(scope.widgetCtrl.insightID, 'frames.' + frame + '.headers');

            // frame has changed so send a new task to paint
            if (frameHeadersCurrent.length !== frameHeadersNew.length) {
                return false;
            }

            // no layout so need a new task
            if (!layout) {
                return false;
            }

            for (let i = 0; i < frameHeadersCurrent.length; i++) {
                headers[frameHeadersCurrent[i].displayName] = true;
            }

            for (let i = 0; i < frameHeadersNew.length; i++) {
                newHeaders.push(frameHeadersNew[i].displayName);
            }

            for (let i = 0; i < newHeaders.length; i++) {
                if (!headers[newHeaders[i]]) {
                    // frame headers have changed so we will set new task to paint
                    return false;
                }
            }

            return true;
        }

        /**
         * @name openPreview
         * @param key - key of the component
         * @desc close the component detail for the selected component and stop editing
         */
        function openPreview(key: string): void {
            scope.pipeline.preview = {
                open: true,
                selected: key
            };

            pipelineContentEle.style.bottom = '20%';
            pipelinePreviewEle.style.top = '80%';
            pipelinePreviewEle.style.height = '20%';
        }


        /**
         * @name closePreview
         * @desc close the component detail for the selected component and stop editing
         */
        function closePreview(): void {
            scope.pipeline.preview = {
                open: false,
                selected: undefined
            };

            pipelineContentEle.style.bottom = '0%';
            pipelinePreviewEle.style.top = '100%';
            pipelinePreviewEle.style.height = '0%';
        }

        /**
         * @name exportCSV
         * @desc exports the frame as a csv
         * @return {void}
         */
        function exportCSV(): void {
            const key = scope.pipeline.preview.selected;

            scope.widgetCtrl.execute([
                {
                    type: 'Pixel',
                    components: [`Frame(frame=[${scope.pipeline.data[key].output[0].value.name}]) | QueryAll() | ToCsv();`],
                    terminal: true
                }
            ]);
        }

        /**
         * @name initialize
         * @desc initialize the module
         * @returns {void}
         */
        function initialize(): void {
            let leftResizable,
                previewResizable,
                pipelineListener: () => void,
                layout = scope.widgetCtrl.getWidget('view.visualization.layout'),
                headers: { displayName: string }[] = [];

            pipelineEle = ele[0];
            pipelineLeftEle = ele[0].querySelector('#pipeline__left');
            pipelineContentEle = ele[0].querySelector('#pipeline__content');
            pipelineWorkspaceContentEle = ele[0].querySelector('#pipeline__content__workspace');
            pipelinePreviewEle = ele[0].querySelector('#pipeline__preview');

            if (layout) {
                headers = scope.widgetCtrl.getFrame('headers');
            }
            scope.pipeline.headersIn = headers.map(header => header.displayName);

            // used when user visualizes to determine if they can go back to a previous visualization or we need to load a grid
            // close right side menu
            semossCoreService.emit('close-menu');
            // set up the resizable
            leftResizable = Resizable({
                available: ['E'],
                unit: '%',
                content: pipelineLeftEle,
                container: pipelineEle,
                restrict: {
                    minimumWidth: '240px',
                    maximumWidth: '70%'
                },
                on: function (top, left, height, width) {
                    pipelineContentEle.style.left = width + '%';
                    pipelinePreviewEle.style.left = width + '%';
                    pipelinePreviewEle.style.width = 100 - width + '%';
                },
                stop: function () {
                    $timeout();
                }
            });

            previewResizable = Resizable({
                available: ['N'],
                unit: '%',
                content: pipelinePreviewEle,
                container: pipelineEle,
                restrict: {
                    minimumHeight: '20%',
                    maximumHeight: '50%'
                },
                on: function (top) {
                    pipelineContentEle.style.bottom = 100 - top + '%';
                },
                stop: function () {
                    $timeout();
                }
            });

            // set up the plumbing
            plumbing = jsPlumb.jsPlumb.getInstance({
                Container: pipelineWorkspaceContentEle
            });

            pipelineListener = semossCoreService.on('initialize-widgets', initializePipeline);

            initializePipeline();

            // cleanup
            scope.$on('$destroy', function () {
                // save the json, right before we destroy it
                savePipeline();
                // remove movable
                for (let key in scope.pipeline.movable) {
                    if (scope.pipeline.movable.hasOwnProperty(key)) {
                        scope.pipeline.movable[key].destroy();
                    }
                }

                if (leftResizable) {
                    leftResizable.destroy();
                }

                if (previewResizable) {
                    previewResizable.destroy();
                }

                pipelineListener();
            });
        }

        initialize();
    }
}
