'use strict';

import { CONNECTORS, PREVIEW_LIMIT } from '@/core/constants.js';

import './pipeline-external.scss';

export default angular.module('app.pipeline.pipeline-external',
    []
).directive('pipelineExternal', pipelineExternalDirective);

pipelineExternalDirective.$inject = ['semossCoreService'];

function pipelineExternalDirective(semossCoreService) {
    pipelineExternalCtrl.$inject = [];
    pipelineExternalLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        template: require('./pipeline-external.directive.html'),
        scope: {},
        require: ['^widget', '^pipelineComponent'],
        controller: pipelineExternalCtrl,
        controllerAs: 'pipelineExternal',
        bindToController: {
            driver: '@'
        },
        link: pipelineExternalLink
    };

    function pipelineExternalCtrl() {}

    function pipelineExternalLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];
        scope.pipelineComponentCtrl = ctrl[1];

        scope.pipelineExternal.frameType = undefined;
        scope.pipelineExternal.customFrameName = {
            name: '',
            valid: true,
            message: ''
        };

        scope.pipelineExternal.drivers = [];

        scope.pipelineExternal.connection = {
            name: '',
            img: '',
            driver: '',
            type: '',
            url: '',
            direct: false,
            hostname: {
                visible: true,
                value: ''
            },
            port: {
                visible: true,
                value: ''
            },
            schema: {
                visible: true,
                value: ''
            },
            username: {
                visible: true,
                value: ''
            },
            password: {
                visible: true,
                value: ''
            },
            service: {
                visible: true,
                value: ''
            },
            string: '',
            query: ''
        };


        scope.pipelineExternal.updateFrame = updateFrame;
        scope.pipelineExternal.updateExternal = updateExternal;
        scope.pipelineExternal.buildExternal = buildExternal;
        scope.pipelineExternal.previewExternal = previewExternal;
        scope.pipelineExternal.importExternal = importExternal;
        scope.pipelineExternal.validateFrameName = validateFrameName;

        /**
         * @name setFrameData
         * @desc set the frame type
         * @return {void}
         */
        function setFrameData() {
            scope.pipelineExternal.frameType = scope.widgetCtrl.getOptions('initialFrameType');
        }

        /**
         * @name updateFrame
         * @param {string} frame - frame
         * @desc update the frame type
         * @return {void}
         */
        function updateFrame(frame) {
            scope.widgetCtrl.setOptions('initialFrameType', frame);
        }


        /**
         * @name setExternal
         * @desc set the initial data
         * @returns {void}
         */
        function setExternal() {
            var connector,
                qsComponent;

            // all of the options
            scope.pipelineExternal.drivers = [];
            for (connector in CONNECTORS) {
                if (CONNECTORS.hasOwnProperty(connector)) {
                    scope.pipelineExternal.drivers.push({
                        'display': String(CONNECTORS[connector].name).replace(/_/g, ' '),
                        'value': CONNECTORS[connector].driver,
                        'image': CONNECTORS[connector].image
                    });
                }
            }

            // if there is a qsComponent and it is valid, we will use that
            qsComponent = scope.pipelineComponentCtrl.getComponent('parameters.QUERY_STRUCT.value');
            if (qsComponent && qsComponent.hasOwnProperty('dbDriver')) {
                scope.pipelineExternal.driver = qsComponent.dbDriver;
            }

            // validate that the driver is possible
            if (!CONNECTORS.hasOwnProperty(scope.pipelineExternal.driver)) {
                scope.pipelineExternal.driver = false;
            }

            // select one if there is nothing selected
            if (!scope.pipelineExternal.driver) {
                scope.pipelineExternal.driver = scope.pipelineExternal.drivers[0].value;
            }

            updateExternal();
        }

        /**
         * @name updateExternal
         * @desc update the external data
         * @return {void}
         */
        function updateExternal() {
            var selectedConnector = CONNECTORS[scope.pipelineExternal.driver];

            scope.pipelineExternal.connection.name = selectedConnector.name;
            scope.pipelineExternal.connection.image = selectedConnector.image;
            scope.pipelineExternal.connection.driver = selectedConnector.driver;
            scope.pipelineExternal.connection.type = selectedConnector.type;
            scope.pipelineExternal.connection.url = selectedConnector.url;

            scope.pipelineExternal.connection.hostname.visible = selectedConnector.url.indexOf('HOSTNAME') > -1;
            scope.pipelineExternal.connection.port.visible = selectedConnector.url.indexOf('PORT') > -1;
            scope.pipelineExternal.connection.schema.visible = selectedConnector.url.indexOf('SCHEMA') > -1;
            scope.pipelineExternal.connection.service.visible = selectedConnector.url.indexOf('SERVICE') > -1;

            scope.pipelineExternal.customFrameName.name = scope.pipelineComponentCtrl.createFrameName(scope.pipelineExternal.connection.driver);

            buildExternal();
        }

        /**
         * @name buildExternal
         * @desc build the external connection string
         * @returns {void}
         */
        function buildExternal() {
            var connectionString = scope.pipelineExternal.connection.url;

            if (scope.pipelineExternal.connection.hostname.visible && scope.pipelineExternal.connection.hostname.value) {
                connectionString = connectionString.replace('<HOSTNAME>', scope.pipelineExternal.connection.hostname.value);
            }
            if (scope.pipelineExternal.connection.port.visible && scope.pipelineExternal.connection.port.value) {
                connectionString = connectionString.replace('<PORT>', scope.pipelineExternal.connection.port.value);
            }
            if (scope.pipelineExternal.connection.schema.visible && scope.pipelineExternal.connection.schema.value) {
                connectionString = connectionString.replace('<SCHEMA>', scope.pipelineExternal.connection.schema.value);
            }
            if (scope.pipelineExternal.connection.service.visible && scope.pipelineExternal.connection.service.value) {
                connectionString = connectionString.replace('<SERVICE>', scope.pipelineExternal.connection.service.value);
            }

            connectionString = connectionString.replace('<USERNAME>', scope.pipelineExternal.connection.username.value);
            connectionString = connectionString.replace('<PASSWORD>', scope.pipelineExternal.connection.password.value);

            scope.pipelineExternal.connection.string = connectionString;
        }

        /**
         * @name validateExternal
         * @desc validate the connection form is valid
         * @return {boolean} is the connection valid or not?
         */
        function validateExternal() {
            if (!scope.pipelineExternal.connection.direct) {
                if (scope.pipelineExternal.connection.hostname.visible && !scope.pipelineExternal.connection.hostname.value) {
                    scope.widgetCtrl.alert('warn', 'Missing Hostname. Please enter value.');
                    return false;
                }
                if (scope.pipelineExternal.connection.port.visible && !scope.pipelineExternal.connection.port.value) {
                    scope.widgetCtrl.alert('warn', 'Missing Port. Please enter value.');
                    return false;
                }
                if (scope.pipelineExternal.connection.schema.visible && !scope.pipelineExternal.connection.schema.value) {
                    scope.widgetCtrl.alert('warn', 'Missing Schema. Please enter value.');
                    return false;
                }
                if (scope.pipelineExternal.connection.service.visible && !scope.pipelineExternal.connection.service.value) {
                    scope.widgetCtrl.alert('warn', 'Missing Service. Please enter value.');
                    return false;
                }
            } else if (!scope.pipelineExternal.connection.string) {
                scope.widgetCtrl.alert('warn', 'Missing Connection String. Please enter value.');
                return false;
            }

            if (!scope.pipelineExternal.connection.query) {
                scope.widgetCtrl.alert('warn', 'Missing Query. Please enter value.');
                return false;
            }

            return true;
        }

        /**
         * @name previewExternal
         * @desc preview the external
         * @returns {void}
         */
        function previewExternal() {
            var parameters = {},
                valid = true;

            if (!validateExternal()) {
                valid = false;
            }

            if (valid) {
                parameters = buildParameters(true);
            }
            scope.pipelineComponentCtrl.previewComponent(parameters);
        }

        /**
         * @name importExternal
         * @param {boolean} visualize 0 if true visualize frame
         * @desc import the query
         * @returns {void}
         */
        function importExternal(visualize) {
            let parameters, callback;

            if (!validateExternal()) {
                return;
            }

            parameters = buildParameters();

            if (visualize) {
                callback = scope.pipelineComponentCtrl.visualizeComponent;
            }

            scope.pipelineComponentCtrl.executeComponent(parameters, {
                name: scope.pipelineExternal.connection.name,
                icon: scope.pipelineExternal.connection.image
            }, callback);
        }

        /**
         * @name buildParameters
         * @param {boolean} preview - true if coming from preview
         * @desc build the parameters for the current module
         * @returns {object} - map of the paramters to value
         */
        function buildParameters(preview) {
            return {
                'IMPORT_FRAME': {
                    'name': scope.pipelineComponentCtrl.getComponent('parameters.IMPORT_FRAME.value.name') || scope.pipelineExternal.customFrameName.name,
                    'type': scope.pipelineComponentCtrl.getComponent('parameters.IMPORT_FRAME.value.type') || scope.widgetCtrl.getOptions('initialFrameType'),
                    'override': true
                },
                'QUERY_STRUCT': {
                    'qsType': 'RAW_JDBC_ENGINE_QUERY',
                    'limit': preview ? PREVIEW_LIMIT : -1,
                    'config': {
                        'dbDriver': scope.pipelineExternal.connection.driver,
                        'connectionString': scope.pipelineExternal.connection.string,
                        'username': scope.pipelineExternal.connection.username.value,
                        'password': scope.pipelineExternal.connection.password.value
                    },
                    'query': scope.pipelineExternal.connection.query
                }
            };
        }

        /**
         * @name validateFrameName
         * @desc checks if the frame name entered by the user is valid
         * @returns {void}
         */
        function validateFrameName() {
            let results = scope.pipelineComponentCtrl.validateFrameName(scope.pipelineExternal.customFrameName.name);
            scope.pipelineExternal.customFrameName.valid = results.valid;
            scope.pipelineExternal.customFrameName.message = results.message;
        }

        /**
         * @name initialize
         * @desc initialize the module
         * @returns {void}
         */
        function initialize() {
            setFrameData();
            setExternal();
            scope.$on('$destroy', function () {});
            scope.pipelineExternal.customFrameName.name = scope.pipelineComponentCtrl.createFrameName(scope.pipelineExternal.connection.driver);
        }

        initialize();
    }
}