'use strict';

import angular from 'angular';
import jsPlumb from 'jsplumb';
import panzoom from 'panzoom';
import {
    PREVIEW_LIMIT
} from '@/core/constants.js';
import Resizable from '../../core/utility/resizable.ts';

import './pipeline-app.scss';

import './pipeline-app-filter/pipeline-app-filter.directive.ts';

export default angular.module('app.pipeline.pipeline-app',
    ['app.pipeline.pipeline-app.filter']
).directive('pipelineApp', pipelineAppDirective);

pipelineAppDirective.$inject = ['$filter', '$timeout', '$compile', 'ENDPOINT', 'semossCoreService', 'CONFIG'];

function pipelineAppDirective($filter, $timeout, $compile, ENDPOINT, semossCoreService, CONFIG) {
    pipelineAppCtrl.$inject = [];
    pipelineAppLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        template: require('./pipeline-app.directive.html'),
        scope: {},
        require: ['^widget', '^pipelineComponent'],
        controller: pipelineAppCtrl,
        controllerAs: 'pipelineApp',
        bindToController: {},
        link: pipelineAppLink
    };

    function pipelineAppCtrl() {}

    function pipelineAppLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];
        scope.pipelineComponentCtrl = ctrl[1];

        var aliasTimeout,
            resizable,
            contentEle,
            leftEle,
            rightEle;

        scope.pipelineApp.valid = false;
        scope.pipelineApp.view = 'list';
        scope.pipelineApp.cache = false;
        scope.pipelineApp.CONFIG = CONFIG;

        scope.pipelineApp.frameType = undefined;
        scope.pipelineApp.customFrameName = {
            name: '',
            valid: true,
            message: ''
        };
        scope.pipelineApp.availableFrames = [];

        scope.pipelineApp.recommendations = {};

        scope.pipelineApp.structure = {
            searchTerm: '',
            loading: false,
            open: false,
            tables: [],
            edges: []
        };

        scope.pipelineApp.metamodel = {
            plumbing: undefined,
            zoom: undefined,
            graph: undefined,
            scope: undefined

        };

        scope.pipelineApp.traversal = {
            selectedQueueTableIdx: -1,
            selectedQueueTable: undefined,
            searchTerm: '',
            options: [],
            searched: [],
            toggle: true
        };

        scope.pipelineApp.selected = {
            queue: []
        };

        scope.pipelineApp.filter = {
            selected: '',
            headers: [],
            comparator: '',
            comparatorOptions: [],
            new: {}
        };

        scope.pipelineApp.joinOptions = {
            'inner.join': {
                display: 'Inner Join',
                img: require('images/inner_join.svg')
            },
            'left.outer.join': {
                display: 'Left Join',
                img: require('images/left_join.svg')
            },
            'right.outer.join': {
                display: 'Right Join',
                img: require('images/right_join.svg')
            },
            'outer.join': {
                display: 'Outer Join',
                img: require('images/outer_join.svg')
            }
        };
        scope.pipelineApp.taskToRemove = '';

        scope.pipelineApp.updateFrame = updateFrame;
        scope.pipelineApp.updateApp = updateApp;
        scope.pipelineApp.updateData = updateData;
        scope.pipelineApp.updateView = updateView;
        scope.pipelineApp.updateTraversal = updateTraversal;
        scope.pipelineApp.searchTraversal = searchTraversal;
        scope.pipelineApp.toggleTraversal = toggleTraversal;
        scope.pipelineApp.checkTraversalToggle = checkTraversalToggle;
        scope.pipelineApp.addAllSelected = addAllSelected;
        scope.pipelineApp.addSelected = addSelected;
        scope.pipelineApp.removeSelected = removeSelected;
        scope.pipelineApp.updateSelectedAlias = updateSelectedAlias;
        scope.pipelineApp.previewSelected = previewSelected;
        scope.pipelineApp.showFilter = showFilter;
        scope.pipelineApp.hideFilter = hideFilter;
        scope.pipelineApp.previewSelected = previewSelected;
        scope.pipelineApp.removeFilter = removeFilter;
        scope.pipelineApp.searchMetamodel = searchMetamodel;
        scope.pipelineApp.selectTable = selectTable;
        scope.pipelineApp.getTableType = getTableType;
        scope.pipelineApp.selectColumn = selectColumn;
        scope.pipelineApp.getColumnType = getColumnType;
        scope.pipelineApp.getColumnButtonClass = getColumnButtonClass;
        scope.pipelineApp.importQueue = importQueue;
        scope.pipelineApp.importAuto = importAuto;
        scope.pipelineApp.exit = exit;
        scope.pipelineApp.validateFrameName = validateFrameName;

        /**
         * @name setInitialFrameType
         * @desc set the frame type
         * @return {void}
         */
        function setInitialFrameType() {
            scope.pipelineApp.frameType = scope.widgetCtrl.getOptions('initialFrameType');
        }

        /**
         * @name updateFrame
         * @param {string} type - new frame type
         * @desc update the frame type
         * @return {void}
         */
        function updateFrame(type) {
            scope.widgetCtrl.setOptions('initialFrameType', type);
        }

        /**
         * @name getFrames
         * @desc gets the available (active) frames
         * @returns {void}
         */
        function getFrames() {
            var frames = scope.widgetCtrl.getShared('frames'),
                frame;

            scope.pipelineApp.availableFrames = [];

            for (frame in frames) {
                if (frames.hasOwnProperty(frame)) {
                    scope.pipelineApp.availableFrames.push(frames[frame]);
                }
            }

            getFrameRecommendations();
        }

        /**
         * @name getFrameRecommendations
         * @desc get recommendations for the frames (this tells us what we can link too)
         * @returns {void}
         */
        function getFrameRecommendations() {
            let pixel,
                callback;

            // clear it out
            scope.pipelineApp.recommendations = {};

            pixel = '';
            for (let frameIdx = 0, frameLen = scope.pipelineApp.availableFrames.length; frameIdx < frameLen; frameIdx++) {
                pixel += `${scope.pipelineApp.availableFrames[frameIdx].name} | GetFrameDatabaseJoins();`;
            }

            if (pixel.length === 0) {
                return;
            }

            callback = (response) => {
                let merged = {};
                for (let returnIdx = 0, returnLen = response.pixelReturn.length; returnIdx < returnLen; returnIdx++) {
                    let output = response.pixelReturn[returnIdx].output,
                        type = response.pixelReturn[returnIdx].operationType;

                    if (type.indexOf('ERROR') > -1) {
                        return;
                    }

                    for (let app in output) {
                        if (output.hasOwnProperty(app)) {
                            // add the app if it doesn't exist
                            if (!merged.hasOwnProperty(app)) {
                                merged[app] = {};
                            }

                            // look at all the concepts
                            for (let concept in output[app]) {
                                if (output[app].hasOwnProperty(concept)) {
                                    // add the concept if it doesn't exist
                                    if (!merged[app].hasOwnProperty(concept)) {
                                        merged[app][concept] = [];
                                    }


                                    // add in all of the frames, its safe to assume that each frame/colun is unique
                                    for (let frameIdx = 0, frameLen = output[app][concept].length; frameIdx < frameLen; frameIdx++) {
                                        merged[app][concept].push(output[app][concept][frameIdx]);
                                    }
                                }
                            }
                        }
                    }
                }

                for (let app in merged) {
                    if (merged.hasOwnProperty(app)) {
                        // look at all the concepts
                        for (let concept in merged[app]) {
                            if (merged[app].hasOwnProperty(concept)) {
                                let message = 'This concept may connect to others that have already been imported (';

                                // add in all of the frames, its safe to assume that each frame/colun is unique
                                for (let frameIdx = 0, frameLen = merged[app][concept].length; frameIdx < frameLen; frameIdx++) {
                                    if (merged[app][concept][frameIdx]) {
                                        let split = merged[app][concept][frameIdx].split('__');
                                        message += `Table ${split[0]} with ${split[1]}, `;
                                    }
                                }

                                // clean up the end
                                message = message.slice(0, -2);
                                message += ')';

                                merged[app][concept] = message;
                            }
                        }
                    }
                }


                // for(let app)
                scope.pipelineApp.recommendations = merged;
            };


            // execute a meta query
            scope.widgetCtrl.meta([{
                type: 'Pixel',
                components: [
                    pixel
                ],
                terminal: true
            }], callback, []);
        }

        /**
         * @name getApps
         * @desc set the initial data
         * @returns {void}
         */
        function getApps() {
            const previousSelected = scope.pipelineComponentCtrl.getComponent('parameters.SELECTED.value');
            let selectedApp = semossCoreService.app.get('selectedApp'),
                keepSelected = false,
                qsComponent,
                reset = true;

            qsComponent = scope.pipelineComponentCtrl.getComponent('parameters.QUERY_STRUCT.value');

            if (scope.pipelineComponentCtrl.getComponent('parameters.SELECTED_APP.value')) {
                scope.pipelineApp.selectedApp = scope.pipelineComponentCtrl.getComponent('parameters.SELECTED_APP.value');
                if (qsComponent && qsComponent.engineName) {
                    reset = false;
                    keepSelected = true;
                    scope.pipelineApp.valid = true;
                }
            } else if (selectedApp && selectedApp !== 'NEWSEMOSSAPP') {
                scope.pipelineApp.selectedApp = selectedApp;
            } else if (!scope.pipelineApp.selectedApp) {
                reset = false;
                keepSelected = false;
            }

            if (previousSelected && Object.keys(previousSelected).length > 0) {
                scope.pipelineApp.selected = previousSelected.selected;
                scope.pipelineApp.traversal = previousSelected.traversal;
                scope.pipelineApp.structure = previousSelected.structure;
                previewSelected(false);
            } else if (reset || keepSelected) {
                updateApp(reset);
            }
        }

        /**
         * @name updateApp
         * @desc update the information based on the selected APP
         * @param {boolean} reset - should we reset the information or use the information from the QUERY_STRUCT
         * @returns {void}
         */
        function updateApp(reset) {
            var callback;
            scope.pipelineApp.customFrameName.name = scope.pipelineComponentCtrl.createFrameName(scope.pipelineApp.selectedApp.display);
            scope.pipelineApp.structure = {
                tables: {},
                edges: []
            };

            // register message to come back to
            callback = function (response) {
                var tableStructure = response.pixelReturn[0].output,
                    metamodel = response.pixelReturn[1].output,
                    nodeIdx,
                    nodeLen,
                    table,
                    edgeIdx,
                    edgeLen,
                    concept;

                if (metamodel.nodes) {
                    for (nodeIdx = 0, nodeLen = metamodel.nodes.length; nodeIdx < nodeLen; nodeIdx++) {
                        table = {
                            alias: String(metamodel.nodes[nodeIdx].conceptualName).replace(/_/g, ' '),
                            table: metamodel.nodes[nodeIdx].conceptualName,
                            position: metamodel.positions && metamodel.positions[metamodel.nodes[nodeIdx].conceptualName] ? metamodel.positions[metamodel.nodes[nodeIdx].conceptualName] : {
                                top: 0,
                                left: 0
                            },
                            columns: {}
                        };

                        concept = metamodel.nodes[nodeIdx].conceptualName;

                        for (let i = 0; i < tableStructure.length; i++) {
                            if (table.table !== tableStructure[i][0]) continue;
                            let columnQs,
                                columnAlias,
                                columnType,
                                columnPk;
                            // 0 index = table
                            // 1 index = column
                            // 2 index = data type
                            // 3 index = isTable?
                            if (tableStructure[i][3]) {
                                columnQs = tableStructure[i][1];
                                columnAlias = String(tableStructure[i][1]).replace(/_/g, ' ');
                                columnPk = true;
                            } else {
                                columnQs = tableStructure[i][0] + '__' + tableStructure[i][1];
                                columnAlias = String(tableStructure[i][1]).replace(/_/g, ' ');
                                columnPk = false;
                            }

                            columnType = getTypeInformation(metamodel.dataTypes[concept], metamodel.additionalDataTypes[concept]);

                            table.columns[columnAlias] = {
                                alias: columnAlias,
                                table: tableStructure[i][0],
                                column: tableStructure[i][1],
                                concept: columnQs,
                                isPrimKey: columnPk,
                                type: tableStructure[i][2],
                                typeFormat: columnType.typeFormat
                            };
                        }

                        scope.pipelineApp.structure.tables[table.table] = table;
                    }
                }

                if (metamodel.edges) {
                    for (edgeIdx = 0, edgeLen = metamodel.edges.length; edgeIdx < edgeLen; edgeIdx++) {
                        scope.pipelineApp.structure.edges.push({
                            source: metamodel.edges[edgeIdx].source,
                            target: metamodel.edges[edgeIdx].target
                        });
                    }
                }

                updateData(reset);

                // update view
                updateView(scope.pipelineApp.view);
            };

            scope.widgetCtrl.query([{
                type: 'getDatabaseTableStructure',
                components: [scope.pipelineApp.selectedApp.value],
                terminal: true
            },
            {
                type: 'getDatabaseMetamodel',
                components: [
                    scope.pipelineApp.selectedApp.value,
                    ['dataTypes', 'positions']
                ],
                terminal: true,
                meta: true
            }
            ], callback);
        }

        /**
         * @name updateData
         * @desc set the selected information (and grab it from QUERY_STRUCT)
         * @param {boolean} reset - should we reset the information or use the information from the QUERY_STRUCT
         * @returns {void}
         */
        function updateData(reset) {
            var qsComponent,
                addedTableIdx,
                addedTables,
                addedColumnIdx,
                addedColumn,
                selectorIdx,
                selectorLen,
                relations,
                relationIdx,
                tableIdx,
                explicitFilters,
                filterIdx,
                filterLen,
                filterValue;

            scope.pipelineApp.selected = {
                queue: []
            };

            scope.pipelineApp.filter = {
                selected: '',
                headers: [],
                comparator: '',
                comparatorOptions: [],
                new: {}
            };

            if (!reset) {
                qsComponent = scope.pipelineComponentCtrl.getComponent('parameters.QUERY_STRUCT.value');

                // grab the selected information from the traversal options
                // (we use this because the QUERY_STRUCT doesn't contain all of the required information)
                addedTables = {};
                for (selectorIdx = 0, selectorLen = qsComponent.selectors.length; selectorIdx < selectorLen; selectorIdx++) {
                    if (qsComponent.selectors[selectorIdx].type === 'COLUMN') {
                        // add in the table to store the location
                        if (!addedTables.hasOwnProperty(qsComponent.selectors[selectorIdx].content.table)) {
                            addedTableIdx = scope.pipelineApp.selected.queue.length;

                            scope.pipelineApp.selected.queue.push({
                                idx: addedTableIdx,
                                table: qsComponent.selectors[selectorIdx].content.table,
                                fromTable: undefined,
                                toTable: undefined,
                                joinType: undefined,
                                columns: []
                            });

                            addedTables[qsComponent.selectors[selectorIdx].content.table] = {
                                idx: addedTableIdx,
                                columns: {}
                            };
                        }

                        if (scope.pipelineApp.structure.tables.hasOwnProperty(qsComponent.selectors[selectorIdx].content.table)) {
                            if (qsComponent.selectors[selectorIdx].content.column === 'PRIM_KEY_PLACEHOLDER') {
                                addedColumn = scope.pipelineApp.structure.tables[qsComponent.selectors[selectorIdx].content.table].columns[qsComponent.selectors[selectorIdx].content.table];
                            } else {
                                addedColumn = scope.pipelineApp.structure.tables[qsComponent.selectors[selectorIdx].content.table].columns[qsComponent.selectors[selectorIdx].content.column];
                            }

                            if (typeof addedColumn !== 'undefined') {
                                addedTables[addedColumn.table].columns[addedColumn.column] = {
                                    idx: scope.pipelineApp.selected.queue[addedTables[addedColumn.table].idx].columns.length
                                };

                                scope.pipelineApp.selected.queue[addedTables[addedColumn.table].idx].columns.push({
                                    alias: qsComponent.selectors[selectorIdx].content.alias,
                                    table: addedColumn.table,
                                    column: addedColumn.column,
                                    concept: addedColumn.concept,
                                    isPrimKey: addedColumn.isPrimKey,
                                    conceptualName: addedColumn.conceptualName,
                                    type: addedColumn.type,
                                    typeFormat: addedColumn.typeFormat,
                                    hasAliasError: false,
                                    filters: [],
                                    havings: [],
                                    orders: []
                                });
                            }
                        }
                    }
                }


                // we go backwards and associate the join with the first table that is there
                relations = qsComponent.relations ? JSON.parse(qsComponent.relations) : [];
                for (relationIdx = relations.length - 1; relationIdx >= 0; relationIdx--) {
                    for (tableIdx = scope.pipelineApp.selected.queue.length - 1; tableIdx >= 0; tableIdx--) {
                        if (scope.pipelineApp.selected.queue[tableIdx].table === relations[relationIdx][2] || scope.pipelineApp.selected.queue[tableIdx].table === relations[relationIdx][0]) {
                            if (typeof scope.pipelineApp.selected.queue[tableIdx].fromTable !== 'undefined' || typeof scope.pipelineApp.selected.queue[tableIdx].toTable !== 'undefined') {
                                console.warn('Relations mapping is not correct for ' + scope.pipelineApp.selected.queue[tableIdx].table);
                            }

                            scope.pipelineApp.selected.queue[tableIdx].fromTable = relations[relationIdx][0];
                            scope.pipelineApp.selected.queue[tableIdx].joinType = relations[relationIdx][1];
                            scope.pipelineApp.selected.queue[tableIdx].toTable = relations[relationIdx][2];
                            break;
                        }
                    }
                }

                // add in the filters... this is specific. I should make it more generic
                explicitFilters = qsComponent.explicitFilters ? qsComponent.explicitFilters : [];
                for (filterIdx = 0, filterLen = explicitFilters.length - 1; filterIdx < filterLen; filterIdx += 2) {
                    if (explicitFilters[filterIdx] === 'SIMPLE') {
                        if (explicitFilters[filterIdx + 1].left.pixelType === 'COLUMN' &&
                            explicitFilters[filterIdx + 1].left.value[0].type === 'COLUMN' &&
                            (
                                explicitFilters[filterIdx + 1].right.pixelType === 'CONSTANT' ||
                                explicitFilters[filterIdx + 1].right.pixelType === 'CONST_STRING' ||
                                explicitFilters[filterIdx + 1].right.pixelType === 'LAMBDA'
                            )) {
                            if (addedTables.hasOwnProperty(explicitFilters[filterIdx + 1].left.value[0].content.table)) {
                                addedTableIdx = addedTables[explicitFilters[filterIdx + 1].left.value[0].content.table].idx;

                                if (explicitFilters[filterIdx + 1].left.value[0].content.column === 'PRIM_KEY_PLACEHOLDER') {
                                    addedColumnIdx = addedTables[explicitFilters[filterIdx + 1].left.value[0].content.table].columns[explicitFilters[filterIdx + 1].left.value[0].content.table];
                                } else {
                                    addedColumnIdx = addedTables[explicitFilters[filterIdx + 1].left.value[0].content.table].columns[explicitFilters[filterIdx + 1].left.value[0].content.column];
                                }

                                if (typeof addedColumnIdx !== 'undefined') {
                                    if (explicitFilters[filterIdx + 1].comparator === '<' || explicitFilters[filterIdx + 1].comparator === '<=' || explicitFilters[filterIdx + 1].comparator === '>' || explicitFilters[filterIdx + 1].comparator === '>=') {
                                        filterValue = explicitFilters[filterIdx + 1].right.value[0];
                                    } else {
                                        filterValue = explicitFilters[filterIdx + 1].right.value;
                                    }

                                    scope.pipelineApp.selected.queue[addedTableIdx].columns[addedColumnIdx.idx].filters.push({
                                        pixelType: explicitFilters[filterIdx + 1].right.pixelType,
                                        comparator: explicitFilters[filterIdx + 1].comparator,
                                        value: filterValue
                                    });
                                }
                            }
                        }
                    }
                }
            }

            // clear
            scope.pipelineApp.filter = {
                selected: '',
                headers: [],
                comparator: '',
                comparatorOptions: [],
                new: {}
            };


            // update the traversals
            updateTraversal(scope.pipelineApp.selected.queue.length - 1);

            // validate that the available are correct (naming)
            validateSelected();

            // update preview
            previewSelected(false);
        }

        /**
         * @name updateView
         * @desc update the view based on selection
         * @param {string} view - view to switch to
         * @returns {void}
         */
        function updateView(view) {
            scope.pipelineApp.view = view;

            if (scope.pipelineApp.view === 'list') {
                closeMetamodel();
            } else if (scope.pipelineApp.view === 'metamodel') {
                openMetamodel();
            }
        }

        /**
         * @name updateTraversal
         * @desc updates the traversal based on the selection
         * @param {number} selectedQueueTableIdx - index of the selected traversal 
         * @returns {void}
         */
        function updateTraversal(selectedQueueTableIdx) {
            var upstreamTables = {},
                downstreamTables = {};

            scope.pipelineApp.traversal = {
                selectedQueueTableIdx: selectedQueueTableIdx,
                selectedQueueTable: selectedQueueTableIdx === -1 ? undefined : scope.pipelineApp.selected.queue[selectedQueueTableIdx],
                searchTerm: '',
                options: [],
                searched: [],
                toggle: true
            };

            // we can load everything, we are starting from fresh
            if (scope.pipelineApp.traversal.selectedQueueTableIdx === -1) {
                for (let table in scope.pipelineApp.structure.tables) {
                    if (scope.pipelineApp.structure.tables.hasOwnProperty(table)) {
                        let tableObj = {
                            alias: scope.pipelineApp.structure.tables[table].alias,
                            table: scope.pipelineApp.structure.tables[table].table,
                            columns: [],
                            open: true,
                            direction: undefined,
                            fromTable: undefined,
                            toTable: undefined
                        };

                        for (let column in scope.pipelineApp.structure.tables[table].columns) {
                            if (scope.pipelineApp.structure.tables[table].columns.hasOwnProperty(column)) {
                                tableObj.columns.push(scope.pipelineApp.structure.tables[table].columns[column]);
                            }
                        }

                        tableObj.columns = semossCoreService.utility.sort(tableObj.columns, 'alias');

                        scope.pipelineApp.traversal.options.push(tableObj);
                    }
                }

                scope.pipelineApp.traversal.options = semossCoreService.utility.sort(scope.pipelineApp.traversal.options, 'alias');

                // filter with the search
                searchTraversal();

                return;
            }

            // look at the connetions, if one of them matches the traversed table, we can connect to the other one
            for (let edgeIdx = 0, edgeLen = scope.pipelineApp.structure.edges.length; edgeIdx < edgeLen; edgeIdx++) {
                // the target is DOWNSTREAM from our traversed table
                if (scope.pipelineApp.structure.edges[edgeIdx].source === scope.pipelineApp.traversal.selectedQueueTable.table) {
                    downstreamTables[scope.pipelineApp.structure.edges[edgeIdx].target] = true;
                }

                // the source is UPSTREAM from our traversed table
                if (scope.pipelineApp.structure.edges[edgeIdx].target === scope.pipelineApp.traversal.selectedQueueTable.table) {
                    upstreamTables[scope.pipelineApp.structure.edges[edgeIdx].source] = true;
                }
            }

            // now we, need to construct the object
            for (let table in scope.pipelineApp.structure.tables) {
                if (scope.pipelineApp.structure.tables.hasOwnProperty(table)) {
                    // the table is there add it in as an upstream table
                    if (upstreamTables.hasOwnProperty(scope.pipelineApp.structure.tables[table].table)) {
                        let tableObj = {
                            alias: scope.pipelineApp.structure.tables[table].alias,
                            table: scope.pipelineApp.structure.tables[table].table,
                            columns: [],
                            open: true,
                            direction: 'upstream',
                            fromTable: scope.pipelineApp.structure.tables[table].table,
                            toTable: scope.pipelineApp.traversal.selectedQueueTable.table
                        };

                        for (let column in scope.pipelineApp.structure.tables[table].columns) {
                            if (scope.pipelineApp.structure.tables[table].columns.hasOwnProperty(column)) {
                                tableObj.columns.push(scope.pipelineApp.structure.tables[table].columns[column]);
                            }
                        }

                        tableObj.columns = semossCoreService.utility.sort(tableObj.columns, 'alias');

                        scope.pipelineApp.traversal.options.push(tableObj);
                    }
                }

                // the table is there add it in as an downstream table
                if (downstreamTables.hasOwnProperty(scope.pipelineApp.structure.tables[table].table)) {
                    let tableObj = {
                        alias: scope.pipelineApp.structure.tables[table].alias,
                        table: scope.pipelineApp.structure.tables[table].table,
                        columns: [],
                        open: true,
                        direction: 'downstream',
                        fromTable: scope.pipelineApp.traversal.selectedQueueTable.table,
                        toTable: scope.pipelineApp.structure.tables[table].table
                    };

                    for (let column in scope.pipelineApp.structure.tables[table].columns) {
                        if (scope.pipelineApp.structure.tables[table].columns.hasOwnProperty(column)) {
                            tableObj.columns.push(scope.pipelineApp.structure.tables[table].columns[column]);
                        }
                    }

                    tableObj.columns = semossCoreService.utility.sort(tableObj.columns, 'alias');

                    scope.pipelineApp.traversal.options.push(tableObj);
                }

                // same table add it in
                if (scope.pipelineApp.traversal.selectedQueueTable.table === scope.pipelineApp.structure.tables[table].table) {
                    let tableObj = {
                        alias: scope.pipelineApp.structure.tables[table].alias,
                        table: scope.pipelineApp.structure.tables[table].table,
                        columns: [],
                        open: true,
                        direction: undefined,
                        fromTable: undefined,
                        toTable: undefined
                    };

                    for (let column in scope.pipelineApp.structure.tables[table].columns) {
                        if (scope.pipelineApp.structure.tables[table].columns.hasOwnProperty(column)) {
                            tableObj.columns.push(scope.pipelineApp.structure.tables[table].columns[column]);
                        }
                    }

                    tableObj.columns = semossCoreService.utility.sort(tableObj.columns, 'alias');

                    scope.pipelineApp.traversal.options.push(tableObj);
                }
            }


            scope.pipelineApp.traversal.options = semossCoreService.utility.sort(scope.pipelineApp.traversal.options, 'alias');

            // filter with the search
            searchTraversal();
        }

        /**
         * @name searchTraversal
         * @desc search the traverse list and update the options
         * @returns {void}
         */
        function searchTraversal() {
            var cleanedSearch,
                tableIdx,
                tableLen,
                table;

            // if nothing is searched we can just used the original
            if (!scope.pipelineApp.traversal.searchTerm) {
                scope.pipelineApp.traversal.searched = semossCoreService.utility.freeze(scope.pipelineApp.traversal.options);
                return;
            }

            // if it is searched, we only check the column
            cleanedSearch = String(scope.pipelineApp.traversal.searchTerm).replace(/_/g, ' '); // name is already w/o spaces

            scope.pipelineApp.traversal.searched = [];
            for (tableIdx = 0, tableLen = scope.pipelineApp.traversal.options.length; tableIdx < tableLen; tableIdx++) {
                table = semossCoreService.utility.freeze(scope.pipelineApp.traversal.options[tableIdx]);
                table.columns = $filter('filter')(semossCoreService.utility.freeze(table.columns), {
                    alias: cleanedSearch
                });

                if (table.columns.length > 0) {
                    scope.pipelineApp.traversal.searched.push(table);
                }
            }
        }

        /**
         * @name toggleTraversal
         * @desc toggle the traversal tables to be open/close
         * @returns {void}
         */
        function toggleTraversal() {
            var tableIdx,
                tableLen;

            scope.pipelineApp.traversal.toggle = !scope.pipelineApp.traversal.toggle;

            for (tableIdx = 0, tableLen = scope.pipelineApp.traversal.searched.length; tableIdx < tableLen; tableIdx++) {
                scope.pipelineApp.traversal.searched[tableIdx].open = scope.pipelineApp.traversal.toggle;
            }
        }

        /**
         * @name checkTraversalToggle
         * @desc check to see if any of the traversal tables are expanded and set the toggle accordingly
         * @returns {void}
         */
        function checkTraversalToggle() {
            var tableIdx,
                tableLen;


            for (tableIdx = 0, tableLen = scope.pipelineApp.traversal.options.length; tableIdx < tableLen; tableIdx++) {
                if (scope.pipelineApp.traversal.options[tableIdx].open) {
                    scope.pipelineApp.traversal.toggle = true;
                    return;
                }
            }

            scope.pipelineApp.traversal.toggle = false;
        }


        /**
         * @name addAllSelected
         * @desc add all the whole traversal table
         * @param {index} selectedTableIdx - traversal table to add
         * @returns {void}
         */
        function addAllSelected(selectedTableIdx) {
            const table = scope.pipelineApp.traversal.searched[selectedTableIdx];

            table.columns.forEach((col) => {
                addSelected(table, col, true);
            });

            // update the traversals (if its the 1st one)
            if (scope.pipelineApp.traversal.selectedQueueTableIdx === -1) {
                updateTraversal(scope.pipelineApp.selected.queue.length - 1);
            }

            // validate that the available are correct (naming)
            validateSelected();

            previewSelected(false);
        }

        /**
         * @name addSelected
         * @desc add the column to the selected queue
         * @param {object} tableObj - add table object
         * @param {object} columnObj - add column object
         * @param {boolean} block - block updating the view
         * @returns {void}
         */
        function addSelected(tableObj, columnObj, block) {
            var addedTableIdx,
                tableIdx,
                tableLen;

            // cleat out filter
            scope.pipelineApp.filter = {
                selected: '',
                headers: [],
                comparator: '',
                comparatorOptions: [],
                new: {}
            };

            // if it is already added, we add it to the same table (we cannot do 'merges')
            addedTableIdx = -1;
            for (tableIdx = 0, tableLen = scope.pipelineApp.selected.queue.length; tableIdx < tableLen; tableIdx++) {
                if (scope.pipelineApp.selected.queue[tableIdx].table === columnObj.table) {
                    addedTableIdx = tableIdx;
                    break;
                }
            }

            // add a new one
            if (addedTableIdx === -1) {
                scope.pipelineApp.selected.queue.push({
                    idx: scope.pipelineApp.selected.queue.length,
                    table: columnObj.table,
                    fromTable: tableObj.fromTable,
                    toTable: tableObj.toTable,
                    joinType: 'inner.join',
                    columns: []
                });

                addedTableIdx = scope.pipelineApp.selected.queue.length - 1;
            }

            // add a column
            scope.pipelineApp.selected.queue[addedTableIdx].columns.push({
                alias: columnObj.alias,
                table: columnObj.table,
                column: columnObj.column,
                concept: columnObj.concept,
                isPrimKey: columnObj.isPrimKey,
                conceptualName: columnObj.conceptualName,
                type: columnObj.type,
                typeFormat: columnObj.typeFormat,
                hasAliasError: false,
                filters: [],
                havings: [],
                orders: []
            });

            if (!block) {
                // update the traversals (if its the 1st one)
                if (scope.pipelineApp.traversal.selectedQueueTableIdx === -1) {
                    updateTraversal(scope.pipelineApp.selected.queue.length - 1);
                }
                // validate that the available are correct (naming)
                validateSelected();

                previewSelected(false);
            }
        }

        /**
         * @name removeSelected
         * @desc remove a column from the selected queue
         * @returns {void}
         */
        function removeSelected() {
            var removedTableIdx,
                removedColumnIdx;

            if (scope.pipelineApp.selected.queue.length === 0) {
                return; // not removed
            }

            removedTableIdx = scope.pipelineApp.selected.queue.length - 1;
            removedColumnIdx = scope.pipelineApp.selected.queue[removedTableIdx].columns.length - 1;


            // remove the column
            if (removedColumnIdx !== -1) {
                scope.pipelineApp.selected.queue[removedTableIdx].columns.splice(removedColumnIdx, 1);
            }

            // if the table is empty, there is no need to keep it there
            if (scope.pipelineApp.selected.queue[removedTableIdx].columns.length === 0) {
                scope.pipelineApp.selected.queue.splice(removedTableIdx, 1);
            }

            // reset the data
            if (scope.pipelineApp.selected.queue.length === 0) {
                updateData(true);
                return;
            }

            // update the traversals
            updateTraversal(scope.pipelineApp.selected.queue.length - 1);

            // validate that the available are correct (naming)
            validateSelected();

            previewSelected(false);
        }

        /**
         * @name validateSelected
         * @desc validate the selected options
         * @returns {void}
         */
        function validateSelected() {
            var alias,
                tableIdx,
                tableLen,
                colIdx,
                colLen,
                upperAlias,
                aliasMap = {},
                aliasIdx,
                aliasLen;

            if (scope.pipelineApp.selected.queue.length === 0) {
                scope.pipelineApp.valid = false;
                return;
            }

            // check for invalid aliases
            // and aggregate alias that match ignoring case
            for (tableIdx = 0, tableLen = scope.pipelineApp.selected.queue.length; tableIdx < tableLen; tableIdx++) {
                for (colIdx = 0, colLen = scope.pipelineApp.selected.queue[tableIdx].columns.length; colIdx < colLen; colIdx++) {
                    // check if the alias itself is valid and save the duplicate
                    scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].hasAliasError = (!scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].alias || !!scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].alias.match(/[-\/\\^$*+?.()|[\]{};!%#@~`]/g) || !!scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].alias.match(/( |_)( |_)/g));

                    upperAlias = scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].alias.toUpperCase();
                    if (!aliasMap.hasOwnProperty(upperAlias)) {
                        aliasMap[upperAlias] = [];
                    }

                    aliasMap[upperAlias].push([tableIdx, colIdx]);
                }
            }

            for (alias in aliasMap) {
                if (aliasMap.hasOwnProperty(alias)) {
                    // duplicates
                    if (aliasMap[alias].length >= 2) {
                        for (aliasIdx = 0, aliasLen = aliasMap[alias].length; aliasIdx < aliasLen; aliasIdx++) {
                            scope.pipelineApp.selected.queue[aliasMap[alias][aliasIdx][0]].columns[aliasMap[alias][aliasIdx][1]].hasAliasError = true;
                        }
                    }
                }
            }

            // set validity
            for (tableIdx = 0, tableLen = scope.pipelineApp.selected.queue.length; tableIdx < tableLen; tableIdx++) {
                for (colIdx = 0, colLen = scope.pipelineApp.selected.queue[tableIdx].columns.length; colIdx < colLen; colIdx++) {
                    if (scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].hasAliasError) {
                        scope.pipelineApp.valid = false;
                        return;
                    }
                }
            }

            scope.pipelineApp.valid = true;
        }

        /**
         * @name updateSelectedAlias
         * @desc when user updates alias we want to update preview but not on each change
         *       so, we create a timeout that is canceled at the beginning of each call
         *       to this function so when the user types, load preview will not be continuously
         *       called. After one second of no typing previewSelected is called
         * @return {void}
         */
        function updateSelectedAlias() {
            validateSelected();

            if (aliasTimeout) {
                $timeout.cancel(aliasTimeout);
            }

            aliasTimeout = $timeout(function () {
                previewSelected(false);
            }, 300);
        }

        /**
         * @name previewSelected
         * @param {boolean} alert - message on errors
         * @desc preview the selected data
         * @returns {void}
         */
        function previewSelected(alert) {
            var parameters = {},
                valid = true;

            if (scope.pipelineApp.selected.queue.length === 0) {
                if (alert) {
                    scope.widgetCtrl.alert('warn', 'Selectors are empty. Please select columns to import.');
                }
                valid = false;
            }

            if (valid) {
                parameters = buildParameters('queue', true);
            }

            scope.pipelineComponentCtrl.previewComponent(parameters);
        }

        /**
         * @name showFilter
         * @desc show screen for a new filter
         * @param {object} column - filter to select
         * @returns {void}
         */
        function showFilter(column) {
            scope.pipelineApp.selectedFilterColumn = column;
            scope.pipelineApp.showFilters = true;
        }

        /**
         * @name hideFilter
         * @desc hide screent for a new filter
         * @returns {void}
         */
        function hideFilter() {
            scope.pipelineApp.showFilters = false;
        }

        /**
         * @name removeFilter
         * @desc remove the selected filter
         * @param {object} columnObj - column to remove filter from
         * @param {number} filterIdx - index of the filter to remove
         * @returns {void}
         */
        function removeFilter(columnObj, filterIdx) {
            if (columnObj.filters[filterIdx]) {
                columnObj.filters.splice(filterIdx, 1);

                previewSelected();
            }
        }

        /**
         * @name openMetamodel
         * @desc open the metamodel
         * @returns {void}
         */
        function openMetamodel() {
            // close before opening
            closeMetamodel();

            scope.pipelineApp.structure.loading = true;
            scope.pipelineApp.structure.searchTerm = '';

            //  get the metamodel
            $timeout(function () {
                scope.pipelineApp.metamodel.graph = ele[0].querySelector('#pipeline-app__metamodel__graph');
                scope.pipelineApp.metamodel.plumbing = jsPlumb.jsPlumb.getInstance({
                    Container: scope.pipelineApp.metamodel.graph
                });

                // add panzoom
                scope.pipelineApp.metamodel.zoom = panzoom(scope.pipelineApp.metamodel.graph);

                drawMetamodel();

                scope.pipelineApp.structure.loading = false;
            });
        }

        /**
         * @name closeMetamodel
         * @desc destroy the metamodel
         * @returns {void}
         */
        function closeMetamodel() {
            // remove connections
            if (scope.pipelineApp.metamodel.plumbing) {
                scope.pipelineApp.metamodel.plumbing.reset();
            }

            if (scope.pipelineApp.metamodel.zoom) {
                scope.pipelineApp.metamodel.zoom.dispose();
                scope.pipelineApp.metamodel.zoom = undefined;
            }

            // destroy the old scope
            if (scope.pipelineApp.metamodel.scope) {
                scope.pipelineApp.metamodel.scope.$destroy();
            }

            // remove the eles
            if (scope.pipelineApp.metamodel.graph) {
                while (scope.pipelineApp.metamodel.graph.firstChild) {
                    if (scope.pipelineApp.metamodel.graph.lastChild) {
                        scope.pipelineApp.metamodel.graph.removeChild(scope.pipelineApp.metamodel.graph.lastChild);
                    }
                }
            }
        }

        /**
         * @name drawMetamodel
         * @desc draw the metamodel
         * @returns {void}
         */
        function drawMetamodel() {
            let html = '',
                eles = {};

            // generate the html
            // add the tables
            html += '<div>';
            for (let table in scope.pipelineApp.structure.tables) {
                if (scope.pipelineApp.structure.tables.hasOwnProperty(table)) {
                    html += generateTable(scope.pipelineApp.structure.tables[table].table);
                }
            }
            html += '</div>';

            // create a new scope
            scope.pipelineApp.metamodel.scope = scope.$new();

            // mount and compile
            scope.pipelineApp.metamodel.graph.appendChild(angular.element(html)[0]);
            $compile(scope.pipelineApp.metamodel.graph)(scope.pipelineApp.metamodel.scope);

            // store all of the eles
            for (let table in scope.pipelineApp.structure.tables) {
                if (scope.pipelineApp.structure.tables.hasOwnProperty(table)) {
                    eles[scope.pipelineApp.structure.tables[table].table] = scope.pipelineApp.metamodel.graph.querySelector(`#pipeline-app__metamodel__graph__table--${scope.pipelineApp.structure.tables[table].table}`);
                }
            }

            // add edges
            for (let edgeIdx = 0, edgeLen = scope.pipelineApp.structure.edges.length; edgeIdx < edgeLen; edgeIdx++) {
                scope.pipelineApp.metamodel.plumbing.connect({
                    source: eles[scope.pipelineApp.structure.edges[edgeIdx].source],
                    target: eles[scope.pipelineApp.structure.edges[edgeIdx].target],
                    detachable: false,
                    anchor: 'AutoDefault',
                    endpoint: 'Blank',
                    connectionsDetachable: false,
                    maxConnections: -1,
                    connector: [
                        'Flowchart',
                        {
                            cssClass: 'pipeline-app__metamodel__graph__edge__connector'
                        }
                    ]
                });
            }
        }

        /**
         * @name searchMetamodel
         * @desc search the metamodel
         * @returns {void}
         */
        function searchMetamodel() {
            if (scope.pipelineApp.metamodel.graph) {
                let tables = scope.pipelineApp.metamodel.graph.querySelectorAll('[metamodel-alias]') || [];
                const len = tables.length;
                if (len > 0) {
                    let searchString = scope.pipelineApp.structure.searchTerm || '';
                    searchString = searchString.toUpperCase().replace(/ /g, '_');

                    for (let i = 0; i < len; i++) {
                        // clear the old
                        let temp = tables[i].getAttribute('metamodel-alias') || '',
                            iconEle = tables[i].querySelector('.pipeline-app__metamodel__graph__table__item__icon') || [],
                            textEle = tables[i].querySelector('.pipeline-app__metamodel__graph__table__item__text') || [];

                        temp = temp.toUpperCase().replace(/ /g, '_');
                        if (temp.indexOf(searchString) === -1 || !searchString) {
                            iconEle.style.color = '';
                            textEle.style.color = '';
                            tables[i].style.backgroundColor = '';
                        } else {
                            iconEle.style.color = '#000000';
                            textEle.style.color = '#000000';
                            tables[i].style.backgroundColor = '#fff9e9';
                        }
                    }
                }
            }
        }

        /**
         * @name generateTable
         * @param {string} id - id of the table to create
         * @desc generates a label for the selected table
         * @return {string}  the html for the table
         */
        function generateTable(id) {
            const table = scope.pipelineApp.structure.tables[id];

            let labelHolder = '';
            labelHolder += `<div id="pipeline-app__metamodel__graph__table--${table.table}" 
                            class="pipeline-app__metamodel__graph__table"
                            ng-class="{'pipeline-app__metamodel__graph__table--light': pipelineApp.traversal.selectedQueueTableIdx !== -1 && !pipelineApp.getTableType('${table.table}'), 'pipeline-app__metamodel__graph__table--selected': pipelineApp.getTableType('${table.table}') === 'selected', 'pipeline-app__metamodel__graph__table--imported': pipelineApp.getTableType('${table.table}') === 'selected' || pipelineApp.getTableType('${table.table}') === 'previous'}" 
                            style="top:${table.position && table.position.hasOwnProperty('top')  ? table.position.top : 0}px;left:${table.position && table.position.hasOwnProperty('left')  ? table.position.left : 0}px">`;
            labelHolder +=
                `<div class="pipeline-app__metamodel__graph__table__item pipeline-app__metamodel__graph__table__item--border smss-clear"
                    title="${table.alias}"
                    metamodel-alias="${table.alias}"> 
                <div class="pipeline-app__metamodel__graph__table__item__icon smss-text smss-left">
                    <i class="fa fa-table"></i>
                </div>
                <div class="pipeline-app__metamodel__graph__table__item__text pipeline-app__metamodel__graph__table__item__text--wide smss-text smss-left">
                    <span class="smss-small">${table.alias}</span>
                </div>
                <smss-btn class="smss-left smss-btn--icon" title="Traverse from: ${table.alias}" ng-show="(pipelineApp.getTableType('${table.table}') === 'selected' || pipelineApp.getTableType('${table.table}') === 'previous')" ng-click="pipelineApp.selectTable('${table.table}')">
                    <i class="fa fa-eye" ng-class="{'smss-color--primary': pipelineApp.traversal.selectedQueueTable.table === '${table.table}'}"></i>
                </smss-btn>
            </div>`;

            // column list
            for (let column in table.columns) {
                if (table.columns.hasOwnProperty(column)) {
                    labelHolder +=
                        `<div class="pipeline-app__metamodel__graph__table__item smss-clear" 
                        title="${table.columns[column].alias}"
                        metamodel-alias="${table.columns[column].alias}">
                        <div class="pipeline-app__metamodel__graph__table__item__icon smss-text smss-left">
                            ${table.columns[column].isPrimKey ? '<i class="fa fa-key"></i>' : ''}
                        </div>
                        <div class="pipeline-app__metamodel__graph__table__item__text smss-text smss-left">
                            <span class="smss-small"> ${table.columns[column].alias} </span>
                        </div>
                        <div class="pipeline-app__metamodel__graph__table__item__icon smss-text smss-left">
                            ${table.columns[column].type === 'STRING' ? '<i class="fa fa-font"></i>' : ''}
                            ${table.columns[column].type === 'DATE' ? '<i class="fa fa-calendar-o"></i>' : ''}
                            ${table.columns[column].type === 'TIMESTAMP' ? '<i class="fa fa-clock-o"></i>' : ''}
                            ${table.columns[column].type === 'NUMBER' ? '<i class="fa fa-hashtag"></i>' : ''}
                        </div>
                        <smss-btn class="smss-left smss-btn--icon" title="${table.columns[column].alias}" ng-show="pipelineApp.getColumnType('${table.columns[column].table}', '${table.columns[column].column}')" ng-click="pipelineApp.selectColumn('${table.columns[column].table}', '${table.columns[column].column}')">
                            <i ng-class="pipelineApp.getColumnButtonClass('${table.columns[column].table}', '${table.columns[column].column}')"></i>
                        </smss-btn>
                    </div>`;
                }
            }
            labelHolder += '</div>';


            return labelHolder;
        }

        /**
         * @name selectTable
         * @param {string} table - table
         * @desc select the table
         * @returns {void}
         */
        function selectTable(table) {
            const option = getTableOptions(table);

            if (option.type === 'previous') {
                updateTraversal(option.tableIdx);
            }
        }

        /**
         * @name getTableType
         * @desc get the class for the metamodel
         * @param {string} table - table
         * @returns {string} preivious, selected
         */
        function getTableType(table) {
            const option = getTableOptions(table);

            return option.type;
        }

        /**
         * @name getTableOptions
         * @desc get the options for the column
         * @param {string} table - table
         * @returns {object} type, tableIdx
         */
        function getTableOptions(table) {
            if (scope.pipelineApp.traversal.selectedQueueTableIdx === -1) {
                return {
                    tableIdx: -1,
                    type: ''
                };
            }

            // no need to traverse if it is already st
            if (scope.pipelineApp.traversal.selectedQueueTable.table === table) {
                return {
                    tableIdx: scope.pipelineApp.traversal.selectedQueueTable,
                    type: 'selected'
                };
            }

            for (let tableIdx = 0, tableLen = scope.pipelineApp.selected.queue.length; tableIdx < tableLen; tableIdx++) {
                if (scope.pipelineApp.selected.queue[tableIdx].table === table) {
                    return {
                        tableIdx: tableIdx,
                        type: 'previous'
                    };
                }
            }

            // check if its is a traversal option
            // this is nasty =(
            for (let tableIdx = 0, tableLen = scope.pipelineApp.traversal.options.length; tableIdx < tableLen; tableIdx++) {
                if (scope.pipelineApp.traversal.options[tableIdx].table === table) {
                    return {
                        tableIdx: -1,
                        type: 'traverse'
                    };
                }
            }

            // can't over
            return {
                tableIdx: -1,
                type: ''
            };
        }


        /**
         * @name selectColumn
         * @desc select the column and trigger the appropriate action
         * @param {string} table - table
         * @param {string} column - column
         * @returns {void}
         */
        function selectColumn(table, column) {
            const option = getColumnOptions(table, column);

            if (option.type === 'add' || option.type === 'remove') {
                if (option.type === 'add') {
                    addSelected(option.tableObj, option.columnObj, false);
                }

                if (option.type === 'remove') {
                    removeSelected();
                }
            }
        }

        /**
         * @name getColumnType
         * @desc get the type for the metamodel
         * @param {string} table - table
         * @param {string} column - column
         * @returns {boolean} show it?
         */
        function getColumnType(table, column) {
            const option = getColumnOptions(table, column);

            return option.type;
        }

        /**
         * @name getColumnButtonClass
         * @desc get the class for the metamodel
         * @param {string} table - table
         * @param {string} column - column
         * @returns {string} button class to show
         */
        function getColumnButtonClass(table, column) {
            const option = getColumnOptions(table, column);

            if (option.type === 'added') {
                return 'fa fa-check';
            } else if (option.type === 'add') {
                return 'fa fa-plus smss-color--success';
            } else if (option.type === 'remove') {
                return 'fa fa-times smss-color--error';
            }

            return '';
        }

        /**
         * @name getColumnOptions
         * @desc get the options for the column
         * @param {string} table - table
         * @param {string} column - column
         * @returns {object} type, columnObj, tableObj
         */
        function getColumnOptions(table, column) {
            // three classes accept, reject, nothing
            if (scope.pipelineApp.traversal.selectedQueueTableIdx === -1) {
                for (let tableIdx = 0, tableLen = scope.pipelineApp.traversal.options.length; tableIdx < tableLen; tableIdx++) {
                    if (scope.pipelineApp.traversal.options[tableIdx].table === table) {
                        for (let colIdx = 0, colLen = scope.pipelineApp.traversal.options[tableIdx].columns.length; colIdx < colLen; colIdx++) {
                            if (scope.pipelineApp.traversal.options[tableIdx].columns[colIdx].column === column) {
                                return {
                                    tableObj: scope.pipelineApp.traversal.options[tableIdx],
                                    columnObj: scope.pipelineApp.traversal.options[tableIdx].columns[colIdx],
                                    type: 'add'
                                };
                            }
                        }
                    }
                }
            }

            // removable one
            if (
                scope.pipelineApp.selected.queue[scope.pipelineApp.selected.queue.length - 1].table === table &&
                scope.pipelineApp.selected.queue[scope.pipelineApp.selected.queue.length - 1].columns[scope.pipelineApp.selected.queue[scope.pipelineApp.selected.queue.length - 1].columns.length - 1].column === column) {
                return {
                    tableObj: scope.pipelineApp.selected.queue[scope.pipelineApp.selected.queue.length - 1],
                    columnObj: scope.pipelineApp.selected.queue[scope.pipelineApp.selected.queue.length - 1].columns[scope.pipelineApp.selected.queue[scope.pipelineApp.selected.queue.length - 1].columns.length - 1],
                    type: 'remove'
                };
            }

            // check if it is added
            for (let tableIdx = 0, tableLen = scope.pipelineApp.selected.queue.length; tableIdx < tableLen; tableIdx++) {
                if (scope.pipelineApp.selected.queue[tableIdx].table === table) {
                    for (let colIdx = 0, colLen = scope.pipelineApp.selected.queue[tableIdx].columns.length; colIdx < colLen; colIdx++) {
                        if (scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].column === column) {
                            return {
                                tableObj: scope.pipelineApp.selected.queue[tableIdx],
                                columnObj: scope.pipelineApp.selected.queue[tableIdx].columns[colIdx],
                                type: 'added'
                            };
                        }
                    }
                }
            }

            // check if its is a traversal option
            // this is nasty =(
            for (let tableIdx = 0, tableLen = scope.pipelineApp.traversal.options.length; tableIdx < tableLen; tableIdx++) {
                if (scope.pipelineApp.traversal.options[tableIdx].table === table) {
                    for (let colIdx = 0, colLen = scope.pipelineApp.traversal.options[tableIdx].columns.length; colIdx < colLen; colIdx++) {
                        if (scope.pipelineApp.traversal.options[tableIdx].columns[colIdx].column === column) {
                            return {
                                tableObj: scope.pipelineApp.traversal.options[tableIdx],
                                columnObj: scope.pipelineApp.traversal.options[tableIdx].columns[colIdx],
                                type: 'add'
                            };
                        }
                    }
                }
            }

            // can't over
            return {
                tableObj: undefined,
                columnObj: undefined,
                type: ''
            };
        }

        /**
         * @name setCacheSettings
         * @param {*} frameName the frame name to cache
         * @param {*} frameType the frame type to cache into
         * @desc if its a native frame, we can cache the frame
         * @returns {void}
         */
        function setCacheSettings(frameName, frameType) {
            if (scope.pipelineApp.cache && (scope.pipelineApp.frameType === 'NATIVE' || CONFIG.defaultFrameType === 'NATIVE')) {
                // add cache pixel
                scope.pipelineComponentCtrl.data.pixel += semossCoreService.pixel.build([{
                    type: 'cacheNativeFrame',
                    components: [frameName, frameType],
                    terminal: true
                }]);
            }
        }

        /**
         * @name checkImportLimits
         * @param {string} type the type of import auto or queue
         * @desc checks to see if the data is over the limit
         * @returns {*} object containing isOverLimit boolean and the warning message
         */
        function checkImportLimits(type) {
            var returnObj = {
                    isOverLimits: false,
                    message: ''
                }, columnCount = 0, queueIndex;

            // no limit set or type is auto where the FE does not have the row count so let it through. BE will have to do a check for limits
            if (!CONFIG.importLimit || !scope.pipelineApp.totalCount || type === 'auto') {
                return returnObj;
            }

            // get the column counts
            if (type === 'queue') {
                for (queueIndex = 0; queueIndex < scope.pipelineApp.selected.queue.length; queueIndex++) {
                    columnCount += scope.pipelineApp.selected.queue[queueIndex].columns.length;
                }
            }

            if (CONFIG.importLimit < (scope.pipelineApp.totalCount * columnCount)) {
                returnObj.message = 'The number data points (' + (scope.pipelineApp.totalCount * columnCount) + ') exceeds the limit of ' + CONFIG.importLimit + '. ';
                returnObj.message += 'Please reduce your data points by filtering or removing columns.';
                returnObj.isOverLimits = true;
            }

            return returnObj;
        }

        /**
         * @name importQueue
         * @param {boolean} visualize - if true visualize frame
         * @desc import the queue
         * @returns {void}
         */
        function importQueue(visualize) {
            let parameters,
                options,
                callback,
                limitCheckObj;

            if (scope.pipelineApp.selected.queue.length === 0) {
                scope.widgetCtrl.alert('warn', 'Selectors are empty. Please select data.');
                return;
            }

            limitCheckObj = checkImportLimits('queue');

            if (limitCheckObj.isOverLimits) {
                scope.widgetCtrl.alert('warn', limitCheckObj.message);
                return;
            }

            parameters = buildParameters('queue');

            options = {};
            options.name = scope.pipelineApp.selectedApp.display;
            options.icon = scope.pipelineApp.selectedApp.image;

            if (visualize) {
                callback = scope.pipelineComponentCtrl.visualizeComponent;
            }

            setCacheSettings(parameters.IMPORT_FRAME.name, 'R');
            scope.pipelineComponentCtrl.executeComponent(parameters, options, callback);
        }

        /**
         * @name importAuto
         * @param {boolean} visualize - if true visualize frame
         * @desc import automatically
         * @returns {void}
         */
        function importAuto(visualize) {
            let parameters,
                options,
                callback,
                limitCheckObj;

            if (
                Object.keys(scope.pipelineApp.structure.tables).length === 0
            ) {
                scope.widgetCtrl.alert('warn', 'There are no tables. Please select a new database.');
                return;
            }

            limitCheckObj = checkImportLimits('auto');

            if (limitCheckObj.isOverLimits) {
                scope.widgetCtrl.alert('warn', limitCheckObj.message);
                return;
            }

            // adding everything to the queue so user can come back to component
            scope.pipelineApp.traversal.options.forEach(table => {
                table.columns.forEach(col => addSelected(table, col, true));
            });

            parameters = buildParameters('auto');

            options = {};
            options.name = scope.pipelineApp.selectedApp.display;
            options.icon = scope.pipelineApp.selectedApp.image;

            if (visualize) {
                callback = scope.pipelineComponentCtrl.visualizeComponent;
            }

            setCacheSettings(parameters.IMPORT_FRAME.name, 'R');
            scope.pipelineComponentCtrl.executeComponent(parameters, options, callback);
        }

        /**
         * @name buildParameters
         * @desc build the parameters for the current module
         * @param {string} type - type of parameter to build
         * @param {boolelan} preview - true if coming from preview
         * @returns {object} - map of the paramters to value
         */
        function buildParameters(type, preview) {
            var relations = [],
                selectors = [],
                explicitFilters = [],
                havingFilters = [],
                alias,
                tableIdx,
                tableLen,
                colIdx,
                colLen,
                filterIdx,
                filterLen,
                added = {},
                join,
                table,
                column,
                edgeIdx,
                edgeLen,
                selected;

            if (type === 'auto') {
                for (table in scope.pipelineApp.structure.tables) {
                    if (scope.pipelineApp.structure.tables.hasOwnProperty(table)) {
                        for (column in scope.pipelineApp.structure.tables[table].columns) {
                            if (scope.pipelineApp.structure.tables[table].columns.hasOwnProperty(column)) {
                                alias = scope.pipelineApp.structure.tables[table].columns[column].alias.replace(/ /g, '_');

                                if (!scope.pipelineApp.structure.tables[table].columns[column].isPrimKey) {
                                    if (added.hasOwnProperty(alias)) {
                                        added[alias] = added[alias] + 1;
                                    } else {
                                        added[alias] = 0;
                                    }
                                    selectors.push({
                                        'type': 'COLUMN',
                                        'content': {
                                            'table': scope.pipelineApp.structure.tables[table].columns[column].table,
                                            'column': scope.pipelineApp.structure.tables[table].columns[column].isPrimKey ? 'PRIM_KEY_PLACEHOLDER' : scope.pipelineApp.structure.tables[table].columns[column].column,
                                            'alias': added[alias] > 0 ? alias + '_' + added[alias] : alias
                                        }
                                    });
                                }
                                // else {
                                //     console.warn('Multiple nodes with the same alias:' + alias);
                                // }

                                // added[alias] = 0;
                            }
                        }
                    }
                }

                added = {};
                for (edgeIdx = 0, edgeLen = scope.pipelineApp.structure.edges.length; edgeIdx < edgeLen; edgeIdx++) {
                    join = scope.pipelineApp.structure.edges[edgeIdx].source + '.' + scope.pipelineApp.structure.edges[edgeIdx].target;

                    if (!added.hasOwnProperty(join)) {
                        relations.push([
                            scope.pipelineApp.structure.edges[edgeIdx].source,
                            'inner.join',
                            scope.pipelineApp.structure.edges[edgeIdx].target
                        ]);
                    } else {
                        console.warn('Multiple nodes with the same join:' + scope.pipelineApp.structure.edges[edgeIdx].source + ' to ' + scope.pipelineApp.structure.edges[edgeIdx].target);
                    }
                }
            } else if (type === 'queue') {
                relations = [];
                for (tableIdx = 0, tableLen = scope.pipelineApp.selected.queue.length; tableIdx < tableLen; tableIdx++) {
                    if (typeof scope.pipelineApp.selected.queue[tableIdx].fromTable !== 'undefined' && typeof scope.pipelineApp.selected.queue[tableIdx].toTable !== 'undefined') {
                        relations.push([
                            scope.pipelineApp.selected.queue[tableIdx].fromTable,
                            scope.pipelineApp.selected.queue[tableIdx].joinType,
                            scope.pipelineApp.selected.queue[tableIdx].toTable
                        ]);
                    }

                    for (colIdx = 0, colLen = scope.pipelineApp.selected.queue[tableIdx].columns.length; colIdx < colLen; colIdx++) {
                        alias = String(scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].alias).replace(/ /g, '_');

                        selectors.push({
                            'type': 'COLUMN',
                            'content': {
                                'table': scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].table,
                                'column': scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].isPrimKey ? 'PRIM_KEY_PLACEHOLDER' : scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].column,
                                'alias': alias
                            }
                        });

                        for (filterIdx = 0, filterLen = scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].filters.length; filterIdx < filterLen; filterIdx++) {
                            explicitFilters.push(
                                'SIMPLE', {
                                    'left': {
                                        'pixelType': 'COLUMN',
                                        'value': [{
                                            'type': 'COLUMN',
                                            'content': {
                                                'table': scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].table,
                                                'column': scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].isPrimKey ? 'PRIM_KEY_PLACEHOLDER' : scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].column,
                                                'alias': alias
                                            }
                                        }]
                                    },
                                    'comparator': scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].filters[filterIdx].comparator,
                                    'right': {
                                        'pixelType': scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].filters[filterIdx].pixelType,
                                        'value': scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].filters[filterIdx].value
                                    }
                                }
                            );
                        }
                    }
                }
            }

            // TODO: rebuild the selected based on the QS
            if (!preview) {
                selected = {
                    traversal: scope.pipelineApp.traversal,
                    selected: scope.pipelineApp.selected,
                    structure: scope.pipelineApp.structure
                };
            } else {
                selected = {};
            }

            return {
                'IMPORT_FRAME': {
                    'name': scope.pipelineComponentCtrl.getComponent('parameters.IMPORT_FRAME.value.name') || scope.pipelineApp.customFrameName.name,
                    'type': scope.pipelineComponentCtrl.getComponent('parameters.IMPORT_FRAME.value.type') || scope.widgetCtrl.getOptions('initialFrameType'),
                    'override': true
                },
                'QUERY_STRUCT': {
                    'qsType': 'ENGINE',
                    'engineName': scope.pipelineApp.selectedApp.value,
                    'isDistinct': true,
                    'limit': preview ? PREVIEW_LIMIT : -1,
                    'offset': -1,
                    'relations': relations,
                    'orders': [],
                    'selectors': selectors,
                    'explicitFilters': explicitFilters,
                    'havingFilters': havingFilters
                },
                'SELECTED': selected,
                'SELECTED_APP': scope.pipelineApp.selectedApp
            };
        }

        /**
         * @name validateFrameName
         * @desc checks if the frame name entered by the user is valid
         * @returns {void}
         */
        function validateFrameName() {
            let results = scope.pipelineComponentCtrl.validateFrameName(scope.pipelineApp.customFrameName.name);
            scope.pipelineApp.customFrameName.valid = results.valid;
            scope.pipelineApp.customFrameName.message = results.message;
        }

        /** Utility */
        /**
         * @name getTypeInformation
         * @param {string} type - data type to set
         * @param {string} typeFormat - typeFormat
         * @returns {object} map - containing the type information
         */
        function getTypeInformation(type, typeFormat) {
            var newType = type,
                newTypeFormat = typeFormat;

            if (newType === 'INT' || newType === 'DOUBLE') {
                // ui considers int and double a type format for number,
                newTypeFormat = type;
                newType = 'NUMBER';
            }

            if ((newType === 'DATE' || newType === 'TIMESTAMP') && !typeFormat) {
                // needs type format, must tell user
            }

            if (!newType) {
                newType = 'STRING';
                newTypeFormat = '';
            }

            return {
                type: newType,
                typeFormat: newTypeFormat
            };
        }

        /**
         * @name exit
         * @desc hide the overlay and if task exists, remove it to remove any connection to the db
         * @returns {void}
         */
        function exit() {
            var components = [];
            scope.pipelineApp.showFilters = false;
            if (scope.pipelineApp.taskToRemove) {
                components.push({
                    type: 'removeTask',
                    components: [
                        scope.pipelineApp.taskToRemove
                    ],
                    terminal: true,
                    meta: true
                });

                scope.pipelineApp.taskToRemove = '';
            }

            if (components.length > 0) {
                scope.widgetCtrl.meta(components);
            }
        }

        /**
         * @name initialize
         * @desc initialize the module
         * @returns {void}
         */
        function initialize() {
            var selectPreviewListener,
                selectedApp = scope.pipelineComponentCtrl.getComponent('parameters.SELECTED_APP.value'),
                totalCountListener;

            selectPreviewListener = scope.widgetCtrl.on('select-preview', function (selected) {
                var tableIdx,
                    tableLen,
                    colIdx,
                    colLen;

                if (selected && selected.length > 0) {
                    // check queue
                    for (tableIdx = 0, tableLen = scope.pipelineApp.selected.queue.length; tableIdx < tableLen; tableIdx++) {
                        for (colIdx = 0, colLen = scope.pipelineApp.selected.queue[tableIdx].columns.length; colIdx < colLen; colIdx++) {
                            if (scope.pipelineApp.selected.queue[tableIdx].columns[colIdx].alias === selected[0].display) {
                                updateTraversal(tableIdx);
                                return;
                            }
                        }
                    }
                }
            });

            totalCountListener = scope.widgetCtrl.on('preview-total-count', function (payload) {
                scope.pipelineApp.totalCount = payload.data;
            });
            contentEle = ele[0].querySelector('#pipeline-app__content');
            leftEle = ele[0].querySelector('#pipeline-app__content__left');
            rightEle = ele[0].querySelector('#pipeline-app__content__right');

            // add the resize events
            resizable = Resizable({
                available: ['W'],
                container: contentEle,
                content: rightEle,
                unit: '%',
                restrict: {
                    minimumWidth: '20%',
                    maximumWidth: '70%'
                },
                start: function () {
                    // trigger digest
                    $timeout();
                },
                on: function () {},
                stop: function (top, left, height, width) {
                    leftEle.style.right = `${100 - left}%`;
                }
            });


            setInitialFrameType();
            if (selectedApp) {
                scope.pipelineApp.selectedApp = selectedApp;
            }
            // if be is taking too long to return app list,
            // pipeline app will not have a selected app and causes a cascade of errors
            // once list comes back in app-list-dropdown it will emit the change necessary to
            // allow the user to work as usual
            if (scope.pipelineApp.selectedApp) {
                getApps();
            }
            getFrames();

            // console.error('FIX Initialization. build off of pipeline inputs');
            // console.error('FIX QUERY_STRUCT. Make Join an array');
            // console.error('FIX QUERY_STRUCT. Genericize filter');

            scope.$on('$destroy', function () {
                selectPreviewListener();

                totalCountListener();

                // remove the resizable;
                if (resizable) {
                    resizable.destroy();
                }
            });

            if (scope.pipelineApp.selectedApp) {
                scope.pipelineApp.customFrameName.name = scope.pipelineComponentCtrl.createFrameName(scope.pipelineApp.selectedApp.display);
            }
        }

        initialize();
    }
}
