import angular from 'angular';

import './pipeline-app-filter.scss';

export default angular.module('app.pipeline.pipeline-app.filter', [])
    .directive('pipelineAppFilter', pipelineAppFilterDirective);

pipelineAppFilterDirective.$inject = ['$timeout', '$q', 'semossCoreService'];

function pipelineAppFilterDirective($timeout: ng.ITimeoutService, $q: ng.IQService, semossCoreService: SemossCoreService) {
    pipelineAppFilterCtrl.$inject = [];
    pipelineAppFilterLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        template: require('./pipeline-app-filter.directive.html'),
        scope: {},
        require: ['^widget', '^pipelineApp'],
        controller: pipelineAppFilterCtrl,
        controllerAs: 'pipelineAppFilter',
        bindToController: {},
        link: pipelineAppFilterLink
    };

    function pipelineAppFilterCtrl() { }

    function pipelineAppFilterLink(scope, ele, attrs, ctrl) {
        let searchTimeout;

        const COMPARATOR = {
            initial: {
                'NUMBER': '==',
                'DOUBLE': '==',
                'INT': '==',
                'STRING': '==',
                'DATE': '==',
                'TIMESTAMP': '=='
            },
            options: [
                {
                    'frame': true,
                    'display': 'Equal To',
                    'value': '==',
                    'types': ['NUMBER', 'DOUBLE', 'INT', 'STRING', 'DATE', 'TIMESTAMP']
                },
                {
                    'frame': true,
                    'display': 'Not Equal To',
                    'value': '!=',
                    'types': ['NUMBER', 'DOUBLE', 'INT', 'STRING', 'DATE', 'TIMESTAMP']
                },
                {
                    'frame': false,
                    'display': 'Contains',
                    'value': '?like',
                    'types': ['STRING']
                },
                {
                    'frame': false,
                    'display': 'Less Than',
                    'value': '<',
                    'types': ['NUMBER', 'DOUBLE', 'INT']
                },
                {
                    'frame': false,
                    'display': 'Less Than or Equal To',
                    'value': '<=',
                    'types': ['NUMBER', 'DOUBLE', 'INT']
                },
                {
                    'frame': false,
                    'display': 'Greater Than',
                    'value': '>',
                    'types': ['NUMBER', 'DOUBLE', 'INT']
                },
                {
                    'frame': false,
                    'display': 'Greater Than or Equal To',
                    'value': '>=',
                    'types': ['NUMBER', 'DOUBLE', 'INT']
                }
            ]
        }

        scope.widgetCtrl = ctrl[0];
        scope.pipelineAppCtrl = ctrl[1];

        scope.pipelineAppFilter.queueFilter = queueFilter;
        scope.pipelineAppFilter.hideFilter = hideFilter;
        scope.pipelineAppFilter.selectComparator = selectComparator;
        scope.pipelineAppFilter.searchEquality = searchEquality;
        scope.pipelineAppFilter.getMoreEquality = getMoreEquality;
        scope.pipelineAppFilter.searchLike = searchLike;
        scope.pipelineAppFilter.getMoreLike = getMoreLike;
        scope.pipelineAppFilter.toggleFrameFilter = toggleFrameFilter;
        scope.pipelineAppFilter.selectFrameFilter = selectFrameFilter;
        scope.pipelineAppFilter.selectFrameFilterColumn = selectFrameFilterColumn;

        scope.pipelineAppFilter.filter = {
            selected: '',
            headers: [],
            comparator: '',
            comparatorOptions: [],
            new: {}
        }

        scope.pipelineAppFilter.frame = {
            open: false
        }


        /** Filter */
        /** 
         * @name resetFilter
         * @desc reset the filter
         */
        function resetFilter() {
            scope.pipelineAppFilter.filter = {
                selected: scope.pipelineAppCtrl.selectedFilterColumn || '',
                headers: [],
                comparator: '',
                comparatorOptions: [],
                new: {}
            }

            // set selected
            selectComparator('');
        }

        /**
         * @name queueFilter
         * @desc queue the selected value
         */
        function queueFilter(): void {
            let value: string | number | undefined,
                pixelType: string | number | undefined;

            if (scope.pipelineAppFilter.frame.open) {
                let pixel = semossCoreService.pixel.build([{
                    type: 'frame',
                    components: [
                        scope.pipelineAppFilter.filter.new.frame.available.selected.name
                    ]
                },
                {
                    type: 'select2',
                    components: [[{
                        alias: scope.pipelineAppFilter.filter.new.frame.column.selected.alias
                    }]]
                },
                {
                    type: 'collect',
                    components: [-1],
                    terminal: true
                }]);

                pixel = pixel.substring(0, pixel.length - 1); // remove last semi-colon

                value = scope.pipelineAppFilter.filter.selected.concept + ' ' + scope.pipelineAppFilter.filter.comparator + ' (' + pixel + ')';
                pixelType = 'LAMBDA';
            } else if (scope.pipelineAppFilter.filter.comparator === '==' || scope.pipelineAppFilter.filter.comparator === '!=') {
                value = scope.pipelineAppFilter.filter.new.equality.model;
                pixelType = 'CONSTANT';
            } else if (scope.pipelineAppFilter.filter.comparator === '<' || scope.pipelineAppFilter.filter.comparator === '<=' || scope.pipelineAppFilter.filter.comparator === '>' || scope.pipelineAppFilter.filter.comparator === '>=') {
                value = scope.pipelineAppFilter.filter.new.numerical.model;
                pixelType = 'CONSTANT';
            } else if (scope.pipelineAppFilter.filter.comparator === '?like') {
                value = scope.pipelineAppFilter.filter.new.like.model;
                pixelType = 'CONSTANT';
            }

            if (typeof value === 'undefined') {
                return;
            }

            scope.pipelineAppCtrl.valid = true;

            for (let tableIdx = 0, tableLen = scope.pipelineAppCtrl.selected.queue.length; tableIdx < tableLen; tableIdx++) {
                if (scope.pipelineAppCtrl.selected.queue[tableIdx].table === scope.pipelineAppFilter.filter.selected.table) {
                    for (let colIdx = 0, colLen = scope.pipelineAppCtrl.selected.queue[tableIdx].columns.length; colIdx < colLen; colIdx++) {
                        if (scope.pipelineAppCtrl.selected.queue[tableIdx].columns[colIdx].column === scope.pipelineAppFilter.filter.selected.column && scope.pipelineAppCtrl.selected.queue[tableIdx].columns[colIdx].alias === scope.pipelineAppFilter.filter.selected.alias) {
                            scope.pipelineAppCtrl.selected.queue[tableIdx].columns[colIdx].filters.push({
                                pixelType: pixelType,
                                comparator: scope.pipelineAppFilter.filter.comparator,
                                value: value
                            });
                            break;
                        }
                    }

                    break;
                }
            }

            // hide
            hideFilter();

            scope.pipelineAppCtrl.previewSelected(false);
        }

        /**
         * @name hideFilter
         */
        function hideFilter(): void {
            scope.pipelineAppCtrl.showFilters = false;
            removeTasks();
        }

        /** Comparator */
        /**
         * @name selectComparator
         * @desc get a list of options available for the filter
         * @param comparator - selected comparator
         */
        function selectComparator(comparator: string): void {
            // set the comparator
            // get the correct list of comparators
            scope.pipelineAppFilter.filter.comparatorOptions = [];

            for (let optIdx = 0, optLen = COMPARATOR.options.length; optIdx < optLen; optIdx++) {
                if (
                    COMPARATOR.options[optIdx].types.indexOf(scope.pipelineAppFilter.filter.selected.type) > -1 &&
                    (
                        !scope.pipelineAppFilter.frame.open ||
                        (scope.pipelineAppFilter.frame.open && scope.pipelineAppFilter.frame.open === COMPARATOR.options[optIdx].frame)
                    )
                ) {
                    scope.pipelineAppFilter.filter.comparatorOptions.push(COMPARATOR.options[optIdx]);
                }
            }

            // set a default comparator
            scope.pipelineAppFilter.filter.comparator = false;
            if (comparator) {
                for (let optIdx = 0, optLen = scope.pipelineAppFilter.filter.comparatorOptions.length; optIdx < optLen; optIdx++) {
                    if (scope.pipelineAppFilter.filter.comparatorOptions[optIdx].value === comparator) {
                        scope.pipelineAppFilter.filter.comparator = comparator;
                    }
                }
            }

            if (!scope.pipelineAppFilter.filter.comparator && scope.pipelineAppFilter.filter.comparatorOptions.length > 0) {
                scope.pipelineAppFilter.filter.comparator = scope.pipelineAppFilter.filter.comparatorOptions[0].value;
            }

            if (scope.pipelineAppFilter.frame.open) {
                resetOptions();
                resetFrameFilter();
            } else if (scope.pipelineAppFilter.filter.comparator === '==' || scope.pipelineAppFilter.filter.comparator === '!=') {
                if (comparator && scope.pipelineAppFilter.filter.new.equality && scope.pipelineAppFilter.filter.new.equality.options.length > 0) {
                    // already filled, so no need to do grab the data again
                    return;
                }
                resetOptions();
                resetEquality();

                let focusTimeout = $timeout(function () {
                    let focusedElement = ele[0].querySelector('#pipeline-app-filter__equality');
                    if (focusedElement) {
                        focusedElement.focus();
                    }
                    $timeout.cancel(focusTimeout);
                });
            } else if (scope.pipelineAppFilter.filter.comparator === '<' || scope.pipelineAppFilter.filter.comparator === '<=' || scope.pipelineAppFilter.filter.comparator === '>' || scope.pipelineAppFilter.filter.comparator === '>=') {
                if (comparator && scope.pipelineAppFilter.filter.new.numerical && scope.pipelineAppFilter.filter.new.numerical.max !== Infinity) {
                    // already filled, so no need to do grab the data again
                    return;
                }
                resetOptions();
                resetNumerical();

                let focusTimeout = $timeout(function () {
                    let focusedElement = ele[0].querySelector('#pipeline-app-filter__numerical');
                    if (focusedElement) {
                        focusedElement.focus();
                    }
                    $timeout.cancel(focusTimeout);
                });
            } else if (scope.pipelineAppFilter.filter.comparator === '?like') {
                if (scope.pipelineAppFilter.filter.new.like.options.length === 0) {
                    scope.pipelineAppFilter.filter.new.like.options = scope.pipelineAppFilter.filter.new.equality.options;
                }
            } else {
                resetOptions();
            }
        }

        /** Options */
        /**
         * @name resetOptions
         * @desc reset the defaults for the filter options
         */
        function resetOptions(): void {
            // reset the object (new qs was selected)
            scope.pipelineAppFilter.filter.new = {
                equality: {
                    loading: false,
                    taskId: false,
                    options: [], // all values on the dom for the alias
                    model: [], // all checked values on the dom for the alias,
                    search: '', // search term used - only used for search term in filtermodel call, nothing else
                    limit: 20, // how many filter values to collect
                    canCollect: true // flag to determine whether another filter model call should be made - infinite scroll,
                },
                numerical: {
                    loading: true,
                    model: undefined,
                    min: -Infinity,
                    max: Infinity
                },
                like: {
                    loading: false,
                    taskId: false,
                    options: [], // all values on the dom for the alias
                    model: undefined, // actual model value
                    limit: 20, // how many values to collect
                    canCollect: true // infinite scroll,
                },
                frame: {
                    available: {
                        options: [],
                        selected: ''
                    },
                    column: {
                        options: [],
                        selected: ''
                    }
                }
            };
        }

        /**
         * @name resetNumerical
         * @desc updates filter model for this filter
         */
        function resetNumerical(): void {
            // remove the old task before running the new task
            removeTasks();
            if (scope.pipelineAppFilter.filter.new.numerical.max) {
                scope.pipelineAppFilter.filter.new.numerical.loading = true;
            }

            // register message to come back to
            let callback = (response: PixelReturnPayload) => {
                let output = response.pixelReturn[0].output;

                scope.pipelineAppFilter.filter.new.numerical.max = output.data.values[0][0];
                scope.pipelineAppFilter.filter.new.numerical.min = output.data.values[0][1];
                scope.pipelineAppFilter.filter.new.numerical.model = undefined;
                scope.pipelineAppFilter.filter.new.numerical.loading = false;
                scope.pipelineAppCtrl.taskToRemove = output.taskId;
            };

            scope.widgetCtrl.meta([
                {
                    type: 'database',
                    components: [
                        scope.pipelineAppCtrl.selectedApp.value
                    ],
                    meta: true
                },
                {
                    type: 'select2',
                    components: [
                        [{
                            math: 'Max',
                            alias: 'Max_of_' + scope.pipelineAppFilter.filter.selected.concept,
                            calculatedBy: scope.pipelineAppFilter.filter.selected.concept
                        },
                        {
                            math: 'Min',
                            alias: 'Min_of_' + scope.pipelineAppFilter.filter.selected.concept,
                            calculatedBy: scope.pipelineAppFilter.filter.selected.concept
                        }
                        ]
                    ]
                },
                {
                    type: 'collect',
                    components: [
                        -1
                    ],
                    terminal: true
                }
            ], callback);
        }

        /**
         * @name resetEquality
         * @desc updates filter model for this filter
         */
        function resetEquality(): void {
            let components: PixelCommand[] = [],
                callback;

            scope.pipelineAppFilter.filter.new.equality.loading = true;
            // remove the old task before running the new task
            removeTasks();
            components.push(
                {
                    type: 'database',
                    components: [
                        scope.pipelineAppCtrl.selectedApp.value
                    ],
                    meta: true
                },
                {
                    type: 'select2',
                    components: [
                        [{
                            alias: scope.pipelineAppFilter.filter.selected.concept
                        }]
                    ]
                });

            if (scope.pipelineAppFilter.filter.new.equality.search) {
                let filterObj = {};
                let values = [scope.pipelineAppFilter.filter.new.equality.search];

                // pass in the version with underscore as well to search for
                if (scope.pipelineAppFilter.filter.new.equality.search !== scope.pipelineAppFilter.filter.new.equality.search.replace(/ /g, '_')) {
                    values.push(scope.pipelineAppFilter.filter.new.equality.search.replace(/ /g, '_'));
                }

                // search
                filterObj[scope.pipelineAppFilter.filter.selected.concept] = {
                    comparator: '?like',
                    value: values
                };

                components.push({
                    type: 'filter',
                    components: [filterObj]
                });
            }

            components.push({
                type: 'collect',
                components: [
                    scope.pipelineAppFilter.filter.new.equality.limit
                ],
                terminal: true
            });

            // register message to come back to
            callback = (response: PixelReturnPayload) => {
                let output = response.pixelReturn[0].output;

                if (output) {
                    // only empty when there is no existing task
                    if (!scope.pipelineAppFilter.filter.new.equality.taskId) {
                        scope.pipelineAppFilter.filter.new.equality.model = [];
                    }

                    // no need to clear out because we are loading infinite
                    scope.pipelineAppFilter.filter.new.equality.options = [];

                    // add new ones
                    for (let valueIdx = 0, valueLen = output.data.values.length; valueIdx < valueLen; valueIdx++) {
                        scope.pipelineAppFilter.filter.new.equality.options.push(output.data.values[valueIdx][0]);
                    }

                    // clean
                    scope.pipelineAppFilter.filter.new.equality.options = cleanValues(scope.pipelineAppFilter.filter.new.equality.options, true);

                    scope.pipelineAppCtrl.taskToRemove = output.taskId;
                    // update filter control variables
                    scope.pipelineAppFilter.filter.new.equality.taskId = output.taskId;
                    scope.pipelineAppFilter.filter.new.equality.canCollect = (output.numCollected === output.data.values.length);
                    scope.pipelineAppFilter.filter.new.equality.loading = false;
                }
            };

            scope.widgetCtrl.meta(components, callback);
        }

        /**
         * @name removeTasks
         * @desc remove old tasks
         */
        function removeTasks() {
            let components: any = [];

            if (scope.pipelineAppCtrl.taskToRemove) {
                components.push({
                    type: 'removeTask',
                    components: [
                        scope.pipelineAppCtrl.taskToRemove
                    ],
                    terminal: true,
                    meta: true
                });

                scope.pipelineAppCtrl.taskToRemove = '';
            }

            if (components.length > 0) {
                scope.widgetCtrl.meta(components);
            }
        }

        /**
         * @name searchEquality
         * @desc search for the options based on searchTerm
         */
        function searchEquality(): void {
            if (searchTimeout) {
                $timeout.cancel(searchTimeout);
            }
            searchTimeout = $timeout(function () {
                // reset vars
                resetEquality();
                $timeout.cancel(searchTimeout);
            }, 500);
        }

        /**
         * @name getMoreEquality
         * @desc get the next set of options
         */
        function getMoreEquality(): void {
            let components: PixelCommand[] = [],
                callback;

            if (scope.pipelineAppFilter.filter.new.equality.canCollect) {
                scope.pipelineAppFilter.filter.new.equality.loading = true;

                components = [{
                    type: 'task',
                    components: [
                        scope.pipelineAppFilter.filter.new.equality.taskId
                    ]
                },
                {
                    type: 'collect',
                    components: [
                        scope.pipelineAppFilter.filter.new.equality.limit
                    ],
                    terminal: true
                }
                ];

                // register message to come back to
                callback = (response: PixelReturnPayload) => {
                    let output = response.pixelReturn[0].output;

                    // add new ones
                    for (let valueIdx = 0, valueLen = output.data.values.length; valueIdx < valueLen; valueIdx++) {
                        scope.pipelineAppFilter.filter.new.equality.options.push(output.data.values[valueIdx][0]);
                    }

                    // clean
                    scope.pipelineAppFilter.filter.new.equality.options = cleanValues(scope.pipelineAppFilter.filter.new.equality.options, true);

                    // update filter control variables
                    scope.pipelineAppFilter.filter.new.equality.taskId = output.taskId;
                    scope.pipelineAppFilter.filter.new.equality.canCollect = (output.numCollected === output.data.values.length);
                    scope.pipelineAppFilter.filter.new.equality.loading = false;
                };

                scope.widgetCtrl.meta(components, callback);
            }
        }

        /**
         * @name resetLike
         * @desc updates filter model for this filter
         */
        function resetLike(): void {
            let components: PixelCommand[] = [],
                callback;

            scope.pipelineAppFilter.filter.new.like.loading = true;
            // remove the old task before running the new task
            removeTasks();
            components.push(
                {
                    type: 'database',
                    components: [
                        scope.pipelineAppCtrl.selectedApp.value
                    ],
                    meta: true
                },
                {
                    type: 'select2',
                    components: [
                        [{
                            alias: scope.pipelineAppFilter.filter.selected.concept
                        }]
                    ]
                });

            if (scope.pipelineAppFilter.filter.new.like.model) {
                let filterObj = {};

                // search
                filterObj[scope.pipelineAppFilter.filter.selected.concept] = {
                    comparator: '?like',
                    value: [scope.pipelineAppFilter.filter.new.like.model]
                };

                components.push({
                    type: 'filter',
                    components: [filterObj]
                });
            }

            components.push({
                type: 'collect',
                components: [
                    scope.pipelineAppFilter.filter.new.like.limit
                ],
                terminal: true
            });

            // register message to come back to
            callback = (response: PixelReturnPayload) => {
                let output = response.pixelReturn[0].output;

                // no need to clear out because we are loading infinite
                scope.pipelineAppFilter.filter.new.like.options = [];

                // add new ones
                for (let valueIdx = 0, valueLen = output.data.values.length; valueIdx < valueLen; valueIdx++) {
                    scope.pipelineAppFilter.filter.new.like.options.push(output.data.values[valueIdx][0]);
                }

                // clean
                scope.pipelineAppFilter.filter.new.like.options = cleanValues(scope.pipelineAppFilter.filter.new.like.options, true);

                scope.pipelineAppCtrl.taskToRemove = output.taskId;
                // update filter control variables
                scope.pipelineAppFilter.filter.new.like.taskId = output.taskId;
                scope.pipelineAppFilter.filter.new.like.canCollect = (output.numCollected === output.data.values.length);
                scope.pipelineAppFilter.filter.new.like.loading = false;
            };

            scope.widgetCtrl.meta(components, callback);
        }

        /**
         * @name searchLike
         * @desc search for the options based on searchTerm
         */
        function searchLike(): void {
            if (searchTimeout) {
                $timeout.cancel(searchTimeout);
            }
            searchTimeout = $timeout(function () {
                // reset vars
                resetLike();
                $timeout.cancel(searchTimeout);
            }, 500);
        }

        /**
         * @name getMoreLike
         * @desc get the next set of options
         */
        function getMoreLike(): void {
            let components: PixelCommand[] = [],
                callback;

            if (scope.pipelineAppFilter.filter.new.like.canCollect) {
                scope.pipelineAppFilter.filter.new.like.loading = true;

                components = [{
                    type: 'task',
                    components: [
                        scope.pipelineAppFilter.filter.new.like.taskId
                    ]
                },
                {
                    type: 'collect',
                    components: [
                        scope.pipelineAppFilter.filter.new.like.limit
                    ],
                    terminal: true
                }
                ];

                // register message to come back to
                callback = function (response: PixelReturnPayload) {
                    let output = response.pixelReturn[0].output;

                    // add new ones
                    for (let valueIdx = 0, valueLen = output.data.values.length; valueIdx < valueLen; valueIdx++) {
                        scope.pipelineAppFilter.filter.new.like.options.push(output.data.values[valueIdx][0]);
                    }

                    // clean
                    scope.pipelineAppFilter.filter.new.like.options = cleanValues(scope.pipelineAppFilter.filter.new.like.options, true);

                    // update filter control variables
                    scope.pipelineAppFilter.filter.new.like.taskId = output.taskId;
                    scope.pipelineAppFilter.filter.new.like.canCollect = (output.numCollected === output.data.values.length);
                    scope.pipelineAppFilter.filter.new.like.loading = false;
                };

                scope.widgetCtrl.meta(components, callback);
            }
        }

        /** Frame */
        /**
         * @name toggleFrameFilter
         * @desc toggle the frame filter
         */
        function toggleFrameFilter(): void {
            scope.pipelineAppFilter.frame.open = !scope.pipelineAppFilter.frame.open;

            // update the selector
            selectComparator(scope.pipelineAppFilter.filter.comparator)
        }

        /**
         * @name resetFrameFilter
         * @desc toggle the frame filter
         */
        function resetFrameFilter(): void {
            scope.pipelineAppFilter.filter.new.frame.available.options = scope.pipelineAppCtrl.availableFrames;
            selectFrameFilter(scope.pipelineAppFilter.filter.new.frame.available.options[0]);
        }

        /**
         * @name selectFrameFilter
         * @desc select the frame filter
         * @param frame - frame to select
         */
        function selectFrameFilter(frame: any): void {
            scope.pipelineAppFilter.filter.new.frame.available.selected = frame;

            scope.pipelineAppFilter.filter.new.frame.column.options = frame.headers.map((header) => {
                return {
                    type: header.dataType,
                    displayName: header.displayName,
                    alias: header.alias,
                }
            });

            selectFrameFilterColumn(scope.pipelineAppFilter.filter.new.frame.column.options[0]);
        }

        /**
         * @name selectFrameFilterColumn
         * @desc select the frame filter
         * @param header - header to select
         */
        function selectFrameFilterColumn(header: any): void {
            scope.pipelineAppFilter.filter.new.frame.column.selected = header;
        }

        /** Utility */
        /**
         * @name cleanValues
         * @desc uniques the values
         * @param values - values to display
         * @param  sort - true or fale for sort
         * @returns values
         */
        function cleanValues<T>(values: T[], sort: boolean): T[] {
            let unique: T[] = [],
                seen: Map<T, boolean> = new Map();

            if (!values) {
                return [];
            }

            for (let i = 0, len = values.length; i < len; i++) {
                if (!seen.get(values[i])) {
                    unique.push(values[i]);
                    seen.set(values[i], true);
                }
            }

            if (!sort) {
                return unique;
            }

            return semossCoreService.utility.sort(unique);
        }

        /**
         * @name initialize
         * @desc initialize the module
         */
        function initialize(): void {
            resetFilter();
        }

        initialize();
    }
}