'use strict';

import angular from 'angular';
import variables from '@/style/src/variables.scss';


import './panel-settings.scss';

import { PANEL_TYPES, PANEL_OPTIONS } from '../../core/constants.js'


export default angular.module('app.panel-settings.directive', [])
    .directive('panelSettings', panelSettingsDirective);


panelSettingsDirective.$inject = ['VIZ_COLORS'];

function panelSettingsDirective(VIZ_COLORS) {
    panelSettingsCtrl.$inject = [];
    panelSettingsLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        template: require('./panel-settings.directive.html'),
        require: ['^insight', '^widget'],
        scope: {},
        controller: panelSettingsCtrl,
        controllerAs: 'panelSettings',
        bindToController: {},
        link: panelSettingsLink
    };

    function panelSettingsCtrl() { }

    function panelSettingsLink(scope, ele, attrs, ctrl) {
        scope.insightCtrl = ctrl[0];
        scope.widgetCtrl = ctrl[1];

        // variables
        scope.panelSettings.panelLabel = '';
        scope.panelSettings.config = {}

        scope.panelSettings.showBackgroundColor = true;
        scope.panelSettings.theme = VIZ_COLORS.COLOR_SEMOSS;
        scope.panelSettings.position = {
            top: {
                number: 0,
                unit: '%'
            },
            left: {
                number: 0,
                unit: '%'
            },
            width: {
                number: 50,
                unit: '%'
            },
            height: {
                number: 50,
                unit: '%'
            }
        };

        // Functions
        scope.panelSettings.change = change;
        scope.panelSettings.save = save;

        /** Actions */

        /**
         * @name change
         * @desc the type has changed update the config
         */
        function change(): void {
            console.log(scope.panelSettings.config.type);

            // set settings based on the new type
            setSettings(scope.panelSettings.config.type, scope.panelSettings.config);
        }

        /**
         * @name save
         * @desc save the changes
         */
        function save(): void {
            let commandList: PixelCommand[] = [],
                changes: any = {};

            const sheetId = scope.insightCtrl.getWorkbook('worksheet'),
                panel = scope.insightCtrl.getPanel(sheetId, scope.widgetCtrl.panelId);


            // If the name has not changed, no need to run the pixel (or if it isn't set)
            if (scope.panelSettings.panelLabel !== panel.panelLabel) {
                commandList.push({
                    type: 'panel',
                    components: [
                        scope.widgetCtrl.panelId
                    ]
                }, {
                    type: 'setPanelLabel',
                    components: [
                        scope.panelSettings.panelLabel
                    ],
                    terminal: true
                });
            }



            // there might be a type change, we need to account for this
            // look at the current type, remove things that aren't valid
            if (scope.panelSettings.config.type === PANEL_TYPES.GOLDEN) {
                for (let option in scope.panelSettings.config) {
                    if (
                        scope.panelSettings.config.hasOwnProperty(option) &&
                        PANEL_OPTIONS.GOLDEN.config.hasOwnProperty(option)
                    ) {
                        changes[option] = scope.panelSettings.config[option];
                    }
                }
            } else if (scope.panelSettings.config.type === PANEL_TYPES.FLOATING) {
                for (let option in scope.panelSettings.config) {
                    if (
                        scope.panelSettings.config.hasOwnProperty(option) &&
                        PANEL_OPTIONS.FLOATING.config.hasOwnProperty(option)
                    ) {
                        changes[option] = scope.panelSettings.config[option];
                    }
                }

                // add in the position changes
                changes.top = `${scope.panelSettings.position.top.number}${scope.panelSettings.position.top.unit}`;
                changes.left = `${scope.panelSettings.position.left.number}${scope.panelSettings.position.left.unit}`;
                changes.width = `${scope.panelSettings.position.width.number}${scope.panelSettings.position.width.unit}`;
                changes.height = `${scope.panelSettings.position.height.number}${scope.panelSettings.position.height.unit}`;
            }

            //this list should be everything that has been changed and should be valid for the PANEL_TYPE
            if (Object.keys(changes).length > 0) {
                commandList.push({
                    type: 'panel',
                    components: [
                        scope.widgetCtrl.panelId
                    ]
                }, {
                    type: 'addPanelConfig',
                    components: [
                        changes
                    ],
                    terminal: true
                });
            }


            if (commandList.length > 0) {
                scope.insightCtrl.execute(commandList);
            }
        }

        /** Utility */
        /**
         * @name setSettings
         * @param type - type of the panel to set
         * @param config - config to merge in
         * @returns set settings based on the type of panel
         */
        function setSettings(type: string, config: any): void {
            if (type === PANEL_TYPES.GOLDEN) {
                scope.panelSettings.config = angular.merge({}, PANEL_OPTIONS.GOLDEN.config, config || {})
            } else if (type === PANEL_TYPES.FLOATING) {
                scope.panelSettings.config = angular.merge({}, PANEL_OPTIONS.FLOATING.config, config || {})

                let splitTop = splitPanelPosition(scope.panelSettings.config.top),
                    splitLeft = splitPanelPosition(scope.panelSettings.config.left),
                    splitWidth = splitPanelPosition(scope.panelSettings.config.width),
                    splitHeight = splitPanelPosition(scope.panelSettings.config.height);

                scope.panelSettings.position = {
                    top: {
                        number: splitTop[0] ? splitTop[0] : 0,
                        unit: splitTop[1]
                    },
                    left: {
                        number: splitLeft[0] ? splitLeft[0] : 0,
                        unit: splitLeft[1]
                    },
                    width: {
                        number: splitWidth[0] ? splitWidth[0] : 50,
                        unit: splitWidth[1]
                    },
                    height: {
                        number: splitHeight[0] ? splitHeight[0] : 50,
                        unit: splitHeight[1]
                    }
                };
            }
        }

        /**
         * @name splitPanelPosition
         * @param {string} value panel position value
         * @returns the value and unit for the position
         */
        function splitPanelPosition(value: string): [number | undefined, string] {
            let i = 0,
                unit: string;

            // convert value to a string
            value = String(value);

            if (!value) {
                return [undefined, '%'];
            }

            for (i = 0; i < value.length; i++) {
                if (value[i] === '%' || value[i] === 'p') {
                    break;
                }
            }

            unit = value.substring(i);
            if (!unit) {
                unit = '%';
            }

            return [Number(value.substring(0, i)), unit];
        }

        /** Updates */
        /**
         * @name updatePanel
         * @desc called to update when the panel changes
         */
        function updatePanel(): void {
            const sheetId = scope.insightCtrl.getWorkbook('worksheet'),
                panel = scope.insightCtrl.getPanel(sheetId, scope.widgetCtrl.panelId);

            scope.panelSettings.panelLabel = panel.panelLabel;

            // set settings based on the original type
            setSettings(panel.config.type, panel.config);

            const layout = scope.widgetCtrl.getWidget('view.visualization.layout');
            if (layout === 'Grid' || layout === 'VivaGraph') {
                scope.panelSettings.showBackgroundColor = false;
            } else {
                scope.panelSettings.showBackgroundColor = true;
            }
        }

        /** Initialize */
        /**
         * @name initialize
         * @desc initializes the panel-settings directive
         */
        function initialize(): void {
            let updatedPanelListener: () => {},
                resetPanelListener: () => {};

            updatedPanelListener = scope.insightCtrl.on('updated-panel', updatePanel);
            resetPanelListener = scope.insightCtrl.on('reset-panel', updatePanel);

            // cleanup
            scope.$on('$destroy', function () {
                updatedPanelListener();
                resetPanelListener();
            });

            updatePanel();
        }

        initialize();
    }
}
