'use strict';

import './kpi.scss';
import EchartsHelper from '@/widget-resources/js/echarts/echarts-helper.js';
import visualizationUniversal from '@/core/store/visualization/visualization.js';

export default angular.module('app.kpi.directive', [])
    .directive('kpi', kpi);

kpi.$inject = ['semossCoreService'];

function kpi(semossCoreService) {
    kpiLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        require: ['^widget', '^visualization'],
        template: require('./kpi.directive.html'),
        link: kpiLink,
        scope: {}
    };

    function kpiLink(scope, ele, attrs, ctrl) {
        let groupEle,
            valueEle,
            titleEle,
            autoscale = false;

        scope.widgetCtrl = ctrl[0];
        scope.visualizationCtrl = ctrl[1];

        /**
         * @name setData
         * @desc setData for the visualization and paints testestest 
         * @returns {void}
         */
        function setData() {
            let keys = scope.widgetCtrl.getWidget('view.visualization.keys.KPI'),
                layerIndex = 0,
                data = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.data'),
                individualTools = scope.widgetCtrl.getWidget('view.visualization.tools.individual.KPI') || {},
                sharedTools = scope.widgetCtrl.getWidget('view.visualization.tools.shared') || {},
                colorByValue = scope.widgetCtrl.getWidget('view.visualization.colorByValue'),
                uiOptions,
                dimensionHeaderIdx,
                fill,
                header,
                value,
                dimensionName,
                formatType,
                formatDimension,
                j,
                colorPalette;

            // clear out all of the old text
            valueEle.textContent = '';
            titleEle.textContent = '';

            // find the required dimensions
            dimensionHeaderIdx = -1;
            // look for the key that is dimension, we will use this to paint the data
            for (let keyIdx = 0, keyLen = keys.length; keyIdx < keyLen; keyIdx++) {
                if (keys[keyIdx].model === 'dimension') {
                    //get primary db format type
                    dimensionName = keys[keyIdx].alias;
                    formatType = visualizationUniversal.mapFormatOpts(keys[keyIdx]);
                    // find the appropriate headerIdx, this will help us match up the actual data value
                    for (let headerIdx = 0, headerLen = data.headers.length; headerIdx < headerLen; headerIdx++) {
                        if (data.headers[headerIdx] === keys[keyIdx].alias) {
                            dimensionHeaderIdx = headerIdx;
                            break;
                        }
                    }

                    break;
                }
            }

            if (dimensionHeaderIdx === -1) {
                console.error('Dimension is required');
                return;
            }

            // get the options to paint with
            uiOptions = angular.extend(sharedTools, individualTools);


            // bind the actual do
            header = data.headers[dimensionHeaderIdx];
            if (data.values.length > 0) {
                value = data.values[0][dimensionHeaderIdx];
            } else {
                value = 0;
            }

            // set the fill. The order of importance is
            // 1. use the default theme color
            // 2. use the settings color
            // 3. use color by value
            fill = '';

            colorPalette = semossCoreService.visualization.getColorPalette(uiOptions.colorName);
            if (colorPalette && colorPalette.length > 0) {
                fill = colorPalette[0];
            }

            if (uiOptions.hasOwnProperty('kpiColor')) {
                fill = uiOptions.kpiColor;
            }

            colorByValue.forEach(function (rule) {
                for (let valueIdx = 0, valueLen = rule.valuesToColor.length; valueIdx < valueLen; valueIdx++) {
                    if (
                        header === rule.colorOn &&
                        value === rule.valuesToColor[valueIdx]
                    ) {
                        fill = rule.color;
                    }
                }
            });


            groupEle.style.fill = fill;

            // bind the actual do
            if (uiOptions && uiOptions.hasOwnProperty('kpiRound')) {
                if (uiOptions.kpiRound) {
                    let shift = 100; // default
                    if (uiOptions.hasOwnProperty('kpiRoundShift')) {
                        shift = Math.pow(10, uiOptions.kpiRoundShift);
                    }
                    value = Math.round(value * shift) / shift;
                }
            }

            //if user has updated formatting rules in widget, override db format types
            if(uiOptions && uiOptions.formatDataValues){
                for(j = 0; j < uiOptions.formatDataValues.formats.length; j++){
                    formatDimension = uiOptions.formatDataValues.formats[j].dimension;
                    if(formatDimension === dimensionName){
                        formatType = uiOptions.formatDataValues.formats[j];
                    }
                }
            }
            value = visualizationUniversal.formatValue(value, formatType);

            if (uiOptions && uiOptions.hasOwnProperty('kpiFormat')) {
                if (uiOptions.kpiFormat === 'None') {
                    valueEle.textContent = value;
                } else if (uiOptions.kpiFormat === '$ USD') {
                    valueEle.textContent = `$${value}`;
                } else if (uiOptions.kpiFormat === '% Percentage') {
                    valueEle.textContent = `${value}%`;
                }
            } else {
                valueEle.textContent = value;
            }

            if (uiOptions && uiOptions.hasOwnProperty('kpiValueSize')) {
                valueEle.style.fontSize = uiOptions.kpiValueSize + 'px';
            } else {
                valueEle.style.fontSize = null;
            }

            if (uiOptions && uiOptions.hasOwnProperty('kpiTitleText') && uiOptions.kpiTitleText) {
                titleEle.textContent = uiOptions.kpiTitleText;
            } else {
                titleEle.textContent = String(header).replace(/_/g, ' ');
            }

            if (uiOptions && uiOptions.hasOwnProperty('kpiTitleSize')) {
                titleEle.style.fontSize = uiOptions.kpiTitleSize + 'px';
            } else {
                titleEle.style.fontSize = null;
            }

            if (uiOptions && uiOptions.hasOwnProperty('kpiAutoScale')) {
                autoscale = uiOptions.kpiAutoScale;
            } else {
                autoscale = false;
            }

            // set the size
            resize();
        }

        /**
         * @name resize
         * @desc resize the visualization
         * @returns {void}
         */
        function resize() {
            let groupEleBB,
                widthTransform,
                heightTransform;

            if (!autoscale) {
                groupEle.setAttribute('transform', 'translate(50,50)scale(1)');
                return;
            }

            groupEleBB = groupEle.getBBox();

            // 100 is the view box
            widthTransform = 100 / groupEleBB.width;
            heightTransform = 100 / groupEleBB.height;

            // scale to the smaller one
            groupEle.setAttribute('transform', `translate(50,50)scale(${widthTransform < heightTransform ? widthTransform : heightTransform})`);
        }

        /**
         * @name initialize
         * @desc creates the visualization on the chart div
         * @returns {void}
         */
        function initialize() {
            let resizeListener,
                updateTaskListener,
                addDataListener,
                updateOrnamentsListener;

            // bind listeners
            resizeListener = scope.widgetCtrl.on('resize-widget', resize);
            updateTaskListener = scope.widgetCtrl.on('update-task', setData);
            addDataListener = scope.widgetCtrl.on('added-data', setData);
            updateOrnamentsListener = scope.widgetCtrl.on('update-ornaments', setData);

            scope.$on('$destroy', function () {
                resizeListener();
                updateTaskListener();
                updateOrnamentsListener();
                addDataListener();
            });

            groupEle = ele[0].querySelector('#kpi__group');
            valueEle = groupEle.querySelector('#kpi__group__value');
            titleEle = groupEle.querySelector('#kpi__group__title');

            setData();
        }

        initialize();
    }
}
