'use strict';

import angular from 'angular';

import './implied-insights.scss';

import { Grid } from 'ag-grid-community';
import '@/widgets/grid-standard/grid-standard.scss';

export default angular.module('app.implied-insights.directive', [])
    .directive('impliedInsights', impliedInsightsDirective);

impliedInsightsDirective.$inject = ['semossCoreService'];

function impliedInsightsDirective(semossCoreService: SemossCoreService) {
    impliedInsightsLink.$inject = [];

    return {
        restrict: 'E',
        template: require('./implied-insights.directive.html'),
        scope: {},
        require: ['^widget'],
        controllerAs: 'impliedInsights',
        bindToController: {},
        link: impliedInsightsLink,
        controller: impliedInsightsCtrl
    };

    function impliedInsightsCtrl() { }

    function impliedInsightsLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];

        let datasetGridEle,
            summaryGridEle,
            outlierGridEle,
            updateTaskListener = scope.widgetCtrl.on('update-task', getDataset);

        scope.impliedInsights.grid = {
            dataset: undefined,
            summary: undefined,
            outlier: undefined
        };

        scope.impliedInsights.data = {
            summary: {},
            outliers: {}
        }

        scope.impliedInsights.columns = [];
        scope.impliedInsights.selectedColumn = '';
        scope.impliedInsights.selectedValues = [];
        scope.impliedInsights.filters = [];
        scope.impliedInsights.filterType = '';
        scope.impliedInsights.filterHtml = '<div class="smss-small">Must select a column to filter.</div>';

        scope.impliedInsights.showOverlay = false;
        scope.impliedInsights.summaryType = '';
        scope.impliedInsights.summaryBtns = [];

        scope.impliedInsights.addFilter = addFilter;
        scope.impliedInsights.removeFilter = removeFilter;
        scope.impliedInsights.updateFilterType = updateFilterType;
        scope.impliedInsights.resetFilter = resetFilter;
        scope.impliedInsights.changeSummaryType = changeSummaryType;
        scope.impliedInsights.toggleOverlay = toggleOverlay;

        /**
         * @name toggleOverlay
         * @desc - show/hide the overlay
         */
        function toggleOverlay():void {
            scope.impliedInsights.showOverlay = !scope.impliedInsights.showOverlay;
        }

        /**
         * @name changeSummaryType
         * @desc changes the data to display in the summary table
         * @param type - what type of data to display
         */
        function changeSummaryType(type: string): void {
            scope.impliedInsights.summaryType = type;
            setData(scope.impliedInsights.grid.summary, scope.impliedInsights.data.summary[scope.impliedInsights.summaryType].viewHeaders, scope.impliedInsights.data.summary[scope.impliedInsights.summaryType].rawData);
        }

        /**
         * @name updateFilterType
         * @desc grabs the options and creates the correct filter when a column is selected
         */
        function updateFilterType(): void {
            let html,
                type = scope.impliedInsights.selectedColumn.dataType;
            scope.impliedInsights.filterType = type;
            scope.impliedInsights.selectedValues = [];

            scope.impliedInsights.filterOptions = scope.impliedInsights.data.outliers[scope.impliedInsights.selectedColumn.alias];
            if (type === 'NUMBER' || type === 'INT' || type === 'DOUBLE') {
                html = buildSliderHTML(type);
            } else {
                html = buildChecklistHTML();
            }
            scope.impliedInsights.filterHtml = html;
        }

        /**
         * @name resetFilter
         * @desc resets the components in the filter overlay
         */
        function resetFilter(): void{
            scope.selectedValues = [];
            scope.impliedInsights.selectedColumn = '';
            scope.impliedInsights.filterOptions = [];
            scope.impliedInsights.filterType = '';
            scope.impliedInsights.filterHtml = '<div class="smss-small">Must select a column to filter.</div>';
        }

        /**
         * @name buildChecklistHTML
         * @desc creates the html for the checklist filter
         */
        function buildChecklistHTML(): string{
            return `
            <smss-checklist model="impliedInsights.selectedValues"
                options="impliedInsights.filterOptions"
                searchable
                multiple
                quickselect
                style="height: 200px">
            </smss-checklist>
            `
        }

        /**
         * @name buildSliderHTML
         * @desc builds html for a number
         */
        function buildSliderHTML(): string {
            let min = parseFloat(scope.impliedInsights.filterOptions[0]),
                max = parseFloat(scope.impliedInsights.filterOptions[1]);
            scope.impliedInsights.selectedValues = [min, max];
            
            return `
            <smss-slider model="impliedInsights.selectedValues"
                multiple
                numerical
                min=${min}
                max=${max}>
            </smss-slider>
            `
        }

        /**
         * @name setColumnDefs
         * @desc creates the column definition for the grid based on the header data
         * @param data - the data for the table
         */
        function setColumnDefs(data): object[] {
            let defs: any = [];

            for (let i = 0; i < data.length; i++){
                defs.push({
                    headerName: String(data[i]).replace(/_/g, ' '),
                    field: data[i]
                })
            }

            return defs;
        }

        /**
         * @name paintGrids
         * @desc initializes an empty grid for each table
         */
        function paintGrids(): void {
            scope.impliedInsights.grid.dataset = new Grid(datasetGridEle, { rowData: [], columnDefs: [] });
            scope.impliedInsights.grid.summary = new Grid(summaryGridEle, { rowData: [], columnDefs: [] });
            scope.impliedInsights.grid.outlier = new Grid(outlierGridEle, { rowData: [], columnDefs: [], localeText: {noRowsToShow: 'There are no outliers in your dataset.',} });
        }

        /**
         * @name setData
         * @desc sets the data for the grid to display
         * @param grid - the grid to set the data for
         * @param columnData - the column headers
         * @param rowData - the data to be displayed
         */
        function setData(grid, columnData, rowData): void {
            let columnDefs = setColumnDefs(columnData);
            grid.gridOptions.api.setColumnDefs(columnDefs);
            grid.gridOptions.api.setRowData(rowData);
        }

        /**
         * @name getDataset
         * @desc retrieves the data for the dataset grid
         */
        function getDataset(): void {
            let data = scope.widgetCtrl.getWidget('view.visualization.tasks.0.data'),
                rowData = semossCoreService.visualization.getTableData(data.headers, data.values, data.headers);
            setData(scope.impliedInsights.grid.dataset, data.headers, rowData.rawData);
            runImpliedInsights();
        }
        /**
         * @name setFilterOptions
         * @desc sets the filter options using the outlier data
         * @param data - the outlier data
         */
        function setFilterOptions(data): void {
            let options = {},
                keys = scope.widgetCtrl.getWidget('view.visualization.keys.Grid'),
                outliers;
            for (let i = 0; i < data.length; i++) {
                let column = data[i].Column,
                    value = data[i].Outlier,
                    type = keys.find((key) => key.alias === column).dataType;
                if (options.hasOwnProperty(column)){
                    if (options[column].indexOf(value) === -1) {
                        options[column].push(value);
                    }
                } else {
                    if (type === "NUMBER" || type === "INT" || type === "DOUBLE") {
                        value = value.slice(1, -1).split(",");
                        options[column] = value;
                    } else {
                        options[column] = [value];
                    }
                }
            }
            outliers = Object.keys(options);
            scope.impliedInsights.columns = keys.filter(function(key) {
                return outliers.indexOf(key.alias) > -1;
            })
            scope.impliedInsights.data.outliers = options;
        }

        /**
         * @name runImpliedInsights
         * @desc retrieves the data for the outliers and summary data
         */
        function runImpliedInsights(): void {
            
            let callback = function(response) {
                const output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType[0];

                if (type.indexOf('ERROR') > -1) {
                    return;
                }
                let summary = output[0].output,
                    outliers = output[1].output;
                
                if (summary.length) {
                    for (let i = 0; i < summary.length; i++) {
                        scope.impliedInsights.data.summary[summary[i].datatype] = semossCoreService.visualization.getTableData(summary[i].headers, summary[i].values, summary[i].headers);
                        scope.impliedInsights.data.summary[summary[i].datatype].rawHeaders = summary[i].headers;
                    }
                    scope.impliedInsights.summaryBtns = Object.keys(scope.impliedInsights.data.summary);
                    scope.impliedInsights.summaryType = scope.impliedInsights.summaryBtns[0];
                    setData(scope.impliedInsights.grid.summary, scope.impliedInsights.data.summary[scope.impliedInsights.summaryType].rawHeaders, scope.impliedInsights.data.summary[scope.impliedInsights.summaryType].rawData);
                }
                if (outliers) {
                    let outlierTableData = semossCoreService.visualization.getTableData(outliers.headers, outliers.values, outliers.headers);
                    setData(scope.impliedInsights.grid.outlier, outliers.headers, outlierTableData.rawData);
                    setFilterOptions(outlierTableData.rawData);
                }
            }

            scope.widgetCtrl.meta([
                {
                    type: 'Pixel',
                    components: [
                        `Panel(${scope.widgetCtrl.panelId}) | RunImpliedInsights()`
                    ],
                    terminal: true
                }
            ], callback);
        }
        /**
         * @name createChecklistFilter
         * @desc creates the pixel command for a checklist filter
         */
        function createChecklistFilter(): PixelCommand[] {
            let components: PixelCommand[] = [],
                limit: number = scope.widgetCtrl.getOptions('limit'),
                taskOptionsComponent = {},
                i: number,
                len: number,
                layout: string = scope.widgetCtrl.getWidget('view.visualization.layout'),
                keys = scope.widgetCtrl.getWidget('view.visualization.keys.' + layout),
                selectComponent: any[] = [],
                selections: string[] = [];

            taskOptionsComponent[scope.widgetCtrl.panelId] = {
                'layout': layout,
                'alignment': {}
            };
            for (i = 0, len = keys.length; i < len; i++) {
                const key = keys.find(function(col) {
                    return col.alias === keys[i].alias;
                });
                if (!taskOptionsComponent[scope.widgetCtrl.panelId].alignment[key.model]) {
                    taskOptionsComponent[scope.widgetCtrl.panelId].alignment[key.model] = [];
                }
                taskOptionsComponent[scope.widgetCtrl.panelId].alignment[key.model].push(keys[i].alias);

                if (key.calculatedBy) {
                    selectComponent.push({
                        calculatedBy: keys.calculatedBy,
                        math: keys.math,
                        alias: keys.alias
                    });
                } else {
                    selectComponent.push({
                        alias: key.alias
                    });
                }
            }

            for (let k = 0; k < scope.impliedInsights.selectedValues.length; k++){
                let value = scope.impliedInsights.selectedValues[k].replace(/_/g, '_');
                selections.push(value);
            }

            components.push(
                {
                    type: 'Pixel',
                    components: [
                        `Panel(${scope.widgetCtrl.panelId})`
                    ]
                },
                {
                    type: 'addPanelFilter',
                    components: [
                        [
                            {
                                type: 'value',
                                alias: scope.impliedInsights.selectedColumn.alias,
                                comparator: '==',
                                values: selections
                            }
                        ]
                    ]
                },
                {
                    'type': 'frame',
                    'components': [
                        scope.widgetCtrl.getFrame('name')
                    ]
                },
                {
                    'type': 'select2',
                    'components': [
                        selectComponent
                    ]
                },
                {
                    type: 'with',
                    components: [
                        scope.widgetCtrl.panelId
                    ]
                },
                {
                    'type': 'format',
                    'components': ['table']
                },
                {
                    'type': 'taskOptions',
                    'components': [
                        taskOptionsComponent
                    ]
                },
                {
                    'type': 'collect',
                    'components': [limit],
                    'terminal': true
                },
                {
                    type: 'Pixel',
                    components: [
                        `Panel(${scope.widgetCtrl.panelId})`
                    ]
                },
                {
                    'type': 'setPanelView',
                    'components': [
                        'implied-insights'
                    ],
                    'terminal': true
                }
            )

            return components;
        }
        /**
         * @name createSliderFilter
         * @desc creates the pixel command for slider filter
         */
        function createSliderFilter(): PixelCommand[] {
            let components: PixelCommand[] = [],
                limit: number = scope.widgetCtrl.getOptions('limit'),
                taskOptionsComponent = {},
                i: number,
                len: number,
                layout: string = scope.widgetCtrl.getWidget('view.visualization.layout'),
                keys = scope.widgetCtrl.getWidget('view.visualization.keys.' + layout),
                selectComponent: any[] = [];
            taskOptionsComponent[scope.widgetCtrl.panelId] = {
                'layout': layout,
                'alignment': {}
            };
            for (i = 0, len = keys.length; i < len; i++) {
                const key = keys.find(function(col) {
                    return col.alias === keys[i].alias;
                });
                if (!taskOptionsComponent[scope.widgetCtrl.panelId].alignment[key.model]) {
                    taskOptionsComponent[scope.widgetCtrl.panelId].alignment[key.model] = [];
                }
                taskOptionsComponent[scope.widgetCtrl.panelId].alignment[key.model].push(keys[i].alias);

                if (key.calculatedBy) {
                    selectComponent.push({
                        calculatedBy: keys.calculatedBy,
                        math: keys.math,
                        alias: keys.alias
                    });
                } else {
                    selectComponent.push({
                        alias: key.alias
                    });
                }
            }

            components.push(
                {
                    type: 'Pixel',
                    components: [
                        `Panel(${scope.widgetCtrl.panelId})`
                    ]
                },
                {
                    type: 'setPanelFilter',
                    components: [
                        [
                            {
                                type: 'value',
                                alias: scope.impliedInsights.selectedColumn.alias,
                                comparator: '>=',
                                values: scope.impliedInsights.selectedValues[0],
                                operator: 'AND'
                            },
                            {
                                type: 'value',
                                alias: scope.impliedInsights.selectedColumn.alias,
                                comparator: '<=',
                                values: scope.impliedInsights.selectedValues[1]
                            }
                        ]
                    ],
                    terminal: true
                },
                {
                    'type': 'frame',
                    'components': [
                        scope.widgetCtrl.getFrame('name')
                    ]
                },
                {
                    'type': 'select2',
                    'components': [
                        selectComponent
                    ]
                },
                {
                    type: 'with',
                    components: [
                        scope.widgetCtrl.panelId
                    ]
                },
                {
                    'type': 'format',
                    'components': ['table']
                },
                {
                    'type': 'taskOptions',
                    'components': [
                        taskOptionsComponent
                    ]
                },
                {
                    'type': 'collect',
                    'components': [limit],
                    'terminal': true
                },
                {
                    type: 'Pixel',
                    components: [
                        `Panel(${scope.widgetCtrl.panelId})`
                    ]
                },
                {
                    'type': 'setPanelView',
                    'components': [
                        'implied-insights'
                    ],
                    'terminal': true
                }
            )
            return components;
        }
        
        /**
         * @name addFilter
         * @desc - called when a user wants to drill down their data by column anomaly
         */
        function addFilter(): void {
            let selected = scope.impliedInsights.selectedColumn.alias;
            scope.impliedInsights.showOverlay = false;
            let components: PixelCommand[] = [],
                callback;
  
            if (scope.impliedInsights.filterType === 'NUMBER' || scope.impliedInsights.filterType === 'INT') {
                components = createSliderFilter();
            } else {
                components = createChecklistFilter();
            }

            callback = function(response) {
                resetFilter();
                const output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType[0];

                if (type.indexOf('ERROR') > -1) {
                    return;
                }
                scope.impliedInsights.filters.push(selected);
                
            }
            scope.widgetCtrl.execute(components, callback);
        }
        
        /**
         * @name removeFilter
         * @param column - the column to unfilter
         * @desc - called when a user wants to remove a previous anomaly to go back to a past state
         */
        function removeFilter(column: string): void {
            let components: PixelCommand[] = [],
                callback,
                limit: number = scope.widgetCtrl.getOptions('limit'),
                taskOptionsComponent = {},
                i: number,
                len: number,
                layout: string = scope.widgetCtrl.getWidget('view.visualization.layout'),
                keys = scope.widgetCtrl.getWidget('view.visualization.keys.' + layout),
                selectComponent: any[] = [];
            taskOptionsComponent[scope.widgetCtrl.panelId] = {
                'layout': layout,
                'alignment': {}
            };
            for (i = 0, len = keys.length; i < len; i++) {
                const key = keys.find(function(col) {
                    return col.alias === keys[i].alias;
                });
                if (!taskOptionsComponent[scope.widgetCtrl.panelId].alignment[key.model]) {
                    taskOptionsComponent[scope.widgetCtrl.panelId].alignment[key.model] = [];
                }
                taskOptionsComponent[scope.widgetCtrl.panelId].alignment[key.model].push(keys[i].alias);

                if (key.calculatedBy) {
                    selectComponent.push({
                        calculatedBy: keys.calculatedBy,
                        math: keys.math,
                        alias: keys.alias
                    });
                } else {
                    selectComponent.push({
                        alias: key.alias
                    });
                }
            }

            components.push(
                {
                    type: 'Pixel',
                    components: [
                        `Panel(${scope.widgetCtrl.panelId})`
                    ]
                },
                {
                    type: 'Pixel',
                    components: [
                        `UnfilterPanel(["${column.replace(/ /g, '_')}"])`
                    ],
                    terminal: true
                },
                {
                    'type': 'frame',
                    'components': [
                        scope.widgetCtrl.getFrame('name')
                    ]
                },
                {
                    'type': 'select2',
                    'components': [
                        selectComponent
                    ]
                },
                {
                    type: 'with',
                    components: [
                        scope.widgetCtrl.panelId
                    ]
                },
                {
                    'type': 'format',
                    'components': ['table']
                },
                {
                    'type': 'taskOptions',
                    'components': [
                        taskOptionsComponent
                    ]
                },
                {
                    'type': 'collect',
                    'components': [limit],
                    'terminal': true
                },
                {
                    type: 'Pixel',
                    components: [
                        `Panel(${scope.widgetCtrl.panelId})`
                    ]
                },
                {
                    'type': 'setPanelView',
                    'components': [
                        'implied-insights'
                    ],
                    'terminal': true
                }
            )
            callback = function(response) {
                
                const output = response.pixelReturn[0].output,
                type = response.pixelReturn[0].operationType[0];

                if (type.indexOf('ERROR') > -1) {
                    return;
                }
                scope.impliedInsights.filters.pop();
            }
            scope.widgetCtrl.execute(components, callback);
        }


        /**
         * @name initialize
         * @desc called when the directive is loaded
         * @returns {void}
         */
        function initialize(): void {
            datasetGridEle = ele[0].querySelector('#dataset-grid');
            summaryGridEle = ele[0].querySelector('#summary-grid');
            outlierGridEle = ele[0].querySelector('#outlier-grid');

            paintGrids();
            getDataset();

            scope.$on('$destroy', function () {
                updateTaskListener();
            })

        }

        initialize();
    }
}
