'use strict';

import './filter.scss';

export default angular.module('app.filter.directive', [])
    .directive('filter', filterDirective);

filterDirective.$inject = ['semossCoreService', '$timeout', 'optionsService'];

function filterDirective(semossCoreService, $timeout, optionsService) {
    filterCtrl.$inject = [];
    filterLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        template: require('./filter.directive.html'),
        scope: {},
        require: ['^widget', '?^pipelineComponent'],
        controllerAs: 'filter',
        bindToController: {},
        controller: filterCtrl,
        link: filterLink
    };

    function filterCtrl() {}

    function filterLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];
        scope.pipelineComponentCtrl = ctrl[1];

        var searchTimeout;

        scope.filter.PIPELINE = scope.pipelineComponentCtrl !== null;

        // functions
        scope.filter.editFrameFilter = editFrameFilter;
        scope.filter.deleteFrameFilter = deleteFrameFilter;
        scope.filter.selectFilter = selectFilter;
        scope.filter.applyFilter = applyFilter;
        scope.filter.previewFilter = previewFilter;
        scope.filter.searchEquality = searchEquality;
        scope.filter.getMoreEquality = getMoreEquality;
        scope.filter.searchLike = searchLike;
        scope.filter.getMoreLike = getMoreLike;
        scope.filter.queueFilter = queueFilter;
        scope.filter.removeFromQueue = removeFromQueue;
        scope.filter.clearQueue = clearQueue;
        scope.filter.closeFilter = closeFilter;

        // variables
        scope.filter.selected = false;
        scope.filter.filter = {};
        scope.filter.comparator = '';
        scope.filter.comparatorOptions = [];
        scope.filter.frameFilters = [];
        scope.filter.queuedFilters = [];
        scope.filter.filterOperators = [
            {
                display: 'AND',
                value: 'AND'
            },
            {
                display: 'OR',
                value: 'OR'
            }
        ];

        /** Applied Filters */
        /**
         * @name getFrameName
         * @param {string} accessor - how do we want to access the frame?
         * @returns {*} frame options
         */
        function getFrame(accessor) {
            if (scope.filter.PIPELINE) {
                return scope.pipelineComponentCtrl.getComponent(accessor ? ('parameters.FRAME.value.' + accessor) : 'parameters.FRAME.value');
            }

            return scope.widgetCtrl.getFrame(accessor);
        }

        /**
         * @name setFrame
         * @returns {void}
         */
        function setFrame() {
            if (!getFrame('name')) {
                closeFilter();
                return;
            }

            // clear queued
            scope.filter.queuedFilters = [];

            getFrameHeaders();
            getFrameFilters();

            if (scope.filter.PIPELINE) {
                previewFilter(false);
            }
        }

        /**
         * @name getFrameFilters
         * @returns {void}
         */
        function getFrameFilters() {
            var callback;

            // register function to come back
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    i, j;

                for (i = 0; i < output.length; i++) {
                    if (output[i].filterObj.filterType === 'SIMPLE') {
                        output[i].filterObj.right.display = '';

                        if (Array.isArray(output[i].filterObj.left.value)) {
                            output[i].filterObj.left.display = output[i].filterObj.left.value.join(', ');
                        } else {
                            output[i].filterObj.left.display = String(output[i].filterObj.left.value);
                        }

                        output[i].filterObj.left.display = output[i].filterObj.left.display.replace(/_/g, ' ');

                        if (Array.isArray(output[i].filterObj.right.value)) {
                            for (j = 0; j < output[i].filterObj.right.value.length; j++) {
                                output[i].filterObj.right.display += output[i].filterObj.right.value[j] || output[i].filterObj.right.value[j] === 0 ? output[i].filterObj.right.value[j] : 'null';
                                if (output[i].filterObj.right.value.length - 1 !== j) {
                                    output[i].filterObj.right.display += ', ';
                                }
                            }
                        } else {
                            output[i].filterObj.right.display = output[i].filterObj.right.value || output[i].filterObj.right.value === 0 ? String(output[i].filterObj.right.value) : 'null';
                        }

                        output[i].filterObj.right.display = output[i].filterObj.right.display.replace(/_/g, ' ');
                    } else {
                        output[i].filterStrDisplay = output[i].filterStr.replace(/_/g, ' ');
                    }
                }

                // set
                scope.filter.frameFilters = output;
            };

            // execute a meta query
            scope.widgetCtrl.meta([
                {
                    type: 'variable',
                    components: [
                        getFrame('name')
                    ]
                },
                {
                    type: 'getFrameFilters',
                    components: [],
                    terminal: true
                }
            ], callback);
        }

        /**
         * @name editFrameFilter
         * @param {number} index - the index of the frame filter
         * @desc edit the existing filter
         * @returns {void}
         */
        function editFrameFilter(index) {
            const comparator = scope.filter.frameFilters[index].filterObj.comparator,
                values = scope.filter.frameFilters[index].filterObj.right.value;
            let callback = function () {
                scope.filter.selectFilter(scope.filter.selected, comparator, values);
            };

            scope.filter.selected = scope.filter.frameFilters[index].filterObj.left.value;

            // if (comparator === '==' || comparator === '!=') {
            //     scope.filter.filter[scope.filter.selected].equality.loading = true;
            // } else if (comparator === '<' || comparator === '<=' || comparator === '>' || comparator === '>=') {
            //     scope.filter.filter[scope.filter.selected].numerical.loading = true;
            // } else if (comparator === '?like') {
            //     scope.filter.filter[scope.filter.selected].like.loading = true;
            // }

            deleteFrameFilter(index, callback);
        }

        /**
         * @name deleteFrameFilter
         * @param {number} index - the index of the frame filter
         * @param {function} callback the callback function to run after deleting
         * @desc unfilter the selected filter
         * @returns {void}
         */
        function deleteFrameFilter(index, callback) {
            var parameters,
                components = [],
                pixel,
                filterString,
                panels,
                i,
                len;

            scope.filter.deleteFilterIndex = index;

            if (!validateFilter('delete', true)) {
                return;
            }

            parameters = buildParameters('delete');
            scope.filter.deleteFilterIndex = undefined;

            if (scope.filter.PIPELINE) {
                scope.pipelineComponentCtrl.executeComponent(parameters, {
                    name: 'Remove Filter'
                });
                return;
            }

            filterString = semossCoreService.pixel.build([{
                type: 'FILTERS',
                components: [
                    parameters.FILTERS
                ]
            }]);
            filterString = filterString.slice(0, -2);
            pixel = `${parameters.FRAME.name} | ${parameters.OPERATION}(${filterString});`;


            components.push({
                type: 'Pixel',
                components: [
                    pixel
                ],
                terminal: true
            });

            // refresh if it isn't pipeline
            panels = scope.widgetCtrl.getShared('panels') || [];
            for (i = 0, len = panels.length; i < len; i++) {
                components.push({
                    'type': 'refresh',
                    'components': [panels[i].widgetId],
                    'terminal': true
                });
            }

            scope.widgetCtrl.execute(components, callback);
        }

        /** New Filters */
        /**
         * @name getFrameHeaders
         * @desc updates filter options
         * @returns {void}
         */
        function getFrameHeaders() {
            scope.filter.headers = getFrame('headers') || [];

            if (scope.filter.headers.length > 0) {
                selectFilter(scope.filter.headers[0].alias, false);
            }
        }

        /**
         * @name selectFilter
         * @param {string} alias - aliased filter to select
         * @param {string} comparator - selected comparator
         * @param {array} selectedValues - list of values to be selected. optional.
         * @desc updates filter model for this filter
         * @returns {void}
         */
        function selectFilter(alias, comparator, selectedValues) {
            var defaultInitialComparatorOptions = {
                    'NUMBER': '==',
                    'STRING': '==',
                    'DATE': '=='
                },
                defaultComparatorOptions = [
                    {
                        'display': 'Equal To',
                        'value': '==',
                        'acceptableTypes': ['NUMBER', 'STRING', 'DATE']
                    },
                    {
                        'display': 'Not Equal To',
                        'value': '!=',
                        'acceptableTypes': ['NUMBER', 'STRING', 'DATE']
                    },
                    {
                        'display': 'Contains',
                        'value': '?like',
                        'acceptableTypes': ['STRING']
                    },
                    {
                        'display': 'Less Than',
                        'value': '<',
                        'acceptableTypes': ['NUMBER']
                    },
                    {
                        'display': 'Less Than or Equal To',
                        'value': '<=',
                        'acceptableTypes': ['NUMBER']
                    },
                    {
                        'display': 'Greater Than',
                        'value': '>',
                        'acceptableTypes': ['NUMBER']
                    },
                    {
                        'display': 'Greater Than or Equal To',
                        'value': '>=',
                        'acceptableTypes': ['NUMBER']
                    }
                ],
                i,
                len,
                j,
                len2,
                focusTimeout;

            // set the selected
            scope.filter.selected = alias;

            // set the comparator
            // get the correct list of comparators
            scope.filter.comparatorOptions = [];
            for (i = 0, len = scope.filter.headers.length; i < len; i++) {
                if (scope.filter.headers[i].alias === scope.filter.selected) {
                    for (j = 0, len2 = defaultComparatorOptions.length; j < len2; j++) {
                        if (defaultComparatorOptions[j].acceptableTypes.indexOf(scope.filter.headers[i].dataType) > -1) {
                            scope.filter.comparatorOptions.push(defaultComparatorOptions[j]);
                        }
                    }
                    break;
                }
            }

            // set a default comparator
            scope.filter.comparator = false;
            if (comparator) {
                for (i = 0, len = scope.filter.comparatorOptions.length; i < len; i++) {
                    if (scope.filter.comparatorOptions[i].value === comparator) {
                        scope.filter.comparator = comparator;
                    }
                }
            }

            if (!scope.filter.comparator && defaultInitialComparatorOptions.hasOwnProperty(scope.filter.headers[i].dataType)) {
                scope.filter.comparator = defaultInitialComparatorOptions[scope.filter.headers[i].dataType];
            }

            if (!scope.filter.comparator && scope.filter.comparatorOptions.length > 0) {
                scope.filter.comparator = scope.filter.comparatorOptions[0].value;
            }

            // reset the object (new alias was selected)
            scope.filter.filter[alias] = {
                equality: {
                    loading: false,
                    taskId: false,
                    options: [], // all values on the dom for the alias
                    model: [], // all checked values on the dom for the alias,
                    search: '', // search term used - only used for search term in filtermodel call, nothing else
                    limit: 50, // how many filter values to collect
                    canCollect: true // flag to determine whether another filter model call should be made - infinite scroll,
                },
                numerical: {
                    loading: true,
                    model: undefined,
                    min: -Infinity,
                    max: Infinity
                },
                like: {
                    loading: false,
                    taskId: false,
                    options: [], // all values on the dom for the alias
                    model: undefined, // actual model value
                    limit: 50, // how many values to collect
                    canCollect: true // infinite scroll,
                }
            };

            if (scope.filter.comparator === '==' || scope.filter.comparator === '!=') {
                resetEquality(selectedValues);

                focusTimeout = $timeout(function () {
                    var focusedElement = ele[0].querySelector('#filter-equality');
                    if (focusedElement) {
                        focusedElement.focus();
                    }
                    $timeout.cancel(focusTimeout);
                });
            } else if (scope.filter.comparator === '<' || scope.filter.comparator === '<=' || scope.filter.comparator === '>' || scope.filter.comparator === '>=') {
                resetNumerical(selectedValues);

                focusTimeout = $timeout(function () {
                    var focusedElement = ele[0].querySelector('#filter-numerical');
                    if (focusedElement) {
                        focusedElement.focus();
                    }
                    $timeout.cancel(focusTimeout);
                });
            } else if (scope.filter.comparator === '?like') {
                resetLike(selectedValues);

                focusTimeout = $timeout(function () {
                    var focusedElement = ele[0].querySelector('#filter-like');
                    if (focusedElement) {
                        focusedElement.focus();
                    }
                    $timeout.cancel(focusTimeout);
                });
            } else {
                console.error('Not built');
            }
        }


        /**
         * @name buildParameters
         * @desc build the parameters for the current module
         * @param {string} type - type of filter
         * @returns {object} - map of the paramters to value
         */
        function buildParameters(type) {
            var filters = [],
                filterType,
                queuedIdx,
                queuedLength;

            if (type === 'delete') {
                filterType = 'DeleteFrameFilter';
                filters.push({
                    type: 'delete',
                    values: scope.filter.deleteFilterIndex
                });
            } else if (type === 'apply') {
                // loop through the queuedFilters
                if (scope.filter.queuedFilters.length > 0) {
                    filterType = 'AddFrameFilter';
                    for (queuedIdx = 0, queuedLength = scope.filter.queuedFilters.length; queuedIdx < queuedLength; queuedIdx++) {
                        // we push the AND/OR into the same array so every other index will be AND/OR
                        // we know whether it's a filter or AND/OR operator by looking at the isFilter boolean
                        if (scope.filter.queuedFilters[queuedIdx].isFilter) {
                            filters.push({
                                type: scope.filter.queuedFilters[queuedIdx].type,
                                operator: scope.filter.queuedFilters[queuedIdx + 1] && scope.filter.queuedFilters[queuedIdx + 1].value ? scope.filter.queuedFilters[queuedIdx + 1].value : '',
                                alias: scope.filter.queuedFilters[queuedIdx].alias,
                                comparator: scope.filter.queuedFilters[queuedIdx].comparator,
                                values: scope.filter.queuedFilters[queuedIdx].values
                            });
                        }
                    }
                } else if (scope.filter.comparator === '==' || scope.filter.comparator === '!=') { // based off the comparator
                    filterType = 'ReplaceFrameFilter';
                    filters.push({
                        type: 'value',
                        operator: '',
                        alias: scope.filter.selected,
                        comparator: scope.filter.comparator,
                        values: scope.filter.filter[scope.filter.selected].equality.model
                    });
                } else if (scope.filter.comparator === '<' || scope.filter.comparator === '<=' || scope.filter.comparator === '>' || scope.filter.comparator === '>=') {
                    filterType = 'ReplaceFrameFilter';
                    filters.push({
                        operator: '',
                        type: 'value',
                        alias: scope.filter.selected,
                        comparator: scope.filter.comparator,
                        values: scope.filter.filter[scope.filter.selected].numerical.model
                    });
                } else if (scope.filter.comparator === '?like') {
                    filterType = 'AddFrameFilter';
                    filters.push({
                        operator: '',
                        type: 'value',
                        alias: scope.filter.selected,
                        comparator: scope.filter.comparator,
                        values: scope.filter.filter[scope.filter.selected].like.model
                    });
                }
            }

            return {
                FRAME: getFrame(),
                OPERATION: filterType,
                FILTERS: filters
            };
        }

        /**
         * @name applyFilter
         * @desc takes current filter options and sends them to sheet to update visualization data
         * @returns {void}
         */
        function applyFilter() {
            var parameters = {},
                components = [],
                pixel,
                filterString,
                panels,
                i,
                len;

            if (!validateFilter('apply', true)) {
                return;
            }

            parameters = buildParameters('apply');
            if (scope.filter.PIPELINE) {
                scope.pipelineComponentCtrl.executeComponent(parameters, {
                    name: 'Add Filter'
                });
                return;
            }

            console.warn('TODO: Maybe I can use the config and rebuild .... if I do that, why dont I put this on the widget?');
            filterString = semossCoreService.pixel.build([{
                type: 'FILTERS',
                components: [
                    parameters.FILTERS
                ]
            }]);
            filterString = filterString.slice(0, -2);
            pixel = `${parameters.FRAME.name} | ${parameters.OPERATION}(${filterString});`;

            components = [{
                type: 'Pixel',
                components: [
                    pixel
                ],
                terminal: true
            }];

            // refresh if it isn't pipeline
            panels = scope.widgetCtrl.getShared('panels') || [];
            for (i = 0, len = panels.length; i < len; i++) {
                components.push({
                    'type': 'refresh',
                    'components': [panels[i].widgetId],
                    'terminal': true
                });
            }

            scope.widgetCtrl.execute(components);
            optionsService.set(scope.widgetCtrl.widgetId, 'filteredColumn', scope.filter.selected);
        }

        /**
         * @name validateFilter
         * @param {boolean} type - type of filter to validate
         * @param {boolean} alert - message on errors
         * @desc validate the filter options
         * @returns {boolean} is the merge valid
         */
        function validateFilter(type, alert) {
            if (type === 'delete') {
                if (typeof scope.filter.deleteFilterIndex === 'undefined' || isNaN(scope.filter.deleteFilterIndex)) {
                    if (alert) {
                        scope.widgetCtrl.alert('warn', 'Delete is not defined. Please select a filter to delete.');
                    }
                    return false; // not valid
                }

                return true; // valid
            } else if (type === 'apply') {
                if (scope.filter.queuedFilters.length > 0) {
                    return true;
                }

                if (!scope.filter.filter[scope.filter.selected]) {
                    if (alert) {
                        scope.widgetCtrl.alert('warn', 'No filter is selected. Please select a filter to add.');
                    }
                    return false;
                }

                if (scope.filter.comparator === '==' || scope.filter.comparator === '!=') {
                    if (!scope.filter.filter[scope.filter.selected].equality) {
                        if (alert) {
                            scope.widgetCtrl.alert('warn', 'No options are selected. Please select options to filter.');
                        }
                        return false;
                    }

                    return scope.filter.filter[scope.filter.selected].equality.model.length !== 0;
                } else if (scope.filter.comparator === '<' || scope.filter.comparator === '<=' || scope.filter.comparator === '>' || scope.filter.comparator === '>=') {
                    if (!scope.filter.filter[scope.filter.selected].numerical) {
                        if (alert) {
                            scope.widgetCtrl.alert('warn', 'No options are selected. Please enter a number to filter by.');
                        }
                        return false;
                    }

                    return typeof scope.filter.filter[scope.filter.selected].numerical.model !== 'undefined';
                } else if (scope.filter.comparator === '?like') {
                    if (!scope.filter.filter[scope.filter.selected].like) {
                        if (alert) {
                            scope.widgetCtrl.alert('warn', 'No options are selected. Please input a string to filter by.');
                        }
                        return false;
                    }

                    return !!scope.filter.filter[scope.filter.selected].like.model;
                }

                return false;
            }

            return false;
        }


        /**
         * @name previewFilter
         * @desc takes current filter options and sends them to sheet to update visualization data
         * @returns {void}
         */
        function previewFilter() {
            if (!scope.filter.PIPELINE) {
                return;
            }

            var parameters = {};

            if (validateFilter('apply', alert)) {
                parameters = buildParameters('apply', alert);
            }

            scope.pipelineComponentCtrl.previewComponent(parameters);
        }

        /**
         * @name queueFilter
         * @desc queues the filter
         * @returns {void}
         */
        function queueFilter() {
            var filterToQueue = {
                isFilter: true,
                alias: scope.filter.selected,
                comparator: scope.filter.comparator
            };

            // based off the comparator
            if (scope.filter.comparator === '==' || scope.filter.comparator === '!=') {
                filterToQueue.values = scope.filter.filter[scope.filter.selected].equality.model;
                filterToQueue.type = 'value';
            } else if (scope.filter.comparator === '<' || scope.filter.comparator === '<=' || scope.filter.comparator === '>' || scope.filter.comparator === '>=') {
                filterToQueue.values = scope.filter.filter[scope.filter.selected].numerical.model;
                filterToQueue.type = 'value';
            } else if (scope.filter.comparator === '?like') {
                filterToQueue.values = scope.filter.filter[scope.filter.selected].like.model;
                filterToQueue.type = 'value';
            }

            if (scope.filter.queuedFilters.length !== 0) {
                scope.filter.queuedFilters.push({
                    isFilter: false,
                    value: 'AND',
                    name: 'AND'
                });
            }

            scope.filter.queuedFilters.push(JSON.parse(JSON.stringify(filterToQueue)));

            // reset selected
            selectFilter(scope.filter.selected, false);
        }

        /**
         * @name removeFromQueue
         * @param {*} index the index to remove from the queue
         * @desc remove the filter from the queue
         * @returns {void}
         */
        function removeFromQueue(index) {
            if (index !== 0) {
                // remove the filter and the operator before it
                scope.filter.queuedFilters.splice(index, 1);
                scope.filter.queuedFilters.splice(index - 1, 1);
            } else {
                scope.filter.queuedFilters.splice(index, 1);
            }
        }

        /**
         * @name clearQueue
         * @param {*} index the index to remove from the queue
         * @desc remove all of the filters from the queue
         * @returns {void}
         */
        function clearQueue() {
            scope.filter.queuedFilters = [];

            // reset selected
            selectFilter(scope.filter.selected, false);
        }

        /** Specific */
        /** Equality */
        /**
         * @name resetEquality
         * @param {array} defaultValues values to set. optional
         * @desc updates filter model for this filter
         * @returns {void}
         */
        function resetEquality(defaultValues) {
            var components = [],
                filterObj = {},
                callback;

            scope.filter.filter[scope.filter.selected].equality.loading = true;

            components.push({
                type: 'frame',
                components: [
                    getFrame('name')
                ],
                meta: true
            }, {
                type: 'select2',
                components: [
                    [{
                        alias: scope.filter.selected
                    }]
                ]
            });

            if (scope.filter.filter[scope.filter.selected].equality.search) {
                // search
                filterObj[scope.filter.selected] = {
                    comparator: '?like',
                    value: [scope.filter.filter[scope.filter.selected].equality.search]
                };

                components.push({
                    type: 'filter',
                    components: [filterObj]
                });
            }

            components.push({
                type: 'collect',
                components: [
                    scope.filter.filter[scope.filter.selected].equality.limit
                ],
                terminal: true
            });

            // register callback
            callback = function (response) {
                var returnData = response.pixelReturn[0].output,
                    i, len;

                // only empty when there is no existing task
                if (!scope.filter.filter[scope.filter.selected].equality.taskId) {
                    scope.filter.filter[scope.filter.selected].equality.model = [];
                }

                if (defaultValues) {
                    scope.filter.filter[scope.filter.selected].equality.model = defaultValues;
                }

                // no need to clear out because we are loading infinite
                scope.filter.filter[scope.filter.selected].equality.options = [];

                // add new ones
                for (i = 0, len = returnData.data.values.length; i < len; i++) {
                    scope.filter.filter[scope.filter.selected].equality.options.push(returnData.data.values[i][0]);
                }

                // clean
                scope.filter.filter[scope.filter.selected].equality.options = cleanValues(scope.filter.filter[scope.filter.selected].equality.options, true);

                // update filter control variables
                scope.filter.filter[scope.filter.selected].equality.taskId = returnData.taskId;
                scope.filter.filter[scope.filter.selected].equality.canCollect = (returnData.numCollected === returnData.data.values.length);
                scope.filter.filter[scope.filter.selected].equality.loading = false;
            };


            scope.widgetCtrl.meta(components, callback);
        }

        /**
         * @name searchEquality
         * @desc search for the options based on searchTerm
         * @returns {void}
         */
        function searchEquality() {
            if (searchTimeout) {
                $timeout.cancel(searchTimeout);
            }
            searchTimeout = $timeout(function () {
                // reset vars
                resetEquality();
                $timeout.cancel(searchTimeout);
            }, 500);
        }

        /**
         * @name getMoreEquality
         * @desc get the next set of options
         * @returns {void}
         */
        function getMoreEquality() {
            var components = [],
                callback;

            if (scope.filter.filter[scope.filter.selected].equality.canCollect) {
                scope.filter.filter[scope.filter.selected].equality.loading = true;

                components = [
                    {
                        'type': 'task',
                        'components': [
                            scope.filter.filter[scope.filter.selected].equality.taskId
                        ]
                    },
                    {
                        'type': 'collect',
                        'components': [
                            scope.filter.filter[scope.filter.selected].equality.limit
                        ],
                        'terminal': true
                    }
                ];

                // register message to come back to
                callback = function (response) {
                    var returnData = response.pixelReturn[0].output,
                        i, len;

                    // add new ones
                    for (i = 0, len = returnData.data.values.length; i < len; i++) {
                        scope.filter.filter[scope.filter.selected].equality.options.push(returnData.data.values[i][0]);
                    }

                    // clean
                    scope.filter.filter[scope.filter.selected].equality.options = cleanValues(scope.filter.filter[scope.filter.selected].equality.options, true);

                    // update filter control variables
                    scope.filter.filter[scope.filter.selected].equality.taskId = returnData.taskId;
                    scope.filter.filter[scope.filter.selected].equality.canCollect = (returnData.numCollected === returnData.data.values.length);
                    scope.filter.filter[scope.filter.selected].equality.loading = false;
                };

                scope.widgetCtrl.meta(components, callback);
            }
        }

        /** Numerical */
        /**
         * @name resetNumerical
         * @param {array} defaultValues the default values to set. optional.
         * @desc updates filter model for this filter
         * @returns {void}
         */
        function resetNumerical(defaultValues) {
            var components,
                callback;

            scope.filter.filter[scope.filter.selected].numerical.loading = true;

            components = [
                {
                    type: 'frame',
                    components: [
                        getFrame('name')
                    ],
                    meta: true
                },
                {
                    type: 'select2',
                    components: [
                        [
                            {
                                math: 'Max',
                                alias: 'Max_of_' + scope.filter.selected,
                                calculatedBy: scope.filter.selected
                            },
                            {
                                math: 'Min',
                                alias: 'Min_of_' + scope.filter.selected,
                                calculatedBy: scope.filter.selected
                            }
                        ]
                    ]
                },
                {
                    type: 'collect',
                    components: [
                        -1
                    ],
                    terminal: true
                }
            ];

            // register message to come back to
            callback = function (response) {
                var returnData = response.pixelReturn[0].output;
                scope.filter.filter[scope.filter.selected].numerical.max = returnData.data.values[0][0];
                scope.filter.filter[scope.filter.selected].numerical.min = returnData.data.values[0][1];
                scope.filter.filter[scope.filter.selected].numerical.model = undefined;
                if (defaultValues) {
                    scope.filter.filter[scope.filter.selected].numerical.model = defaultValues;
                }
                scope.filter.filter[scope.filter.selected].numerical.loading = false;
            };

            scope.widgetCtrl.meta(components, callback);
        }


        /** Like */
        /**
         * @name resetLike
         * @param {string} defaultValue the default value to set. optional.
         * @desc updates filter model for this filter
         * @returns {void}
         */
        function resetLike(defaultValue) {
            var components = [],
                filterObj = {},
                callback;

            scope.filter.filter[scope.filter.selected].like.loading = true;

            if (defaultValue) {
                scope.filter.filter[scope.filter.selected].like.model = defaultValue;
            }

            components.push({
                type: 'frame',
                components: [
                    getFrame('name')
                ],
                meta: true
            }, {
                type: 'select2',
                components: [
                    [{
                        alias: scope.filter.selected
                    }]
                ]
            });

            if (scope.filter.filter[scope.filter.selected].like.model) {
                // search
                filterObj[scope.filter.selected] = {
                    comparator: '?like',
                    value: [scope.filter.filter[scope.filter.selected].like.model]
                };

                components.push({
                    type: 'filter',
                    components: [filterObj]
                });
            }

            components.push({
                type: 'collect',
                components: [
                    scope.filter.filter[scope.filter.selected].like.limit
                ],
                terminal: true
            });

            callback = function (response) {
                var returnData = response.pixelReturn[0].output,
                    i, len;

                // no need to clear out because we are loading infinite
                scope.filter.filter[scope.filter.selected].like.options = [];

                // add new ones
                for (i = 0, len = returnData.data.values.length; i < len; i++) {
                    scope.filter.filter[scope.filter.selected].like.options.push(returnData.data.values[i][0]);
                }

                // clean
                scope.filter.filter[scope.filter.selected].like.options = cleanValues(scope.filter.filter[scope.filter.selected].like.options, true);

                // update filter control variables
                scope.filter.filter[scope.filter.selected].like.taskId = returnData.taskId;
                scope.filter.filter[scope.filter.selected].like.canCollect = (returnData.numCollected === returnData.data.values.length);
                scope.filter.filter[scope.filter.selected].like.loading = false;
            };

            scope.widgetCtrl.meta(components, callback);
        }

        /**
         * @name searchLike
         * @desc search for the options based on searchTerm
         * @returns {void}
         */
        function searchLike() {
            if (searchTimeout) {
                $timeout.cancel(searchTimeout);
            }
            searchTimeout = $timeout(function () {
                // reset vars
                resetLike();
                $timeout.cancel(searchTimeout);
            }, 500);
        }

        /**
         * @name getMoreLike
         * @desc get the next set of options
         * @returns {void}
         */
        function getMoreLike() {
            var components = [],
                callback;

            if (scope.filter.filter[scope.filter.selected].like.canCollect) {
                scope.filter.filter[scope.filter.selected].like.loading = true;

                components = [
                    {
                        'type': 'task',
                        'components': [
                            scope.filter.filter[scope.filter.selected].like.taskId
                        ]
                    },
                    {
                        'type': 'collect',
                        'components': [
                            scope.filter.filter[scope.filter.selected].like.limit
                        ],
                        'terminal': true
                    }
                ];

                // register message to come back to
                callback = function (response) {
                    var returnData = response.pixelReturn[0].output,
                        i, len;

                    // add new ones
                    for (i = 0, len = returnData.data.values.length; i < len; i++) {
                        scope.filter.filter[scope.filter.selected].like.options.push(returnData.data.values[i][0]);
                    }

                    // clean
                    scope.filter.filter[scope.filter.selected].like.options = cleanValues(scope.filter.filter[scope.filter.selected].like.options, true);

                    // update filter control variables
                    scope.filter.filter[scope.filter.selected].like.taskId = returnData.taskId;
                    scope.filter.filter[scope.filter.selected].like.canCollect = (returnData.numCollected === returnData.data.values.length);
                    scope.filter.filter[scope.filter.selected].like.loading = false;
                };

                scope.widgetCtrl.meta(components, callback);
            }
        }

        /** Pipeline */
        /**
         * @name closeFilter
         * @desc close the filter when the pipeline is closed
         * @returns {void}
         */
        function closeFilter() {
            if (scope.filter.PIPELINE) {
                scope.pipelineComponentCtrl.closeComponent();
            }
        }
        /** Helpers */
        /**
         * @name cleanValues
         * @desc uniques the values
         * @param {array} values - values to display
         * @param {boolean} sort - true or fale for sort
         * @returns {array} values
         */
        function cleanValues(values, sort) {
            var unique = [],
                seen = {},
                i, len;

            if (!values) {
                return [];
            }

            for (i = 0, len = values.length; i < len; i++) {
                if (!seen[values[i]]) {
                    unique.push(values[i]);
                    seen[values[i]] = true;
                }
            }

            if (!sort) {
                return unique;
            }

            return unique.sort(function (a, b) {
                var textA = typeof a === 'string' ? a.toUpperCase() : a,
                    textB = typeof b === 'string' ? b.toUpperCase() : b;
                if (textA < textB) {
                    return -1;
                }
                if (textA > textB) {
                    return 1;
                }

                return 0;
            });
        }

        /**
         * @name initialize
         * @desc function that is called on directive load
         * @returns {void}
         */
        function initialize() {
            var filterUpdateTaskListener,
                // filterUpdateFrameListener,
                selectedDataListener,
                filteredColumn = optionsService.get(scope.widgetCtrl.widgetId, 'filteredColumn');

            // append listeners
            filterUpdateTaskListener = scope.widgetCtrl.on('update-frame-filter', function () {
                if (scope.filter.selected) {
                    selectFilter(scope.filter.selected, scope.filter.comparator);
                }
                getFrameFilters();
                clearQueue();
                // setFrame();
            });

            // filterUpdateFrameListener = scope.widgetCtrl.on('update-frame', function () {
            //     setFrame();
            // });

            selectedDataListener = scope.widgetCtrl.on('update-selected', function () {
                var selected = scope.widgetCtrl.getSelected('selected');
                if (selected.length > 0) {
                    selectFilter(selected[0].alias, false);
                }
            });


            // cleanup
            scope.$on('$destroy', function () {
                console.log('destroying filter...');
                filterUpdateTaskListener();
                // filterUpdateFrameListener();
                selectedDataListener();
            });

            setFrame();
            if (filteredColumn) {
                scope.filter.selected = filteredColumn;
                selectFilter(filteredColumn, false);
            }
        }

        initialize();
    }
}
