import angular from 'angular';

import './embed-link.scss';

export default angular.module('app.embed-link.directive', [])
    .directive('embedLink', embedLinkDirective);

embedLinkDirective.$inject = ['$location', '$window', 'semossCoreService'];

function embedLinkDirective($location, $window, semossCoreService) {
    embedLinkCtrl.$inject = [];
    embedLinkLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'EA',
        scope: {},
        require: ['^insight'],
        controllerAs: 'embedLink',
        bindToController: {},
        template: require('./embed-link.directive.html'),
        controller: embedLinkCtrl,
        link: embedLinkLink
    };

    function embedLinkCtrl() { }

    function embedLinkLink(scope, ele, attrs, ctrl) {
        scope.insightCtrl = ctrl[0];

        scope.embedLink.embed = {
            url: '',
            code: '',
            config: {
                resizable: false
            },
            custom: {
                open: false,
                valid: false,
                path: ''
            }
        }

        scope.embedLink.toggleCustomEmbed = toggleCustomEmbed;
        scope.embedLink.saveCustomEmbed = saveCustomEmbed;
        scope.embedLink.copyEmbed = copyEmbed;


        /**
      * @name toggleCustomEmbed
      * @desc toggles the custom
      */
        function toggleCustomEmbed(): void {
            scope.embedLink.embed.custom.path = '';
            scope.embedLink.embed.custom.valid = false;

            if (!scope.embedLink.embed.custom.open) {
                updateEmbed();
            }
        }

        /**
         * @name saveCustomEmbed
         * @desc save the custom
         */
        function saveCustomEmbed(): void {
            let callback,
                appId: string,
                appInsightId: string;

            if (!scope.embedLink.embed.custom.path.match(/^[a-zA-Z0-9-_]+$/)) {
                semossCoreService.emit('alert', {
                    color: 'error',
                    text: 'Path is not a valid URL'
                });
                return;
            }

            callback = function (response: PixelReturnPayload) {
                let output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') > -1) {
                    return;
                }

                semossCoreService.emit('alert', {
                    color: 'success',
                    text: output
                });

                scope.embedLink.embed.custom.valid = true;

                updateEmbed();
            };

            appId = scope.insightCtrl.getShared('insight.app_id');
            appInsightId = scope.insightCtrl.getShared('insight.app_insight_id');

            scope.insightCtrl.meta([{
                type: 'Pixel',
                components: [
                    `badd(fancy=["${scope.embedLink.embed.custom.path}"], embed=["<encode>#!/insight?engine=${appId}&id=${appInsightId}</encode>"] )`
                ],
                meta: true,
                terminal: true
            }], callback);
        }

        /**
         * @name copyEmbed
         * @param {string} content - content to copy
         */
        function copyEmbed(content: string): void {
            // For IE.
            if ($window.clipboardData) {
                $window.clipboardData.setData('Text', content);

                semossCoreService.emit('alert', {
                    color: 'success',
                    text: 'Successfully copied to clipboard'
                });
            } else {
                let exportElement = angular.element("<textarea style='position:fixed;left:-1000px;top:-1000px;'>" + content + '</textarea>');
                ele.append(exportElement);
                (exportElement as any).select();


                if (document.execCommand('copy')) {
                    exportElement.remove();

                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully copied to clipboard'
                    });
                } else {
                    exportElement.remove();
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'Unsuccessfully copied to clipboard'
                    });
                }
            }
        }


        /** Updates */
        /**
         * @name updateEmbed
         * @desc updates the embed string
         */
        function updateEmbed(): void {
            let appId = scope.insightCtrl.getShared('insight.app_id'),
                appInsightId = scope.insightCtrl.getShared('insight.app_insight_id');

            scope.embedLink.embed.url = $location.absUrl().split('#')[0];
            if (scope.embedLink.embed.custom.open) {
                scope.embedLink.embed.url += `#!/r/${scope.embedLink.embed.custom.path}`;
            } else {
                scope.embedLink.embed.url += `#!/insight?engine=${appId}&id=${appInsightId}`
            }

            scope.embedLink.embed.code = `<iframe frameborder="0" width="1000" height="600" style="border: 1px solid #ccc; ${scope.embedLink.embed.config.resizable ? 'resize: both;overflow: auto;' : ''}"
                src="${scope.embedLink.embed.url}">/<iframe>`
        }


        /**
         * @name initialize
         * @desc initialize the directive
         * @returns {void}
         */
        function initialize() {
            let savedInsightListener: () => {};


            // register listeners
            savedInsightListener = scope.insightCtrl.on('saved-insight', updateEmbed);

            scope.$on('$destroy', function () {
                console.log('destroying embed-link....');
                savedInsightListener();
            });


            updateEmbed();
        }

        initialize();
    }
}