'use strict';

import EchartsHelper from '@/widget-resources/js/echarts/echarts-helper.js';

import './chicklet.scss';

export default angular.module('app.chicklet-standard.directive', [])
    .directive('chickletStandard', chicletStandard);

chicletStandard.$inject = ['$timeout', 'semossCoreService'];

function chicletStandard($timeout, semossCoreService) {
    chicletStandardLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        require: ['^widget'],
        priority: 300,
        link: chicletStandardLink,
        controllerAs: 'chiclet',
        controller: chicletStandardCtrl,
        template: require('./chicklet-standard.directive.html'),
        scope: {}
    };

    function chicletStandardCtrl() {}

    function chicletStandardLink(scope, ele, attrs, ctrl) {
        var chartEle = ele[0],
            updateTaskListener,
            updateOrnamentsListener,
            addDataListener,
            modeListener,
            clickTimer,
            hoverTimer,
            yAxis;

        scope.widgetCtrl = ctrl[0];
        scope.chiclet.headers = null;
        scope.chiclet.values = null;
        scope.chiclet.headerColor = null;
        scope.chiclet.valuesColor = null;
        // 2d array, inner arrays contain group name and height for axis,
        // groups ordered top to bottom
        scope.chiclet.yAxisSegments = [];
        scope.chiclet.hasYAxis = false;
        scope.chiclet.toggleYAxis = false;
        scope.chiclet.toggleColorPicker = false;
        scope.chiclet.editingEle = null;
        scope.chiclet.editingEleCategory = '';
        scope.chiclet.editColor = null;
        scope.chiclet.documents = {
            documentsToLink: [],
            documentsToUnlink: [],
            linkedDocuments: [],
            display: false,
            instance: '',
            apps: [],
            selectedApp: '',
            tables: [],
            selectedTable: '',
            rowId: '',
            documents: [],
            selectedDocument: '',
            content: ''
        };
        scope.chiclet.insight = {
            selectedApp: '',
            apps: [],
            insights: [],
            eventId: '',
            value: '',
            selectedInsight: '',
            isHeader: false
        };

        scope.chiclet.applyEdit = applyEdit;
        scope.chiclet.setLinks = setLinks;
        scope.chiclet.setTables = setTables;
        scope.chiclet.setDocuments = setDocuments;
        scope.chiclet.linkSelectedDocuments = linkSelectedDocuments;
        scope.chiclet.unlinkSelectedDocuments = unlinkSelectedDocuments;
        scope.chiclet.viewDocument = viewDocument;
        scope.chiclet.resetLinkData = resetLinkData;
        scope.chiclet.contentToDocuments = contentToDocuments;
        scope.chiclet.updateContent = updateContent;
        scope.chiclet.setInsights = setInsights;
        scope.chiclet.linkInsight = linkInsight;

        /**
         * @name setData
         * @desc entry function to configure necessary data to paint the chart
         * @returns {void}
         */
        function setData() {
            var layerIndex = 0,
                active = scope.widgetCtrl.getWidget('active'),
                values = scope.widgetCtrl.getWidget('view.' + active + '.tasks.' + layerIndex + '.data.values');
            scope.chiclet.headers = scope.widgetCtrl.getWidget('view.' + active + '.tasks.' + layerIndex + '.data.headers');
            scope.chiclet.values = values;
            configureYAxis();
            setColors();
            setOrnaments();
            setEvents();
        }

        function setEditMode(e) {
            var className = e.target.className;

            if (
                className.indexOf('chiclet__edit') > -1 ||
                className.indexOf('chiclet__edit__text') > -1 ||
                className.indexOf('chiclet__edit__axis') > -1 ||
                className.indexOf('chiclet__edit__axis__text') > -1
            ) {
                $timeout(function () {
                    scope.chiclet.toggleColorPicker = !scope.chiclet.toggleColorPicker;
                    scope.chiclet.editingEle = e.target;
                    if (className.indexOf('axis') > -1) {
                        scope.chiclet.editingEleCategory = 'Axis';
                    } else if (className.indexOf('text') > -1) {
                        scope.chiclet.editingEleCategory = ' Text';
                    } else {
                        scope.chiclet.editingEleCategory = ' Background';
                    }
                });
            }
        }

        /**
         * @name setEvents
         * @desc entry function to set events
         * @returns {void}
         */
        function setEvents() {
            var eventListeners = {
                click: click,
                mouseover: mouseover,
                mouseout: mouseout
            };
            scope.chiclet.currentMode = EchartsHelper.getCurrentMode(scope.widgetCtrl.getMode('selected'));
            scope.chiclet.callbacks = scope.widgetCtrl.getEventCallbacks();
            scope.chiclet.comments = scope.widgetCtrl.getWidget('view.visualization.commentData');

            EchartsHelper.initializeCommentMode({
                comments: scope.chiclet.comments,
                currentMode: scope.chiclet.currentMode,
                saveCb: scope.chiclet.callbacks.commentMode.onSave
            });

            Object.keys(eventListeners).forEach(function (event) {
                chartEle.addEventListener(event, eventListeners[event]);
            });
        }

        /**
         * @name mouseover
         * @param {Event} event - event object
         * @desc function for mouseover event
         * @returns {void}
         */
        function mouseover(event) {
            if (hoverTimer) {
                clearTimeout(hoverTimer);
            }
            hoverTimer = setTimeout(eventCallback.bind(null, event, 'onHover'), 2000);
        }

        /**
         * @name mouseout
         * @desc function for mouseout event
         * @returns {void}
         */
        function mouseout() {
            var currentEvent = scope.widgetCtrl.getEvent('currentEvent');
            if (currentEvent.type === 'onHover') {
                eventCallback(event, 'onMouseOut');
            }
        }

        /**
         * @name click
         * @param {Event} event - event object
         * @desc function for click event
         * @returns {void}
         */
        function click(event) {
            if (clickTimer) {
                clearTimeout(clickTimer);
                eventCallback(event, 'onDoubleClick');
            } else {
                clickTimer = setTimeout(eventCallback.bind(null, event, 'onClick'), 250);
            }
        }

        /**
         * @name eventCallback
         * @param {Event} event - event object
         * @param {string} type - type of event
         * @desc callback after event fires
         * @returns {void}
         */
        function eventCallback(event, type) {
            var returnObj = {
                    data: {}
                },
                containers = [
                    document.querySelector('.chiclet__cols'),
                    document.querySelector('.chiclet__headers')
                ],
                header, i, child;
            if (scope.chiclet.currentMode === 'editMode' && type === 'onClick') {
                setEditMode(event);
            } else {
                containers.forEach(function (container) {
                    // can trigger event sometimes before the html exists
                    // make sure they do
                    if (container) {
                        for (i = 0; i < container.children.length; i++) {
                            child = container.children[i];
                            if (child.contains(event.target)) {
                                header = scope.chiclet.headers[i];
                                break;
                            }
                        }
                    }
                });

                if (header) {
                    returnObj.data[header] = [event.target.innerText.trim().replace(/ /g, '_')];
                }

                scope.chiclet.callbacks.defaultMode[type](returnObj);
            }
            clickTimer = null;
            hoverTimer = null;
        }

        /**
         * @name setOrnaments
         * @desc sets ornaments relevant to the chiclet chart
         * @returns {void}
         */
        function setOrnaments() {
            var colorBy = scope.widgetCtrl.getWidget('view.visualization.colorByValue'),
                hilite = scope.widgetCtrl.getWidget('view.visualization.tools.shared.highlight'),
                documents = scope.widgetCtrl.getWidget('view.visualization.tools.shared.chicletDocuments'),
                fontSize = scope.widgetCtrl.getWidget('view.visualization.tools.shared.fontSize'),
                fontColor = scope.widgetCtrl.getWidget('view.visualization.tools.shared.fontColor');
            scope.chiclet.toggleYAxis = scope.widgetCtrl.getWidget('view.visualization.tools.shared.toggleYAxis');

            if (colorBy) {
                colorByValue(colorBy);
            }

            if (Object.keys(hilite).length > 0) {
                highlight(hilite.data);
            }
            if (documents && scope.chiclet.documents.instance) {
                setLinkedDocuments(scope.chiclet.documents.instance);
            }

            // minimum is 16 as this is a menu
            if (convertToNum(fontSize) >= 16) {
                scope.chiclet.fontSize = fontSize;
            } else {
                scope.chiclet.fontSize = '16px';
            }

            if (fontColor) {
                scope.chiclet.fontColor = fontColor;
            } else {
                scope.chiclet.fontColor = 'white';
            }
        }

        /**
         * @name convertToNum
         * @param {string} font font size + px
         * @desc takes a font size and just returns number portion
         * @return {number} the font size as a number
         */
        function convertToNum(font) {
            return Number(font.substring(0, font.indexOf('p')));
        }

        /**
         * @name highlight
         * @param {object} highlightMap - highlight data
         * @desc logic for highlighting selected elements of the chiclet chart
         * @returns {void}
         */
        function highlight(highlightMap) {
            var containers = [
                    document.querySelector('.chiclet__cols'),
                    document.querySelector('.chiclet__headers')
                ],
                highlightClass = 'chiclet--highlight';

            // clear old highlight classes before creating new ones
            containers.forEach(function (container, idx) {
                var i, j, child, grandchild;
                for (i = 0; i < container.children.length; i++) {
                    child = container.children[i];
                    // headers dont need to check grandchildren
                    if (idx === 1) {
                        removeHighlight(child, highlightClass);
                    } else {
                        for (j = 0; j < child.children.length; j++) {
                            grandchild = child.children[j];
                            removeHighlight(grandchild, highlightClass);
                        }
                    }
                }
            });

            Object.keys(highlightMap).forEach(function (header) {
                var headerIdx = scope.chiclet.headers.indexOf(header);
                highlightMap[header].forEach(function (value) {
                    containers.forEach(function (container, idx) {
                        var i, child;
                        child = container.children[headerIdx];
                        // headers dont need to check grandchildren
                        if (idx === 1) {
                            if (child.innerText.trim().replace(/ /g, '_') === String(value)) {
                                child.className += ' ' + highlightClass;
                            }
                        } else {
                            for (i = 0; i < child.children.length; i++) {
                                if (child.children[i].innerText.trim().replace(/ /g, '_') === String(value)) {
                                    child.children[i].className += ' ' + highlightClass;
                                }
                            }
                        }
                    });
                });
            });
        }

        /**
         * @name removeHighlight
         * @param {DOM_Element} highlightEle - element to remove highlight from
         * @param {string} highlightClass - class used to indicate highlight
         * @desc removes class that applies highlight style from element
         * @returns {void}
         */
        function removeHighlight(highlightEle, highlightClass) {
            var i, newClass = [];
            for (i = 0; i < highlightEle.classList.length; i++) {
                if (highlightEle.classList[i] !== highlightClass) {
                    newClass.push(highlightEle.classList[i]);
                }
            }

            highlightEle.className = newClass.join(' ');
        }

        /**
         * @name colorByValue
         * @param {array} rules - CBV rules
         * @desc updates colors on chiclet chart based on CBV rules
         * @returns {void}
         */
        function colorByValue(rules) {
            rules.forEach(function (rule) {
                rule.valuesToColor.forEach(function (value) {
                    scope.chiclet.valuesColor[value + rule.colorOn] = rule.color;
                });
            });
        }

        function applyEdit() {
            var editingEle = scope.chiclet.editingEle,
                editingEleClasses = editingEle.className,
                editingEleStyle = editingEle.style;
            if (editingEleClasses.indexOf('chiclet__edit__axis__text') > -1) {
                editingEle.style.color = scope.chiclet.editColor;
                editingEle.parentElement.style.borderRight = '1px solid ' + scope.chiclet.editColor;
            } else if (editingEleClasses.indexOf('chiclet__edit__axis') > -1) {
                editingEle.style.borderRight = '1px solid ' + scope.chiclet.editColor;
                editingEle.children[0].style.color = scope.chiclet.editColor;
            } else if (editingEleClasses.indexOf('chiclet__edit__text') > -1) {
                editingEleStyle.color = scope.chiclet.editColor;
            } else if (editingEleClasses.indexOf('chiclet__edit') > -1) {
                editingEleStyle.backgroundColor = scope.chiclet.editColor;
            }

            scope.chiclet.toggleColorPicker = false;
            scope.chiclet.editColor = null;
        }

        /**
         * @name setColors
         * @desc sets chart colors
         * @returns {void}
         */
        function setColors() {
            var colorPalette,
                colorName = scope.widgetCtrl.getWidget('view.visualization.tools.shared.colorName'),
                // color = scope.widgetCtrl.getWidget('view.visualization.tools.shared.color'),
                colorGroups = scope.widgetCtrl.getWidget('view.visualization.tools.shared.chicletColorGroups');

            colorPalette = semossCoreService.visualization.getColorPalette(colorName);
            scope.chiclet.headerColor = {
                'background-color': colorPalette[0]
            };

            if (!colorGroups && scope.chiclet.hasYAxis) {
                setGroupColors(colorPalette);
            } else {
                setValuesColor(colorPalette);
            }

            setEditModeColors();
        }

        function setGroupColors(palette) {
            var colorGroupsOrnament = {},
                groups = Object.keys(yAxis),
                callback;

            groups.forEach(function (group, idx) {
                var colorIdx = (idx + 1) % palette.length;
                colorGroupsOrnament[group] = palette[colorIdx];
            });

            callback = function () {
                setValuesColor(palette);
            };

            scope.widgetCtrl.execute(
                [{
                    type: 'panel',
                    components: [
                        scope.widgetCtrl.panelId
                    ]
                },
                {
                    type: 'addPanelOrnaments',
                    components: [{
                        tools: {
                            shared: {
                                chicletColorGroups: colorGroupsOrnament
                            }
                        }
                    }],
                    terminal: true
                },
                {
                    type: 'panel',
                    components: [
                        scope.widgetCtrl.panelId
                    ]
                },
                {
                    type: 'retrievePanelOrnaments',
                    components: ['tools'],
                    terminal: true
                }
                ], callback);
        }

        function setValuesColor(palette) {
            var active = scope.widgetCtrl.getWidget('active'),
                layerIndex = 0,
                values = scope.widgetCtrl.getWidget('view.' + active + '.tasks.' + layerIndex + '.data.values'),
                headers = scope.widgetCtrl.getWidget('view.' + active + '.tasks.' + layerIndex + '.data.headers'),
                colorGroups = scope.widgetCtrl.getWidget('view.visualization.tools.shared.chicletColorGroups');

            scope.chiclet.valuesColor = {};
            values.forEach(function (row) {
                row.forEach(function (data, idx) {
                    // the instance + header
                    var valuesColorKey = data + headers[idx],
                        groupName = row[row.length - 1];

                    if (scope.chiclet.hasYAxis) {
                        scope.chiclet.valuesColor[valuesColorKey] = colorGroups[groupName];
                        if (!scope.chiclet.valuesColor[groupName]) {
                            scope.chiclet.valuesColor[groupName + '_axis'] = colorGroups[groupName];
                        }
                    } else {
                        scope.chiclet.valuesColor[valuesColorKey] = palette[1];
                    }
                });
            });
        }

        function setEditModeColors() {
            var colors = scope.widgetCtrl.getWidget('view.visualization.tools.shared.chicletEditColors');

            if (colors) {
                Object.keys(colors).forEach(function (key) {
                    scope.chiclet.valuesColor[key] = colors[key];
                });
            }
        }

        /**
         * @name configureYAxis
         * @desc sets up y axis
         * @returns {void}
         */
        function configureYAxis() {
            var active = scope.widgetCtrl.getWidget('active'),
                layout = scope.widgetCtrl.getWidget('view.' + active + '.layout'),
                keys = scope.widgetCtrl.getWidget('view.' + active + '.keys.' + layout),
                tileHeight = 150;

            yAxis = {};

            scope.chiclet.yAxisSegments = [];
            scope.chiclet.hasYAxis = false;

            keys.forEach(function (key) {
                if (key.model === 'groupY') {
                    scope.chiclet.hasYAxis = true;
                }
            });

            if (scope.chiclet.hasYAxis) {
                // last header is the group category
                scope.chiclet.headers.pop();

                scope.chiclet.values.forEach(function (value) {
                    var groupKey = value.pop();

                    if (!yAxis[groupKey]) {
                        yAxis[groupKey] = [];
                        scope.chiclet.yAxisSegments.push([groupKey]);
                    }

                    if (value[0] || value[0] === 0) {
                        yAxis[groupKey].push(
                            value[0]
                        );
                    }
                });

                scope.chiclet.yAxisSegments.sort(function (a, b) {
                    var groupA = a[0],
                        groupB = b[0];

                    if (groupA < groupB) {
                        return -1;
                    }
                    if (groupA > groupB) {
                        return 1;
                    }

                    return 0;
                });

                scope.chiclet.yAxisSegments.forEach(function (group) {
                    var groupKey = group[0],
                        height = yAxis[groupKey].length * tileHeight + 'px';
                    group.push(height);
                });

                sortValues();
            }
        }

        /**
         * @name sortValues
         * @desc sorts chart values by group
         * @returns {void}
         */
        function sortValues() {
            var layerIndex = 0,
                active = scope.widgetCtrl.getWidget('active'),
                values = scope.widgetCtrl.getWidget('view.' + active + '.tasks.' + layerIndex + '.data.values'),
                sortedChicletValues = [];

            scope.chiclet.yAxisSegments.forEach(function (group) {
                var groupKey = group[0];
                sortedChicletValues = sortedChicletValues.concat(
                    values.filter(function (row) {
                        return row[row.length - 1] === groupKey;
                    })
                );
            });

            scope.chiclet.values = sortedChicletValues;
        }

        /**
         * @name toggleMode
         * @desc switches the jv mode to the new specified mode
         * @returns {void}
         */
        function toggleMode() {
            scope.chiclet.currentMode = EchartsHelper.getCurrentMode(scope.widgetCtrl.getMode('selected'));
        }

        function setLinks(e, instance, header) {
            e.stopPropagation();
            scope.chiclet.documents.display = true;
            scope.chiclet.documents.instance = instance + header;
            scope.chiclet.insight.eventId = instance + header;
            scope.chiclet.insight.value = instance;
            if (instance === 'CHICLET_HEADER') {
                scope.chiclet.insight.isHeader = true;
                scope.chiclet.linkTo = 'insight';
                scope.chiclet.insight.value = header;
            } else {
                setLinkedDocuments(scope.chiclet.documents.instance);
            }
        }

        function setApps() {
            var callback;

            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    i,
                    len;

                for (i = 0, len = output.length; i < len; i++) {
                    scope.chiclet.documents.apps.push(output[i].app_name);
                    scope.chiclet.insight.apps.push(output[i].app_name);
                }
            };

            scope.widgetCtrl.query([{
                meta: true,
                type: 'getDatabaseList',
                components: [],
                terminal: true
            }], callback);
        }

        function setTables() {
            var callback;

            callback = function (response) {
                var i = 0,
                    output = response.pixelReturn[0].output;

                for (i = 0; i < output.length; i++) {
                    scope.chiclet.documents.tables.push(output[i]);
                }
            };

            scope.widgetCtrl.query([{
                type: 'getDatabaseConcepts',
                components: [scope.chiclet.documents.selectedApp],
                terminal: true
            }], callback);
        }

        function setDocuments() {
            var callback;

            // reset the actions
            scope.gridOptions = {
                readOnly: [0],
                actions: [],
                actionIndex: -1
            };


            callback = function (response) {
                var output, i,
                    frame = semossCoreService.utility.random('FRAME'),
                    selectors = [],
                    callback2;

                output = response.pixelReturn[0].output.filter(function (selector) {
                    return selector[0] === scope.chiclet.documents.selectedTable;
                });

                for (i = 0; i < output.length; i++) {
                    selectors.push({
                        'alias': isNaN(output[i][1]) ? '"' + String(output[i][1]).replace(/_/g, ' ') + '"' : output[i][1],
                        'selector': output[i][3] ? output[i][1] : output[i][0] + '__' + output[i][1]
                    });
                }

                callback2 = function (responseData) {
                    var values = responseData.pixelReturn[2].output.data.values;
                    // TODO: GENERICIZE
                    scope.chiclet.documents.documents = values.map(function (row) {
                        return row[2];
                    });
                };

                scope.widgetCtrl.meta([{
                    type: 'createSource',
                    components: ['Frame', 'GRID', frame, true],
                    terminal: true
                },
                {
                    type: 'database',
                    components: [scope.chiclet.documents.selectedApp]
                },
                {
                    type: 'select2',
                    components: [selectors]
                },
                {
                    type: 'import',
                    components: [frame],
                    terminal: true
                },
                {
                    type: 'frame',
                    components: [frame]
                },
                {
                    type: 'queryAll',
                    components: []
                },
                {
                    type: 'Pixel',
                    components: ['CollectAll()'],
                    terminal: true
                }
                ], callback2);
            };

            scope.widgetCtrl.meta([{
                type: 'getDatabaseTableStructure',
                components: [scope.chiclet.documents.selectedApp],
                terminal: true
            }], callback);
        }

        function setLinkedDocuments(instance) {
            var documents = scope.widgetCtrl.getWidget('view.visualization.tools.shared.chicletDocuments');
            if (documents) {
                scope.chiclet.documents.linkedDocuments = documents[instance];
            }
        }

        function linkSelectedDocuments() {
            var linkDocsOrnament = scope.widgetCtrl.getWidget('view.visualization.tools.shared.chicletDocuments') || {};

            if (linkDocsOrnament[scope.chiclet.documents.instance]) {
                linkDocsOrnament[scope.chiclet.documents.instance] = linkDocsOrnament[scope.chiclet.documents.instance].concat(scope.chiclet.documents.documentsToLink);
                linkDocsOrnament[scope.chiclet.documents.instance] = linkDocsOrnament[scope.chiclet.documents.instance].filter(function (doc, idx, self) {
                    return self.indexOf(doc) === idx;
                });
            } else {
                linkDocsOrnament[scope.chiclet.documents.instance] = scope.chiclet.documents.documentsToLink;
            }
            scope.widgetCtrl.execute(
                [{
                    type: 'panel',
                    components: [
                        scope.widgetCtrl.panelId
                    ]
                },
                {
                    type: 'addPanelOrnaments',
                    components: [{
                        tools: {
                            shared: {
                                chicletDocuments: linkDocsOrnament
                            }
                        }
                    }],
                    terminal: true
                },
                {
                    type: 'panel',
                    components: [
                        scope.widgetCtrl.panelId
                    ]
                },
                {
                    type: 'retrievePanelOrnaments',
                    components: ['tools'],
                    terminal: true
                }
                ]
            );

            scope.chiclet.documents.documentsToLink = [];
        }

        function unlinkSelectedDocuments() {
            var currentLinkedDocuments = scope.widgetCtrl.getWidget('view.visualization.tools.shared.chicletDocuments'),
                newDocs = [],
                selected = scope.chiclet.documents.documentsToUnlink.map(function (cur) {
                    return cur;
                });

            currentLinkedDocuments[scope.chiclet.documents.instance].forEach(function (doc) {
                if (selected.indexOf(doc) === -1) {
                    newDocs.push(doc);
                }
            });

            currentLinkedDocuments[scope.chiclet.documents.instance] = newDocs;

            scope.widgetCtrl.execute(
                [{
                    type: 'panel',
                    components: [
                        scope.widgetCtrl.panelId
                    ]
                },
                {
                    type: 'addPanelOrnaments',
                    components: [{
                        tools: {
                            shared: {
                                chicletDocuments: currentLinkedDocuments
                            }
                        }
                    }],
                    terminal: true
                },
                {
                    type: 'panel',
                    components: [
                        scope.widgetCtrl.panelId
                    ]
                },
                {
                    type: 'retrievePanelOrnaments',
                    components: ['tools'],
                    terminal: true
                }
                ]);

            scope.chiclet.documents.documentsToUnlink = [];
        }

        function viewDocument(e, title) {
            var callback;

            e.stopPropagation();

            // reset the actions
            scope.gridOptions = {
                readOnly: [0],
                actions: [],
                actionIndex: -1
            };

            callback = function (response) {
                var output, i,
                    frame = semossCoreService.utility.random('FRAME'),
                    selectors = [],
                    callback2;

                output = response.pixelReturn[0].output.filter(function (selector) {
                    return selector[0] === scope.chiclet.documents.selectedTable;
                });

                for (i = 0; i < output.length; i++) {
                    selectors.push({
                        'alias': isNaN(output[i][1]) ? '"' + String(output[i][1]).replace(/_/g, ' ') + '"' : output[i][1],
                        'selector': output[i][3] ? output[i][1] : output[i][0] + '__' + output[i][1]
                    });
                }

                callback2 = function (responseData) {
                    var values = responseData.pixelReturn[2].output.data.values;
                    scope.chiclet.documents.selectedDocument = title;
                    scope.chiclet.documents.rowId = values[0][1];
                    scope.chiclet.documents.content = values[0][0].replace(/_/g, ' ');
                };

                scope.widgetCtrl.meta([{
                    type: 'createSource',
                    components: ['Frame', 'GRID', frame, true],
                    terminal: true
                },
                {
                    type: 'database',
                    components: [scope.chiclet.documents.selectedApp]
                },
                {
                    type: 'select2',
                    components: [selectors]
                },
                {
                    type: 'import',
                    components: [frame],
                    terminal: true
                },
                {
                    type: 'frame',
                    components: [frame]
                },
                {
                    type: 'queryAll',
                    components: []
                },
                {
                    type: 'filter',
                    components: [{
                        Title: {
                            comparator: '==',
                            value: title
                        }
                    }]
                },
                {
                    type: 'collect',
                    components: [1],
                    terminal: true
                }
                ], callback2);
            };

            scope.widgetCtrl.meta([{
                type: 'getDatabaseTableStructure',
                components: [scope.chiclet.documents.selectedApp],
                terminal: true
            }], callback);
        }

        function resetLinkData() {
            scope.chiclet.documents.selectedApp = '';
            scope.chiclet.documents.selectedTable = '';
            scope.chiclet.documents.content = '';
            scope.chiclet.documents.selectedDocument = '';
            scope.chiclet.documents.instance = '';
            scope.chiclet.documents.display = false;
            scope.chiclet.documents.rowId = '';
            scope.chiclet.documents.documents = [];
            scope.chiclet.documents.tables = [];
            scope.chiclet.insight.selectedApp = '';
            scope.chiclet.insight.insights = [];
            scope.chiclet.insight.selectedInsight = '';
            scope.chiclet.insight.eventId = '';
            scope.chiclet.insight.value = '';
            scope.chiclet.insight.isHeader = false;
        }

        function contentToDocuments() {
            scope.chiclet.documents.selectedDocument = '';
        }

        function updateContent() {
            var pixel = 'Database(database=["' + scope.chiclet.documents.selectedApp + '"]) |',
                table = scope.chiclet.documents.selectedTable;
            pixel += 'Update(columns=[' + table + '__Content],';
            pixel += 'values=["' + scope.chiclet.documents.content + '"]) |';
            pixel += 'Filter(' + table + '__Title == ["' + scope.chiclet.documents.selectedDocument + '"]) |';
            pixel += 'ExecQuery()';

            scope.widgetCtrl.execute([{
                type: 'Pixel',
                components: [pixel],
                meta: true,
                terminal: true
            }]);

            contentToDocuments();
        }

        function setInsights() {
            var callback;

            callback = function (response) {
                scope.chiclet.insight.insights = response.pixelReturn[0].output.map(function (insight) {
                    return {
                        id: insight.app_insight_id,
                        name: insight.name
                    };
                });
            };

            scope.widgetCtrl.meta(
                [{
                    type: 'getInsights',
                    components: [scope.chiclet.insight.selectedApp],
                    meta: true,
                    terminal: true
                }], callback);
        }

        function linkInsight() {
            var eventId = scope.chiclet.insight.eventId,
                event = {},
                query = '<encode>if ((\"<SelectedValue>\" ==';

            query += '\"' + scope.chiclet.insight.value + '\"), ';
            query += '(OpenInsight(app=[\"' + scope.chiclet.insight.selectedApp + '\"], ';
            query += 'id=[\"' + scope.chiclet.insight.selectedInsight + '\"]);));</encode>';
            event.onSingleClick = {};
            event.onSingleClick[eventId] = [{
                disabledVisuals: [],
                options: {},
                panel: scope.widgetCtrl.panelId,
                query: query,
                refresh: false
            }];
            scope.widgetCtrl.emit('add-event', {
                event: event
            });


            scope.widgetCtrl.alert('success', 'Event Added');
        }

        function initialize() {
            addListeners();
            setData();
            setApps();
            scope.$on('$destroy', destroy);
        }

        function addListeners() {
            updateTaskListener = scope.widgetCtrl.on('update-task', setData);
            updateOrnamentsListener = scope.widgetCtrl.on('update-ornaments', setData);
            addDataListener = scope.widgetCtrl.on('added-data', setData);
            modeListener = scope.widgetCtrl.on('update-mode', toggleMode);
        }

        function destroy() {
            updateTaskListener();
            updateOrnamentsListener();
            addDataListener();
            modeListener();
            chartEle.removeEventListener('click', click);
            chartEle.removeEventListener('mouseover', mouseover);
            chartEle.removeEventListener('mouseout', mouseout);
        }

        initialize();
    }
}
