'use strict';

import echarts from 'echarts';
import EchartsHelper from '@/widget-resources/js/echarts/echarts-helper.js';

import './bar-line-echarts.service.js';

export default angular.module('app.bar-line-echarts.directive', [
    'app.bar.service'
]).directive('barLineEcharts', barLineEcharts);

barLineEcharts.$inject = ['VIZ_COLORS', 'barLineService'];

function barLineEcharts(VIZ_COLORS, barLineService) {
    barChartLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        require: ['^widget', '^visualization'],
        priority: 300,
        link: barChartLink
    };

    function barChartLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];
        scope.visualizationCtrl = ctrl[1];

        /** ************* Main Event Listeners ************************/
        var resizeListener,
            updateTaskListener,
            updateOrnamentsListener,
            addDataListener,
            modeListener,
            removeBrushListener,
            /** *************** ECharts ****************************/
            eChartsConfig,
            barChart,
            clientHeight,
            clientWidth,
            destroyListeners;

        /** **************** Destory Listener *************************/

        /**
         * @name initialize
         * @desc creates the visualization on the chart div
         * @returns {void}
         */
        function initialize() {
            // bind listeners
            resizeListener = scope.widgetCtrl.on('resize-widget', resizeViz);
            updateTaskListener = scope.widgetCtrl.on('update-task', setData);
            updateOrnamentsListener = scope.widgetCtrl.on('update-ornaments', setData);
            addDataListener = scope.widgetCtrl.on('added-data', setData);
            modeListener = scope.widgetCtrl.on('update-mode', toggleMode);
            removeBrushListener = scope.widgetCtrl.on('remove-brush', paint);

            // clean up
            scope.$on('$destroy', destroy);

            setData();
        }

        /**
         * @name setData
         * @desc setData for the visualization and paints
         * @returns {void}
         */
        function setData() {
            var layerIndex = 0,
                selectedLayout = scope.widgetCtrl.getWidget('view.visualization.layout'),
                individualTools = scope.widgetCtrl.getWidget('view.visualization.tools.individual.' + selectedLayout) || {},
                sharedTools = scope.widgetCtrl.getWidget('view.visualization.tools.shared'),
                colorBy = scope.widgetCtrl.getWidget('view.visualization.colorByValue'),
                keys = scope.widgetCtrl.getWidget('view.visualization.keys.' + selectedLayout),
                data = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.data'),
                groupBy = {},
                groupByInstance,
                groupedData,
                groupByInfo = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.groupByInfo'),
                uiOptions,
                n;

            uiOptions = angular.extend(sharedTools, individualTools);
            uiOptions.colorByValue = colorBy;
            uiOptions.rotateAxis = sharedTools.rotateAxis;

            if (groupByInfo && groupByInfo.viewType) {
                if (groupByInfo.viewType === 'Individual Instance') {
                    groupBy = formatDataForGroupByIndividual(data, groupByInfo);
                    data = groupBy.data;
                    groupByInstance = groupBy.name;
                } else if (groupByInfo.viewType === 'All Instances') {
                    groupedData = formatDataForGroupByAll(data, groupByInfo, uiOptions.facetHeaders);
                    groupByInfo.tempData = groupedData.data;
                    groupByInfo.maxXLabels = groupedData.maxXLabels;
                    // Remove empty groups (if filtered)
                    groupByInfo.uniqueInstances = [];
                    for (n = 0; n < Object.keys(groupedData.data).length; n++) {
                        groupByInfo.uniqueInstances.push(Object.keys(groupedData.data)[n]);
                    }
                }
            }

            eChartsConfig = barLineService.getConfig('bar', data, uiOptions, colorBy, groupByInstance, groupByInfo, keys);
            eChartsConfig.currentMode = EchartsHelper.getCurrentMode(scope.widgetCtrl.getMode('selected'));
            eChartsConfig.callbacks = scope.widgetCtrl.getEventCallbacks();
            eChartsConfig.comments = scope.widgetCtrl.getWidget('view.visualization.commentData');
            eChartsConfig.echartsMode = EchartsHelper.getEchartsMode(scope.widgetCtrl.getMode('selected'));
            eChartsConfig.groupByInfo = groupByInfo;
            determineResize();
            uiOptions = EchartsHelper.setColors(uiOptions, eChartsConfig.legendHeaders, VIZ_COLORS);

            paint();
        }

        /**
         * @name formatDataForGroupByIndividual
         * @desc formats data when Group By exists
         * @param {object} data orginial data
         * @param {object} groupBy groupBy object
         * @returns {void}
         */
        function formatDataForGroupByIndividual(data, groupBy) {
            var formattedData = data,
                groupByIndex,
                name,
                i,
                instanceIdx,
                returnObj = {};

            if (groupBy.viewType === 'Individual Instance') {
                groupByIndex = data.headers.indexOf(groupBy.selectedDim);
                if (groupByIndex === -1) {
                    // return data;
                    groupByIndex = data.headers.length;
                }

                if (typeof groupBy.instanceIndex === 'string') {
                    instanceIdx = parseInt(groupBy.instanceIndex, 10);
                }
                // Create name for title
                name = groupBy.selectedDim + ' : ' + groupBy.uniqueInstances[instanceIdx];
                // Remove Group By dimension from data headers and values
                formattedData.headers.splice(groupByIndex, 1);
                formattedData.rawHeaders.splice(groupByIndex, 1);

                // Remove any added data from brush/click
                for (i = 0; i < data.values.length; i++) {
                    if (data.values[i][groupByIndex] !== groupBy.uniqueInstances[instanceIdx]) {
                        data.values.splice(i, 1);
                        i--;
                    }
                }

                for (i = 0; i < data.values.length; i++) {
                    data.values[i].splice(groupByIndex, 1);
                }
                returnObj.name = name;
                returnObj.data = data;
            }

            return returnObj;
        }

        /**
         * @name formatDataForGroupByAll
         * @desc formats data when Group By All Instances exists
         * @param {object} data orginial data
         * @param {object} groupBy groupBy object
         * @param {object} facetHeaders uiOptions.facetHeaders
         * @returns {void}
         */
        function formatDataForGroupByAll(data, groupBy, facetHeaders) {
            var groupByIndex,
                i,
                n,
                j,
                dataObj = {},
                axisLabels = [],
                uniqueAxisLabels = [],
                exists,
                emptyArr,
                maxXLabels = 0;

            groupByIndex = data.headers.indexOf(groupBy.selectedDim);
            data.headers.splice(groupByIndex, 1);
            data.rawHeaders.splice(groupByIndex, 1);

            if (facetHeaders.commonAxis) {
                for (j = 0; j < data.values.length; j++) {
                    axisLabels.push(data.values[j][0]);
                }
                // Get unique values axisLabels
                uniqueAxisLabels = axisLabels.filter(onlyUnique);
            }

            for (n = 0; n < groupBy.uniqueInstances.length; n++) {
                dataObj[groupBy.uniqueInstances[n]] = [];
                for (i = 0; i < data.values.length; i++) {
                    if (data.values[i][groupByIndex] === groupBy.uniqueInstances[n]) {
                        data.values[i].splice(groupByIndex, 1);
                        dataObj[groupBy.uniqueInstances[n]].push(data.values[i]);
                    }
                }
                if (facetHeaders.commonAxis) {
                    if (dataObj[groupBy.uniqueInstances[n]].length !== uniqueAxisLabels.length) {
                        for (i = 0; i < uniqueAxisLabels.length; i++) {
                            exists = false;
                            for (j = 0; j < dataObj[groupBy.uniqueInstances[n]].length; j++) {
                                if (dataObj[groupBy.uniqueInstances[n]][j][0] === uniqueAxisLabels[i]) {
                                    exists = true;
                                }
                            }
                            if (!exists) {
                                emptyArr = [uniqueAxisLabels[i], 0, 0];
                                dataObj[groupBy.uniqueInstances[n]].splice(i, 0, emptyArr);
                            }
                        }
                    }
                }

                if (dataObj[groupBy.uniqueInstances[n]].length > maxXLabels) {
                    maxXLabels = dataObj[groupBy.uniqueInstances[n]].length;
                }
                if (dataObj[groupBy.uniqueInstances[n]].length === 0) {
                    delete dataObj[groupBy.uniqueInstances[n]];
                }
            }

            return {
                'data': dataObj,
                'maxXLabels': maxXLabels
            };
        }

        function onlyUnique(value, index, self) {
            return self.indexOf(value) === index;
        }

        /**
         * @name paint
         * @desc paints the visualization
         * @returns {void}
         */
        function paint() {
            if (barChart) {
                barChart.clear();
                barChart.dispose();
            }

            // TODO also think abou abstracting some of these options to variables for more customizabilty from uiOptions
            barChart = echarts.init(ele[0].firstElementChild);

            var option = {},
                i,
                dataEmpty,
                facetLayout,
                titleFontSize;

            if (eChartsConfig.options.facetHeaders && eChartsConfig.options.facetHeaders.titleFontSize) {
                titleFontSize = eChartsConfig.options.facetHeaders.titleFontSize;
            } else {
                titleFontSize = 18;
            }

            if (eChartsConfig.groupByInfo && eChartsConfig.groupByInfo.viewType === 'Individual Instance') {
                dataEmpty = true;
                for (i = 0; i < eChartsConfig.data.length; i++) {
                    if (eChartsConfig.data[i].data.length !== 0) {
                        dataEmpty = false;
                        break;
                    }
                }

                option.graphic = [];
                if (dataEmpty) {
                    option.graphic = option.graphic.concat({
                        id: 'textGroup',
                        type: 'group',
                        right: 'center',
                        top: 'center',
                        children: [{
                            type: 'rect',
                            top: 'center',
                            right: 'center',
                            shape: {
                                width: 200,
                                height: 40
                            },
                            style: {
                                fill: '#fff',
                                stroke: '#999',
                                lineWidth: 2,
                                shadowBlur: 8,
                                shadowOffsetX: 3,
                                shadowOffsetY: 3,
                                shadowColor: 'rgba(0,0,0,0.3)'
                            }
                        },
                        {
                            type: 'text',
                            right: 'center',
                            top: 'center',
                            style: {
                                text: 'There is no data for this instance.',
                                textAlign: 'center'
                            }
                        }
                        ]
                    });
                }
            }

            if (eChartsConfig.backgroundColorStyle) {
                option.backgroundColor = eChartsConfig.backgroundColorStyle;
            }

            option.toolbox = {
                show: false,
                left: '2',
                bottom: '5',
                orient: 'vertical',
                feature: {
                    brush: {
                        // type: ['rect', 'polygon', 'lineX'],
                        type: ['rect', 'polygon'],
                        title: {
                            rect: 'Brush',
                            polygon: 'Polygon Brush' // ,
                            // lineX: 'Horizontal'
                        }
                    }
                },
                iconStyle: {
                    emphasis: {
                        textPosition: 'right',
                        textAlign: 'left'
                    }
                }
            };

            if (eChartsConfig.groupByInfo) {
                // TODO enable brush for Facet All Instances (pass in correct labelIndex)
                if (eChartsConfig.groupByInfo.viewType === 'Individual Instance') {
                    option.brush = {
                        brushStyle: {
                            borderWidth: 1,
                            color: 'rgba(120,140,180,0.15)',
                            borderColor: 'rgba(120,140,180,0.35)'
                        },
                        xAxisIndex: 0
                    };
                }
            } else {
                option.brush = {
                    brushStyle: {
                        borderWidth: 1,
                        color: 'rgba(120,140,180,0.15)',
                        borderColor: 'rgba(120,140,180,0.35)'
                    },
                    xAxisIndex: 0
                };
            }

            option.color = eChartsConfig.options.echartsColor;

            option.tooltip = {
                show: eChartsConfig.options.showTooltips,
                formatter: function (info) {
                    var returnArray = [],
                        tooltipName = info[0].name,
                        j;

                    if (tooltipName) {
                        returnArray.push('<b>' + cleanValue(tooltipName) + '</b>' + '<br>');
                    }

                    for (j = 0; j < info.length; j++) {
                        if (!info[j].data) {
                            continue;
                        }
                        if (info[j].marker) {
                            returnArray.push(info[j].marker);
                        }
                        if (info[j].seriesName) {
                            returnArray.push('' + cleanValue(info[j].seriesName) + ': ' + cleanValue(info[j].value) + '<br>');
                        }
                    }

                    if (info[0].data.tooltip) {
                        for (j = 0; j < info[0].data.tooltip.length; j++) {
                            if (eChartsConfig.legendHeaders.indexOf(info[0].data.tooltip[j].header) === -1) {
                                returnArray.push('' + cleanValue(info[0].data.tooltip[j].header) + ': ' + cleanValue(info[0].data.tooltip[j].value || 0) + '<br>');
                            }
                        }
                    }

                    return returnArray.join('');
                },
                trigger: 'axis',
                axisPointer: {
                    type: eChartsConfig.axisPointer
                },
                confine: true
            };

            // option.title = {
            //     show: true,
            //     text: 'This is an example of a title',
            //     subtext: 'this is an example of a sub text'
            // };

            option.legend = {
                data: eChartsConfig.legendHeaders,
                show: eChartsConfig.showLegend,
                type: 'scroll',
                orient: 'horizontal',
                left: 'left',
                pageButtonPosition: 'start',
                formatter: function (value) {
                    return cleanValue(value);
                }
            };

            // Options specific to Facet All Instances
            if (eChartsConfig.groupByInfo && eChartsConfig.groupByInfo.viewType === 'All Instances') {
                facetLayout = customizeFacetLayout();

                option.grid = facetLayout.grid;
                option.title = facetLayout.title.concat([{
                    text: eChartsConfig.options.facetHeaders.titleName || 'All Instances of ' + cleanValue(eChartsConfig.groupByInfo.selectedDim),
                    top: '20px',
                    left: 'center',
                    textStyle: {
                        fontSize: titleFontSize
                    }
                }]);
                option.xAxis = facetLayout.xAxis;
                option.yAxis = facetLayout.yAxis;
                option.series = facetLayout.series;

                option.barWidth = null;


                // TODO check for max/min canvas sizes

                // TODO handle this...
                // if (eChartsConfig.options.rotateAxis || eChartsConfig.groupByInfo.flipAxis) {
                //     option.barWidth = null;
                // } else {
                //     option.barWidth = getFacetBarWidth(facetLayout.grid[0].width, eChartsConfig.barWidth) || null;
                // }
            } else {
                option.xAxis = eChartsConfig.xAxisConfig;
                option.yAxis = eChartsConfig.yAxisConfig;
                option.series = eChartsConfig.data;
                option.grid = setGrid(eChartsConfig.options.toggleZoomX, eChartsConfig.options.toggleZoomY);
                option.barWidth = eChartsConfig.barWidth;
            }

            option.dataZoom = toggleZoom(eChartsConfig.options.toggleZoomX, eChartsConfig.options.toggleZoomY, eChartsConfig.options.dataZoomXstart, eChartsConfig.options.dataZoomXend, eChartsConfig.options.dataZoomYstart, eChartsConfig.options.dataZoomYend);
            option.textStyle = {
                fontFamily: 'Libre Franklin'
            };
            // use configuration item and data specified to show chart
            EchartsHelper.setOption(barChart, option);
            // barChart.resize();

            // Add event listeners
            initializeEvents();
        }

        function cleanValue(item) {
            if (typeof item === 'string') {
                return item.replace(/_/g, ' ');
            } else if (typeof item === 'number') {
                return item.toLocaleString(undefined, {
                    minimumFractionDigits: 0,
                    maximumFractionDigits: 3
                });
            }
            return item;
        }

        /**
         * @name customizeFacetLayout
         * @desc defines grid, title, xAxis, yAxis, and series arrays used in echarts options configuration
         * @returns {obj} - object of grid dimensions
         */
        function customizeFacetLayout() {
            var i, n,
                gridPixel,
                gridPercent,
                gridArray = [],
                title,
                titleArray = [],
                xAxisArray = [],
                yAxisArray = [],
                seriesArray = [],
                rowIndex,
                columnIndex,
                calculatedGridWidth,
                calculatedGridHeight,
                calculatedSpacingX,
                calculatedSpacingY,
                facetPadding = { // padding within the top and left of the panel
                    top: 100,
                    right: 120,
                    bottom: 100,
                    left: 120
                };

            for (i = 0; i < eChartsConfig.groupByInfo.uniqueInstances.length; i++) {
                rowIndex = Math.floor(i / eChartsConfig.options.facetHeaders.numberColumns);
                columnIndex = (i % eChartsConfig.options.facetHeaders.numberColumns);

                calculatedGridWidth = eChartsConfig.options.facetHeaders.grid.width;
                calculatedGridHeight = eChartsConfig.options.facetHeaders.grid.height;
                calculatedSpacingX = eChartsConfig.options.facetHeaders.spacing.x;
                calculatedSpacingY = eChartsConfig.options.facetHeaders.spacing.y;

                if (eChartsConfig.options.facetHeaders.unitType === '%') {
                    calculatedGridWidth = (ele[0].clientWidth * eChartsConfig.options.facetHeaders.grid.width) / 100;
                    calculatedGridHeight = (ele[0].clientHeight * eChartsConfig.options.facetHeaders.grid.height) / 100;
                    calculatedSpacingX = (ele[0].clientWidth * eChartsConfig.options.facetHeaders.spacing.x) / 100;
                    calculatedSpacingY = (ele[0].clientHeight * eChartsConfig.options.facetHeaders.spacing.y) / 100;
                }

                gridPixel = {
                    top: facetPadding.top + (calculatedGridHeight + calculatedSpacingY) * rowIndex + 'px',
                    left: facetPadding.left + (calculatedGridWidth + calculatedSpacingX) * columnIndex + 'px',
                    width: calculatedGridWidth + 'px',
                    height: calculatedGridHeight + 'px'
                };

                if (eChartsConfig.options.facetHeaders.customLayout) {
                    // Center everything
                    gridPixel.left = parseFloat(gridPixel.left) + (clientWidth - facetPadding.left - facetPadding.right - (calculatedGridWidth) * (eChartsConfig.options.facetHeaders.numberColumns) - (calculatedSpacingX) * (eChartsConfig.options.facetHeaders.numberColumns - 1)) / 2 + 'px';
                }

                gridPercent = {
                    top: (parseFloat(gridPixel.top) / clientHeight) * 100 + '%',
                    left: (parseFloat(gridPixel.left) / clientWidth) * 100 + '%',
                    width: (parseFloat(gridPixel.width) / clientWidth) * 100 + '%',
                    height: (parseFloat(gridPixel.height) / clientHeight) * 100 + '%'
                };

                gridArray.push(gridPercent);

                title = getFacetHeaders(eChartsConfig.groupByInfo.uniqueInstances[i], gridPercent, gridPixel, eChartsConfig.options.facetHeaders);
                titleArray.push(title);

                // Format Axes
                if (columnIndex === 0) {
                    if (eChartsConfig.options.editYAxis) {
                        eChartsConfig.yAxisConfig[i][0].axisLabel.show = eChartsConfig.options.editYAxis.values;
                    } else {
                        eChartsConfig.yAxisConfig[i][0].axisLabel.show = true;
                        eChartsConfig.yAxisConfig[i][0].name = null;
                    }
                } else {
                    eChartsConfig.yAxisConfig[i][0].name = null;
                    eChartsConfig.yAxisConfig[i][0].axisLabel.show = false;
                }
                if (!eChartsConfig.options.editXAxis) {
                    eChartsConfig.xAxisConfig[i][0].name = null;
                    eChartsConfig.xAxisConfig[i][0].axisLabel.show = false;
                }

                xAxisArray.push(eChartsConfig.xAxisConfig[i][0]);
                yAxisArray.push(eChartsConfig.yAxisConfig[i][0]);

                for (n = 0; n < eChartsConfig.data[i].length; n++) {
                    seriesArray.push(eChartsConfig.data[i][n]);
                }
            }

            return {
                grid: gridArray,
                title: titleArray,
                xAxis: xAxisArray,
                yAxis: yAxisArray,
                series: seriesArray
            };
        }

        /**
         * @name getFacetHeaders
         * @desc dynamically adjusts header size to not overlap grid
         * @param {string} text - header text
         * @param {obj} gridPercent - grid object (percent)
         * @param {obj} gridPixel - grid object (percent)
         * @param {obj} facetHeaders - uiOptions.facetHeaders
         * @returns {obj} - object of grid dimensions
         */
        function getFacetHeaders(text, gridPercent, gridPixel, facetHeaders) {
            var title = {},
                fontSize;

            if (typeof text === 'string') {
                title.text = cleanValue(text);
            } else {
                title.text = text;
            }

            title.left = parseFloat(gridPercent.left) + (parseFloat(gridPercent.width) / 2) + '%';

            if (facetHeaders && facetHeaders.headerFontSize) {
                fontSize = facetHeaders.headerFontSize;
                title.top = Number(gridPixel.top.substring(0, gridPixel.top.indexOf('px'))) - facetHeaders.headerFontSize - 8 + 'px';
            } else {
                fontSize = 14;
                title.top = Number(gridPixel.top.substring(0, gridPixel.top.indexOf('px'))) - fontSize - 8 + 'px';
            }

            title.textAlign = 'center';
            title.textStyle = {
                fontSize: fontSize
            };

            return title;
        }

        /**
         * @name setGrid
         * @desc sets grid dimensions based on whether or not datazoom is present
         * @param {bool} showX - boolean to show x zoom or not
         * @param {bool} showY - boolean to show y zoom or not
         * @returns {obj} - object of grid dimensions
         */
        function setGrid(showX, showY) {
            var grid = {
                top: 60,
                right: 45,
                bottom: 45,
                left: 40,
                containLabel: true
            };

            if (showX) {
                grid.bottom += 15;
            }

            if (showY) {
                grid.right += 15;
            }

            if (eChartsConfig.groupByInfo && eChartsConfig.groupByInfo.viewType === 'Individual Instance') {
                grid.bottom += 40;
            }

            return grid;
        }

        /**
         * @name initializeEvents
         * @desc creates the event layer
         * @returns {void}
         */
        function initializeEvents() {
            var pictorialBarExists = false;

            if (typeof destroyListeners === 'function') {
                destroyListeners();
            }

            if (eChartsConfig.options.barImage.symbol && eChartsConfig.options.barImage.symbol !== 'Default (Bar)') {
                eChartsConfig.echartsMode = 'rect';
                pictorialBarExists = true;

                if (eChartsConfig.options.rotateAxis) {
                    barChart.flipped = true;
                } else {
                    barChart.flipped = false;
                }
            }

            // Context Menu
            barChart.on('contextmenu', function (e) {
                scope.visualizationCtrl.setContextMenuDataFromClick(e, {
                    name: [eChartsConfig.legendLabels]
                });
            });
            barChart._dom.addEventListener('contextmenu', scope.visualizationCtrl.openContextMenu);

            if (eChartsConfig.echartsMode) {
                barChart.dispatchAction({
                    type: 'takeGlobalCursor',
                    key: 'brush',
                    brushOption: {
                        brushType: eChartsConfig.echartsMode
                    }
                });
            }

            if (eChartsConfig.currentMode === 'defaultMode' || eChartsConfig.currentMode === 'polygonBrushMode') {
                EchartsHelper.initializeBrush(barChart, {
                    xLabels: getLabels(),
                    legendLabels: eChartsConfig.legendLabels,
                    brushCb: eChartsConfig.callbacks.defaultMode.onBrush,
                    repaint: paint,
                    type: 'bar',
                    openContextMenu: scope.visualizationCtrl.openContextMenu,
                    setContextMenuDataFromBrush: scope.visualizationCtrl.setContextMenuDataFromBrush,
                    pictorial: pictorialBarExists
                });

                destroyListeners = EchartsHelper.initializeClickHoverKeyEvents(barChart, {
                    cb: eChartsConfig.callbacks.defaultMode,
                    header: eChartsConfig.legendLabels,
                    getCurrentEvent: function () {
                        return scope.widgetCtrl.getEvent('currentEvent');
                    }
                });
            }

            // it is necessary to initialize comment mode so the nodes are painted
            EchartsHelper.initializeCommentMode({
                comments: eChartsConfig.comments,
                currentMode: eChartsConfig.currentMode,
                saveCb: eChartsConfig.callbacks.commentMode.onSave
            });
        }

        /**
         * @name getLabels
         * @desc determines labels to pass into eChartsService for Brush
         * @returns {Array} - array of objects defining Data Zoom settings
         */
        function getLabels() {
            if (eChartsConfig.options.rotateAxis) {
                if (eChartsConfig.yAxisConfig[0]) {
                    if (eChartsConfig.yAxisConfig[0][0]) {
                        // TODO Facet: pass in yAxisIndex
                        return eChartsConfig.yAxisConfig[0][0].data;
                    }
                    return eChartsConfig.yAxisConfig[0].data;
                }
                return [];
            }
            if (eChartsConfig.xAxisConfig[0]) {
                if (eChartsConfig.xAxisConfig[0][0]) {
                    // TODO Facet: pass in yAxisIndex
                    return eChartsConfig.xAxisConfig[0][0].data;
                }
                return eChartsConfig.xAxisConfig[0].data;
            }
            return [];
        }

        /**
         * @name toggleZoom
         * @desc toggles Data Zoom feature
         * @param {bool} showX - boolean to show x zoom or not
         * @param {bool} showY - boolean to show y zoom or not
         * @param {number} xStart - start value for x
         * @param {number} xEnd - end value for x
         * @param {number} yStart - start value for y
         * @param {number} yEnd - end value for y
         * @returns {Array} - array of objects defining Data Zoom settings
         */
        function toggleZoom(showX, showY, xStart, xEnd, yStart, yEnd) {
            var dataZoom = [],
                xSlider,
                xInside,
                ySlider,
                i,
                xAxisIndex,
                yAxisIndex,
                bottom = 20,
                yInside;

            if (eChartsConfig.groupByInfo && eChartsConfig.groupByInfo.viewType === 'All Instances') {
                xAxisIndex = [];
                yAxisIndex = [];
                for (i = 0; i < eChartsConfig.groupByInfo.uniqueInstances.length; i++) {
                    xAxisIndex.push(i);
                    yAxisIndex.push(i);
                }
            } else {
                xAxisIndex = 0;
                yAxisIndex = 0;
            }

            if (eChartsConfig.groupByInfo && eChartsConfig.groupByInfo.viewType === 'Individual Instance') {
                bottom += 40;
            }

            if (showX) {
                xSlider = {
                    type: 'slider',
                    show: true,
                    // xAxisIndex: 0,
                    xAxisIndex: xAxisIndex,
                    bottom: bottom + 'px',
                    filterMode: 'empty',
                    showDetail: false,
                    // CustomStyle
                    backgroundColor: 'rgba(255,255,255,1)',
                    borderColor: '#EFEFEF',
                    dataBackground: {
                        lineStyle: {
                            color: 'rgb(210,210,210)'
                        },
                        areaStyle: {
                            color: 'rgb(210,210,210)'
                        }
                    },
                    start: xStart || 0,
                    end: xEnd || 100,
                    height: 20,
                    handleIcon: 'M10.7,11.9H9.3c-4.9,0.3-8.8,4.4-8.8,9.4c0,5,3.9,9.1,8.8,9.4h1.3c4.9-0.3,8.8-4.4,8.8-9.4C19.5,16.3,15.6,12.2,10.7,11.9z M13.3,24.4H6.7v-1.2h6.6z M13.3,22H6.7v-1.2h6.6z M13.3,19.6H6.7v-1.2h6.6z',
                    handleSize: 24,
                    handleStyle: {
                        color: 'rgba(79, 164, 222, 0.6)'
                    },
                    fillerColor: 'rgba(154,201,248,0.3)'
                };
                xInside = {
                    type: 'inside',
                    xAxisIndex: xAxisIndex,
                    filterMode: 'empty'
                };
                dataZoom.push(xSlider, xInside);
            }
            if (showY) {
                ySlider = {
                    type: 'slider',
                    show: true,
                    yAxisIndex: yAxisIndex,
                    right: '20px',
                    filterMode: 'empty',
                    showDetail: false,
                    backgroundColor: 'rgba(255,255,255,1)',
                    borderColor: '#EFEFEF',
                    dataBackground: {
                        lineStyle: {
                            color: 'rgb(210,210,210)'
                        },
                        areaStyle: {
                            color: 'rgb(210,210,210)'
                        }
                    },
                    start: yStart || 0,
                    end: yEnd || 100,
                    width: 20,
                    handleIcon: 'M10.7,11.9H9.3c-4.9,0.3-8.8,4.4-8.8,9.4c0,5,3.9,9.1,8.8,9.4h1.3c4.9-0.3,8.8-4.4,8.8-9.4C19.5,16.3,15.6,12.2,10.7,11.9z M13.3,24.4H6.7v-1.2h6.6z M13.3,22H6.7v-1.2h6.6z M13.3,19.6H6.7v-1.2h6.6z',
                    handleSize: 24,
                    handleStyle: {
                        color: 'rgba(79, 164, 222, 0.6)'
                    },
                    fillerColor: 'rgba(154,201,248,0.3)'
                };
                yInside = {
                    type: 'inside',
                    yAxisIndex: yAxisIndex,
                    filterMode: 'empty'
                };
                dataZoom.push(ySlider, yInside);
            }
            return dataZoom;
        }

        /**
         * @name toggleMode
         * @desc switches the mode to the new specified mode
         * @returns {void}
         */
        function toggleMode() {
            var layerIndex = 0;
            eChartsConfig.echartsMode = EchartsHelper.getEchartsMode(scope.widgetCtrl.getMode('selected'));
            eChartsConfig.currentMode = EchartsHelper.getCurrentMode(scope.widgetCtrl.getMode('selected'));
            eChartsConfig.groupByInfo = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.groupByInfo');

            initializeEvents();
        }

        /**
         * @name determineResize
         * @desc detemin parent and chart container dimensions if Facet exists
         * @returns {void}
         */
        function determineResize() {
            var chartContainer = ele[0].childNodes[0],
                parent = ele[0],
                numRows,
                numColumns,
                containerHeight,
                containerWidth,
                calculatedGridWidth,
                calculatedGridHeight,
                calculatedSpacingX,
                calculatedSpacingY,
                facetPadding = { // padding within the top and left of the panel
                    top: 100,
                    right: 120,
                    bottom: 100,
                    left: 120
                };

            parent.style.position = '';
            parent.style.top = '';
            parent.style.right = '';
            parent.style.bottom = '';
            parent.style.left = '';
            parent.style.overflowY = '';
            chartContainer.style.width = '';
            chartContainer.style.height = '';

            if (eChartsConfig.groupByInfo && eChartsConfig.groupByInfo.viewType === 'All Instances') {
                numColumns = eChartsConfig.options.facetHeaders.numberColumns;
                numRows = Math.ceil(eChartsConfig.groupByInfo.uniqueInstances.length / numColumns);
                calculatedGridWidth = eChartsConfig.options.facetHeaders.grid.width;
                calculatedGridHeight = eChartsConfig.options.facetHeaders.grid.height;
                calculatedSpacingX = eChartsConfig.options.facetHeaders.spacing.x;
                calculatedSpacingY = eChartsConfig.options.facetHeaders.spacing.y;

                if (eChartsConfig.options.facetHeaders.customLayout && eChartsConfig.options.facetHeaders.unitType === '%') {
                    calculatedGridWidth = (chartContainer.clientWidth * eChartsConfig.options.facetHeaders.grid.width) / 100;
                    calculatedGridHeight = (chartContainer.clientHeight * eChartsConfig.options.facetHeaders.grid.height) / 100;
                    calculatedSpacingX = (chartContainer.clientWidth * eChartsConfig.options.facetHeaders.spacing.x) / 100;
                    calculatedSpacingY = (chartContainer.clientHeight * eChartsConfig.options.facetHeaders.spacing.y) / 100;
                }

                if (!eChartsConfig.options.facetHeaders.customLayout) {
                    calculatedGridWidth = (chartContainer.clientWidth - facetPadding.left - ((numColumns - 1) * calculatedSpacingX) - facetPadding.right) / numColumns;
                    eChartsConfig.options.facetHeaders.grid.width = calculatedGridWidth;
                }

                containerHeight = facetPadding.top + (numRows * calculatedGridHeight) + ((numRows - 1) * calculatedSpacingY) + facetPadding.bottom;
                containerWidth = facetPadding.left + (numColumns * calculatedGridWidth) + ((numColumns - 1) * calculatedSpacingX) + facetPadding.right;

                parent.style.position = 'absolute';
                parent.style.top = '0';
                parent.style.right = '0';
                parent.style.bottom = '0';
                parent.style.left = '0';
                parent.style.overflowY = 'auto';

                if (chartContainer.clientWidth < containerWidth) {
                    chartContainer.style.width = '' + containerWidth + 'px';
                    clientWidth = containerWidth;
                } else {
                    chartContainer.style.width = '';
                    clientWidth = chartContainer.clientWidth;
                }

                if (chartContainer.clientHeight < containerHeight) {
                    chartContainer.style.height = '' + containerHeight + 'px';
                    clientHeight = containerHeight;
                } else {
                    chartContainer.style.height = '';
                    clientHeight = chartContainer.clientHeight;
                }
            }
        }

        /**
         * @name resizeViz
         * @desc reruns the jv paint function
         * @returns {void}
         */
        function resizeViz() {
            if (barChart) {
                barChart.resize();
            }

            if (eChartsConfig.groupByInfo && eChartsConfig.groupByInfo.viewType === 'All Instances') {
                determineResize();
                paint();
            }
        }

        /**
         * @name destroy
         * @desc destroys listeners and dom elements outside of the scope
         * @returns {void}
         */
        function destroy() {
            resizeListener();
            updateTaskListener();
            updateOrnamentsListener();
            addDataListener();
            modeListener();
            removeBrushListener();
        }

        // Start Visualization Creation
        initialize();
    }
}
