/** *  jvTip ***/
import * as d3 from 'd3';
import visualizationUniversal from '../../../../core/store/visualization/visualization';
class jvTip {
    constructor(configObj) {
        let tip = this,
            defaultConfig = {
                type: 'simple'
            };
        tip.tipConfig = configObj.tipConfig || defaultConfig;
        tip.chartDiv = configObj.chartDiv;
        tip.options = configObj.uiOptions;

        // Create initial div
        tip.chartDiv.select('.jv-tooltip').remove();

        tip.chartDiv.append('div')
            .attr('class', 'tooltip jv-tooltip')
            .style('pointer-events', 'none')
            .style('opacity', 0);
    }

    showTip(transitionDuration = 50) {
        let tip = this,
            left = 'auto',
            top = 'auto',
            mouse = d3.mouse(tip.chartDiv.select('svg').node()),
            // Logic to determine where tooltip will be placed on page
            leftOfMouse = mouse[0] > (tip.chartDiv._groups[0][0].clientWidth / 2),
            topOfMouse = mouse[1] < (tip.chartDiv._groups[0][0].clientHeight / 2),
            tooltipHeight = tip.toolTip._groups[0][0].clientHeight === 0 ? 75 : tip.toolTip._groups[0][0].clientHeight,
            tooltipWidth = tip.toolTip._groups[0][0].clientWidth,
            t;

        if (leftOfMouse) {
            if (tooltipWidth === 0) {
                tooltipWidth = 250;
            }
            left = mouse[0] - tooltipWidth;
        } else {
            left = mouse[0];
        }
        if (topOfMouse) {
            top = mouse[1];
        } else {
            if (tooltipHeight === 0) {
                tooltipHeight = 75;
            }
            top = mouse[1] - tooltipHeight;
        }

        if (!leftOfMouse && topOfMouse) {
            left = mouse[0] + 13;
        }


        // COOL CURSOR, a function of the height and width of the container
        // var container = tip.chartDiv.select('.bar-container').node().getBoundingClientRect();
        // svgMouse = d3.mouse(tip.chartDiv.select('.bar-container').node());

        // var tooltipHeight = tip.toolTip._groups[0][0].clientHeight === 0 ? 75 : tip.toolTip._groups[0][0].clientHeight;
        // top = mouse[1] - (tooltipHeight * svgMouse[1] / container.height);

        // var tooltipWidth = tip.toolTip._groups[0][0].clientWidth;
        // left = mouse[0] - (tooltipWidth * svgMouse[0] / container.width);

        // STICKY CURSOR IN THE BOTTOM RIGHT
        // top = mouse[1];
        // left = mouse[0];
        // set max left
        // if(left > container.width - tooltipWidth + container.left) {
        // left = container.width - tooltipWidth + container.left;
        // }

        // // set max top
        // if (top > container.height - tooltipHeight + container.top) {
        // top = container.height - tooltipHeight + container.top;
        // }

        t = d3.transition()
            .duration(transitionDuration)
            .ease(d3.easeLinear);

        tip.toolTip
            .transition(t)
            .style('left', left + 'px')
            .style('top', top + 'px')
            .style('display', 'block')
            .style('opacity', 1);
    }

    hideTip() {
        let tip = this,
            t = d3.transition()
                .duration('100')
                .ease(d3.easeLinear);
        if (tip.toolTip) {
            tip.toolTip.transition(t).style('display', 'none');
        }
    }

    generateSimpleTip(dataObj, dataTable, dataHeaders) {
        let tip = this,
            tooltipHtml = '';

        if (dataObj.hasOwnProperty('title') && dataObj.title === '') {
            dataObj.title = 'Empty';
        }

        if (dataObj.viz === 'clusters' || dataObj.viz === 'circleviewplot' || dataObj.viz === 'scatterplot' || dataObj.viz === 'treemap' || dataObj.viz === 'singleaxis') {
            tooltipHtml = generateSingleColorHTML(dataObj, dataTable, dataHeaders, tip);
        } else if (dataObj.viz === 'radial' || dataObj.viz === 'pie') {
            tooltipHtml = generatePieHTML(dataObj, dataTable, dataHeaders, tip);
        } else if (dataObj.viz === 'circlepack' || dataObj.viz === 'sunburst') {
            tooltipHtml = generatePackHTML(dataObj, dataHeaders, tip);
        } else if (dataObj.viz === 'heatmap' || dataObj.viz === 'cloud') {
            tooltipHtml = generateHeatmapHTML(dataObj, dataHeaders, tip);
        } else if (dataObj.viz === 'sankey') {
            tooltipHtml = generateSankeyHTML(dataObj, dataHeaders, tip);
        } else if (dataObj.viz === 'bubble') {
            tooltipHtml = generateBubbleHTML(dataObj, dataHeaders, tip);
        } else if (dataObj.viz === 'boxwhisker') {
            tooltipHtml = generateBoxHTML(dataObj, dataHeaders, tip);
        } else if (dataObj.viz === 'clustergram') {
            tooltipHtml = generateClustergramHTML(dataObj, dataHeaders, tip);
        } else if (dataObj.viz === 'gantt') {
            tooltipHtml = generateGanttHTML(dataObj, dataTable, tip);
        } else {
            tooltipHtml = generateSimpleHTML(dataObj, dataHeaders, tip);
        }

        // add content to tooltip
        tip.toolTip = tip.chartDiv.select('.tooltip')
            .html(tooltipHtml);

        // paint the tooltip
        tip.showTip(0);

        return tip.tooltip;
    }
}

/* ***********************************************  Declare jv tip components *******************************************************************************/
var jvHr = '';

function getValueContent(item, value, colorTile) {
    let valueString = value.toString() ? `: ${value}` : '',
        colorTileString = colorTile ? colorTile : '';
    return `<div class='jv-tip-content '>${colorTileString}<span class="jv-tip-item-text">${item}</span>${valueString}</div>`;
}

function getTitleTemplate(dataObj) {
    return `<div class='title jv-top-margin jv-inline'><b>${dataObj.title}</b></div>${jvHr}`;
}

function getColorTile(color) {
    // Case that we either have an rgba object with a toString or a hex color string
    if ((color && Object.getPrototypeOf(color).rgb) || typeof color === 'string') {
        return `<div class='d3-tooltip-circle jv-inline jv-tip-color-tile' style='background:${color}'></div>`;
    // Case that we have a custom rgba object
    } else if (color) {
        return `<div class='d3-tooltip-circle jv-inline jv-tip-color-tile' style='background:rgba(${color.r}, ${color.g}, ${color.b}, ${color.opacity})'></div>`;
    }

    // No color to return
    return "<div class='jv-inline jv-full-width'>";
}


/* ************************************************ Viz Specific Functions **********************************************************************************/

function generateSimpleHTML(dataObj, dataHeaders, tip) {
    let tooltipText;
    tooltipText = `<div><div class='title jv-tip-container jv-top-margin'><b>${dataObj.title}</b></div>${jvHr}`;

    for (let item in dataObj.tipData) {
        if (visualizationUniversal) {
            let type = dataHeaders ? getDataType(item, dataHeaders) : null,
                additionalFormat,
                formatted = dataObj.tipData[item];
            if (tip.options && tip.options.formatDataValues) {
                additionalFormat = getAdditionalFormat(item, tip.options.formatDataValues.formats);
                formatted = visualizationUniversal.formatValue(formatted, additionalFormat);
            } else if (type) {
                formatted = visualizationUniversal.formatValue(formatted, type);
            }
            tooltipText += getValueContent(item, formatted, getColorTile(dataObj.color[item]));
        } else {
            tooltipText += getValueContent(item, dataObj.tipData[item], getColorTile(dataObj.color[item]));
        }
    }

    tooltipText += '</div>';
    return tooltipText;
}

function generateSingleColorHTML(dataObj, dataTable, dataHeaders, tip) {
    let tooltipText,
        tooltipColor,
        showColorCircle = true,
        colorCircle = '';

    if (!!dataObj.color[dataObj.data[dataTable.series]]) {
        tooltipColor = dataObj.color[dataObj.data[dataTable.series]];
    } else if (!!dataObj.color[dataTable.label] && dataObj.viz !== 'singleaxis') {
        tooltipColor = dataObj.color[dataTable.label];
    } else {
        showColorCircle = false;
    }

    if (showColorCircle) {
        colorCircle = getColorTile(tooltipColor);
    } else {
        colorCircle = getColorTile();
    }

    tooltipText = `<div class='jv-inline jv-full-width'>${colorCircle}<div class='title jv-inline jv-full-width jv-top-margin'><b>${dataObj.title}</b></div>${jvHr}`;

    for (let item in dataObj.tipData) {
        if (dataObj.tipData.hasOwnProperty(item)) {
            let type = getDataType(item, dataHeaders),
                formatted = dataObj.tipData[item],
                additionalFormat;
            if (tip.options && tip.options.formatDataValues) {
                additionalFormat = getAdditionalFormat(item, tip.options.formatDataValues.formats);
                formatted = visualizationUniversal.formatValue(formatted, additionalFormat);
            } else if (type) {
                formatted = visualizationUniversal.formatValue(formatted, type);
            }
            tooltipText += getValueContent(item, formatted);
        }
    }
    tooltipText += '</div>';
    return tooltipText;
}

function generatePackHTML(dataObj, dataHeaders, tip) {
    let tooltipText;
    tooltipText = `<div class='jv-inline jv-full-width'>
        ${getColorTile(dataObj.data.color)}
        ${getTitleTemplate(dataObj)}`;

    for (let item in dataObj.tipData) {
        if (dataObj.tipData.hasOwnProperty(item)) {
            let type = getDataType(item, dataHeaders),
                formatted = dataObj.tipData[item],
                additionalFormat = null;
            if (tip.options && tip.options.formatDataValues) {
                additionalFormat = getAdditionalFormat(item, tip.options.formatDataValues.formats);
                formatted = visualizationUniversal.formatValue(formatted, additionalFormat);
            } else if (type) {
                formatted = visualizationUniversal.formatValue(formatted, type);
            }
            tooltipText += getValueContent(item, formatted);
        }
    }
    tooltipText += '</div>';
    return tooltipText;
}

function generateBubbleHTML(dataObj, dataHeaders, tip) {
    let tooltipText;
    tooltipText = `<div class='jv-inline jv-full-width'>
        ${getColorTile(dataObj.data.color)}
        ${getTitleTemplate(dataObj)}`;

    for (let item in dataObj.tipData) {
        if (item === 'color') {
            continue;
        }
        let type = getDataType(item, dataHeaders),
            additionalFormat = null,
            formatted = dataObj.tipData[item];

        if (tip.options && tip.options.formatDataValues) {
            additionalFormat = getAdditionalFormat(item, tip.options.formatDataValues.formats);
            formatted = visualizationUniversal.formatValue(formatted, additionalFormat);
        } else if (type) {
            formatted = visualizationUniversal.formatValue(formatted, type);
        }
        tooltipText += getValueContent(item, formatted);
    }
    tooltipText += '</div>';
    return tooltipText;
}

function generateBoxHTML(dataObj, dataHeaders, tip) {
    let tooltipText;
    tooltipText = `<div class="jv-inline jv-full-width"> <div><div class='title jv-tip-container jv-top-margin'><b>${dataObj.tipData.Label}</b></div>`;

    for (let item in dataObj.tipData) {
        if (dataObj.tipData.hasOwnProperty(item) && item !== 'Value' && item !== 'Label') {
            let type = getDataType(dataObj.tipData.Value, dataHeaders),
                formatted = dataObj.tipData[item],
                additionalFormat;
            if (tip.options && tip.options.formatDataValues) {
                additionalFormat = getAdditionalFormat(dataObj.tipData.Value, tip.options.formatDataValues.formats);
                formatted = visualizationUniversal.formatValue(formatted, additionalFormat);
            } else if (type) {
                formatted = visualizationUniversal.formatValue(formatted, type);
            }
            tooltipText += getValueContent(item, formatted);
        }
    }
    tooltipText += '</div>';
    return tooltipText;
}

function generateHeatmapHTML(dataObj, dataHeaders, tip) {
    let tooltipText;
    if (dataObj.xAxisCat) {
        tooltipText = `<div class='jv-inline jv-full-width'>
            ${getColorTile(dataObj.color)}` +
            "<div class='title jv-top-margin jv-inline jv-full-width'><b>" + dataObj.data.xAxisName + "</b></div><hr style='margin:3px 0 3px 0;'/>";

        tooltipText += "<span class='jv-tip-content'>" + dataObj.xAxisCat + '</span><br/>';
        tooltipText += '</div>';
        return tooltipText;
    } else if (dataObj.yAxisCat) {
        tooltipText = `<div class='jv-inline jv-full-width'>
            ${getColorTile(dataObj.color)}` +
            "<div class='title jv-top-margin jv-inline jv-full-width'><b>" + dataObj.data.yAxisName + "</b></div><hr style='margin:3px 0 3px 0;'/>";

        tooltipText += "<span class='jv-tip-content'>" + dataObj.yAxisCat + '</span><br/>';
        tooltipText += '</div>';
        return tooltipText;
    }
    tooltipText = `<div class='jv-inline jv-full-width'>
            ${getColorTile(dataObj.color)}
            ${getTitleTemplate(dataObj)}`;

    for (let item in dataObj.tipData) {
        if (dataObj.tipData.hasOwnProperty(item)) {
            let type = getDataType(item, dataHeaders),
                formatted = dataObj.tipData[item],
                additionalFormat;
            if (tip.options && tip.options.formatDataValues) {
                additionalFormat = getAdditionalFormat(item, tip.options.formatDataValues.formats);
                formatted = visualizationUniversal.formatValue(formatted, additionalFormat);
            } else if (type) {
                formatted = visualizationUniversal.formatValue(formatted, type);
            }
            tooltipText += getValueContent(item, formatted);
        }
    }
    tooltipText += '</div>';
    return tooltipText;
}

function generateClustergramHTML(dataObj, dataHeaders, tip) {
    let tooltipText;
    dataObj.title = dataObj.title.replace(/_/g, ' ');
    tooltipText = `<div class='jv-inline jv-full-width'>
    ${getColorTile(dataObj.color)}
    ${getTitleTemplate(dataObj)}`;

    for (let item in dataObj.tipData) {
        if (dataObj.tipData.hasOwnProperty(item)) {
            let type = getDataType(item, dataHeaders),
                formatted = dataObj.tipData[item],
                additionalFormat;
            if (tip.options && tip.options.formatDataValues) {
                additionalFormat = getAdditionalFormat(item, tip.options.formatDataValues.formats);
                formatted = visualizationUniversal.formatValue(formatted, additionalFormat);
            } else if (type) {
                formatted = visualizationUniversal.formatValue(formatted, type);
            }
            tooltipText += getValueContent(item, formatted);
        }
    }
    tooltipText += '</div>';

    return tooltipText;
}

function generateGanttHTML(dataObj, dataHeaders, tip) {
    var tooltipText;

    dataObj.title = String(dataObj.title).replace(/_/g, ' ');

    tooltipText = `<div class='jv-inline jv-full-width'>
    ${getTitleTemplate(dataObj)}`;

    for (let item in dataObj.tipData) {
        if (dataObj.tipData.hasOwnProperty(item)) {
            let type = getDataType(item, dataHeaders),
                additionalFormat = null,
                formatted = dataObj.tipData[item];
            if (tip.options && tip.options.formatDataValues) {
                additionalFormat = getAdditionalFormat(item, tip.options.formatDataValues.formats);
                formatted = visualizationUniversal.formatValue(formatted, additionalFormat);
            } else if (type) {
                formatted = visualizationUniversal.formatValue(formatted, type);
            }
            tooltipText += getValueContent(item, formatted);
        }
    }

    tooltipText += '</div>';

    return tooltipText;
}

function generatePieHTML(dataObj, dataTable, dataHeaders, tip) {
    let tooltipText;
    tooltipText = `<div class='jv-inline jv-full-width'>${getColorTile(dataObj.color[dataObj.data[dataTable.label]])}${getTitleTemplate(dataObj)}`;

    for (let item in dataObj.tipData) {
        if (dataObj.tipData.hasOwnProperty(item)) {
            let type = getDataType(item, dataHeaders),
                additionalFormat = null,
                formatted = dataObj.tipData[item];
            if (tip.options && tip.options.formatDataValues) {
                additionalFormat = getAdditionalFormat(item, tip.options.formatDataValues.formats);
                formatted = visualizationUniversal.formatValue(formatted, additionalFormat);
            } else if (type) {
                formatted = visualizationUniversal.formatValue(formatted, type);
            }
            tooltipText += getValueContent(item, formatted);
        }
    }
    tooltipText += '</div>';
    return tooltipText;
}

function generateSankeyHTML(dataObj, dataHeaders, tip) {
    let tooltipText;
    tooltipText = `<div class='jv-inline jv-full-width'>${getTitleTemplate(dataObj)}`;

    for (let item in dataObj.tipData) {
        if (dataObj.tipData.hasOwnProperty(item)) {
            let type = getDataType(item, dataHeaders),
                formatted = dataObj.tipData[item],
                additionalFormat;
            if (tip.options && tip.options.formatDataValues) {
                additionalFormat = getAdditionalFormat(item, tip.options.formatDataValues.formats);
                formatted = visualizationUniversal.formatValue(formatted, additionalFormat);
            } else if (type) {
                formatted = visualizationUniversal.formatValue(formatted, type);
            }
            tooltipText += getValueContent(item, formatted);
        }
    }
    tooltipText += '</div>';
    return tooltipText;
}
/* ************************************************ Utility Functions **********************************************************************************/
function getDataType(item, dataHeaders) {
    let type;
    for (let i = 0; i < dataHeaders.length; i++) {
        if (dataHeaders[i].name === item.replace(/ /g, '_')) {
            type = dataHeaders[i].additionalDataType ? dataHeaders[i].additionalDataType : '';
            break;
        }
    }
    return type;
}

function getAdditionalFormat(item, options) {
    let format;
    for (let i = 0; i < options.length; i++) {
        if (options[i].dimension === item.replace(/ /g, '_')) {
            format = options[i];
            break;
        }
    }
    return format;
}

export default jvTip;
