const REACTORS = {
    /**
     * @name Pixel
     * @param {string} query - raw query
     * @returns {string} query
     */
    Pixel: function (query) {
        return query;
    },

    /**
     * @name variable
     * @param {string} query - pixel variable
     * @returns {string} query
     */
    variable: function (query) {
        return query;
    },

    /**
     * @name if
     * @param {array} conditionalComponentsArray - conditional component array
     * @param {array} trueComponentsArray - true component array
     * @param {array} falseComponentsArray - falsestatement component array
     * @returns {string} query
     */
    if: function (conditionalComponentsArray, trueComponentsArray, falseComponentsArray) {
        var query = '',
            trueIdx,
            falseIdx;

        query += 'if(  (';
        query += this.trim(this.build(conditionalComponentsArray), ';');
        query += ') , (';
        if (trueComponentsArray && trueComponentsArray.length > 0) {
            for (trueIdx = 0; trueIdx < trueComponentsArray.length; trueIdx++) {
                if (trueIdx === 0 || (trueComponentsArray[trueIdx - 1].terminal)) {
                    query += '(';
                }
                query += this.trim(this.build([trueComponentsArray[trueIdx]]), ';');
                if (trueComponentsArray[trueIdx].terminal) {
                    query += ')';
                }
                if (trueIdx !== trueComponentsArray.length - 1 && trueComponentsArray[trueIdx].terminal) {
                    query += ',';
                }
            }
        } else {
            query += 'true';
        }
        query += ') , (';
        if (falseComponentsArray && falseComponentsArray.length > 0) {
            for (falseIdx = 0; falseIdx < falseComponentsArray.length; falseIdx++) {
                if (falseIdx === 0 || (falseComponentsArray[falseIdx - 1].terminal)) {
                    query += '(';
                }
                query += this.trim(this.build([falseComponentsArray[falseIdx]]), ';');
                if (falseComponentsArray[falseIdx].terminal) {
                    query += ')';
                }
                if (falseIdx !== falseComponentsArray.length - 1 && falseComponentsArray[falseIdx].terminal) {
                    query += ',';
                }
            }
        } else {
            query += 'false';
        }
        query += ') )';

        return query;
    },

    /**
     * @name date
     * @returns {string} query
     */
    date: function () {
        var query = '';
        query += 'Date()';
        return query;
    },

    /**
     * @name getUserInfo
     * @returns {string} query
     */
    getUserInfo: function () {
        var query = '';
        query += 'GetUserInfo()';
        return query;
    },

    /**
     * @name if
     * @param {array} conditionalComponentsArray - conditional component array
     * @param {array} trueComponentsArray - true component array
     * @returns {string} query
     */
    ifError: function (conditionalComponentsArray, trueComponentsArray) {
        var query = '';

        query += 'ifError(  (';
        query += this.trim(this.build(conditionalComponentsArray), ';');
        query += ') , (';

        if (trueComponentsArray.length === 0) {
            query += 'true';
        } else {
            query += this.trim(this.build(trueComponentsArray), ';');
        }

        query += ') )';

        return query;
    },

    /** *Insight */
    /**
     * @name openEmptyInsight
     * @desc open an empty insight
     * @param {string} recipe - recipe to run
     * @param {object} params - params to add to the recipe
     * @returns {string} query
     */
    openEmptyInsight: function (recipe, params) {
        var query = '';

        query += 'OpenEmptyInsight(';

        if (recipe) {
            query += 'recipe=[\"<sEncode>' + recipe + '</sEncode>\"]';
        }


        if (params) {
            query += ' params=[\"' + JSON.stringify(params) + '\"] ';
        }

        query += ');';

        return query;
    },

    /**
     * @name OpenInsight
     * @param {string} app - app to grab the insight from
     * @param {string} rdbmsId - id of the insight
     * @param {object} params - params to add to the recipe
     * @param {string} postQuery - pixel to run after opening
     * @desc open an insight
     * @returns {string} query
     */
    openInsight: function (app, rdbmsId, params, postQuery) {
        var query = '';

        query += 'OpenInsight(app=[\"' + app + '\"], id=["' + rdbmsId + '"]';


        if (params) {
            query += ',params=[\"' + params + '\"]';
        }

        if (postQuery) {
            query += ',additionalPixels=[\"' + postQuery + '\"]';
        }

        query += ');';

        return query;
    },

    /**
     * @name databaseRecommendations
     * @desc get all recommended apps
     * @returns {string} query
     */
    databaseRecommendations: function () {
        var query = '';

        query += 'DatabaseRecommendations()';

        return query;
    },

    /**
     * @name widgetTracking
     * @desc send widget tracking (user clicks)
     * @param {array} insightChanges - boolean to only show local apps; if false include external apps
     * @returns {string} query
     */
    widgetTracking: function (insightChanges) {
        var query = '';

        query += 'WidgetT(' + JSON.stringify(insightChanges) + ')';

        return query;
    },

    /**
     * @name clearInsight
     * @desc clears an insight
     * @returns {string} query
     */
    clearInsight: function () {
        var query = '';

        query = 'ClearInsight()';

        return query;
    },
    /**
     * @name dropInsight
     * @desc drops an insight
     * @returns {string} query
     */
    dropInsight: function () {
        var query = '';

        query = 'DropInsight()';

        return query;
    },

    /**
     * @name getSpecificInsightMeta
     * @param {string} app the app id
     * @param {string} insightId the insight id
     * @desc get the specific meta information for this insight
     * @returns {string} the pixel component
     */
    getSpecificInsightMeta: function (app, insightId) {
        // GetSpecificInsightMeta ( app=["ccebb957-ec6e-4622-9d71-640016744ad5"], id=["551983e5-34ac-4e26-a1f5-92f919b6615c"] ) ;
        var query = '';

        query = `GetSpecificInsightMeta(app=["${app}"], id=["${insightId}"])`;

        return query;
    },

    /**
     * @name setInsightDescription
     * @param {string} app the app id
     * @param {string} insightId the insight id
     * @param {string} description the description to set for this insight
     * @desc set the insight description
     * @returns {string} the pixel component
     */
    setInsightDescription: function (app, insightId, description) {
        // SetInsightDescription(app=["ccebb957-ec6e-4622-9d71-640016744ad5"], id=["551983e5-34ac-4e26-a1f5-92f919b6615c"], description=["testing description"]);
        var query = '';

        query = `SetInsightDescription(app=["${app}"], id=["${insightId}"], description=["${description}"])`;

        return query;
    },

    /**
     * @name setInsightTags
     * @param {string} app the app id
     * @param {string} insightId the insight id
     * @param {array} tags the list of tags to set for this insight
     * @desc set the insight tags
     * @returns {string} the pixel component
     */
    setInsightTags: function (app, insightId, tags) {
        // SetInsightTags(app=["ccebb957-ec6e-4622-9d71-640016744ad5"], id=["551983e5-34ac-4e26-a1f5-92f919b6615c"], tags=["a","b","c"]);
        var query = '';

        query = `SetInsightTags(app=["${app}"], id=["${insightId}"], tags=${JSON.stringify(tags)})`;

        return query;
    },

    /**
     * @name getAvailableTags
     * @param {string} app the app id
     * @desc get the list of available tags for this insight
     * @returns {string} the pixel component
     */
    getAvailableTags: function (app) {
        // GetAvailableTags(app=["ccebb957-ec6e-4622-9d71-640016744ad5"]);
        var query = '';

        query = `GetAvailableTags(app=["${app}"])`;

        return query;
    },

    /** Data**/

    /**
     * @name filePaste
     * @param {string} pastedData - type of file
     * @param {string} delimiter - the delimiter to use to parse the data
     * @returns {string} query
     */
    filePaste: function (pastedData, delimiter) {
        var query = '';

        query = 'TextInput(fileData=["<encode>' + pastedData + '</encode>"],';
        query += 'delim=["' + delimiter + '"]';

        query += ')';

        return query;
    },
    /**
     * @name fileRead
     * @param {string} fileType - type of file
     * @param {string} path - path of the file
     * @param {string} sheetName - name of the sheet to read for excel
     * @param {string} sheetRange - range of sheet
     * @param {object} dataTypeMap - the types for the headers
     * @param {string} delimiter - the delimiter to use to parse the data
     * @param {array} newHeaders - the new header names
     * @param {string} plainFileName - file name without path, timestamp, and extension
     * @param {object} additionalDataTypes - if data type has formatting, we map the formatting to the data type here
     * @returns {string} query
     */
    fileRead: function (fileType, path, sheetName, sheetRange, dataTypeMap, delimiter, newHeaders, plainFileName, additionalDataTypes) {
        var query = '';

        query = 'FileRead(filePath=["' + path + '"],';
        query += 'dataTypeMap=[' + JSON.stringify(dataTypeMap) + '],';
        query += 'delimiter=["' + delimiter + '"],';
        query += 'newHeaders=[' + JSON.stringify(newHeaders) + '],';

        if (fileType === 'excel') {
            query += 'sheetName=["' + sheetName + '"],';
            query += 'sheetRange=["' + sheetRange + '"], ';
        }

        query += 'fileName=["' + plainFileName + '"], ';

        query += 'additionalDataTypes=[' + JSON.stringify(additionalDataTypes) + ']';

        query += ')';

        return query;
    },
    /**
     * @name createSource
     * @param {string} sourceType - type of source e.g Database or Frame
     * @param {string} sourceName - name of the source you are pulling from
     * @param {string} alias - used to identify newly created source
     * @desc create the source component e.g. CreateFrame(grid).as(['frame'])
     * @returns {string} query
     */
    createSource: function (sourceType, sourceName, alias) {
        var query = '';

        query += 'Create' + sourceType + '(' + sourceName + ')';

        if (alias) {
            query += '.as([\'' + alias + '\'])';
        }

        return query;
    },

    /**
     * @name removeFrame
     * @param {string} frameName the name of the frame to remove
     * @desc remove the frame
     * @returns {string} query
     */
    removeFrame: function (frameName) {
        var query = '';

        query += 'RemoveFrame("' + frameName + '")';

        return query;
    },

    /**
     * @name getFrames
     * @desc get the available (active) frames
     * @returns {string} query
     */
    getFrames: function () {
        var query = '';

        query += 'GetFrames()';

        return query;
    },

    /**
     * @name removeVariable
     * @param {string} variable - variable to remove
     * @desc removes a variable
     * @returns {string} query
     */
    removeVariable: function (variable) {
        var query = '';

        query += 'RemoveVariable(\"' + variable + '\")';

        return query;
    },

    /**
     * @name currentVariables
     * @desc gets the variables and their values on backend
     * @return {void}
     */
    currentVariables: function () {
        var query = '';

        query += 'CurrentVariables()';

        return query;
    },

    /**
     * @name source
     * @param {string} sourceType - type of source e.g Database or Frame
     * @param {string} sourceName - name of the source you are pulling from
     * @desc create the source component e.g. Database("Movie_DB")
     * @returns {string} query
     */
    source: function (sourceType, sourceName) {
        var query = '';

        query += sourceType;

        query += '(';

        if (sourceName) {
            query += sourceName;
        }

        query += ')';

        return query;
    },

    /**
     * @name frame
     * @param {string} frame - frame that you are importing
     * @desc create the frame component e.g. Frame(ABCD)
     * @returns {string} query
     */
    frame: function (frame) {
        var query = '';

        query += 'Frame(';
        if (frame) {
            query += ' frame=[' + frame + '] ';
        }
        query += ')';

        return query;
    },

    /**
     * @name database
     * @param {string} database - database that you are importing
     * @desc create the frame component e.g. Database(Movie_DB)
     * @returns {string} query
     */
    database: function (database) {
        var query = '';

        query += 'Database(';
        query += ' database=[\"' + database + '\"] ';
        query += ')';

        return query;
    },

    /**
     * @name jdbcSource
     * @param {string} dbDriver - database driver
     * @param {string} connectionString - custom JDBC url
     * @param {string} username - username to access db
     * @param {string} password - password for user
     * @desc create the frame component e.g. Database(Movie_DB)
     * @returns {string} query
     */
    jdbcSource: function (dbDriver, connectionString, username, password) {
        var query = '';

        query += 'JdbcSource(';
        query += 'dbDriver=["' + (dbDriver || '') + '"]';
        query += ', connectionString=["' + (connectionString || '') + '"]';
        if (username) {
            query += ', username=["' + username + '"]';
        }
        if (password) {
            query += ', password=["' + password + '"]';
        }

        query += ')';

        return query;
    },

    /**
     * @name query
     * @param {string} queryText - raw query you want to query
     * @desc create the query component e.g. Query("Select....")
     * @returns {string} query
     */
    query: function (queryText) {
        var query = '';
        query += 'Query("<encode>';
        query += queryText;
        query += '</encode>")';
        return query;
    },

    /**
     * @name select2
     * @param {array} selectors - list of the selectors to add
     * @desc create the select component e.g. Select(Title,Studio)
     * @returns {string} query
     */
    select2: function (selectors) {
        var query = '',
            i,
            hasAlias = true,
            len = selectors.length;

        query += 'Select(';

        for (i = 0; i < len; i++) {
            // Add grouping to query if applicable
            if (selectors[i].selector) {
                query += selectors[i].selector;
            } else if (selectors[i].math && selectors[i].calculatedBy) {
                query += (selectors[i].math + '(' + selectors[i].calculatedBy + ')');
            } else {
                query += selectors[i].alias;
            }

            // can't add an alias if it isn't there or missing...
            if (!selectors[i].alias) {
                hasAlias = false;
                console.error('Alias is needed!');
            }
            query += ', ';
        }

        // trim trailing comma
        query = this.trim(query, ',');

        query += ')'; // Select(Title)

        // add the alias
        if (hasAlias) {
            query += '.as([';
            for (i = 0; i < len; i++) {
                if (selectors[i].alias) {
                    query += selectors[i].alias;
                } else {
                    query += selectors[i].selector; // Fallback... you should pass in an alias
                }
                query += ', ';
            }

            // trim trailing comma
            query = this.trim(query, ',');

            query += '])';
        }

        return query;
    },

    /**
     * @name filter
     * @param {object} filterObj - object containing the selectors to add
     * @desc create the filter component e.g. Filter(Studio=["WB", "Paramount"])
     * @returns {string} query
     */
    filter: function (filterObj) {
        var query = '',
            filter,
            itemIdx;

        for (filter in filterObj) {
            if (filterObj.hasOwnProperty(filter)) {
                if (filterObj[filter].isFilterString) {
                    // complex filters means utilizing AND/OR filters aka custom filters, which we will just append.
                    query += filterObj[filter].value;
                } else {
                    // filter filterObj[filter].comparator
                    query += filter; // Filter(Studio
                    query += filterObj[filter].comparator; // Filter(Studio=

                    // handle array different instead of JSON.stringifying because '\' gets escaped. so will manually construct
                    if (Array.isArray(filterObj[filter].value)) {
                        query += '[';
                        for (itemIdx = 0; itemIdx < filterObj[filter].value.length; itemIdx++) {
                            if (typeof filterObj[filter].value[itemIdx] === 'string') {
                                query += '"';
                                // escape special characters (quotes and backslash)
                                query += filterObj[filter].value[itemIdx].replace(/"/g, '\\"');
                                query += '"';
                            } else {
                                query += filterObj[filter].value[itemIdx];
                            }

                            // if not last
                            if (itemIdx !== filterObj[filter].value.length - 1) {
                                query += ', ';
                            }
                        }
                        query += ']';
                    } else if (typeof filterObj[filter].value === 'string') {
                        if (filterObj[filter].isVariable) {
                            query += filterObj[filter].value;
                        } else {
                            query += '"' + filterObj[filter].value.replace(/"/g, '\\"') + '"'; // Filter(Studio=["WB", "Paramount"]
                        }
                    } else {
                        query += filterObj[filter].value; // Filter(MovieBudget>1000000
                    }
                }

                query += ',';
            }
        }

        // trim trailing comma
        query = this.trim(query, ',');

        if (query) {
            query = 'Filter(' + query + ')'; // Filter(Studio=["WB", "Paramount"])
        }

        return query;
    },

    /**
     * @name filter
     * @param {object} filterObj - object containing the selectors to add
     * @desc create the filter component e.g. Filter(Studio=["WB", "Paramount"])...specific to JSON API
     *       only difference is this one wraps keys and values in double quotes
     * @returns {string} query
     */
    jsonFilter: function (filterObj) {
        var query = '',
            subValue,
            filter;

        for (filter in filterObj) {
            if (filterObj.hasOwnProperty(filter)) {
                if (!Array.isArray(filterObj[filter].value) &&
                    typeof filterObj[filter].value === 'object') {
                    for (subValue in filterObj[filter].value) {
                        if (filterObj[filter].value.hasOwnProperty(subValue)) {
                            query += '"' + filter + '"'; // Filter(Studio
                            query += filterObj[filter].comparator; // Filter(Studio=
                            query += '"' + subValue + '"';
                            query += ',';
                        }
                    }
                } else {
                    // filter filterObj[filter].comparator
                    query += '"' + filter + '"'; // Filter(Studio
                    query += filterObj[filter].comparator; // Filter(Studio=

                    if (Array.isArray(filterObj[filter].value)) {
                        query += JSON.stringify(filterObj[filter].value); // Filter(Studio=["WB", "Paramount"]
                    } else {
                        query += '"' + filterObj[filter].value + '"'; // Filter(MovieBudget>1000000
                    }

                    query += ',';
                }
            }
        }

        // trim trailing comma
        query = this.trim(query, ',');

        query = 'Filter(' + query + ')'; // Filter(Studio=["WB", "Paramount"])

        return query;
    },

    /**
     * @name join
     * @param {array} joinList - array of the join information
     * @desc create the join component e.g. Join(Title inner.join Studio)
     * @returns {string} query
     */
    join: function (joinList) {
        var query = '',
            i,
            len = joinList.length;

        if (len > 0) {
            for (i = 0; i < len; i++) {
                if (joinList[i].fromColumn && joinList[i].toColumn) {
                    query += '(';
                    query += joinList[i].fromColumn + ', '; // Join(Title
                    query += this.convertJoin(joinList[i].joinType) + ', '; // Join(Title inner.join
                    query += joinList[i].toColumn; // Join(Title inner.join Studio
                    query += '), ';
                }
            }

            // trim trailing comma
            query = this.trim(query, ',');
            query = 'Join(' + query + ')';
        }

        return query;
    },
    /**
     * @name merge
     * @param {array} tableJoins - array of the join information
     * @desc create the merge component e.g. Merge(Title inner.join Title); this is exactly the same as creating the join component. but will keep separate because they could change in the future
     * @param {string} frame - frame that you are mergin into
     * @returns {string} query
     */
    merge: function (tableJoins, frame) {
        var query = '',
            i,
            len;

        query += 'Merge(';

        if (tableJoins) {
            query += 'joins=[';
            for (i = 0, len = tableJoins.length; i < len; i++) {
                query += '(';
                query += tableJoins[i].fromColumn + ', '; // Join(Title
                query += this.convertJoin(tableJoins[i].joinType) + ', '; // Join(Title inner.join
                query += tableJoins[i].toColumn; // Join(Title inner.join Studio
                query += '), ';
            }
            // trim trailing comma
            query = this.trim(query, ',');
            query += ']';
        }

        if (frame) {
            query += ', frame=[' + frame + '] ';
        }
        query += ')';

        return query;
    },

    /**
     * @name limit
     * @param {number} limit - limit the select
     * @desc create a limt component that determines the datas's output
     * @returns {string} query
     */
    limit: function (limit) {
        var query = '';

        query = 'Limit(' + limit + ')';

        return query;
    },


    /**
     * @name import
     * @desc create the import component e.g. Import() to import the data into the frame
     * @param {string} frame - frame that you are importing
     * @returns {string} query
     */
    import: function (frame) {
        var query = '';

        query += 'Import(';

        if (frame) {
            query += ' frame=[' + frame + '] ';
        }
        query += ')';

        return query;
    },

    /**
     * @name queryAll
     * @desc create the query all component to automatically get the whole frame
     * @returns {string} query
     */
    queryAll: function () {
        var query = '';

        query += 'QueryAll()';

        return query;
    },

    /**
     * @name execute
     * @desc create the execute component e.g. Execute() to query the db and not import it
     * @returns {string} query
     */
    execute: function () {
        var query = '';

        query += 'Execute()';

        return query;
    },

    /**
     * @name convert
     * @desc create the convert component e.g. Convert() to convert a frame from one type to another
     * @param {string} frameType - frame type to convert the current frame to
     * @param {string} alias - used to identify newly created source
     * @returns {string} query
     */
    convert: function (frameType, alias) {
        var query = '';

        query += 'Convert(';
        query += ' frameType=[' + frameType + ']';
        query += ')';

        if (alias) {
            query += '.as([\'' + alias + '\'])';
        }

        return query;
    },

    /**
     * @name purge
     * @desc create the purge component e.g. Purge() to truncate the data
     * @returns {string} query
     */
    purge: function () {
        var query = '';

        query += 'Purge()';

        return query;
    },


    /**
     * @name toCsv
     * @desc create the toCsv component e.g. ToCsv() to query the db and export it
     * @returns {string} query
     */
    toCsv: function () {
        var query = '';

        query += 'ToCsv()';

        return query;
    },

    /**
     * @name toTsv
     * @desc create the toTsv component e.g. ToTsv() to query the db and export it
     * @returns {string} query
     */
    toTsv: function () {
        var query = '';

        query += 'ToTsv()';

        return query;
    },

    /**
     * @name toTxt
     * @param {string} delimiter the delimiter to use in the generated text file
     * @desc create the toTxt component e.g. ToTxt() to query the db and export it
     * @returns {string} query
     */
    toTxt: function (delimiter) {
        var query = '';

        query += 'ToTxt("' + delimiter + '")';

        return query;
    },

    /**
     * @name toExcel
     * @desc create the toExcel component e.g. ToExcel() to query the db and export it
     * @returns {string} query
     */
    toExcel: function () {
        var query = '';

        query += 'ToExcel()';

        return query;
    },

    /**
     * @name iterate
     * @desc create a iterate component that initializes the iterator
     * @returns {string} query
     */
    iterate: function () {
        var query = '';

        query = 'Iterate()';

        return query;
    },

    /**
     * @name format
     * @param {string} type - type of format that you will pipe the data into
     * @param {object} options - format options
     * @desc create a format component that determines the datas's output
     * @returns {string} query
     */
    format: function (type, options) {
        var query = '';

        query = 'Format(type=[\'' + type + '\']';

        if (options) {
            query += ', options=' + JSON.stringify(options);
        }

        query += ')';

        return query;
    },

    /**
     * @name removeLayer
     * @param {string} panelId - panel id layer belongs to
     * @param {string} layerId - layer id to remove
     * @desc removes a layer
     * @returns {string} query
     */
    removeLayer: function (panelId, layerId) {
        var query = '';

        query += 'RemoveLayer(';
        query += 'panel=[\"' + panelId + '\"], ';
        query += 'layer=[\"' + layerId + '\"] ';
        query += ')';

        return query;
    },

    /**
     * @name task
     * @param {string} taskId - id of the tast to select
     * @desc creates a task component that sets options associated with the view
     * @returns {string} query
     */
    task: function (taskId) {
        var query = '';
        query += 'Task(\"' + taskId + '\")';
        return query;
    },

    /**
     * @name removeTask
     * @param {string} taskId - id of the task
     * @param {boolean} drop - drop now}
     * @desc removes a task
     * @returns {string} query
     */
    removeTask: function (taskId, drop) {
        var query = '';

        query += 'RemoveTask(\"';
        query += taskId;
        query += '\"';
        if (typeof drop !== 'undefined') {
            query += ', ';
            query += drop;
        }
        query += ')';

        return query;
    },

    /**
     * @name taskOptions
     * @param {object} options - options to associate with the view
     * @desc creates a taskOptions component that sets options associated with the view
     * @returns {string} query
     */
    taskOptions: function (options) {
        var query = '';

        query = 'TaskOptions(' + JSON.stringify(options) + ')';

        return query;
    },

    /**
     * @name autoTaskOptions
     * @param {string} panelId - panelId
     * @param {string} layout - layout
     * @desc automatically sets the taskOptions component that sets options associated with the view
     * @returns {string} query
     */
    autoTaskOptions: function (panelId, layout) {
        var query = '';

        query += 'AutoTaskOptions(';
        query += 'panel=[\"' + panelId + '\"], ';
        query += 'layout=[\"' + layout + '\"] ';
        query += ')';

        return query;
    },

    /**
     * @name sortOptions
     * @param {array} sortOptions the sort options array
     * @desc builds sort options query: sort(cols=[col1, col2....], dirs=[asc, desc, asc....])
     * @return {string} query
     */
    sortOptions: function (sortOptions) {
        var query = '',
            i,
            len = sortOptions.length,
            cols = [],
            dirs = [];

        for (i = 0; i < len; i++) {
            cols.push(sortOptions[i].alias);
            dirs.push(sortOptions[i].dir);
        }

        if (cols.length > 0 && cols.length === dirs.length) {
            query = 'Sort(columns=' + JSON.stringify(cols) + ', sort=' + JSON.stringify(dirs) + ')';
        }

        return query;
    },

    /**
     * @name collect
     * @param {number} amount - amount to collect
     * @desc create a collect component that pulls from the iterator
     * @returns {string} query
     */
    collect: function (amount) {
        var query = '';

        query = 'Collect(' + amount + ')';

        return query;
    },

    /**
     * @name collectGraph
     * @param {string} frame frame name
     * @desc create a collect graph component that pulls all of the nodes and edges
     * @returns {string} query
     */
    collectGraph: function (frame) {
        var query = 'CollectGraph(';

        if (frame) {
            query += 'frame=[' + frame + ']';
        }
        query += ')';

        return query;
    },

    /**
     * @name collectPivot
     * @param {*} rows the rows for the pivot table
     * @param {*} columns the columns for the pivot table
     * @param {*} calculations the calculations for the intersections
     * @desc create a pivot table from r using the passed in params
     * @returns {string} query
     */
    collectPivot: function (rows, columns, calculations) {
        var query = 'CollectPivot(';

        query += 'rowGroups=' + JSON.stringify(rows);
        query += ', ';
        query += 'columns=' + JSON.stringify(columns);
        query += ', ';
        query += 'values=' + JSON.stringify(calculations);
        query += ')';

        return query;
    },

    /**
     * @name collectGGPlot
     * @param {*} ggplot ggplot script
     * @desc create ggplot graph
     * @returns {string} query
     */
    collectGGPlot: function (ggplot) {
        var query = 'CollectGGPlot(';

        query += ggplot;
        query += ')';

        return query;
    },

    /**
     * @name collectSeaborn
     * @param {*} seaborn seaborn script
     * @desc create ggplot graph
     * @returns {string} query
     */
    collectSeaborn: function (seaborn) {
        var query = 'CollectSeaborn(';

        query += seaborn;
        query += ')';

        return query;
    },

    /**
     * @name changeGraphLayout
     * @param {string} layout - specified tinker layout
     * @param {string} frame frame name
     * @desc create a change graph layout component to specify graph tinker layout
     * @returns {string} query
     */
    changeGraphLayout: function (layout, frame) {
        var query = '';

        if (frame) {
            query += frame + ' | ';
        }
        query += 'ChangeGraphLayout(graphLayout=["layout.' + layout + '"])';

        return query;
    },

    /**
     * @name clusterGraph
     * @param {string} method - specified tinker cluster method
     * @param {string} frame frame name
     * @desc create a cluster graph component to specify graph tinker cluster
     * @returns {string} query
     */
    clusterGraph: function (method, frame) {
        var query = '';

        if (frame) {
            query += frame + ' | ';
        }
        query = 'ClusterGraph(routine=["' + method + '"])';

        return query;
    },

    /**
     * @name checkRPackages
     * @desc identifies all installed packages (to be compared to required packages for specific widgets)
     * @param {bool} reload - whether or not to reload
     * @returns {string} query
     */
    checkRPackages: function (reload) {
        var query = '';

        query = 'CheckRPackages(';

        if (reload) {
            query += 'reload=["' + reload + '"]';
        }

        query += ')';

        return query;
    },

    /** *Frame */
    /**
     * @name frameHeaders
     * @desc gets the frameHeaders
     * @param {string} frame - passed in frame
     * @param {string} type - which type of headers to get. Ie STRING, NUMBER
     * @returns {string} query
     */
    frameHeaders: function (frame, type) {
        var query = '';
        query = 'FrameHeaders(';
        if (frame) {
            query += ' frame=[' + frame + '] ';
        }

        if (type) {
            query += ', "' + type + '"';
        }

        query += ')';

        return query;
    },

    /**
     * @name frameFilterModelFilteredValues
     * @param {string} title - column to filter on
     * @param {string} search - query to match instances with
     * @param {number} limit - max amount of instances to get back
     * @param {number} offset - where to start getting instances from
     * @desc retrieves values that are filtered out of the frame based on parameters
     * @return {string} the query
     */
    frameFilterModelFilteredValues: function (title, search, limit, offset) {
        var query = '',
            off;

        if (offset === undefined) {
            off = 0;
        } else {
            off = offset;
        }

        query += 'FrameFilterModelFilteredValues(column=["' + title + '"]';

        if (typeof search !== 'undefined' && search) {
            query += ', filterWord=[\"' + search + '\"]';
        }

        query += ', limit=[' + limit + '], offset=[' + off + '])';


        return query;
    },

    /**
     * @name frameFilterModelVisibleValues
     * @param {string} title - column to filter on
     * @param {string} search - query to match instances with
     * @param {number} limit - max amount of instances to get back
     * @param {number} offset - where to start getting instances from
     * @desc retrieves values that are in the frame based on parameters
     * @return {string} the query
     */
    frameFilterModelVisibleValues: function (title, search, limit, offset) {
        var query = '',
            off;

        if (offset === undefined) {
            off = 0;
        } else {
            off = offset;
        }

        query += 'FrameFilterModelVisibleValues(column=["' + title + '"]';

        if (typeof search !== 'undefined' && search) {
            query += ', filterWord=[\"' + search + '\"]';
        }

        query += ', limit=[' + limit + '], offset=[' + off + '])';


        return query;
    },

    /**
     * @name frameFilterModelNumericRange
     * @param {string} title - column to get range for
     * @desc retrieves the min and max of the column
     * @return {string} the query
     */
    frameFilterModelNumericRange: function (title) {
        var query = '';

        query += 'FrameFilterModelNumericRange(column=["' + title + '"])';

        return query;
    },

    /**
     * @name frameFilterModel
     * @param {string} alias - used to select the column to filter on
     * @param {string} search - search term used to filter the model
     * @param {number} limit - used to grab a certain portion of the model
     * @param {number} offset - used to grab a certain amount of the model
     * @desc gets the frame filtermodel
     * @returns {string} query
     */
    frameFilterModel: function (alias, search, limit, offset) {
        var query = '';

        query += 'FrameFilterModel(column=[';
        query += alias;
        query += ']';

        if (typeof search !== 'undefined' && search) {
            query += ', filterWord=[\"' + search + '\"]';
        }

        if (typeof limit !== 'undefined') {
            query += ', limit=[' + limit + ']';
        }

        if (typeof offset !== 'undefined') {
            query += ', offset=[' + offset + ']';
        }

        query += ')';

        return query;
    },

    /**
     * @name getFrameFilterState
     * @param {string} alias - used to select the column to filter on
     * @param {string} search - search term used to filter the model
     * @param {number} limit - used to grab a certain portion of the model
     * @param {number} offset - used to grab a certain amount of the model
     * @param {boolean} dynamic - is the filter dynamic?
     * @param {arrray} panels - panels to run this on
     * @desc gets the frame filtermodel
     * @returns {string} query
     */
    getFrameFilterState: function (alias, search, limit, offset, dynamic, panels) {
        var query = '';

        query += 'GetFrameFilterState(column=[';
        query += alias;
        query += ']';

        if (typeof search !== 'undefined' && search) {
            // escape special characters (quotes and backslash)
            query += ', filterWord=[\"' + search.replace(/"/g, '\\"') + '\"]';
        }

        if (typeof limit !== 'undefined') {
            query += ', limit=[' + limit + ']';
        }

        if (typeof offset !== 'undefined') {
            query += ', offset=[' + offset + ']';
        }

        if (typeof dynamic !== 'undefined') {
            query += ', dynamic=[' + dynamic + ']';
        }

        if (typeof panels !== 'undefined' && Array.isArray(panels)) {
            query += ', panel=' + JSON.stringify(panels);
        }


        query += ')';

        return query;
    },

    /**
     * @name getFrameFilterRange
     * @param {string} alias - used to select the column to filter on
     * @param {boolean} dynamic - is the filter dynamic?
     * @param {arrray} panels - panels to run this on
     * @desc gets the frame filtermodel
     * @returns {string} query
     */
    getFrameFilterRange: function (alias, dynamic, panels) {
        var query = '';

        query += 'GetFrameFilterRange(column=[';
        query += alias;
        query += ']';
        if (typeof dynamic !== 'undefined') {
            query += ', dynamic=[' + dynamic + ']';
        }
        if (typeof panels !== 'undefined' && Array.isArray(panels)) {
            query += ', panel=' + JSON.stringify(panels);
        }
        query += ')';

        return query;
    },

    /**
     * @name addFrameFilter
     * @param {array} filters - filters to construct from
     * @desc adds a filter to the frame
     * @returns {string} query
     */
    addFrameFilter: function (filters) {
        var query = '',
            filterIdx,
            filterLen;

        query += 'AddFrameFilter(';
        for (filterIdx = 0, filterLen = filters.length; filterIdx < filterLen; filterIdx++) {
            if (filters[filterIdx].type === 'value') {
                query += `(${filters[filterIdx].alias} ${filters[filterIdx].comparator} ${JSON.stringify(filters[filterIdx].values)})`;
            } else if (filters[filterIdx].type === 'range') {
                query += `("${filters[filterIdx].values[0]}" <= ${filters[filterIdx].alias}), (${filters[filterIdx].alias} <= "${filters[filterIdx].values[1]}")`;
            } else {
                console.warn('Unrecognized type');
            }

            if (filters[filterIdx].operator) {
                query += ' ' + filters[filterIdx].operator + ' ';
            }
        }
        query += ')';

        return query;
    },

    /**
     * @name removeFrameFilter
     * @param {array} filters - filters to construct from
     * @desc removes a filter to the frame
     * @returns {string} query
     */
    removeFrameFilter: function (filters) {
        var query = '',
            filterIdx,
            filterLen;

        query += 'RemoveFrameFilter(';
        for (filterIdx = 0, filterLen = filters.length; filterIdx < filterLen; filterIdx++) {
            if (filters[filterIdx].type === 'value') {
                query += `(${filters[filterIdx].alias} ${filters[filterIdx].comparator} ${JSON.stringify(filters[filterIdx].values)})`;
            } else if (filters[filterIdx].type === 'range') {
                query += `("${filters[filterIdx].values[0]}" <= ${filters[filterIdx].alias}), (${filters[filterIdx].alias} <= "${filters[filterIdx].values[1]}")`;
            } else {
                console.warn('Unrecognized type');
            }

            if (filters[filterIdx].operator) {
                query += ' ' + filters[filterIdx].operator + ' ';
            }
        }
        query += ')';

        return query;
    },

    /**
     * @name replaceFrameFilter
     * @param {array} filters - filters to construct from
     * @desc replaces a filter to the frame (adds if it doesn't exist)
     * @returns {string} query
     */
    replaceFrameFilter: function (filters) {
        var query = '',
            filterIdx,
            filterLen;

        query += 'ReplaceFrameFilter(';
        for (filterIdx = 0, filterLen = filters.length; filterIdx < filterLen; filterIdx++) {
            if (filters[filterIdx].type === 'value') {
                query += `(${filters[filterIdx].alias} ${filters[filterIdx].comparator} ${JSON.stringify(filters[filterIdx].values)})`;
            } else if (filters[filterIdx].type === 'range') {
                query += `("${filters[filterIdx].values[0]}" <= ${filters[filterIdx].alias}), (${filters[filterIdx].alias} <= "${filters[filterIdx].values[1]}")`;
            } else {
                console.warn('Unrecognized type');
            }

            if (filters[filterIdx].operator) {
                query += ' ' + filters[filterIdx].operator + ' ';
            }
        }
        query += ')';

        return query;
    },

    /**
     * @name deleteFrameFilter
     * @param {array} filters - array of filters to remove
     * @desc removes a filter from the frame
     * @returns {string} query
     */
    deleteFrameFilter: function (filters) {
        var query = '',
            filterIdx,
            filterLen;

        query += 'DeleteFrameFilter ( index';
        query += ' = [';
        for (filterIdx = 0, filterLen = filters.length; filterIdx < filterLen; filterIdx++) {
            query += (String(filters[filterIdx].index) + ', ');
        }
        query = this.trim(query, ',');
        query += '])';

        return query;
    },

    /**
     * @name setFrameFilter
     * @param {array} filters - filters to construct from
     * @desc sets a filter to the frame
     * @returns {string} query
     */
    setFrameFilter: function (filters) {
        var query = '',
            filterIdx,
            filterLen;

        query += 'SetFrameFilter(';
        for (filterIdx = 0, filterLen = filters.length; filterIdx < filterLen; filterIdx++) {
            if (filters[filterIdx].type === 'value') {
                query += `(${filters[filterIdx].alias} ${filters[filterIdx].comparator} ${JSON.stringify(filters[filterIdx].values)})`;
            } else if (filters[filterIdx].type === 'range') {
                query += `("${filters[filterIdx].values[0]}" <= ${filters[filterIdx].alias}), (${filters[filterIdx].alias} <= "${filters[filterIdx].values[1]}")`;
            } else {
                console.warn('Unrecognized type');
            }

            if (filters[filterIdx].operator) {
                query += ' ' + filters[filterIdx].operator + ' ';
            }
        }
        query += ')';

        return query;
    },

    /**
     * @name addPanelFilter
     * @param {array} filters - filters to construct from
     * @desc adds a filter to the frame
     * @returns {string} query
     */
    addPanelFilter: function (filters) {
        var query = '',
            filterIdx,
            filterLen;

        query += 'AddPanelFilter(';
        for (filterIdx = 0, filterLen = filters.length; filterIdx < filterLen; filterIdx++) {
            if (filters[filterIdx].type === 'value') {
                query += `(${filters[filterIdx].alias} ${filters[filterIdx].comparator} ${JSON.stringify(filters[filterIdx].values)})`;
            } else if (filters[filterIdx].type === 'range') {
                query += `("${filters[filterIdx].values[0]}" <= ${filters[filterIdx].alias}), (${filters[filterIdx].alias} <= "${filters[filterIdx].values[1]}")`;
            } else {
                console.warn('Unrecognized type');
            }

            if (filters[filterIdx].operator) {
                query += ' ' + filters[filterIdx].operator + ' ';
            }
        }
        query += ')';

        return query;
    },

    /**
     * @name removePanelFilter
     * @param {array} filters - filters to construct from
     * @desc removes a filter to the frame
     * @returns {string} query
     */
    removePanelFilter: function (filters) {
        var query = '',
            filterIdx,
            filterLen;

        query += 'RemovePanelFilter(';
        for (filterIdx = 0, filterLen = filters.length; filterIdx < filterLen; filterIdx++) {
            if (filters[filterIdx].type === 'value') {
                query += `(${filters[filterIdx].alias} ${filters[filterIdx].comparator} ${JSON.stringify(filters[filterIdx].values)})`;
            } else if (filters[filterIdx].type === 'range') {
                query += `("${filters[filterIdx].values[0]}" <= ${filters[filterIdx].alias}), (${filters[filterIdx].alias} <= "${filters[filterIdx].values[1]}")`;
            } else {
                console.warn('Unrecognized type');
            }

            if (filters[filterIdx].operator) {
                query += ' ' + filters[filterIdx].operator + ' ';
            }
        }
        query += ')';

        return query;
    },


    /**
     * @name setPanelFilter
     * @param {array} filters - filters to construct from
     * @desc sets a filter to the frame
     * @returns {string} query
     */
    setPanelFilter: function (filters) {
        var query = '',
            filterIdx,
            filterLen;

        query += 'SetPanelFilter(';
        for (filterIdx = 0, filterLen = filters.length; filterIdx < filterLen; filterIdx++) {
            if (filters[filterIdx].type === 'value') {
                query += `(${filters[filterIdx].alias} ${filters[filterIdx].comparator} ${JSON.stringify(filters[filterIdx].values)})`;
            } else if (filters[filterIdx].type === 'range') {
                query += `("${filters[filterIdx].values[0]}" <= ${filters[filterIdx].alias}), (${filters[filterIdx].alias} <= "${filters[filterIdx].values[1]}")`;
            } else {
                console.warn('Unrecognized type');
            }

            if (filters[filterIdx].operator) {
                query += ' ' + filters[filterIdx].operator + ' ';
            }
        }
        query += ')';

        return query;
    },

    /**
     * @name unfilterFrame
     * @param {string} alias - used to select the column to filter on
     * @desc unfilters the frame
     * @returns {string} query
     */
    unfilterFrame: function (alias) {
        var query = '';

        query += 'UnfilterFrame(';
        if (alias) {
            query += alias;
        }
        query += ')';

        return query;
    },

    /**
     * @name unfilterPanel
     * @param {string} alias - used to select the column to filter on
     * @desc unfilters the panel
     * @returns {string} query
     */
    unfilterPanel: function (alias) {
        var query = '';

        query += 'UnfilterPanel(';
        if (alias) {
            query += alias;
        }
        query += ')';

        return query;
    },

    /**
     * @name frameHeaderExists
     * @param {string} frame the id of the frame
     * @param {string} header the name of the header to check
     * @desc check to see if column exists in frame
     * @returns {string} query
     */
    frameHeaderExists: function (frame, header) {
        var query = '';

        if (frame) {
            query += 'FrameHeaderExists(frame=[' + frame + '], column=[' + header + '])';
        } else {
            query += 'FrameHeaderExists(' + header + ')';
        }

        return query;
    },

    /** *R */
    /**
     * @name synchronizeGridToRDataTable
     * @param {string} name - frame name
     * @desc unfilters the frame
     * @returns {string} query
     */
    synchronizeGridToRDataTable: function (name) {
        var query = '';

        query += '<j><encode>synchronizeGridToRDataTable(\"';
        query += name;
        query += '\")</encode></j>';

        return query;
    },

    /** *PANEL */

    /**
     * @name addPanel
     * @param {string} panelId - panel id to create
     * @param {string} sheetId - sheet to add the panel to
     * @desc creates a new panel
     * @returns {string} query
     */
    addPanel: function (panelId, sheetId) {
        let query = '';

        query = `AddPanel(panel=[${panelId}] ${sheetId ? `, sheet=["${sheetId}"]` : ''})`;

        return query;
    },

    /**
     * @name addPanelIfAbsent
     * @param {string} panelId - panel id to create
     * @param {string} sheetId - sheet to add the panel to
     * @desc creates a new panel if it does not already exist
     * @returns {string} query
     */
    addPanelIfAbsent: function (panelId, sheetId) {
        let query = '';

        query = `AddPanelIfAbsent(panel=[${panelId}] ${sheetId ? `, sheet=["${sheetId}"]` : ''})`;

        return query;
    },

    /**
     * @name clonePanel
     * @param {number} id - new panel id to clone to
     * @param {number} cloneId - If specified, this will be the new panel id for the cloned panel
     * @param {number} sheetId - If specified, this will be the sheet where the panel is cloned to
     * @desc creates and clones a new panel
     * @returns {string} query
     */
    clonePanel: function (id, cloneId, sheetId) {
        var query = '';
        if (cloneId) {
            query = `Clone(panel=["${id}"], cloneId=["${cloneId}"] ${sheetId ? `, sheet=["${sheetId}"]` : ''})`;
        } else {
            query = 'Clone("' + id + '")';
        }
        return query;
    },


    /**
     * @name closePanel
     * @param {number} id - panel id to close
     * @desc closes a panel
     * @returns {string} query
     */
    closePanel: function (id) {
        var query = '';

        query = 'ClosePanel(' + id + ')';

        return query;
    },

    /**
     * @name panel
     * @param {number} id - panel id to create
     * @desc selects a panel
     * @returns {string} query
     */
    panel: function (id) {
        var query = '';

        query = 'Panel(' + id + ')';

        return query;
    },

    /**
     * @name setPanelView
     * @param {string} view - new view to set the panel to
     * @param {object} options - view options
     * @desc sets the panel view
     * @returns {string} query
     */
    setPanelView: function (view, options) {
        var query = '';

        query += 'SetPanelView(\"' + view + '\"';

        if (options) {
            query += ', \"<encode>' + JSON.stringify(options) + '</encode>\"';
        }

        query += ')';

        return query;
    },

    /**
     * @name setPanelLabel
     * @param {string} label - new view to set the panel to
     * @desc sets the panel label
     * @returns {string} query
     */
    setPanelLabel: function (label) {
        var query = '';

        query += 'SetPanelLabel(\"' + label + '\")';

        return query;
    },
    /**
     * @name addPanelOrnaments
     * @param {object} ornaments - ornaments to add to the panel
     * @desc adds and saves ornaments to the panel
     * @returns {string} query
     */
    addPanelOrnaments: function (ornaments) {
        var query = '';

        query = 'AddPanelOrnaments(' + JSON.stringify(ornaments) + ')';

        return query;
    },

    /**
     * @name addPanelConfig
     * @param {object} config - panel configuration
     * @return {string} query
     */
    addPanelConfig: function (config) {
        var query = '';

        query += 'AddPanelConfig(config=[' + JSON.stringify(config) + '])';

        return query;
    },

    /**
     * @name setPanelPosition
     * @param {object} position - panel position values
     * @return {string} query
     */
    setPanelPosition: function (position) {
        var query = '',
            positionArr = [];

        for (let prop in position) {
            if (position.hasOwnProperty(prop)) {
                let value = position[prop];
                if (typeof value === 'string') {
                    value = `"${value}"`;
                }
                positionArr.push(`"${prop}": ${value}`);
            }
        }

        query += 'SetPanelPosition({';
        query += positionArr.join(',');
        query += '})';

        return query;
    },

    /**
     * @name removePanelOrnaments
     * @param {string} accessor - string to get to the object. In the form of 'a.b.c'
     * @desc removes ornaments from the panel
     * @returns {string} query
     */
    removePanelOrnaments: function (accessor) {
        var query = '';

        query = 'RemovePanelOrnaments(\"' + accessor + '\")';

        return query;
    },

    /**
     * @name retrievePanelOrnaments
     * @param {string} accessor - string to get to the object. In the form of 'a.b.c'
     * @desc retrieves saved ornaments from the panel
     * @returns {string} query
     */
    retrievePanelOrnaments: function (accessor) {
        var query = '';

        query = 'RetrievePanelOrnaments(\"' + accessor + '\")';

        return query;
    },

    /**
     * @name addPanelColorByValue
     * @param {string} variable -  variable to add
     * @param {object} options - optional colorby value options
     * @desc adds and saves colorbyvalue to the panel
     * @returns {string} query
     */
    addPanelColorByValue: function (variable, options) {
        var query = '';

        query += 'AddPanelColorByValue(';
        query += 'name=["' + variable + '"], ';
        query += 'qs=[' + variable + '], ';
        query += 'options=[' + JSON.stringify(options) + ']';
        query += ')';

        return query;
    },

    /**
     * @name retrievePanelColorByValue
     * @param {string} variable -  variable to remove
     * @desc adds and saves colorbyvalue to the panel
     * @returns {string} query
     */
    retrievePanelColorByValue: function (variable) {
        var query = '';

        query += 'RetrievePanelColorByValue(';
        query += 'name=["' + variable + '"]';
        query += ')';

        return query;
    },

    /**
     * @name removePanelColorByValue
     * @param {string} variable -  variable to remove
     * @desc adds and saves colorbyvalue to the panel
     * @returns {string} query
     */
    removePanelColorByValue: function (variable) {
        var query = '';

        query += 'RemovePanelColorByValue(';
        query += 'name=["' + variable + '"]';
        query += ')';

        return query;
    },

    /**
     * @name setPanelSort
     * @param {array} sortOptions the sort options array
     * @desc builds the setpanelsort query
     * @returns {string} query
     */
    setPanelSort: function (sortOptions) {
        var query = '',
            i,
            len = sortOptions.length,
            cols = [],
            dirs = [];

        for (i = 0; i < len; i++) {
            cols.push(sortOptions[i].alias);
            dirs.push(sortOptions[i].dir);
        }

        if (cols.length > 0 && cols.length === dirs.length) {
            query = 'SetPanelSort(columns=' + JSON.stringify(cols) + ', sort=' + JSON.stringify(dirs) + ')';
        }
        return query;
    },

    /**
     * @name unsortPanel
     * @desc removes sorts in the panel
     * @returns {string} query
     */
    unsortPanel: function () {
        var query = '';

        query += 'UnsortPanel()';

        return query;
    },

    /**
     * @name addPanelComment
     * @param {object} comment - comment object to add to the panel
     * @desc adds comment to panel
     * @returns {string} query
     */
    addPanelComment: function (comment) {
        var query = '';
        comment.commentText = encodeURIComponent(comment.commentText);
        query = 'AddPanelComment(' + JSON.stringify(comment) + ')';

        return query;
    },

    /**
     * @name removePanelComment
     * @param {object} commentId - comment id to remove
     * @desc removes panel comment by id
     * @returns {string} query
     */
    removePanelComment: function (commentId) {
        var query = '';

        query = 'RemovePanelComment(' + JSON.stringify(commentId) + ')';

        return query;
    },

    /**
     * @name RetrievePanelComment
     * @desc retrieves saved comments for the panel
     * @returns {string} query
     */
    retrievePanelComments: function () {
        var query = '';

        query = 'RetrievePanelComment()';

        return query;
    },

    /**
     * @name addPanelEvents
     * @param {object} event - events to add to the panel
     * @desc adds and saves ornaments to the panel
     * @returns {string} query
     */
    addPanelEvents: function (event) {
        var query = '',
            queryObj = {},
            action, eventName, eventIdx, uniqueName, queryId;

        // we're going to pull out all of the queries in events we're adding...
        // because we don't want to stringify the queries themselves...
        // that causes issues with double escapes if the query is doing (for example) set panel view of a default-handle
        for (action in event) {
            if (event.hasOwnProperty(action)) {
                for (eventName in event[action]) {
                    if (event[action].hasOwnProperty(eventName)) {
                        for (eventIdx = 0; eventIdx < event[action][eventName].length; eventIdx++) {
                            uniqueName = action + eventName + eventIdx;
                            queryObj['<' + uniqueName + '>'] = event[action][eventName][eventIdx].query;
                            event[action][eventName][eventIdx].query = '<' + uniqueName + '>';
                        }
                    }
                }
            }
        }

        query = 'AddPanelEvents(' + JSON.stringify(event) + ')';

        for (queryId in queryObj) {
            if (queryObj.hasOwnProperty(queryId)) {
                query = query.replace(queryId, queryObj[queryId]);
            }
        }

        return query;
    },

    /**
     * @name removePanelEvents
     * @param {object} event - event to remove from the panel
     * @desc removes the specified event panel
     * @returns {string} query
     */
    removePanelEvents: function (event) {
        var query = '';

        query = 'RemovePanelEvents(\"' + event + '\")';

        return query;
    },

    /**
     * @name retrievePanelOrnaments
     * @param {string} accessor - string to get to the object. In the form of 'a.b.c'
     * @desc retrieves saved ornaments from the panel
     * @returns {string} query
     */
    retrievePanelEvents: function (accessor) {
        var query = '';

        query = 'RetrievePanelEvents(';

        if (accessor) {
            query += '\"' + accessor + '\"';
        }

        query += ')';

        return query;
    },

    /**
     * @name with
     * @param {object} panelId - panelId
     * @desc creates with panel id query
     * @returns {string} query
     */
    with: function (panelId) {
        var query = '';

        query = 'With(Panel(' + panelId + '))';

        return query;
    },

    /**
     * @name hasDuplicates
     * @param {array} fields - fields to check duplicates on
     * @desc check whether duplicates exist for the parameters
     * @returns {string} query
     */
    hasDuplicates: function (fields) {
        var query = '',
            i;
        query += 'HasDuplicates(';
        for (i = 0; i < fields.length; i++) {
            query += fields[i] + ',';
        }
        query = this.trim(query, ',');
        query += ')';

        return query;
    },

    /**
     * @name group
     * @desc specifies a grouping for a math operation
     * @param {string} group the field to perform the grouping on
     * @returns {string} query
     */
    group: function (group) {
        var query = '',
            i,
            len = group.length;

        if (group.length > 0) {
            for (i = 0; i < len; i++) {
                query += group[i] + ',';
            }

            query = this.trim(query, ',');
            query = 'Group(' + query + ')';
        }

        return query;
    },

    /**
     * @name sort
     * @param {array} cols the columns to sort
     * @param {array} directions the directions of the columns
     * @returns {string} query
     */
    sort: function (cols, directions) {
        var query = '';

        query = 'Sort(columns=' + JSON.stringify(cols);

        if (directions) {
            query += ', sort=' + JSON.stringify(directions);
        }

        query += ')';
        return query;
    },

    /**
     * @name saveInsight
     * @param {string} app - app to save to
     * @param {string} insightName - new question name
     * @param {boolean} hidden - hide the insight?
     * @param {array} steps - steps to save
     * @param {string} layout - layout to save as
     * @param {string} image - image of the saved insight
     * @param {array} params - the parameter you want to set for this insight
     * @param {string} appVar - the variable containing the app name
     * @param {object} pipeline - the pipeline config
     * @param {array} tags - list of tags
     * @param {string} description - the description for this insight
     * @desc constructs a save pixel
     * @returns {string} query
     */
    saveInsight: function (app, insightName, hidden, steps, layout, image, params, appVar, pipeline, tags, description) {
        var query = '',
            i,
            len, canvasImg;

        query += 'SaveInsight(';
        if (app) {
            query += 'app=[\"' + app + '\"], ';
        } else if (appVar) {
            query += 'app=[' + appVar + '], ';
        }
        query += 'insightName=[\"' + insightName + '\"], ';
        query += 'hidden=[' + hidden + '], ';
        query += 'recipe=[';

        for (i = 0, len = steps.length; i < len; i++) {
            query += ('\"<sEncode>' + steps[i].expression + '</sEncode>\"' + ', ');
        }

        query = this.trim(query, ',');
        query += '], ';

        if (image) {
            query += 'layout=[\"' + layout + '\"], ';
            canvasImg = image.substr(22);
            query += 'image=[\"' + canvasImg + '\"]';
        } else {
            query += 'layout=[\"' + layout + '\"]';
        }

        if (params && params.length > 0) {
            query += ', params=[' + params + ']';
        }

        if (pipeline) {
            query = this.trim(query, ',');
            query += ', pipeline=[' + JSON.stringify(pipeline) + ']';
        }

        if (tags && tags.length > 0) {
            query += ', tags=' + JSON.stringify(tags);
        }

        if (description) {
            query += ', description=["' + description + '"]';
        }

        query += ')';

        return query;
    },
    /**
     * @name updateInsight
     * @param {string} app - app to save to
     * @param {string} insightName - new question name
     * @param {boolean} hidden - hide the insight?
     * @param {array} steps - steps to save
     * @param {string} layout - layout to save as
     * @param {string} image - image of the saved insight
     * @param {string} rdbmsId - id of the insisght db
     * @param {array} params - the parameters to set for this insight
     * @param {string} appVar - the variable containing the app name
     * @param {object} pipeline - the pipeline config
     * @param {array} tags - list of tags
     * @param {string} description - the description for this insight
     * @desc constructs an update insight pixel
     * @returns {string} query
     */
    updateInsight: function (app, insightName, hidden, steps, layout, image, rdbmsId, params, appVar, pipeline, tags, description) {
        var query = '',
            i,
            len, canvasImg;

        query += 'UpdateInsight(';

        if (app) {
            query += 'app=[\"' + app + '\"], ';
        } else if (appVar) {
            query += 'app=[' + appVar + '], ';
        }

        query += 'insightName=[\"' + insightName + '\"], ';
        query += 'hidden=[' + hidden + '], ';
        query += 'recipe=[';

        for (i = 0, len = steps.length; i < len; i++) {
            query += ('\"<sEncode>' + steps[i].expression + '</sEncode>\"' + ', ');
        }

        query = this.trim(query, ',');

        query += '], ';
        query += 'layout=[\"' + layout + '\"], ';
        if (image) {
            canvasImg = image.substr(22);
            query += 'image=[\"' + canvasImg + '\"], ';
        }
        query += 'id=["' + rdbmsId + '"] ';

        if (params.length > 0) {
            query += ', params=[' + params + ']';
        }

        if (pipeline) {
            query += ', pipeline=[' + JSON.stringify(pipeline) + ']';
        }

        if (tags && tags.length > 0) {
            query += ', tags=' + JSON.stringify(tags);
        }

        if (description) {
            query += ', description=["' + description + '"]';
        }

        query += ')';

        return query;
    },
    savePipeline: function (app, insightName, pipeline) {
        var query = '';

        query += 'SaveInsight(';

        query += 'app=[\"' + app + '\"], ';
        query += 'insightName=[\"' + insightName + '\"], ';
        query += 'pipeline=[' + JSON.stringify(pipeline) + ']';

        query += ')';

        return query;
    },
    /**
     * @name dashboardInsightConfig
     * @param {array} insightArray - array of insightInfo and insightId in the current dashbaord
     * @param {object} currentLayout - dashboard layout
     * @param {string} selectedLayout - dashboard selectedLayout
     * @desc constructs a dashboardInsightConfig pixel
     * @returns {string} query
     */
    dashboardInsightConfig: function (insightArray, currentLayout, selectedLayout) {
        var query = '',
            i,
            len = insightArray.length;

        query += 'DashboardInsightConfig(';
        query += 'insights=[';
        for (i = 0; i < len; i++) {
            query += ('\"' + insightArray[i].app_id + '__' + insightArray[i].app_insight_id + '\"' + ', ');
        }
        query = this.trim(query, ',');
        query += '], ';
        query += 'oldIds=[';
        for (i = 0; i < len; i++) {
            query += ('\"' + insightArray[i].insightID + '\"' + ', ');
        }
        query = this.trim(query, ',');
        query += '], ';
        query += 'layout=[\"<encode>';
        query += JSON.stringify({
            currentLayout: currentLayout,
            selectedLayout: selectedLayout
        });
        query += '</encode>\"';
        query += '] ';
        query += ')';

        return query;
    },

    /* Meta Pixel*/
    /**
     * @name getDatabaseMetamodel
     * @param {string} database - name of the database to get metamodel for
     * @param {array} options - array of options
     * @returns {string} query
     */
    getDatabaseMetamodel: function (database, options) {
        var query = '';

        query += 'GetDatabaseMetamodel(';
        query += ' database=[\"' + database + '\"]';
        if (options && options.length > 0) {
            query += ', options=' + JSON.stringify(options) + '';
        }
        query += ')';

        return query;
    },

    /**
     * @name getDatabaseTableStructure
     * @param {string} database - name of the database to get table info for
     * @returns {string} query
     */
    getDatabaseTableStructure: function (database) {
        var query = '';

        query = 'GetDatabaseTableStructure(database=[\"' + database + '\"])';

        return query;
    },

    /**
     * @name predictDataTypes
     * @param {string} path - path to the file to upload
     * @param {string} delimiter - delimiter to use for the file
     * @param {boolean} rowCount - rowCount
     * @desc predict the datatypes based on a file
     * @returns {string} the query
     */
    predictDataTypes: function (path, delimiter, rowCount) {
        var query = '';

        query += 'PredictDataTypes(';
        query += 'filePath=["' + path + '"], ';
        query += 'delimiter=["' + delimiter + '"], ';
        query += 'rowCount=[' + rowCount + ']);';
        return query;
    },

    /**
     * @name predictExcelDataTypes
     * @param {string} path - path to the file to upload
     * @desc predict the datatypes based on a file
     * @returns {string} the query
     */
    predictExcelDataTypes: function (path) {
        var query = '';

        query += 'PredictExcelDataTypes(';
        query += 'filePath=["' + path + '"]);';
        return query;
    },

    /**
     * @name predictMetamodel
     * @param {string} path - path to the file to upload
     * @param {string} delimiter - delimiter to use for the file
     * @param {boolean} rowCount - rowCount
     * @desc predict the metamodel based on a file
     * @returns {string} the query
     */
    predictMetamodel: function (path, delimiter, rowCount) {
        var query = '';

        query += 'PredictMetamodel(';
        query += 'filePath=["' + path + '"], ';
        query += 'delimiter=["' + delimiter + '"], ';
        query += 'rowCount=[' + rowCount + ']);';
        return query;
    },

    /**
     * @name parseMetamodel
     * @param {string} path - path to the file to upload
     * @param {string} delimiter - delimiter to use for the file
     * @param {boolean} rowCount - rowCount
     * @param {string} prop - path to the prop file
     * @desc predict the metamodel basd on a csv
     * @returns {string} the query
     */
    parseMetamodel: function (path, delimiter, rowCount, prop) {
        var query = '';

        query += 'ParseMetamodel(';
        query += 'filePath=["' + path + '"], ';
        query += 'delimiter=["' + delimiter + '"], ';
        query += 'rowCount=[' + rowCount + '], ';
        query += 'propFile=["' + prop + '"]);';
        return query;
    },

    /**
     * @name rdbmsUploadTableData
     * @param {string} app - new app name
     * @param {string} path - path to the file to upload
     * @param {string} delimiter - delimiter to use for the file
     * @param {object} dataTypeMap - map of the dataTypes
     * @param {object} newHeaders - map of the new headers
     * @param {object} additionalDataTypes - map of the additional types
     * @param {object} descriptionMap - map of descriptions
     * @param {object} logicalNamesMap - map of logical names
     * @param {boolean} existing - add to existing?
     * @param {string} appVar - the variable containing the app name
     * @desc upload a CSV or TSV as a new database
     * @returns {string} the query
     */
    rdbmsUploadTableData: function (app, path, delimiter, dataTypeMap, newHeaders, additionalDataTypes, descriptionMap, logicalNamesMap, existing, appVar) {
        var query = '';

        query += 'RdbmsUploadTableData(';
        if (appVar) {
            query += 'app=[' + appVar + '], ';
        } else {
            query += 'app=["' + app + '"], ';
        }
        query += 'filePath=["' + path + '"], ';
        query += 'delimiter=["' + delimiter + '"], ';
        query += 'dataTypeMap=[' + JSON.stringify(dataTypeMap) + '], ';
        query += 'newHeaders=[' + JSON.stringify(newHeaders) + '], ';
        query += 'additionalDataTypes=[' + JSON.stringify(additionalDataTypes) + '], ';
        query += 'descriptionMap=[' + JSON.stringify(descriptionMap) + '], ';
        query += 'logicalNamesMap=[' + JSON.stringify(logicalNamesMap) + '], ';
        query += 'existing=[' + existing + ']);';

        return query;
    },

    /**
     * @name rdbmsUploadExcelData
     * @param {string} app - new app name
     * @param {string} path - path to the file to upload
     * @param {object} dataTypeMap - map of the dataTypes
     * @param {object} newHeaders - map of the new headers
     * @param {object} additionalDataTypes - map of the additional types
     * @param {object} descriptionMap - map of descriptions
     * @param {object} logicalNamesMap - map of logical names
     * @param {boolean} existing - add to existing?
     * @param {string} appVar - the variable containing the app name
     * @desc upload a CSV or TSV as a new database
     * @returns {string} the query
     */
    rdbmsUploadExcelData: function (app, path, dataTypeMap, newHeaders, additionalDataTypes, descriptionMap, logicalNamesMap, existing, appVar) {
        var query = '';

        query += 'RdbmsUploadExcelData(';
        if (appVar) {
            query += 'app=[' + appVar + '], ';
        } else {
            query += 'app=["' + app + '"], ';
        }
        query += 'filePath=["' + path + '"], ';
        query += 'dataTypeMap=[' + JSON.stringify(dataTypeMap) + '], ';
        query += 'newHeaders=[' + JSON.stringify(newHeaders) + '], ';
        query += 'additionalDataTypes=[' + JSON.stringify(additionalDataTypes) + '], ';
        query += 'descriptionMap=[' + JSON.stringify(descriptionMap) + '], ';
        query += 'logicalNamesMap=[' + JSON.stringify(logicalNamesMap) + '], ';
        query += 'existing=[' + existing + ']);';

        return query;
    },

    /**
     * @name rdbmsCsvUpload
     * @param {string} app - new app name
     * @param {string} path - path to the file to upload
     * @param {string} delimiter - delimiter
     * @param {string} propFile - path to propfile
     * @param {object} metamodel - map of the metamodel
     * @param {object} dataTypeMap - map of the dataTypes
     * @param {object} newHeaders - map of the new headers
     * @param {object} additionalDataTypes - map of the additional types
     * @param {object} descriptionMap - map of descriptions
     * @param {object} logicalNamesMap - map of logical names
     * @param {boolean} existing - add to existing?
     * @param {string} appVar - the variable containing the app name
     * @desc upload a CSV or TSV as a new database
     * @returns {string} the query
     */
    rdbmsCsvUpload: function (app, path, delimiter, propFile, metamodel, dataTypeMap, newHeaders, additionalDataTypes, descriptionMap, logicalNamesMap, existing, appVar) {
        var query = '';

        query += 'RdbmsCsvUpload(';
        if (appVar) {
            query += 'app=[' + appVar + '], ';
        } else {
            query += 'app=["' + app + '"], ';
        }
        query += 'filePath=["' + path + '"], ';
        query += 'delimiter=["' + delimiter + '"], ';
        if (propFile) {
            query += 'propFile=["' + propFile + '"], ';
        }
        query += 'metamodel=[' + JSON.stringify(metamodel) + '], ';
        query += 'dataTypeMap=[' + JSON.stringify(dataTypeMap) + '], ';
        query += 'newHeaders=[' + JSON.stringify(newHeaders) + '], ';
        query += 'additionalDataTypes=[' + JSON.stringify(additionalDataTypes) + '], ';
        query += 'descriptionMap=[' + JSON.stringify(descriptionMap) + '], ';
        query += 'logicalNamesMap=[' + JSON.stringify(logicalNamesMap) + '], ';
        query += 'existing=[' + existing + ']);';

        return query;
    },

    /**
     * @name rdfCsvUpload
     * @param {string} app - new app name
     * @param {string} path - path to the file to upload
     * @param {string} delimiter - delimiter
     * @param {string} propFile - path to propfile
     * @param {object} metamodel - map of the metamodel
     * @param {object} dataTypeMap - map of the dataTypes
     * @param {object} newHeaders - map of the new headers
     * @param {object} additionalDataTypes - map of the additional types
     * @param {object} descriptionMap - map of descriptions
     * @param {object} logicalNamesMap - map of logical names
     * @param {string} uri - custom base uri?
     * @param {boolean} existing - add to existing?
     * @param {string} appVar - the variable containing the app name
     * @desc upload a CSV or TSV as a new database
     * @returns {string} the query
     */
    rdfCsvUpload: function (app, path, delimiter, propFile, metamodel, dataTypeMap, newHeaders, additionalDataTypes, descriptionMap, logicalNamesMap, uri, existing, appVar) {
        var query = '';

        query += 'RdfCsvUpload(';
        if (appVar) {
            query += 'app=[' + appVar + '], ';
        } else {
            query += 'app=["' + app + '"], ';
        }
        query += 'filePath=["' + path + '"], ';
        query += 'delimiter=["' + delimiter + '"], ';
        if (propFile) {
            query += 'propFile=["' + propFile + '"], ';
        }
        query += 'metamodel=[' + JSON.stringify(metamodel) + '], ';
        query += 'dataTypeMap=[' + JSON.stringify(dataTypeMap) + '], ';
        query += 'newHeaders=[' + JSON.stringify(newHeaders) + '], ';
        query += 'additionalDataTypes=[' + JSON.stringify(additionalDataTypes) + '], ';
        query += 'descriptionMap=[' + JSON.stringify(descriptionMap) + '], ';
        query += 'logicalNamesMap=[' + JSON.stringify(logicalNamesMap) + '], ';
        query += 'customBaseURI=["' + uri + '"], ';
        query += 'existing=[' + existing + ']);';

        return query;
    },
    /**
     * @name tinkerCsvUpload
     * @param {string} app - new app name
     * @param {string} path - path to the file to upload
     * @param {string} delimiter - delimiter
     * @param {string} propFile - path to propfile
     * @param {object} metamodel - map of the metamodel
     * @param {object} dataTypeMap - map of the dataTypes
     * @param {object} newHeaders - map of the new headers
     * @param {object} additionalDataTypes - map of the additional types
     * @param {object} descriptionMap - map of descriptions
     * @param {object} logicalNamesMap - map of logical names
     * @param {string} tinkerDriver - type of tinker driver to create?
     * @param {boolean} existing - add to existing?
     * @param {string} appVar - the variable containing the app name
     * @desc upload a CSV or TSV as a new database
     * @returns {string} the query
     */
    tinkerCsvUpload: function (app, path, delimiter, propFile, metamodel, dataTypeMap, newHeaders, additionalDataTypes, descriptionMap, logicalNamesMap, tinkerDriver, existing, appVar) {
        var query = '';

        query += 'TinkerCsvUpload(';
        if (appVar) {
            query += 'app=[' + appVar + '], ';
        } else {
            query += 'app=["' + app + '"], ';
        }
        query += 'filePath=["' + path + '"], ';
        query += 'delimiter=["' + delimiter + '"], ';
        if (propFile) {
            query += 'propFile=["' + propFile + '"], ';
        }
        query += 'metamodel=[' + JSON.stringify(metamodel) + '], ';
        query += 'dataTypeMap=[' + JSON.stringify(dataTypeMap) + '], ';
        query += 'newHeaders=[' + JSON.stringify(newHeaders) + '], ';
        query += 'additionalDataTypes=[' + JSON.stringify(additionalDataTypes) + '], ';
        query += 'descriptionMap=[' + JSON.stringify(descriptionMap) + '], ';
        query += 'logicalNamesMap=[' + JSON.stringify(logicalNamesMap) + '], ';
        query += 'tinkerDriver=["' + tinkerDriver + '"], ';
        query += 'existing=[' + existing + ']);';

        return query;
    },

    /**
     * @name rdfLoaderSheetUpload
     * @param {string} app - new app name
     * @param {string} path - path to the file to upload
     * @param {string} uri - custom base uri?
     * @param {boolean} existing - add to existing?
     * @param {string} appVar - the variable containing the app name
     * @desc upload a loadersheet as a new database
     * @returns {string} the query
     */
    rdfLoaderSheetUpload: function (app, path, uri, existing, appVar) {
        var query = '';

        query += 'RdfLoaderSheetUpload(';
        if (appVar) {
            query += 'app=[' + appVar + '], ';
        } else {
            query += 'app=["' + app + '"], ';
        }
        query += 'filePath=["' + path + '"], ';
        query += 'customBaseURI=["' + uri + '"], ';
        query += 'existing=[' + existing + ']);';

        return query;
    },

    /**
     * @name rdbmsLoaderSheetUpload
     * @param {string} app - new app name
     * @param {string} path - path to the file to upload
     * @param {boolean} existing - add to existing?
     * @param {string} appVar - the variable containing the app name
     * @desc upload a loadersheet as a new database
     * @returns {string} the query
     */
    rdbmsLoaderSheetUpload: function (app, path, existing, appVar) {
        var query = '';

        query += 'RdbmsLoaderSheetUpload(';
        if (appVar) {
            query += 'app=[' + appVar + '], ';
        } else {
            query += 'app=["' + app + '"], ';
        }
        query += 'filePath=["' + path + '"], ';
        query += 'existing=[' + existing + ']);';

        return query;
    },

    /**
     * @name rdbmsReplaceDatabaseCsvUpload
     * @param {string} app the app to replace
     * @param {string} filePath the filePath to grab file
     * @param {string} delimiter the delimiter to use for csv file
     * @param {object} propFile the property file to use
     * @param {object} metamodel the metamodel for this database
     * @param {object} dataTypeMap the mapping from column to type
     * @param {object} newHeaders new headers to add
     * @param {object} additionalDataTypes additional information on data type such as date format
     * @param {object} descriptionMap the description mapping for columns
     * @param {object} logicalNamesMap the logical name mapping
     * @desc replace database for a rdbms db via csv
     * @returns {string} pixel query
     */
    rdbmsReplaceDatabaseCsvUpload: function (app, filePath, delimiter, propFile, metamodel, dataTypeMap, newHeaders, additionalDataTypes, descriptionMap, logicalNamesMap) {
        var query = '';

        query += 'rdbmsReplaceDatabaseCsvUpload(';

        query += 'app=["' + app + '"], ';
        query += 'filePath=["' + filePath + '"], ';
        query += 'delimiter=["' + delimiter + '"], ';
        if (propFile) {
            query += 'propFile=["' + propFile + '"], ';
        }
        query += 'metamodel=[' + JSON.stringify(metamodel) + '], ';
        query += 'dataTypeMap=[' + JSON.stringify(dataTypeMap) + '], ';
        query += 'newHeaders=[' + JSON.stringify(newHeaders) + '], ';
        query += 'additionalDataTypes=[' + JSON.stringify(additionalDataTypes) + '], ';
        query += 'descriptionMap=[' + JSON.stringify(descriptionMap) + '], ';
        query += 'logicalNamesMap=[' + JSON.stringify(logicalNamesMap) + ']';

        query += ')';

        return query;
    },

    /**
     * @name rdbmsReplaceDatabaseExcelUpload
     * @param {string} app the app to replace
     * @param {string} filePath the filePath to grab file
     * @param {object} dataTypeMap the mapping from column to type
     * @param {object} newHeaders new headers to add
     * @param {object} additionalDataTypes additional information on data type such as date format
     * @param {object} descriptionMap the description mapping for columns
     * @param {object} logicalNamesMap the logical name mapping
     * @desc replace database for a rdbms db via csv
     * @returns {string} pixel query
     */
    rdbmsReplaceDatabaseExcelUpload: function (app, filePath, dataTypeMap, newHeaders, additionalDataTypes, descriptionMap, logicalNamesMap) {
        var query = '';

        query += 'RdbmsReplaceDatabaseExcelUpload(';

        query += 'app=["' + app + '"], ';
        query += 'filePath=["' + filePath + '"], ';
        query += 'dataTypeMap=[' + JSON.stringify(dataTypeMap) + '], ';
        query += 'newHeaders=[' + JSON.stringify(newHeaders) + '], ';
        query += 'additionalDataTypes=[' + JSON.stringify(additionalDataTypes) + '], ';
        query += 'descriptionMap=[' + JSON.stringify(descriptionMap) + '], ';
        query += 'logicalNamesMap=[' + JSON.stringify(logicalNamesMap) + ']';

        query += ')';

        return query;
    },

    /**
     * @name rdbmsReplaceDatabaseLoaderSheetUpload
     * @param {string} app - new app name
     * @param {string} filePath - path to the file to upload
     * @param {boolean} existing - add to existing?
     * @param {string} appVar - the variable containing the app name
     * @desc upload a loadersheet as a new database
     * @returns {string} the query
     */
    rdbmsReplaceDatabaseLoaderSheetUpload: function (app, filePath) {
        var query = '';

        query += 'RdbmsReplaceDatabaseLoaderSheetUpload(';
        query += 'app=["' + app + '"], ';
        query += 'filePath=["' + filePath + '"]';

        query += ')';

        return query;
    },

    /**
     * @name rdbmsReplaceDatabaseUploadTable
     * @param {string} app - new app name
     * @param {string} path - path to the file to upload
     * @param {string} delimiter - delimiter to use for the file
     * @param {object} dataTypeMap - map of the dataTypes
     * @param {object} newHeaders - map of the new headers
     * @param {object} additionalDataTypes - map of the additional types
     * @param {object} descriptionMap - map of descriptions
     * @param {object} logicalNamesMap - map of logical names
     * @param {boolean} existing - add to existing?
     * @param {string} appVar - the variable containing the app name
     * @desc upload a CSV or TSV as a new database
     * @returns {string} the query
     */
    rdbmsReplaceDatabaseUploadTable: function (app, path, delimiter, dataTypeMap, newHeaders, additionalDataTypes, descriptionMap, logicalNamesMap) {
        var query = '';

        query += 'RdbmsReplaceDatabaseUploadTable(';

        query += 'app=["' + app + '"], ';
        query += 'filePath=["' + path + '"], ';
        query += 'delimiter=["' + delimiter + '"], ';
        query += 'dataTypeMap=[' + JSON.stringify(dataTypeMap) + '], ';
        query += 'newHeaders=[' + JSON.stringify(newHeaders) + '], ';
        query += 'additionalDataTypes=[' + JSON.stringify(additionalDataTypes) + '], ';
        query += 'descriptionMap=[' + JSON.stringify(descriptionMap) + '], ';
        query += 'logicalNamesMap=[' + JSON.stringify(logicalNamesMap) + ']';

        query += ')';
        return query;
    },

    /**
     * @name rdfReplaceDatabaseLoaderSheetUpload
     * @param {string} app the app to replace
     * @param {string} filePath the path to grab file
     * @param {string} customBaseURI the uri to use
     * @desc replace database for rdf via a loader sheet
     * @returns {string} pixel query
     */
    rdfReplaceDatabaseLoaderSheetUpload: function (app, filePath, customBaseURI) {
        var query = '';

        query += 'RdfReplaceDatabaseLoaderSheetUpload(';

        query += 'app=["' + app + '"], ';
        query += 'filePath=["' + filePath + '"], ';
        query += 'customBaseURI=["' + customBaseURI + '"]';

        query += ')';
        return query;
    },

    /**
     * @name rdfReplaceDatabaseCsvUpload
     * @param {string} app the app to replace
     * @param {string} filePath to path to grab csv file
     * @param {string} delimiter the delimiter to use
     * @param {object} propFile the property file to use
     * @param {object} metamodel the metamodel of db
     * @param {object} dataTypeMap the type mapping from column to type
     * @param {object} newHeaders new headers to add
     * @param {object} additionalDataTypes additional informatino on the data type such as date format
     * @param {object} descriptionMap description of columns
     * @param {object} logicalNamesMap mapping of logical names
     * @param {string} customBaseURI the base uri to use
     * @desc replace the rdf database via a csv file
     * @returns {string} the pixel query
     */
    rdfReplaceDatabaseCsvUpload: function (app, filePath, delimiter, propFile, metamodel, dataTypeMap, newHeaders, additionalDataTypes, descriptionMap, logicalNamesMap, customBaseURI) {
        var query = '';

        query += 'RdfReplaceDatabaseCsvUpload(';

        query += 'app=["' + app + '"], ';
        query += 'filePath=["' + filePath + '"], ';
        query += 'delimiter=["' + delimiter + '"], ';
        if (propFile) {
            query += 'propFile=["' + propFile + '"], ';
        }
        query += 'metamodel=[' + JSON.stringify(metamodel) + '], ';
        query += 'dataTypeMap=[' + JSON.stringify(dataTypeMap) + '], ';
        query += 'newHeaders=[' + JSON.stringify(newHeaders) + '], ';
        query += 'additionalDataTypes=[' + JSON.stringify(additionalDataTypes) + '], ';
        query += 'descriptionMap=[' + JSON.stringify(descriptionMap) + '], ';
        query += 'logicalNamesMap=[' + JSON.stringify(logicalNamesMap) + '], ';
        query += 'customBaseURI=["' + customBaseURI + '"]';

        query += ')';

        return query;
    },

    /**
     * @name rReplaceDatabaseCsvUpload
     * @param {string} app - new app name
     * @param {string} path - path to the file to upload
     * @param {string} delimiter - delimiter
     * @param {object} dataTypeMap - map of the dataTypes
     * @param {object} newHeaders - map of the new headers
     * @param {object} additionalDataTypes - map of the additional types
     * @param {object} descriptionMap - map of descriptions
     * @param {object} logicalNamesMap - map of logical names
     * @desc upload a CSV as an R Native App
     * @returns {string} the query
     */
    rReplaceDatabaseCsvUpload: function (app, path, delimiter, dataTypeMap, newHeaders, additionalDataTypes, descriptionMap, logicalNamesMap) {
        var query = '';

        query += 'rReplaceDatabaseCsvUpload(';

        query += 'app=["' + app + '"], ';
        query += 'filePath=["' + path + '"], ';
        query += 'delimiter=["' + delimiter + '"], ';
        query += 'dataTypeMap=[' + JSON.stringify(dataTypeMap) + '], ';
        query += 'newHeaders=[' + JSON.stringify(newHeaders) + '], ';
        query += 'additionalDataTypes=[' + JSON.stringify(additionalDataTypes) + '], ';
        query += 'descriptionMap=[' + JSON.stringify(descriptionMap) + '], ';
        query += 'logicalNamesMap=[' + JSON.stringify(logicalNamesMap) + ']';
        query += ');';

        return query;
    },

    /**
     * @name tinkerReplaceDatabaseCsvUpload
     * @param {string} app - new app name
     * @param {string} path - path to the file to upload
     * @param {string} delimiter - delimiter
     * @param {string} propFile - path to propfile
     * @param {object} metamodel - map of the metamodel
     * @param {object} dataTypeMap - map of the dataTypes
     * @param {object} newHeaders - map of the new headers
     * @param {object} additionalDataTypes - map of the additional types
     * @param {object} descriptionMap - map of descriptions
     * @param {object} logicalNamesMap - map of logical names
     * @param {string} tinkerDriver - type of tinker driver to create?
     * @param {boolean} existing - add to existing?
     * @param {string} appVar - the variable containing the app name
     * @desc upload a CSV or TSV as a new database
     * @returns {string} the query
     */
    tinkerReplaceDatabaseCsvUpload: function (app, path, delimiter, propFile, metamodel, dataTypeMap, newHeaders, additionalDataTypes, descriptionMap, logicalNamesMap, tinkerDriver) {
        var query = '';

        query += 'tinkerReplaceDatabaseCsvUpload(';

        query += 'app=["' + app + '"], ';
        query += 'filePath=["' + path + '"], ';
        query += 'delimiter=["' + delimiter + '"], ';
        if (propFile) {
            query += 'propFile=["' + propFile + '"], ';
        }
        query += 'metamodel=[' + JSON.stringify(metamodel) + '], ';
        query += 'dataTypeMap=[' + JSON.stringify(dataTypeMap) + '], ';
        query += 'newHeaders=[' + JSON.stringify(newHeaders) + '], ';
        query += 'additionalDataTypes=[' + JSON.stringify(additionalDataTypes) + '], ';
        query += 'descriptionMap=[' + JSON.stringify(descriptionMap) + '], ';
        query += 'logicalNamesMap=[' + JSON.stringify(logicalNamesMap) + '], ';
        query += 'tinkerDriver=["' + tinkerDriver + '"]';

        query += ')';

        return query;
    },

    /**
     * @name externalJdbcTablesAndViews
     * @param {string} driver - database driver
     * @param {string} host - database host
     * @param {string} port - database port
     * @param {string} schema - database name
     * @param {string} username - username to access db
     * @param {string} password - password for user
     * @param {string} additional - additional connection
     * @param {string} connectionUrl - custom JDBC url
     * @desc connects to external database and returns its contents
     * @return {object} database contents
     */
    externalJdbcTablesAndViews: function (driver, host, port, schema, username, password, additional, connectionUrl) {
        var query = '',
            newDriver = driver ? driver : '',
            newHost = host ? host : '',
            newPort = port ? port : '',
            newSchema = schema ? schema : '',
            newUser = username ? username : '',
            newPass = password ? password : '',
            newConn = connectionUrl ? connectionUrl : '';

        query += 'ExternalJdbcTablesAndViews(';
        query += 'dbDriver=["' + newDriver + '"], ';
        query += 'connectionString=["' + newConn + '"], ';
        query += 'host=["' + newHost + '"], ';
        query += 'port=["' + newPort + '"], ';
        query += 'schema=["' + newSchema + '"], ';
        query += 'username=["' + newUser + '"], ';
        query += 'password=["' + newPass + '"]';
        if (additional) {
            query += ', connParams=["' + additional + '"]';
        }
        query += ')';

        return query;
    },

    /**
     * @name externalJdbcSchema
     * @param {string} driver - database driver
     * @param {string} host - database host
     * @param {string} port - database port
     * @param {string} schema - database name
     * @param {string} username - username to access db
     * @param {string} password - password for user
     * @param {string} additional - additional connection
     * @param {string} connectionUrl - custom JDBC url
     * @param {array} filters - filter information
     * @desc connects to external database and returns its contents
     * @return {object} database contents
     */
    externalJdbcSchema: function (driver, host, port, schema, username, password, additional, connectionUrl, filters) {
        var query = '',
            newDriver = driver ? driver : '',
            newHost = host ? host : '',
            newPort = port ? port : '',
            newSchema = schema ? schema : '',
            newUser = username ? username : '',
            newPass = password ? password : '',
            newConn = connectionUrl ? connectionUrl : '';

        query += 'ExternalJdbcSchema(';
        query += 'dbDriver=["' + newDriver + '"], ';
        query += 'connectionString=["' + newConn + '"], ';
        query += 'host=["' + newHost + '"], ';
        query += 'port=["' + newPort + '"], ';
        query += 'schema=["' + newSchema + '"], ';
        query += 'username=["' + newUser + '"], ';
        query += 'password=["' + newPass + '"]';
        if (additional) {
            query += ', connParams=["' + additional + '"]';
        }
        if (filters) {
            query += ', filters=' + JSON.stringify(filters) + '';
        }

        query += ')';

        return query;
    },

    /**
     * @name rdbmsExternalUpload
     * @param {string} driver - database driver
     * @param {string} host - database host
     * @param {string} port - database port
     * @param {string} schema - database name
     * @param {string} username - username to access db
     * @param {string} password - pass for user
     * @param {string} additional - additional connection
     * @param {string} connectionUrl - custom JDBC url
     * @param {string} app - new app name
     * @param {string} metamodel - ?
     * @desc delete the cache so it can recache when running
     * @returns {string} the query
     */
    rdbmsExternalUpload: function (driver, host, port, schema, username, password, additional, connectionUrl, app, metamodel) {
        var query = '',
            newDriver = driver ? driver : '',
            newHost = host ? host : '',
            newPort = port ? port : '',
            newSchema = schema ? schema : '',
            newUser = username ? username : '',
            newPass = password ? password : '',
            newConn = connectionUrl ? connectionUrl : '';

        query += 'RdbmsExternalUpload(';
        query += 'dbDriver=["' + newDriver + '"], ';
        query += 'connectionString=["' + newConn + '"], ';
        query += 'host=["' + newHost + '"], ';
        query += 'port=["' + newPort + '"], ';
        query += 'schema=["' + newSchema + '"], ';
        query += 'username=["' + newUser + '"], ';
        query += 'password=["' + newPass + '"], ';
        if (additional) {
            query += 'connParams=["' + additional + '"], ';
        }
        query += 'app=["' + app + '"], ';
        query += 'metamodel=[' + JSON.stringify(metamodel) + ']';
        query += ')';

        return query;
    },

    /**
     * @name rCsvUpload
     * @param {string} app - new app name
     * @param {string} path - path to the file to upload
     * @param {string} delimiter - delimiter
     * @param {object} dataTypeMap - map of the dataTypes
     * @param {object} newHeaders - map of the new headers
     * @param {object} additionalDataTypes - map of the additional types
     * @param {object} descriptionMap - map of descriptions
     * @param {object} logicalNamesMap - map of logical names
     * @param {string} appVar - the variable containing the app name
     * @desc upload a CSV as an R Native App
     * @returns {string} the query
     */
    rCsvUpload: function (app, path, delimiter, dataTypeMap, newHeaders, additionalDataTypes, descriptionMap, logicalNamesMap, appVar) {
        var query = '';

        query += 'RCsvUpload(';
        if (appVar) {
            query += 'app=[' + appVar + '], ';
        } else {
            query += 'app=["' + app + '"], ';
        }
        query += 'filePath=["' + path + '"], ';
        query += 'delimiter=["' + delimiter + '"], ';
        query += 'dataTypeMap=[' + JSON.stringify(dataTypeMap) + '], ';
        query += 'newHeaders=[' + JSON.stringify(newHeaders) + '], ';
        query += 'additionalDataTypes=[' + JSON.stringify(additionalDataTypes) + '], ';
        query += 'descriptionMap=[' + JSON.stringify(descriptionMap) + '], ';
        query += 'logicalNamesMap=[' + JSON.stringify(logicalNamesMap) + ']';
        query += ');';

        return query;
    },

    /**
     * @name getGraphProperties
     * @param {string} path - path
     * @desc connects to external database and returns its contents
     * @return {object} database contents
     */
    getGraphProperties: function (path) {
        var query = '',
            newPath = path ? path : '';

        query += 'GetGraphProperties(';
        query += 'filePath=["' + newPath + '"] ';
        query += ')';

        return query;
    },

    /**
     * @name getGraphMetaModel
     * @param {string} path - path
     * @param {string} graphType - graphType for user
     * @desc connects to external database and returns its contents
     * @return {object} database contents
     */
    getGraphMetaModel: function (path, graphType) {
        var query = '',
            newPath = path ? path : '',
            newGraphType = graphType ? graphType : '';

        query += 'GetGraphMetaModel(';
        query += 'filePath=["' + newPath + '"], ';
        query += 'graphTypeId=["' + newGraphType + '"] ';
        query += ')';

        return query;
    },

    /**
     * @name CreateExternalGraphDatabase
     * @param {string} path - path
     * @param {string} graphType - graphType
     * @param {string} graphName - graphName
     * @param {object} metamodel - metamodel to import
     * @param {string} app - new app name
     * @desc connects to external database and returns its contents
     * @return {object} database contents
     */
    createExternalGraphDatabase: function (path, graphType, graphName, metamodel, app) {
        var query = '',
            newPath = path ? path : '',
            newGraphType = graphType ? graphType : '',
            newGraphName = graphName ? graphName : '';

        query += 'CreateExternalGraphDatabase(';
        query += 'filePath=["' + newPath + '"], ';
        query += 'graphTypeId=["' + newGraphType + '"], ';
        query += 'graphNameId=["' + newGraphName + '"], ';
        query += 'graphMetamodel=[' + JSON.stringify(metamodel) + '], ';
        query += 'app=["' + app + '"]';
        query += ')';

        return query;
    },
    /**
     * @name getDSEGraphProperties
     * @param {string} host - database host
     * @param {string} port - database port
     * @param {string} graph - graph name
     * @param {string} username - username to access db
     * @param {string} password - password for user
     * @desc connects to external database and returns its contents
     * @return {object} database contents
     */
    getDSEGraphProperties: function (host, port, graph, username, password) {
        var query = '',
            newHost = host ? host : '',
            newPort = port ? port : '',
            newGraph = graph ? graph : '',
            newUser = username ? username : '',
            newPass = password ? password : '';

        query += 'GetDSEGraphProperties(';
        query += 'host=["' + newHost + '"], ';
        query += 'port=["' + newPort + '"], ';
        query += 'graphName=["' + newGraph + '"], ';
        query += 'username=["' + newUser + '"], ';
        query += 'password=["' + newPass + '"]';
        query += ')';

        return query;
    },

    /**
     * @name getDSEGraphMetaModel
     * @param {string} host - database host
     * @param {string} port - database port
     * @param {string} graph - graph name
     * @param {string} username - username to access db
     * @param {string} password - password for user
     * @param {string} graphType - graphType for user
     * @desc connects to external database and returns its contents
     * @return {object} database contents
     */
    getDSEGraphMetaModel: function (host, port, graph, username, password, graphType) {
        var query = '',
            newHost = host ? host : '',
            newPort = port ? port : '',
            newGraph = graph ? graph : '',
            newUser = username ? username : '',
            newPass = password ? password : '',
            newGraphType = graphType ? graphType : '';

        query += 'GetDSEGraphMetaModel(';
        query += 'host=["' + newHost + '"], ';
        query += 'port=["' + newPort + '"], ';
        query += 'graphName=["' + newGraph + '"], ';
        query += 'username=["' + newUser + '"], ';
        query += 'password=["' + newPass + '"], ';
        query += 'graphTypeId=["' + newGraphType + '"] ';
        query += ')';

        return query;
    },

    /**
     * @name insightMetamodel
     * @desc gets the frame metamodel
     * @returns {string} query
     */
    insightMetamodel: function () {
        var query = '';

        query = 'InsightMetamodel()';

        return query;
    },

    /**
     * @name getConnected
     * @param {array} conceptualNames a list of logical names to get conencted concepts for
     * @param {boolean} isLogical boolean to indicate if its a list of logical names
     * @returns {string} query
     */
    getTraversalOptions: function (conceptualNames, isLogical) {
        var query = '';

        query = 'GetTraversalOptions(values=' + JSON.stringify(conceptualNames);

        if (isLogical) {
            query += ', logical=[true]';
        }

        query += ')';

        return query;
    },

    /**
     * @name getConceptProperties
     * @param {array} logicalNames a list of logical names to get properties for
     * @param {array} database - the database to search for
     * @returns {string} query
     */
    getConceptProperties: function (logicalNames, database) {
        var query = '';

        query = 'GetConceptProperties(concepts=' + JSON.stringify(logicalNames);

        if (database) {
            query += ', database=[\"' + database + '\"]';
        }

        query += ')';

        return query;
    },

    /**
     * @name getDatabaseConcepts
     * @param {string} database the database to get concepts for
     * @returns {string} query
     */
    getDatabaseConcepts: function (database) {
        var query = '';

        query = 'GetDatabaseConcepts(database=[\"' + database + '\"])';

        return query;
    },

    /**
     * @name getDatabaseConnections
     * @param {array} columns - columns to
     * @param {string} app - app
     * @returns {string} query
     */
    getDatabaseConnections: function (columns, app) {
        var query = '',
            i, len;

        query = 'GetDatabaseConnections(';
        query += 'columns=[';
        for (i = 0, len = columns.length; i < len; i++) {
            query += ('\"' + columns[i] + '\"' + ', ');
        }
        query = this.trim(query, ',');
        query += ']';

        if (app) {
            query += ', app=["' + app + '"]';
        }

        query += ')';

        return query;
    },

    /**
     * @name toDatabase
     * @param {string} targetDatabase the database to write to
     * @param {string} targetTable the table to write to
     * @param {boolean} override whether to override the data in existing table
     * @param {boolean} insertId whether to generate an id column
     * @desc save frame to a database table
     * @returns {string} query
     */
    toDatabase: function (targetDatabase, targetTable, override, insertId) {
        var query = '';

        query = 'ToDatabase(targetDatabase=["' + targetDatabase + '"], targetTable=["' + targetTable + '"], override=[' + override + '], insertId=[' + insertId + '])';

        return query;
    },

    /**
     * @name WRITE_TO_DATABASE
     * @param {object} inputVars input variables
     * @return {string} pipeline string
     */
    WRITE_TO_DATABASE: function (inputVars) {
        var inputString = '';

        inputString += 'targetDatabase=["' + inputVars.targetDatabase + '"],';
        inputString += 'targetTable=["' + inputVars.targetTable + '"],';
        inputString += 'override=[' + inputVars.override + '],';
        inputString += 'insertId=[' + inputVars.insertId + ']';

        return inputString;
    },

    /**
     * @name getSpecificConceptProperties
     * @param {string} database - the database to get properties for
     * @param {string} concept - the concept to get properties for
     * @returns {string} query
     */
    getSpecificConceptProperties: function (database, concept) {
        var query = '';

        query = 'GetSpecificConceptProperties(database=[\"' + database + '\"], concept=[' + concept + '])';

        return query;
    },

    /**
     * @name getDatabaseList
     * @returns {string} query
     */
    getDatabaseList: function () {
        var query = '';

        query = 'GetDatabaseList()';

        return query;
    },

    /**
     * @name getFrameFilters
     * @returns {string} query
     */
    getFrameFilters: function () {
        var query = '';

        query = 'GetFrameFilters()';

        return query;
    },

    /**
     * @name updateInsightImage
     * @param {string} app -  app to update
     * @param {number} rdbmsId - the rdbmsId of the insight to be updated
     * @param {string} url - the url for BE to fetch the image from
     * @returns {string} query for updating an insights image on feed
     */
    updateInsightImage: function (app, rdbmsId, url) {
        var query = '';

        query += 'UpdateInsightImage(';
        query += 'app=[\"' + app + '\"], ';
        query += 'id=[\"' + rdbmsId + '\"], ';
        query += 'url=[\"' + url + '\"]';
        query += ')';

        return query;
    },

    /**
     * @name updateRowValues
     * @param {string} col The column to perform replacement on
     * @param {number | string | date} replacementValue The value to be used for replacement
     * @param {Array} replacementArray The values in col to be replaced with replacementValue
     * @returns {string} The updateRowValues query to be run
     */
    updateRowValues: function (col, replacementValue, replacementArray) {
        var query = '',
            i;

        query = 'UpdateRowValues(' + col + ', ' + '"' + replacementValue + '", Filter(' + col + ' == [';

        for (i = 0; i < replacementArray.length; i++) {
            query += '"' + replacementArray[i] + '"';

            if (i !== (replacementArray.length - 1)) {
                query += ', ';
            }
        }

        query += ']))';

        return query;
    },

    // App calls

    /**
     * @name myApps
     * @desc create the pixel to get my apps
     * @returns {string} the pixel query
     */
    myApps: function () {
        var query = '';
        query += 'MyApps()';
        return query;
    },

    /**
     * @name appInfo
     * @desc create the pixel to get metadata for the app
     * @param {string} app - the app name
     * @returns {string} the pixel query
     */
    appInfo: function (app) {
        var query = '';

        query += 'AppInfo(';
        query += 'app=[\"' + app + '\"]';
        query += ')';

        return query;
    },

    /**
     * @name appUsers
     * @desc create the pixel to get users of the app
     * @param {string} appId - the app id
     * @returns {string} the pixel query
     */
    appUsers: function (appId) {
        var query = '';

        query += 'AppUsers(';
        query += 'app=[\"' + appId + '\"]';
        query += ')';

        return query;
    },

    /**
     * @name naturalLanguageSearch
     * @desc create the pixel to get the insights for an app
     * @param {string} searchQuery - the search query
     * @param {array} app - the app names
     * @param {boolean} global - scope of the query. If true will be all DB (string query), if false will be at frame level (template query).
     * @param {number} panelId - the ID to create the new panel with
     * @returns {string} the pixel query
     */
    naturalLanguageSearch: function (searchQuery, app, global, panelId) {
        var query = '';

        query += 'NaturalLanguageSearch(';
        if (!global) {
            query += 'query=[\"' + JSON.stringify(searchQuery).replace(/"/g, '\\"') + '\"], ';
        } else {
            query += 'query=[\"' + searchQuery + '\"], ';
        }
        query += 'app=' + JSON.stringify(app) + ', ';
        query += 'global = [' + global + ']';
        if (panelId) {
            query += ', panel = [' + panelId + ']';
        } else {
            query += ', panel = [0]';
        }
        query += ')';

        return query;
    },

    /**
     * @name naturalLanguagePredictVizType
     * @desc predict the visualization type for the nlp suggested insight
     * @param {string} appId - selected app ID
     * @param {array} columns - columns to include in viz
     * @returns {string} the pixel query
     */
    naturalLanguagePredictVizType: function (appId, columns) {
        var query = '';
        // query += 'Frame ( frame = [ ' + frame + ' ] )| ';
        query += 'PredictViz(';
        query += 'app=[\"' + appId + '\"], ';
        query += 'columns=' + JSON.stringify(columns) + '';
        query += ')';

        return query;
    },

    /**
     * @name nlsQueryHelper
     * @desc create the pixel to get the insights for an app
     * @param {string} searchQuery - the search query
     * @param {array} app - the app names
     * @param {boolean} global - scope of the query. If true will be all DB (string query), if false will be at frame level (template query).
     * @returns {string} the pixel query
     */
    nlsQueryHelper: function (searchQuery, app, global) {
        var query = '';

        query += 'NLSQueryHelper(';
        if (!global) {
            query += 'query=[\"' + JSON.stringify(searchQuery).replace(/"/g, '\\"') + '\"], ';
        } else {
            query += 'query=[\"' + searchQuery + '\"], ';
        }
        query += 'app=' + JSON.stringify(app) + ', ';
        query += 'helpOn = [ true ], ';
        query += 'global = [  ' + global + ']';
        query += ')';

        return query;
    },

    /**
     * @name getInsights
     * @desc create the pixel to get the insights for an app
     * @param {string | array} app - the app name
     * @param {string} limit - requested limit
     * @param {string} offset - requested offset
     * @param {string} filter - filter workd
     * @param {array} tags - tags to search on
     * @returns {string} the pixel query
     */
    getInsights: function (app, limit, offset, filter, tags) {
        var query = '',
            j;

        query += 'GetInsights(';

        if (typeof app !== undefined && app) {
            if (typeof app === 'object') {
                if (app.length > 0) {
                    query += 'app=[';
                    for (j = 0; j < app.length; j++) {
                        if (j === app.length - 1) {
                            query += '\"' + app[j] + '\"';
                        } else {
                            query += '\"' + app[j] + '\"' + ', ';
                        }
                    }
                    query += '], ';
                }
            } else {
                query += 'app=[\"' + app + '\"], ';
            }
        }

        if (typeof limit !== 'undefined' && typeof offset !== 'undefined') {
            query += 'limit=[\"' + limit + '\"], ';
            query += 'offset=[\"' + offset + '\"],';
        }

        if (typeof filter !== 'undefined' && filter) {
            query += 'filterWord=[\"' + filter + '\"], ';
        }

        if (typeof tags !== 'undefined' && tags.length > 0) {
            query += ` tags=${JSON.stringify(tags)},`;
        }

        query = this.trim(query, ',');

        query += ')';

        return query;
    },

    /**
     * @name openApp
     * @desc create the pixel to open an app
     * @param {string} app - apps to delete
     * @returns {string} the pixel query
     */
    openApp: function (app) {
        var query = '';

        query += 'OpenApp(';
        query += 'app=[\"';
        query += app;
        query += '\"] ';
        query += ')';

        return query;
    },

    /**
     * @name exportApp
     * @desc create the pixel to export an app
     * @param {string} app - apps to delete
     * @returns {string} the pixel query
     */
    exportApp: function (app) {
        var query = '';

        query += 'ExportApp(';
        query += 'app=[\"';
        query += app;
        query += '\"] ';
        query += ')';

        return query;
    },

    /**
     * @name deleteApp
     * @desc create the pixel to delete an app
     * @param {array} apps - apps to delete
     * @returns {string} the pixel query
     */
    deleteApp: function (apps) {
        var query = '',
            i, len;

        query += 'DeleteApp(';
        query += 'app=[';
        for (i = 0, len = apps.length; i < len; i++) {
            query += ('\"' + apps[i] + '\"' + ', ');
        }
        query = this.trim(query, ',');
        query += '] ';
        query += ')';

        return query;
    },

    /**
     * @name deleteInsight
     * @desc create the pixel to delete insights for an app
     * @param {string} appId the app id to delete from
     * @param {array} insights insights to delete
     * @returns {string} the pixel query
     */
    deleteInsight: function (appId, insights) {
        var query = '',
            i, len;

        query += 'DeleteInsight(app=[\"' + appId + '\"], ';
        query += 'id=[';
        for (i = 0, len = insights.length; i < len; i++) {
            query += ('\"' + insights[i] + '\"' + ', ');
        }
        query = this.trim(query, ',');
        query += '] ';
        query += ')';

        return query;
    },

    /**
     * @name fuzzyMatches
     * @param {string} outputFrame - frame for matches to go to (random string)
     * @param {string} column - name of column to match
     * @param {string} frame - frame to get matches from
     * @desc gets fuzzy matches for a column
     * @return {string} query
     */
    fuzzyMatches: function (outputFrame, column, frame) {
        var query = '';

        query += 'FuzzyMatches(';
        if (frame) {
            query += 'frame=[' + frame + '], ';
        }
        query += 'outputFrame=["' + outputFrame + '"], ';
        query += 'frameCol=[' + column + '])';

        return query;
    },
    /**
     * @name fuzzyMerge
     * @param {object} params all the params
     * @returns {string} the pixel query
     */
    FUZZY_MERGE: function (params) {
        var query = '',
            param;

        for (param in params) {
            if (params.hasOwnProperty(param)) {
                query += param + '=[' + params[param] + '],';
            }
        }

        // remove trailing comma
        query = query.slice(0, -1);

        return query;
    },

    /**
     * @name federationBestMatches
     * @param {string} appName - name of the selected app to pull from
     * @param {string} conceptName - name of the selected column to pull from
     * @param {string} columnName - name of the selected column to pull from
     * @param {string} frameColumnName - name of the frameColumn tha tyou will interact with
     * @param {string} outputFrame - name of frame to use from output
     * @returns {string} the pixel query
     */
    federationBestMatches: function (appName, conceptName, columnName, frameColumnName, outputFrame) {
        var query = '';

        query += 'FederationBestMatches(';
        query += 'app=[\"' + appName + '\"], ';
        query += 'concept=[\"' + conceptName + '\"], ';
        query += 'column=[\"' + columnName + '\"], ';
        query += 'frameCol=[\"' + frameColumnName + '\"] ';
        if (outputFrame) {
            query += ',';
            query += 'outputFrame=[\"' + outputFrame + '\"] ';
        }
        query += ')';

        return query;
    },

    /**
     * @name FederationBlend
     * @param {string} appName - name of the selected app to pull from
     * @param {string} conceptName - name of the selected column to pull from
     * @param {string} columnName - name of the selected column to pull from
     * @param {string} fedFrame - name of the frame to federate with
     * @param {string} frameColumnName - name of the frameColumn tha tyou will interact with
     * @param {string} joinType - type of join
     * @param {array} acceptedMatches - accepted matches to use
     * @param {array} rejectedMatches - reject matches to use
     * @param {number} propagation - propagation score
     * @param {string} additionalCols - additional columns to bring in
     * @param {string} frame - frame to import into
     * @returns {string} the pixel query
     */
    federationBlend: function (appName, conceptName, columnName, fedFrame, frameColumnName, joinType, acceptedMatches, rejectedMatches, propagation, additionalCols, frame) {
        var query = '',
            i, len;

        query += 'FederationBlend(';
        query += 'app=[\"' + appName + '\"], ';
        query += 'concept=[\"' + conceptName + '\"], ';
        query += 'column=[\"' + columnName + '\"], ';
        query += 'fedFrame=[\"' + fedFrame + '\"], ';
        query += 'frameCol=[\"' + frameColumnName + '\"], ';
        query += 'joinType=[\"' + joinType + '\"], ';
        query += 'matches=[';
        for (i = 0, len = acceptedMatches.length; i < len; i++) {
            query += ('\"' + acceptedMatches[i] + '\"' + ', ');
        }
        query = this.trim(query, ',');
        query += '], ';
        query += 'nonMatches=[';
        for (i = 0, len = rejectedMatches.length; i < len; i++) {
            query += ('\"' + rejectedMatches[i] + '\"' + ', ');
        }
        query = this.trim(query, ',');
        query += '], ';
        query += 'propagation=[' + propagation + '], ';
        query += 'additionalCols=[';
        for (i = 0, len = additionalCols.length; i < len; i++) {
            query += ('\"' + additionalCols[i] + '\"' + ', ');
        }
        query = this.trim(query, ',');
        query += '] ';
        if (frame) {
            query += ', frame=[' + frame + '] ';
        }
        query += ')';

        return query;
    },
    // Git calls
    /**
     * @name isGit
     * @desc checks if an app is Gitified
     * @param {string} appName the name you want to use as the app name
     * @returns {string} the pixel query
     */
    isGit: function (appName) {
        var query = '';

        query += 'IsGit(';
        query += 'app=[\"' + appName + '\"] ';
        query += ')';

        return query;
    },

    /**
     * @name gitStatus
     * @desc checks the status of the app
     * @param {string} appName the name you want to use as the app name
     * @returns {string} the pixel query
     */
    gitStatus: function (appName) {
        var query = '';

        query += 'GitStatus(';
        query += 'app=[\"' + appName + '\"] ';
        query += ')';

        return query;
    },

    /**
     * @name copyAppRepo
     * @param {string} appName the name of the app you want to synchronize
     * @param {string} gitLocation the url link to your git
     * @desc create the pixel for to copy a 'git' app
     * @returns {string} the pixel query
     */

    copyAppRepo: function (appName, gitLocation) {
        var query = '';

        query += 'CopyAppRepo(';
        query += 'app=[\"' + appName + '\"], ';
        query += 'repository=[\"' + gitLocation + '\"] ';
        query += ')';

        return query;
    },
    /**
     * @name uploadAppRepo
     * @param {string} filePath - the path of the file
     * @param {string} fileSpace  - optional: the space where the file is located
     * @desc imports an app via a zip file
     * @returns {string} the pixel query
     */
    uploadAppRepo: function (filePath, fileSpace) {
        var query = '';
        query = `UploadApp(filePath=["${filePath}"], space=["${fileSpace ? fileSpace : ''}"])`;
        return query;
    },

    /* uploadAppRepo: function (filePath, fileSpace) {
        var query = '';
        query = `UploadApp(filePath=["${filePath}"], space=["${fileSpace ? fileSpace : ''}"])`;
        return query;
    },*/

    /**
     * @name initAppRepo
     * @param {string} appName the name of the app you want to synchronize
     * @param {string} gitLocation the url link to your git
     * @param {string} databaseSync  - sync the database?
     * @desc create the pixel for to create a new 'git' app
     * @returns {string} the pixel query
     */

    initAppRepo: function (appName, gitLocation, databaseSync) {
        var query = '';

        query += 'InitAppRepo(';
        query += 'app=[\"' + appName + '\"], ';
        query += 'repository=[\"' + gitLocation + '\"], ';
        query += 'syncDatabase=[\"' + databaseSync + '\"] ';
        query += ')';

        return query;
    },

    /**
     * @name syncApp
     * @param {string} appName - the name of the app you want to synchronize
     * @param {string} gitLocation - the url link to your git
     * @param {string} dualSync - true/false the dictates whether you are just pulling (false) or pulling and then pushing (true)
     * @param {string} databaseSync  - sync the database?
     * @desc create the pixel for synchronizing the app to/from git
     * @returns {string} the pixel query
     */
    syncApp: function (appName, gitLocation, dualSync, databaseSync) {
        var query = '';

        query += 'SyncApp(';
        query += 'app=[\"' + appName + '\"], ';
        query += 'repository=[\"' + gitLocation + '\"], ';
        query += 'dual=[\"' + dualSync + '\"], ';
        query += 'syncDatabase=[\"' + databaseSync + '\"] ';
        query += ')';

        return query;
    },

    /**
     * @name syncAppFilesO
     * @param {string} appName - the name of the app you want to synchronize
     * @param {string} gitLocation - the url link to your git
     * @param {string} dualSync - true/false the dictates whether you are just pulling (false) or pulling and then pushing (true)
     * @param {array} files - files to sync
     * @param {string} databaseSync  - sync the database?
     * @desc create the pixel for synchronizing the app to/from git
     * @returns {string} the pixel query
     */
    syncAppFilesO: function (appName, gitLocation, dualSync, files, databaseSync) {
        var query = '',
            i, len;

        query += 'SyncAppFilesO(';
        query += 'app=[\"' + appName + '\"], ';
        query += 'repository=[\"' + gitLocation + '\"], ';
        query += 'dual=[\"' + dualSync + '\"], ';
        query += 'syncDatabase=[\"' + databaseSync + '\"], ';
        query += 'files=[';
        for (i = 0, len = files.length; i < len; i++) {
            query += ('\"' + files[i] + '\"' + ', ');
        }
        query = this.trim(query, ',');
        query += '] ';
        query += ')';

        return query;
    },

    /**
     * @name deleteAppRepo
     * @param {string} appName the name of the app you want to synchronize
     * @param {string} gitLocation the url link to your git
     * @desc create the pixel for to create a delete 'git' app (local + remote)
     * @returns {string} the pixel query
     */

    deleteAppRepo: function (appName, gitLocation) {
        var query = '';

        query += 'DeleteAppRepo(';
        query += 'app=[\"' + appName + '\"], ';
        query += 'repository=[\"' + gitLocation + '\"]';
        query += ')';

        return query;
    },

    /**
     * @name dropAppRepo
     * @param {string} appName the name of the app you want to synchronize
     * @param {string} gitLocation the url link to your git
     * @desc create the pixel for to a drop 'git' app (local)
     * @returns {string} the pixel query
     */

    dropAppRepo: function (appName, gitLocation) {
        var query = '';

        query += 'DropAppRepo(';
        query += 'app=[\"' + appName + '\"], ';
        query += 'repository=[\"' + gitLocation + '\"] ';
        query += ')';

        return query;
    },

    /**
     * @name listAppRemotes
     * @desc get a list of remote repos associated with an app
     * @param {string} appName the name of the app you want to synchronize
     * @returns {string} the pixel query
     */
    listAppRemotes: function (appName) {
        var query = '';

        query += 'ListAppRemotes(';
        query += 'app=[\"' + appName + '\"] ';
        query += ')';

        return query;
    },

    /**
     * @name listAppCollaborators
     * @desc create the pixel to get all of the collabarators for this app/repo
     * @param {string} gitLocation the url link to your git
     * @returns {string} the pixel query
     */
    listAppCollaborators: function (gitLocation) {
        var query = '';

        query += 'ListAppCollaborators(';
        query += 'repository=[\"' + gitLocation + '\"]';
        query += ')';
        return query;
    },

    /**
     * @name searchAppCollaborator
     * @desc create the pixel to search for a collabarators
     * @param {string} search search string
     * @returns {string} the pixel query
     */
    searchAppCollaborator: function (search) {
        var query = '';

        query += 'SearchAppCollaborator(';
        query += 'collaborator=[\"' + search + '\"]';
        query += ')';

        return query;
    },

    /**
     * @name AddAppCollaborator
     * @param {string} gitLocation the url location of the git repository
     * @param {string} collaboratorAccName the name of the collaborator you want to add
     * @desc create the pixel for adding a collaborator to the repo
     * @returns {string} the pixel query
     */
    addAppCollaborator: function (gitLocation, collaboratorAccName) {
        var query = '';

        query += 'AddAppCollaborator(';
        query += 'repository=[\"' + gitLocation + '\"], ';
        query += 'collaborator=[\"' + collaboratorAccName + '\"]';
        query += ')';

        return query;
    },

    /**
     * @name removeAppCollaborator
     * @param {string} gitLocation the url location of the git repository
     * @param {string} collaboratorAccName the name of the collaborator you want to add
     * @desc create the pixel for removing a collaborator to the repo
     * @returns {string} the pixel query
     */
    removeAppCollaborator: function (gitLocation, collaboratorAccName) {
        var query = '';

        query += 'RemoveAppCollaborator(';
        query += 'repository=[\"' + gitLocation + '\"], ';
        query += 'collaborator=[\"' + collaboratorAccName + '\"]';
        query += ')';
        return query;
    },

    /**
     * @name scheduleJob
     * @param {string} jobName name of job to schedule
     * @param {string} jobGroup name of group job belongs to
     * @param {string} cronExpression a cron expression
     * @param {string} recipe pixel to run for job
     * @param {boolean} onLoad whether to trigger on load
     * @param {string} parameters stringified object that BE will send back when listing jobs
     * @desc schedules a job
     * @return {string} the pixel query
     */
    scheduleJob: function (jobName, jobGroup, cronExpression, recipe, onLoad, parameters) {
        var query = '';

        query += 'ScheduleJob(';
        query += 'jobName=["' + jobName + '"], ';
        query += 'jobGroup=["' + jobGroup + '"], ';
        query += 'cronExpression=["' + cronExpression + '"], ';
        query += 'recipe=["' + recipe + '"], ';
        query += 'parameters=["' + parameters + '"], ';
        query += 'triggerOnLoad=[' + onLoad + '], triggerNow=[false]';
        query += ')';

        return query;
    },

    /**
     * @name deleteJob
     * @param {string} jobName name of job to schedule
     * @param {string} jobGroup name of group job belongs to
     * @desc unschedules a job
     * @return {string} the pixel query
     */
    deleteJob: function (jobName, jobGroup) {
        var query = '';

        query += 'RemoveJobFromDB(';
        query += 'jobName=["' + jobName + '"], ';
        query += 'jobGroup=["' + jobGroup + '"]) ';
        return query;
    },

    /**
     * @name listAllJobs
     * @desc lists all available jobs in semoss
     * @return {string} the pixel query
     */
    listAllJobs: function () {
        var query = '';

        query += 'ListAllJobs()';

        return query;
    },

    /**
     * @name insightRecipe
     * @param {string} app Name of the app your insight is tied to
     * @param {string} rdbmsId The RDBMS ID of your insight
     * @returns {string} Pixel query to run
     * @desc creates and returns the pixel query for getting an insight's recipe
     */
    insightRecipe: function (app, rdbmsId) {
        var query = '';

        query += 'InsightRecipe(app=["' + app + '"], id=["' + rdbmsId + '"])';

        return query;
    },

    /**
     * @name googleListFiles
     * @desc lists all of the google sheets files
     * @returns {string} the query
     */
    googleListFiles: function () {
        var query = '';

        query += 'GoogleListFiles()';

        return query;
    },

    /**
     * @name googleSheetSource
     * @param {string} sheetId unique id of the sheet
     * @param {string} sheetType the type of the sheet
     * @desc sets the google sheets as the data source
     * @returns {string} the query
     */
    googleFileRetriever: function (sheetId, sheetType) {
        var query = '';

        query += 'GoogleFileRetriever(id=["' + sheetId + '"],';

        query += 'type=["' + sheetType + '"])';

        return query;
    },

    /**
     * @name dropBoxListFiles
     * @desc lists all of the dropbox files (text/csv)
     * @returns {string} the query
     */
    dropBoxListFiles: function () {
        var query = '';

        query += 'DropBoxListFiles()';

        return query;
    },

    /**
     * @name dropBoxFileRetriever
     * @param {string} path path of the file
     * @desc sets the file as the data source
     * @returns {string} the query
     */
    dropBoxFileRetriever: function (path) {
        var query = '';

        query += 'DropBoxFileRetriever(path=["' + path + '"])';

        return query;
    },

    /**
     * @name oneDriveListFiles
     * @desc lists all of the one drive files (text/csv)
     * @returns {string} the query
     */
    oneDriveListFiles: function () {
        var query = '';

        query += 'OneDriveListFiles()';

        return query;
    },

    /**
     * @name oneDriveFileRetriever
     * @param {string} id unique id of the file
     * @desc sets the file as the data source
     * @returns {string} the query
     */
    oneDriveFileRetriever: function (id) {
        var query = '';

        query += 'OneDriveFileRetriever(id=["' + id + '"])';

        return query;
    },

    /**
     * @name api
     * @param {string} type - type of API to parse (JSON, XML, etc.)
     * @param {string} url -  API url to hit
     * @param {string} method - API method, post, etc.
     * @param {object} headers - headers for the call
     * @param {object} body - body for the call
     * @param {object} roots - roots of the path
     * @returns {string} Pixel query to run
     * @desc Creates and returns the API pixel query necessary for the jsonPath UI
     */
    api: function (type, url, method, headers, body, roots) {
        var query = '';

        query += 'API(';
        query += 'api_type=["' + type + '"]';
        query += ', aliasMap=[' + JSON.stringify(angular.merge({
            input_url: url,
            input_method: method
        }, roots || {})) + ']';

        if (headers && Object.keys(headers).length > 0) {
            query += ', headersMap=[' + JSON.stringify(headers) + ']';
        }

        // Case that we are passing headers
        if (body && Object.keys(body).length > 0) {
            query += ', bodyMap=[' + JSON.stringify(body) + ']';
        }

        query += ')';

        return query;
    },

    /**
     * @name jsonPath
     * @param {string} apiType JSON, XML, etc.
     * @param {string} inputURL API url to hit
     * @param {string} inputMethod GET, POST, etc.
     * @param {array} aliases Array of alias strings to use
     * @param {array} jsonPaths Array of json path strings corresponding to aliases
     * @param {Object} headerMap Map of headers to pass through call
     * @returns {string} Pixel query to run
     * @desc Creates and returns the API pixel query necessary for the jsonPath UI
     */
    jsonPath: function (apiType, inputURL, inputMethod, aliases, jsonPaths, headerMap) {
        var query = '',
            key,
            i = 0;

        query += 'API(api_type=["' + apiType + '"], aliasMap=[{"input_url": "' + inputURL +
            '", "input_method": "' + inputMethod.toLocaleLowerCase() + '", ';

        for (i; i < aliases.length; i++) {
            query += '"' + aliases[i] + '": "' + jsonPaths[i] + '", ';

            if (i === (aliases.length - 1)) {
                query = query.slice(0, -2);
            }
        }

        if (headerMap && Object.keys(headerMap).length > 0) {
            // Close our aliasMap array and open our
            // headersMap array
            query += '}], headersMap=[{';

            for (key in headerMap) {
                if (headerMap.hasOwnProperty(key)) {
                    query += '"' + key + '": ';
                    query += '"' + headerMap[key] + '"';
                    query += ', ';
                }
            }

            // Remove last comma space (, )
            query = query.slice(0, -2);
        }

        query += '}])'; // TODO::: multiple urls?

        return query;
    },

    /**
     * @name jsonPath
     * @param {string} apiType JSON, XML, etc.
     * @param {string} inputURL API url to hit
     * @param {string} inputMethod GET, POST, etc.
     * @param {string} root JMES Path for the root
     * @param {array} aliasCombos Array of alias combos to pass into the API call
     * @param {Object} headerMap Map of headers to pass through call
     * @returns {string} Pixel query to run
     * @desc Creates and returns the API pixel query necessary for the jsonPath UI
     */
    jmesPath: function (apiType, inputURL, inputMethod, root, aliasCombos, headerMap) {
        var query = '',
            key,
            i;

        query += 'API(api_type=["' + apiType + '"], aliasMap=[{"input_url": "' + inputURL +
            '", "input_method": "' + inputMethod.toLocaleLowerCase() + '", "root": ' + root;

        if (aliasCombos) {
            for (i = 0; i < aliasCombos.length; i++) {
                query += ', ';
                query += '"' + aliasCombos[i].alias + '":';
                query += '"' + aliasCombos[i].property + '"';
            }
        }

        if (headerMap && (Object.keys(headerMap).length > 0)) {
            // Close our aliasMap array and open our
            // headersMap array
            query += '}], headersMap=[{';

            for (key in headerMap) {
                if (headerMap.hasOwnProperty(key)) {
                    query += '"' + key + '": ';
                    query += '"' + headerMap[key] + '"';
                    query += ', ';
                }
            }

            // Get rid of last comma
            query = query.slice(0, -2);
        }

        // Close our array
        query += '}])';

        return query;
    },

    /**
     * @name getNumTable
     * @param {string} URL URL string to hit
     * @param {string} inputMethod Type of input to use (GET, POST, etc.)
     * @param {Object} headerMap Optional object composed of headers to add to our URL
     * @returns {string} query
     * @desc Create the GetNumTable pixel to fetch the number of tables
     *       at a given URL
     */
    getNumTable: function (URL, inputMethod, headerMap) {
        var query = '',
            key;

        query += 'GetNumTable(url=["' + URL + '"], ';
        query += 'aliasMap=[{"input_method": "' + inputMethod.toLowerCase() + '"';

        if (headerMap && (Object.keys(headerMap).length > 0)) {
            query += '}], headersMap=[{';

            for (key in headerMap) {
                if (headerMap.hasOwnProperty(key)) {
                    query += '"' + key + '": ';
                    query += '"' + headerMap[key] + '"';
                    query += ', ';
                }
            }

            // Get rid of last comma
            query = query.slice(0, -2);
        }

        query += '}])';

        return query;
    },

    /**
     * @name getTableHeader
     * @param {string} URL URL to hit
     * @param {string} inputMethod Type of input to use (GET, POST, etc.)
     * @param {number} tableNumber The number of the table to pull eata from
     * @param {Object} headerMap Optional object composed of headers to add to our URL
     * @returns {string} query
     * @desc Creates the GetTableHeader pixel that fetches
     *       the names of a certain table on a web page.
     */
    getTableHeader: function (URL, inputMethod, tableNumber, headerMap) {
        var query = '',
            key;

        query += 'GetTableHeader(url=["' + URL + '"], ';
        query += 'aliasMap=[{"input_method": "' + inputMethod.toLowerCase() + '", ';
        query += '"table_number": ' + tableNumber;

        if (headerMap && (Object.keys(headerMap).length > 0)) {
            query += '}], headersMap=[{';

            for (key in headerMap) {
                if (headerMap.hasOwnProperty(key)) {
                    query += '"' + key + '": ';
                    query += '"' + headerMap[key] + '"';
                    query += ', ';
                }
            }

            // Get rid of last comma
            query = query.slice(0, -2);
        }

        query += '}])';

        return query;
    },

    /**
     * @name scrapeURL
     * @param {string} URL URL to hit
     * @param {string} inputMethod Type of input to use (GET, POST, etc.)
     * @param {number} tableNumber The number of the table to pull data from
     * @param {Object} headerMap Optional object composed of headers to add to the URL
     * @returns {string} query
     * @desc Creates the WEB API call to pull in data from a web page.
     */
    scrapeURL: function (URL, inputMethod, tableNumber, headerMap) {
        var query = '',
            key;

        query += 'API(api_type=["WEB"], ';
        query += 'aliasMap=[{"input_url": "' + URL + '", ';
        query += '"input_method": "' + inputMethod.toLowerCase() + '", ';
        query += '"table_number": ' + tableNumber;

        if (headerMap && (Object.keys(headerMap).length > 0)) {
            query += '}], headersMap=[{';

            for (key in headerMap) {
                if (headerMap.hasOwnProperty(key)) {
                    query += '"' + key + '": ';
                    query += '"' + headerMap[key] + '"';
                    query += ', ';
                }
            }

            // Get rid of last comma
            query = query.slice(0, -2);
        }

        query += '}])';

        return query;
    },

    /**
     * @name getRequest
     * @param {string} URL url to fetch data from
     * @param {object} headers -  map of headers for URL
     * @returns {string} pixel query to run
     * @desc Creates a simple GetRequest query that fetches data from an endpoint
     */
    getRequest: function (URL, headers) {
        var query = '';

        query += 'GetRequest(';
        query += 'url=["' + URL + '"]';

        // Case that we are passing headers
        if (headers && Object.keys(headers).length > 0) {
            query += ', headersMap=[' + JSON.stringify(headers) + ']';
        }

        query += ')';

        return query;
    },

    /**
     * @name postRequest
     * @param {string} URL url to fetch data from
     * @param {object} headers -  map of headers for URL
     * @param {object} body - map of body params for request
     * @returns {string} Pixel query to run
     * @desc Creates a simple GetRequest query that fetches data from an endpoint
     */
    postRequest: function (URL, headers, body) {
        var query = '';


        query += 'PostRequest(';
        query += 'url=["' + URL + '"]';

        // Case that we are passing headers
        if (headers && Object.keys(headers).length > 0) {
            query += ', headersMap=[' + JSON.stringify(headers) + ']';
        }

        // Case that we are passing body
        if (body && Object.keys(body).length > 0) {
            query += ', bodyMap=[' + JSON.stringify(body) + ']';
        }

        query += ')';

        return query;
    },

    /**
     * @name getInsightDataSources
     * @desc grabs all of the data source pixels in the insight
     * @returns {string} Pixel query
     */
    getInsightDatasources: function () {
        var query = '';

        query += 'GetInsightDatasources()';

        return query;
    },

    /**
     * @name modifyInsightDatasource
     * @param {array} options of datasources to replace along with their data {"index": 4, "pixel": "pixel"}
     * @desc modifies the data source used in this insight
     * @returns {string} the pixel query to run
     */
    modifyInsightDatasource: function (options) {
        var query = '';

        query += 'ModifyInsightDatasource(options=' + JSON.stringify(options) + ')';

        return query;
    },

    /**
     * @name generateNewApp
     * @param {string} appName the name of the app
     * @desc generate an empty app with insight
     * @returns {string} the query
     */
    generateEmptyApp: function (appName) {
        var query = '';

        query = 'GenerateEmptyApp(' + appName + ')';

        return query;
    },

    discretize: function (column, breaks, labels) {
        var query = '';

        query = 'Discretize([{"column":' + '"' + column + '", ';

        if (breaks) {
            query += '"breaks":"' + breaks + '"' + ', ';
        }

        if (labels) {
            // TODO
        } else {
            query = query.slice(0, -2);
            query += '}])';
        }

        return query;
    },

    /**
     * @name matchColumnValues
     * @param {string} col Name of a column in the frame
     * @returns {string} Pixel query to run
     * @desc Creates the MatchColumnValues pixel to get all of the matched
     *       column values in a particular column.
     */
    matchColumnValues: function (col) {
        var query = '';

        query += 'MatchColumnValues(column=["' + col + '"])';

        return query;
    },

    /**
     * @name updateMatchColumnValues
     * @param {string} col Name of the column in the frame
     * @param {array} matchArray Array of matched value objects to use in update
     * @param {string} tableName The name of the R table to update these values for
     * @returns {string} Pixel query to run
     * @desc Creates the UpdateMatchColumnValues pixel to update the R table
     *       containing the matches that a user has indicated.
     */
    updateMatchColumnValues: function (col, matchArray, tableName) {
        var query = '',
            i;

        query += 'UpdateMatchColumnValues(column=["' + col + '"], ';
        query += 'matches=[';

        for (i = 0; i < matchArray.length; i++) {
            query += '"' + matchArray[i].left + ' == ' + matchArray[i].right + '"';

            if (i !== matchArray.length - 1) {
                query += ', ';
            }
        }

        query += '], ';
        query += 'matchesTable=["' + tableName + '"])';

        return query;
    },

    /**
     * @name semanticBlending
     * @param {array} cols The columns to run semantic blending with (required)
     * @param {number} display The display parameter (optional, default = 3)
     * @param {number} randomVals The randomVals parameter (optional, default = 20)
     * @param {boolean} genFrame The genFrame parameter (optional, default = false)
     * @param {string} frameName The frame to run this on (option, default = 'predictionFrame'...only needed when genFrame = true)
     * @return {string} Pixel query
     * @desc Creates and returns the pixel query for running semantic blending
     */
    semanticBlending: function (cols, display, randomVals, genFrame, frameName) {
        var query = '',
            finalDisplay = display || 3,
            finalRandomVals = randomVals || 20,
            finalGenFrame = false,
            finalFrameName = frameName || 'predictionFrame',
            i;

        // Have to do this one separately since || only works for truthy vals
        if (genFrame) {
            finalGenFrame = genFrame;
        }

        query += 'SemanticBlending(columns=["';

        for (i = 0; i < cols.length; i++) {
            if (i === (cols.length - 1)) {
                query += cols[i] + '"], ';
            } else {
                query += cols[i] + '", "';
            }
        }

        query += 'display =["' + finalDisplay + '"], ';
        query += 'randomVals = ["' + finalRandomVals + '"], ';
        query += 'genFrame = [' + finalGenFrame + '], ';
        query += 'frameName = ["' + finalFrameName + '"])';

        return query;
    },

    /**
     * @name addInsightComment
     * @param {string} comment The insight comment String to be saved
     * @returns {string} Pixel query
     * @desc Creates and returns the pixel query for adding an insight comment
     */
    addInsightComment: function (comment) {
        var query = '';

        query += 'AddInsightComment("' + comment + '")';

        return query;
    },

    /**
     * @name modifyInsightComment
     * @param {string} id The id of the insight to modify
     * @param {string} newComment The new comment to replace the old comment with
     * @returns {string} pixel queyr
     * @desc Creates and returns the pixel query for modifying an insight comment
     */
    modifyInsightComment: function (id, newComment) {
        var query = '';

        query += 'ModifyInsightComment("' + id + '", "' + newComment + '")';

        return query;
    },

    /**
     * @name getInsightComments
     * @returns {string} Pixel query
     * @desc Creates and returns the pixel query for getting all insight comments
     */
    getInsightComments: function () {
        var query = '';

        query += 'GetInsightComments()';

        return query;
    },

    /**
     * @name getRuleTypes
     * @returns {string} Pixel query to run
     * @desc Creates the GetEditRuleTypes pixel to fetch all the pre-defined
     *       business rules held in the back-end.
     */
    getRuleTypes: function () {
        var query = '';

        query += 'GetRuleTypes()';

        return query;
    },

    /**
     * @name predictExcelRangeMetadata
     * @param {string} fileLocation the location of the file
     * @param {string} sheetName name of the sheet to parse
     * @param {string} range the range to get metadata for
     * @desc get the metadata for the range specified in the sheet
     * @returns {string} the full query
     */
    predictExcelRangeMetadata: function (fileLocation, sheetName, range) {
        var query = '';

        query += 'PredictExcelRangeMetadata(filePath=["' + fileLocation + '"], sheetName=["' + sheetName + '"], sheetRange=["' + range + '"])';

        return query;
    },

    /**
     * @name extractAppMeta
     * @param {string} app name of app to store unique values
     * @param {boolean} descriptions true if called when user clicks optimize button instead of on upload
     * @param {string} appVar - the variable containing the app name
     * @desc used in optimizing apps
     * @returns {string} the query
     */
    extractAppMeta: function (app, descriptions, appVar) {
        var pixel = 'ExtractAppMeta( ';

        if (app) {
            pixel += 'app=[\"' + app + '\"]';
        } else if (appVar) {
            pixel += 'app=[' + appVar + ']';
        }

        if (descriptions) {
            pixel += ', descriptions = [ true ]';
        }

        pixel += ')';

        return pixel;
    },

    /**
     * @name frameCache
     * @param {boolean} bool whether to turn frame cache on or off
     * @desc turns the frame cache on/off
     * @returns {string} the query
     */
    frameCache: function (bool) {
        var pixel = 'FrameCache(';

        pixel += '"' + bool + '"';

        pixel += ')';

        return pixel;
    },

    /**
     * @name frameType
     * @desc checks the frame type
     * @returns {string} the query
     */
    frameType: function () {
        var pixel = 'FrameType()';

        return pixel;
    },

    /**
     * @name deleteInsightCache
     * @param {string} app the name of the app to delete cache from
     * @param {string} insightId the id of the insight to delete cache for
     * @desc delete the cache so it can recache when running
     * @returns {string} the query
     */
    deleteInsightCache: function (app, insightId) {
        var pixel = 'DeleteInsightCache(app=[\"' + app + '\"], id=[\"' + insightId + '\"])';

        return pixel;
    },

    /**
     * @name setInsightCacheable
     * @param {string} app the app id the insight is from
     * @param {string} id the insight id
     * @param {boolean} cache set cacheable to true/false
     * @desc turning on/off the insight cache
     * @returns {string} the query
     */
    setInsightCacheable: function (app, id, cache) {
        var pixel = '';

        pixel += 'SetInsightCacheable("' + app + '", "' + id + '", ' + cache + ')';

        return pixel;
    },

    /**
     * @name cacheNativeFrame
     * @param {string} frame the frame to cache
     * @param {string} frameType the frame type of the cache
     * @desc starts a separate thread to perform the query + swap to the cached frame
     * @returns {string} the query
     */
    cacheNativeFrame: function (frame, frameType) {
        var pixel = '';

        pixel += 'CacheNativeFrame(';

        pixel += 'frame=[' + frame + '],';
        pixel += 'frameType=["' + frameType + '"]';

        pixel += ')';

        return pixel;
    },

    /**
     * @name generateFrameFromRVariable
     * @param {string} rVariable the name of the variable to bring into frame
     * @desc bring the data from the r variable to the frame
     * @returns {string} the pixel to run
     */
    generateFrameFromRVariable: function (rVariable) {
        var pixel = 'GenerateFrameFromRVariable("' + rVariable + '")';

        return pixel;
    },

    /**
     * @name generateFrameFromPyVariable
     * @param {string} pyVariable the name of the variable to bring into frame
     * @desc bring the data from the py variable to the frame
     * @returns {string} the pixel to run
     */
    generateFrameFromPyVariable: function (pyVariable) {
        var pixel = 'GenerateFrameFromPyVariable("' + pyVariable + '")';

        return pixel;
    },

    /**
     * @name databaseColumnUnique
     * @param {string} database the app id
     * @param {array} columns list of columns to check
     * @desc checks to see if the columns passed in will yield a unique row
     * @returns {string} the pixel to run
     */
    databaseColumnUnique: function (database, columns) {
        var pixel = 'DatabaseColumnUnique(database=["' + database + '"], columns=' + JSON.stringify(columns) + ')';

        return pixel;
    },

    /**
     * @name getImageAsset
     * @param {string} filePath the the file path
     * @desc get the file image in base64 format
     * @returns {string} the pixel to run
     */
    getImageAsset: function (filePath) {
        var pixel = 'GetImageAsset(filePath=["';

        pixel += filePath;

        pixel += '"])';

        return pixel;
    },

    /** OWL **/
    /**
     * @name addOwlConcept
     * @param {string} app - app
     * @param {string} concept - concept
     * @param {string} column - column
     * @param {string} dataType - dataType
     * @param {string} additionalDataType - additionalDataTypes
     * @param {string} conceptual - conceptual
     * @param {string} description - description
     * @param {array} logicalNames - logicalNames
     * @param {string} appVar - the variable containing the app name
     * @desc add a concept to the owl
     * @returns {string} query
     */
    addOwlConcept: function (app, concept, column, dataType, additionalDataType, conceptual, description, logicalNames, appVar) {
        var query = '';

        query += 'AddOwlConcept(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', concept=[\"' + concept + '\"]';
        query += ', column=[\"' + column + '\"]';
        query += ', dataType=[\"' + dataType + '\"]';
        if (typeof additionalDataType === 'string') {
            query += ', additionalDataType=[\"' + additionalDataType + '\"]';
        } else {
            query += ', additionalDataType=[\'' + JSON.stringify(additionalDataType) + '\']';
        }
        query += ', conceptual=[\"' + conceptual + '\"]';
        if (description) {
            query += ', description=[\"' + description + '\"]';
        }
        if (logicalNames && logicalNames.length > 0) {
            query += ', logicalNames=' + JSON.stringify(logicalNames) + ' ';
        }

        query += ');';

        return query;
    },
    /**
     * @name removeOwlConcept
     * @param {string} app - app
     * @param {string} concept - concept
     * @param {string} appVar - the variable containing the app name
     * @desc remove a concept from the owl
     * @returns {string} query
     */
    removeOwlConcept: function (app, concept, appVar) {
        var query = '';

        query += 'RemoveOwlConcept(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', concept=[\"' + concept + '\"]';
        query += ');';

        return query;
    },
    /**
     * @name editOwlConceptDataType
     * @param {string} app - app
     * @param {string} concept - concept
     * @param {string} dataType - dataType
     * @param {string} additionalDataType - additionalDataTypes
     * @param {string} appVar - the variable containing the app name
     * @desc edit a concept's datatype in the owl
     * @returns {string} query
     */
    editOwlConceptDataType: function (app, concept, dataType, additionalDataType, appVar) {
        var query = '';

        query += 'EditOwlConceptDataType(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', concept=[\"' + concept + '\"]';
        query += ', dataType=[\"' + dataType + '\"]';
        if (typeof additionalDataType === 'string') {
            query += ', additionalDataType=[\"' + additionalDataType + '\"]';
        } else {
            query += ', additionalDataType=[\'' + JSON.stringify(additionalDataType) + '\']';
        }
        query += ');';

        return query;
    },
    /**
     * @name editOwlConceptConceptualName
     * @param {string} app - app
     * @param {string} concept - concept
     * @param {string} conceptual - conceptual
     * @param {string} appVar - the variable containing the app name
     * @desc edit a concept's conceptural name in the owl
     * @returns {string} query
     */
    editOwlConceptConceptualName: function (app, concept, conceptual, appVar) {
        var query = '';

        query += 'EditOwlConceptConceptualName(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', concept=[\"' + concept + '\"]';
        query += ', conceptual=[\"' + conceptual + '\"]';
        query += ');';

        return query;
    },
    /**
     * @name addOwlProperty
     * @param {string} app - app
     * @param {string} concept - concept
     * @param {string} column - column
     * @param {string} dataType - dataType
     * @param {string} additionalDataType - additionalDataTypes
     * @param {string} conceptual - conceptual
     * @param {string} description - description
     * @param {array} logicalNames - logicalNames
     * @param {string} appVar - the variable containing the app name
     * @desc add a property to the owl
     * @returns {string} query
     */
    addOwlProperty: function (app, concept, column, dataType, additionalDataType, conceptual, description, logicalNames, appVar) {
        var query = '';

        query += 'AddOwlProperty(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', concept=[\"' + concept + '\"]';
        query += ', column=[\"' + column + '\"]';
        query += ', dataType=[\"' + dataType + '\"]';
        if (typeof additionalDataType === 'string') {
            query += ', additionalDataType=[\"' + additionalDataType + '\"]';
        } else {
            query += ', additionalDataType=[\'' + JSON.stringify(additionalDataType) + '\']';
        }
        query += ', conceptual=[\"' + conceptual + '\"]';
        if (description) {
            query += ', description=[\"' + description + '\"]';
        }
        if (logicalNames && logicalNames.length > 0) {
            query += ', logicalNames=' + JSON.stringify(logicalNames) + ' ';
        }

        query += ');';

        return query;
    },
    /**
     * @name removeOwlProperty
     * @param {string} app - app
     * @param {string} concept - concept
     * @param {string} column - column
     * @param {string} appVar - the variable containing the app name
     * @desc remove a property from the owl
     * @returns {string} query
     */
    removeOwlProperty: function (app, concept, column, appVar) {
        var query = '';

        query += 'RemoveOwlProperty(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', concept=[\"' + concept + '\"]';
        query += ', column=[\"' + column + '\"]';
        query += ');';

        return query;
    },

    /**
     * @name editOwlPropertyDataType
     * @param {string} app - app
     * @param {string} concept - concept
     * @param {string} column - column
     * @param {string} dataType - dataType
     * @param {string} additionalDataType - additionalDataTypes
     * @param {string} appVar - the variable containing the app name
     * @desc edit a property's datatype in the owl
     * @returns {string} query
     */
    editOwlPropertyDataType: function (app, concept, column, dataType, additionalDataType, appVar) {
        var query = '';

        query += 'EditOwlPropertyDataType(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', concept=[\"' + concept + '\"]';
        query += ', column=[\"' + column + '\"]';
        query += ', dataType=[\"' + dataType + '\"]';
        if (typeof additionalDataType === 'string') {
            query += ', additionalDataType=[\"' + additionalDataType + '\"]';
        } else {
            query += ', additionalDataType=[\'' + JSON.stringify(additionalDataType) + '\']';
        }
        query += ');';

        return query;
    },
    /**
     * @name editOwlPropertyConceptualName
     * @param {string} app - app
     * @param {string} concept - concept
     * @param {string} column - column
     * @param {string} conceptual - conceptual
     * @param {string} appVar - the variable containing the app name
     * @desc edit a property's conceptural name in the owl
     * @returns {string} query
     */
    editOwlPropertyConceptualName: function (app, concept, column, conceptual, appVar) {
        var query = '';

        query += 'EditOwlPropertyConceptualName(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', concept=[\"' + concept + '\"]';
        query += ', column=[\"' + column + '\"]';
        query += ', conceptual=[\"' + conceptual + '\"]';
        query += ');';

        return query;
    },
    /**
     * @name predictOwlDescription
     * @param {string} app - app
     * @param {string} concept - concept
     * @param {string} column - column
     * @param {string} appVar - the variable containing the app name
     * @desc predict descriptions in the owl
     * @returns {string} query
     */
    predictOwlDescription: function (app, concept, column, appVar) {
        var query = '';

        query += 'PredictOwlDescription(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', concept=[\"' + concept + '\"]';
        if (column) {
            query += ', column=[\"' + column + '\"]';
        }
        query += ');';

        return query;
    },
    /**
     * @name editOwlDescription
     * @param {string} app - app
     * @param {string} concept - concept
     * @param {string} column - column
     * @param {string} description - description
     * @param {string} appVar - the variable containing the app name
     * @desc edit descriptions in the owl
     * @returns {string} query
     */
    editOwlDescription: function (app, concept, column, description, appVar) {
        var query = '';

        query += 'EditOwlDescription(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', concept=[\"' + concept + '\"]';
        if (column) {
            query += ', column=[\"' + column + '\"]';
        }
        if (description) {
            query += ', description=[\"' + description + '\"]';
        }
        query += ');';

        return query;
    },
    /**
     * @name predictOwlLogicalNames
     * @param {string} app - app
     * @param {string} concept - concept
     * @param {string} column - column
     * @param {string} appVar - the variable containing the app name
     * @desc predict logical names in the owl
     * @returns {string} query
     */
    predictOwlLogicalNames: function (app, concept, column, appVar) {
        var query = '';

        query += 'PredictOwlLogicalNames(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', concept=[\"' + concept + '\"]';
        if (column) {
            query += ', column=[\"' + column + '\"]';
        }
        query += ');';

        return query;
    },
    /**
     * @name editOwlLogicalNames
     * @param {string} app - app
     * @param {string} concept - concept
     * @param {string} column - column
     * @param {array} logicalNames - logicalNames
     * @param {string} appVar - the variable containing the app name
     * @desc edit the logical names in the owl
     * @returns {string} query
     */
    editOwlLogicalNames: function (app, concept, column, logicalNames, appVar) {
        var query = '';

        query += 'EditOwlLogicalNames(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', concept=[\"' + concept + '\"]';
        if (column) {
            query += ', column=[\"' + column + '\"]';
        }
        if (logicalNames && logicalNames.length > 0) {
            query += ', logicalNames=' + JSON.stringify(logicalNames) + ' ';
        }

        query += ');';

        return query;
    },
    /**
     * @name addOwlRelationship
     * @param {string} app - app name
     * @param {array} startT - array of start table names
     * @param {array} startC - array of start column names
     * @param {array} endT - array of end table names
     * @param {array} endC - array of end column names
     * @param {string} store - where to store the results
     * @param {string} appVar - the variable containing the app name
     * @desc add a relationship to the owl
     * @returns {string} query
     */
    addOwlRelationship: function (app, startT, startC, endT, endC, store, appVar) {
        var query = '';

        query += 'AddOwlRelationship(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', startT=' + JSON.stringify(startT) + ' ';
        query += ', startC=' + JSON.stringify(startC) + ' ';
        query += ', endT=' + JSON.stringify(endT) + ' ';
        query += ', endC=' + JSON.stringify(endC) + ' ';
        if (store) {
            query += ', store=[' + store + ']';
        }
        query += ');';

        return query;
    },
    /**
     * @name addBulkOwlRelationships
     * @param {string} app - app name
     * @param {string} frame - frame to add from
     * @param {number} propagation - propagation threshold
     * @param {string} store - where to store the results
     * @param {string} appVar - the variable containing the app name
     * @desc add a relationship to the owl
     * @returns {string} query
     */
    addBulkOwlRelationships: function (app, frame, propagation, store, appVar) {
        var query = '';

        query += 'AddBulkOwlRelationships(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', frame=[' + frame + ']';
        query += ', propagation=[' + propagation + ']';
        if (store) {
            query += ', store=[' + store + ']';
        }
        query += ');';

        return query;
    },
    /**
     * @name removeOwlRelationship
     * @param {string} app - app name
     * @param {array} startT - array of start table names
     * @param {array} startC - array of start column names
     * @param {array} endT - array of end table names
     * @param {array} endC - array of end column names
     * @param {string} store - where to store the results
     * @param {string} appVar - the variable containing the app name
     * @desc add a relationship to the owl
     * @returns {string} query
     */
    removeOwlRelationship: function (app, startT, startC, endT, endC, store, appVar) {
        var query = '';

        query += 'RemoveOwlRelationship(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ', startT=' + JSON.stringify(startT) + ' ';
        query += ', startC=' + JSON.stringify(startC) + ' ';
        query += ', endT=' + JSON.stringify(endT) + ' ';
        query += ', endC=' + JSON.stringify(endC) + ' ';
        if (store) {
            query += ', store=[' + store + ']';
        }
        query += ');';

        return query;
    },
    /**
     * @name saveOwlPositions
     * @param {string} app - app
     * @param {string} positions - map of the positions {table to x,y}
     * @param {string} appVar - the variable containing the app name
     * @desc edit the logical names in the owl
     * @returns {string} query
     */
    saveOwlPositions: function (app, positions, appVar) {
        var query = '';

        query += 'SaveOwlPositions (';
        if (appVar) {
            query += 'app=[' + appVar + '], ';
        } else {
            query += 'app=["' + app + '"], ';
        }
        query += 'positionMap=[' + JSON.stringify(positions) + '] ';

        query += ');';

        return query;
    },
    /**
     * @name syncAppWithLocalMaster
     * @param {string} app - app name
     * @param {string} appVar - the variable containing the app name
     * @desc sync the owl to local master
     * @returns {string} query
     */
    syncAppWithLocalMaster: function (app, appVar) {
        var query = '';

        query += 'SyncAppWithLocalMaster(';
        if (appVar) {
            query += 'app=[' + appVar + ']';
        } else {
            query += 'app=["' + app + '"]';
        }
        query += ');';

        return query;
    },
    /**
     * @name runDataQuality
     * @param {string} ruleId - rule identifier
     * @param {string} columns - column to run the rule on
     * @param {array} options - specified formatting options for the rule
     * @param {string} inputTable - R data frame
     * @param {array} newFrame - whether or not to create a new frame
     * @desc run data quality pixel
     * @returns {string} query
     */
    runDataQuality: function (ruleId, columns, options, inputTable, newFrame) {
        var query = '',
            i;

        query += 'RunDataQuality ( rule = [ "' + ruleId + '" ] , ';
        query += ' column = [ "' + columns + '" ] , ';

        query += 'options = [ ';
        if (options && options.length > 0) {
            for (i = 0; i < options.length; i++) {
                query += '"' + options[i] + '"';
                query += ', ';
            }

            // trim trailing comma
            query = this.trim(query, ',');
        } else {
            query += '""';
        }
        query += ' ] , ';

        if (newFrame) {
            query += 'inputTable = ["' + inputTable + '"] ) ;';
        } else {
            query += 'inputTable = [' + inputTable + ' ] ) ;';
        }

        return query;
    },

    /**
     * @name runDocumentSummarization
     * @param {string} fileOrigin - input method
     * @param {string} userInput - document or text to summarize
     * @param {array} numSentences - max number of sentences to include in the summary
     * @param {string} numTopics - max number of topics to include in the summary
     * @param {array} numTopicTerms - max number of keywords to include in the summary
     * @desc run document summarization pixel
     * @returns {string} query
     */
    runDocumentSummarization: function (fileOrigin, userInput, numSentences, numTopics, numTopicTerms) {
        var query = '';

        query += 'RunDocumentSummarization ( fileOrigin = [ "' + fileOrigin + '" ] , ';
        query += ' userInput = [ "' + userInput + '" ] , ';
        query += ' numSentences = [ "' + numSentences + '" ] , ';
        query += ' numTopics = [ "' + numTopics + '" ] , ';
        query += 'numTopicTerms = [ "' + numTopicTerms + '" ] ) ;';

        return query;
    },

    /**
     * @name dateAddValue
     * @desc creates param string for add date value pixel FOR PIPELINE
     * @param {string} srcCol values for dateAddValue
     * @param {string} newCol the new column
     * @param {string} unit unit
     * @param {string} valToAdd value to add
     * @return {string} query
     */
    dateAddValue: function (srcCol, newCol, unit, valToAdd) {
        var query = 'DateAddValue( ';

        query += 'column = [ "' + srcCol + '"], ';
        query += 'new_col = ["' + newCol + '"], ';
        query += 'unit = ["' + unit + '"], ';
        query += 'val_to_add = [' + valToAdd + '] )';

        return query;
    },

    /**
     * @name DATE_ADD_VALUE
     * @desc creates param string for add date value pixel FOR PIPELINE
     * @param {object} dateAddValue values for dateAddValue
     * @return {string} query
     */
    DATE_ADD_VALUE: function (dateAddValue) {
        var query = '';

        query += 'column = [ "' + dateAddValue.srcCol + '"], ';
        query += 'new_col = ["' + dateAddValue.new_col + '"], ';
        query += 'unit = ["' + dateAddValue.unit + '"], ';
        query += 'val_to_add = [' + dateAddValue.val_to_add + ']  ';

        return query;
    },

    /**
     * @name DateDifference
     * @desc creates the param string for date difference pixel FOR PIPELINE
     * @param {string} startColumn start column
     * @param {string} endColumn end column
     * @param {string} inputUse input
     * @param {string} inputDate date
     * @param {string} unit unit
     * @param {string} newCol new column
     * @return {string} query
     */
    dateDifference: function (startColumn, endColumn, inputUse, inputDate, unit, newCol) {
        var query = 'DateDifference( ';

        query += 'start_column = [ "' + startColumn + '"], ';
        query += 'end_column = ["' + endColumn + '"], ';
        query += 'input_use = ["' + inputUse + '"], ';
        query += 'input_date = ["' + inputDate + '"], ';
        query += 'unit = ["' + unit + '"], ';
        query += 'newCol = ["' + newCol + '"] )';

        return query;
    },

    /**
     * @name DATE_DIFFERENCE
     * @desc creates the param string for date difference pixel FOR PIPELINE
     * @param {object} dateDifference -
     * @return {string} query
     */
    DATE_DIFFERENCE: function (dateDifference) {
        var query = '';

        query += 'start_column = [ "' + dateDifference.start_column + '"], ';
        query += 'end_column = ["' + dateDifference.end_column + '"], ';
        query += 'input_use = ["' + dateDifference.input_use + '"], ';
        query += 'input_date = ["' + dateDifference.input_date + '"], ';
        query += 'unit = ["' + dateDifference.unit + '"], ';
        query += 'newCol = ["' + dateDifference.newCol + '"] ';

        return query;
    },

    /** NEW */
    /**
     * @name PIXEL
     * @param {string} query - raw query
     * @returns {string} query
     */
    PIXEL: function (query) {
        return query;
    },
    /**
     * @name QUERY_STRUCT
     * @param {object} qs - the query struct
     * @desc generate pixel from query struct
     * @returns {string} query
     */
    QUERY_STRUCT: function (qs) {
        var pixel = '',
            tempFilter = {},
            tempColumn,
            filterIdx,
            selectorComponents = [],
            joinComponents = [],
            groupComponents = [],
            pixelComponents = [],
            selectorIdx,
            componentIdx,
            relationIdx,
            groupIdx;

        console.warn('CONSRUCT PIXEL COMPONENTS ON THE FLY');

        // start building with what we have
        if (qs.qsType === 'ENGINE') {
            pixelComponents.push({
                type: 'database',
                components: [
                    qs.engineName
                ]
            });
        } else if (qs.qsType === 'RAW_JDBC_ENGINE_QUERY') {
            pixelComponents.push({
                type: 'jdbcSource',
                components: [
                    qs.config.dbDriver,
                    qs.config.connectionString,
                    qs.config.username,
                    qs.config.password
                ]
            });
        } else if (qs.qsType === 'RAW_ENGINE_QUERY') {
            pixelComponents.push({
                type: 'database',
                components: [
                    qs.engineName
                ]
            });
        } else if (qs.qsType === 'CSV_FILE') {
            pixelComponents.push({
                type: 'fileRead',
                components: [
                    'csv',
                    qs.filePath,
                    '',
                    '',
                    qs.columnTypes,
                    qs.delimiter,
                    qs.newHeaderNames,
                    qs.fileName,
                    qs.additionalTypes
                ]
            });
        } else if (qs.qsType === 'EXCEL_FILE') {
            pixelComponents.push({
                type: 'fileRead',
                components: [
                    'excel',
                    qs.filePath,
                    qs.sheetName,
                    qs.sheetRange,
                    qs.columnTypes,
                    qs.delimiter,
                    qs.newHeaderNames,
                    qs.fileName,
                    qs.additionalTypes
                ]
            });
        } else if (qs.qsType === 'FRAME') {
            pixelComponents.push({
                type: 'frame',
                components: [
                    qs.frameName
                ]
            });
        } else if (qs.qsType === 'API') {
            console.error('TODO: Standardize with the backend');
            pixelComponents.push({
                type: 'api',
                components: [
                    qs.type,
                    qs.url,
                    qs.method,
                    qs.headers,
                    qs.body,
                    qs.roots
                ]
            });
        } else if (qs.qsType === 'SOCIAL') {
            // console.error('TODO: Standardize with the backend');
            if (qs.provider === 'google') {
                pixelComponents.push({
                    type: 'googleFileRetriever',
                    components: [
                        qs.fileId,
                        qs.fileType
                    ]
                });
            } else if (qs.provider === 'dropbox') {
                pixelComponents.push({
                    type: 'dropBoxFileRetriever',
                    components: [
                        qs.filePath
                    ]
                });
            } else if (qs.provider === 'ms') {
                pixelComponents.push({
                    type: 'oneDriveFileRetriever',
                    components: [
                        qs.fileId
                    ]
                });
            }
            // pixelComponents.push({
            //     type: 'social',
            //     components: [
            //         qs.fileName,
            //         qs.fileId,
            //         qs.fileType,
            //         qs.provider
            //     ]
            // });
        }

        if (qs.query) {
            pixelComponents.push({
                type: 'query',
                components: [
                    qs.query
                ]
            });
        } else {
            if (qs.selectors && qs.selectors.length > 0) {
                for (selectorIdx = 0; selectorIdx < qs.selectors.length; selectorIdx++) {
                    // the prim key is referred to as 'PRIM_KEY_PLACEHOLDER' as the column...
                    if (qs.selectors[selectorIdx].content.column === 'PRIM_KEY_PLACEHOLDER') {
                        // todo add column
                        // qs.selectors[selectorIdx].content.table + '__' + qs.selectors[selectorIdx].content.table
                        selectorComponents.push({
                            selector: qs.selectors[selectorIdx].content.table,
                            alias: qs.selectors[selectorIdx].content.alias
                        });
                    } else if (qs.selectors[selectorIdx].content.math && qs.selectors[selectorIdx].content.calculatedBy) {
                        selectorComponents.push({
                            math: qs.selectors[selectorIdx].content.math,
                            calculatedBy: qs.selectors[selectorIdx].content.calculatedBy
                        });
                    } else {
                        let selectorComponent = {
                            alias: qs.selectors[selectorIdx].content.alias
                        };
                        // in graph frame we just want to use alias cant use frame header
                        if (qs.selectors[selectorIdx].content.table) {
                            selectorComponent.selector = qs.selectors[selectorIdx].content.table + '__' + qs.selectors[selectorIdx].content.column;
                        }

                        selectorComponents.push(selectorComponent);
                    }
                }

                pixelComponents.push({
                    type: 'select2',
                    components: [
                        selectorComponents
                    ]
                });
            }

            if (qs.relations && qs.relations.length > 0) {
                for (relationIdx = 0; relationIdx < qs.relations.length; relationIdx++) {
                    joinComponents.push({
                        fromColumn: qs.relations[relationIdx][0],
                        joinType: qs.relations[relationIdx][1],
                        toColumn: qs.relations[relationIdx][2]
                    });
                }

                pixelComponents.push({
                    type: 'join',
                    components: [
                        joinComponents
                    ]
                });
            }

            if (qs.groups && qs.groups.length > 0) {
                for (groupIdx = 0; groupIdx < qs.groups.length; groupIdx++) {
                    if (qs.groups[groupIdx].column === 'PRIM_KEY_PLACEHOLDER') {
                        groupComponents.push(qs.groups[groupIdx].table + '__' + qs.groups[groupIdx].table);
                    } else {
                        groupComponents.push(qs.groups[groupIdx].table + '__' + qs.groups[groupIdx].column);
                    }
                }

                pixelComponents.push({
                    type: 'group',
                    components: [
                        groupComponents
                    ]
                });
            }

            // TODO: fix the filter to properly use left/right
            // filter
            if (qs.explicitFilters && qs.explicitFilters.length > 0) {
                // qs for filter is structured oddly so we need to increment by 2 for a filter
                for (filterIdx = 1; filterIdx < qs.explicitFilters.length; filterIdx += 2) {
                    if (qs.explicitFilters[filterIdx].left.pixelType === 'COLUMN') {
                        tempColumn = qs.explicitFilters[filterIdx].left.value[0].content.table + '__' + qs.explicitFilters[filterIdx].left.value[0].content.column;
                    } else if (qs.explicitFilters[filterIdx].left.pixelType === 'CONSTANT') {
                        tempColumn = qs.explicitFilters[filterIdx].left.value;
                    } else if (qs.explicitFilters[filterIdx].left.pixelType === 'CONST_STRING') {
                        tempColumn = qs.explicitFilters[filterIdx].left.value;
                    }

                    tempFilter = {};

                    if (
                        !qs.explicitFilters[filterIdx].right.pixelType ||
                        qs.explicitFilters[filterIdx].right.pixelType === 'CONSTANT' ||
                        qs.explicitFilters[filterIdx].right.pixelType === 'CONST_STRING'
                    ) {
                        // {MOVIE_DATA__Director: {comparator: '==', value: ['Adam_McKay']}}
                        tempFilter[tempColumn] = {
                            comparator: qs.explicitFilters[filterIdx].comparator,
                            value: qs.explicitFilters[filterIdx].right.value
                        };
                    } else if (qs.explicitFilters[filterIdx].right.pixelType === 'LAMBDA') {
                        tempFilter[tempColumn] = {
                            comparator: qs.explicitFilters[filterIdx].comparator,
                            value: `(${qs.explicitFilters[filterIdx].right.value})`,
                            isFilterString: true
                        };
                    }

                    // this will allow you to have multiple filters
                    if (Object.keys(tempFilter).length !== 0) {
                        pixelComponents.push({
                            type: 'filter',
                            components: [
                                tempFilter
                            ]
                        });
                    }
                }
            }
        }

        if (qs.limit > -1) {
            pixelComponents.push({
                type: 'limit',
                components: [qs.limit]
            });
        }

        for (componentIdx = 0; componentIdx < pixelComponents.length; componentIdx++) {
            pixel += this.REACTORS[pixelComponents[componentIdx].type].apply(this, pixelComponents[componentIdx].components);
            pixel = this.append(pixel, this.ENUMS.PIPE);
        }

        return pixel;
    },
    /**
     * @name CREATE_FRAME
     * @param {object} frame - the frame object
     * @desc generate pixel from a frame
     * @returns {string} query
     */
    CREATE_FRAME: function (frame) {
        var frameType = frame.type,
            override = false;

        // TODO: remove when backend can set the default
        // if (frameType === 'default') {
        //     frameType = 'GRID';
        // }

        if (frame.override) {
            override = true;
        }

        return `CreateFrame(frameType=[${frameType}], override=[${override}]).as(["${frame.name}"])`;
    },
    /**
     * @name JOINS
     * @param {array} joins - array of the join information
     * @desc create a joins component
     * @returns {string} query
     */
    JOINS: function (joins) {
        var query = '',
            i,
            len;


        if (joins) {
            for (i = 0, len = joins.length; i < len; i++) {
                // ORDER IS ON PURPOSE DO NOT CHANGE
                query += '(';
                query += joins[i].from + ', '; // Join(Title
                query += this.convertJoin(joins[i].join) + ', '; // Join(Title inner.join
                query += joins[i].to; // Join(Title inner.join Studio
                query += '), ';
            }
            // trim trailing comma
            query = this.trim(query, ',');
        }

        return query;
    },
    /**
     * @name RDF_FILE_SOURCE
     * @param {object} file - path and format
     * @return {string} query
     */
    RDF_FILE_SOURCE: function (file) {
        return `RDFFileSource(filePath=["${file.path}"], rdfType=["${file.format}"])`;
    },
    /**
     * @name FILTERS
     * @param {array} filters - array of the filter information
     * @desc create a joins component
     * @returns {string} query
     */
    FILTERS: function (filters) {
        var query = '',
            filterIdx,
            filterLen;

        for (filterIdx = 0, filterLen = filters.length; filterIdx < filterLen; filterIdx++) {
            if (filters[filterIdx].type === 'value') {
                query += `( ${filters[filterIdx].alias} ${filters[filterIdx].comparator} ${JSON.stringify(filters[filterIdx].values)})`;
            } else if (filters[filterIdx].type === 'range') {
                query += `( ${ filters[filterIdx].values[0]} <= ${filters[filterIdx].alias}), ( ${filters[filterIdx].alias} <= ${filters[filterIdx].values[1]})`;
            } else if (filters[filterIdx].type === 'delete') {
                query += `( ${filters[filterIdx].values} )`;
            } else {
                console.warn('Unrecognized type');
            }

            if (filters[filterIdx].operator) {
                query += ' ' + filters[filterIdx].operator + ' ';
            }
        }

        return query;
    },
    /**
     * @name UPDATE_ROW
     * @param {object} updateRow - values for update row
     * @desc creates param string for update row pixel
     * @return {string} query
     */
    UPDATE_ROW: function (updateRow) {
        var query = '',
            value;
        if (isNaN(updateRow.newVal)) {
            value = `"${updateRow.newVal}"`;
        } else {
            value = updateRow.newVal;
        }
        query += `${updateRow.newCol}, ${value}`;

        return query;
    },
    /**
     * @name RENAME_COLUMN
     * @param {object} renameCol - values for rename col
     * @desc creates param string for rename col pixel
     * @return {string} query
     */
    RENAME_COLUMN: function (renameCol) {
        var query = '';

        query += `column=["${renameCol.selected}"], newCol=["${renameCol.new}", matchesTable=["${renameCol.matchTable}"]]`;

        return query;
    },
    /**
     * @name COLUMN_CLEANER
     * @param {object} colCleaner - values for column cleaner
     * @desc creates param string for column cleaner pixel
     * @return {string} query
     */
    COLUMN_CLEANER: function (colCleaner) {
        var query = '';

        query += `column=["${colCleaner.col}"], matches=[${colCleaner.matches}], matchesTable=["${colCleaner.matchTable}"]`;

        return query;
    }
};

module.exports = REACTORS;
