import Utility from '../../utility/utility.js';
import angular from 'angular';

export default angular.module('app.external.service', [])
    .factory('externalService', externalService);

externalService.$inject = ['$stateParams', 'workbookService', 'messageService', 'storeService', 'widgetService'];

function externalService($stateParams, workbookService: WorkbookService, messageService: MessageService, storeService: StoreService) {
    const
        /** Public */
        /** Private */
        _state = {},
        _actions = {
        };


    /**
     * @name panelFilter
     * @param filters {}
     * @desc construct the pixel to filter the panel
     * @returns {void}
     */
    function panelFilter(filters: any): void {
        let finalPixelComponents: any = [],
            refreshComponent: any = [],
            panelFilterComponent: any = [],
            innerIfTrue: any = [],
            innerIfFalse: any = [],
            filter: string,
            insightId: string = $stateParams.insight,
            workbook: any = workbookService.getWorkbook(insightId),
            worksheet: any = workbookService.getWorksheet(insightId, workbook.worksheet),
            panelId: string = worksheet.panel,
            widgetId: string = worksheet.panels[worksheet.panel].widgetId,
            cleanedRefreshPixel: string = storeService.generate(
                'refresh',
                {
                    widgetIds: [widgetId]
                }
            );

        cleanedRefreshPixel = cleanedRefreshPixel.replace(/;META\|/g, ',')

        // set up the set panel filter component
        for (filter in filters) {
            if (filters.hasOwnProperty(filter)) {
                panelFilterComponent.push({
                    type: 'panel',
                    components: [panelId],
                    terminal: false
                });
                panelFilterComponent.push({
                    type: 'setPanelFilter',
                    components: [
                        [
                            {
                                type: 'value',
                                alias: filter,
                                comparator: '==',
                                values: filters[filter]
                            }
                        ]
                    ],

                    terminal: true
                });
            }
        }

        // set up the refresh component
        refreshComponent.push({
            type: 'Pixel',
            components: [cleanedRefreshPixel],
            terminal: true
        });

        // add refresh to the panel filter component if we have panel filters to apply
        if (panelFilterComponent.length > 0) {
            panelFilterComponent = panelFilterComponent.concat(refreshComponent);
        }

        // set up the if statement: if there are panel filters, we will run the panel filter pixel along with the refresh
        // if there are no panel filters, we will just run a refresh
        innerIfTrue = [
            {
                type: 'if',
                components: [
                    [
                        {
                            type: 'Pixel',
                            components: [panelFilterComponent.length + ' > 0'],
                            terminal: true
                        }
                    ], // condition
                    panelFilterComponent, // true
                    refreshComponent //false
                ],
                terminal: true
            }
        ];

        innerIfFalse = [
            {
                type: 'if',
                components: [
                    [
                        {
                            type: 'Pixel',
                            components: [panelFilterComponent.length + ' > 0'],
                            terminal: true
                        }
                    ], // condition
                    panelFilterComponent, // true
                    [] //false
                ],
                terminal: true
            }
        ];

        // finally we wrap another if condition for handling unfilter panel.
        // if the panel has been unfiltered, we run the inner if statement to check filter components and then running the refresh
        finalPixelComponents = [{
            type: 'if',
            components: [
                [
                    {
                        type: 'panel',
                        components: [panelId],
                        terminal: false
                    }, {
                        type: 'unfilterPanel',
                        components: [],
                        terminal: true
                    }
                ], // condition
                innerIfTrue, // true
                innerIfFalse //false
            ],
            terminal: true
        }];

        messageService.emit('execute-pixel', {
            insightID: insightId,
            commandList: finalPixelComponents
        });
    }

    /**
     * @name processExternalMessage
     * @desc process the message coming from external window
     * @returns {void}
     */
    function processExternalMessage(message: any): void {
        let messageObject: any;
        try {
            messageObject = JSON.parse(message.data)
        } catch (e) {
            // not a valid message we're expecting so don't process it
            return;
        }

        // make sure we are working with an object
        if (typeof messageObject === 'object') {
            // check the message to see what we need to do
            if (messageObject.message === 'semoss-panel-filter') {
                panelFilter(messageObject.data);
            }
        }
    }

    /**
     * @name initialize
     * @desc called when the module is loaded
     */
    function initialize(): void {
        // register the mode to force conformity
        for (let a in _actions) {
            if (_actions.hasOwnProperty(a)) {
                messageService.on(a, _actions[a]);
            }
        }

        window.addEventListener('message', processExternalMessage, false);
    }

    /**
     * @name get
     * @param widgetId - the widget id to grab
     * @param accessor - string to get to the object. In the form of 'a.b.c'
     * @desc function that gets data from the store
     * @returns value of the requested object
     */
    function get(widgetId: string, accessor?: string): any {
        return Utility.getter(_state[widgetId], accessor);
    }

    return {
        initialize: initialize
    };
}
