import Utility from '../../utility/utility.js';
import variables from '@/style/src/variables.scss';
import angular from 'angular';
import GetThemesReturn from './GetThemesReturn';
import Theme from './Theme';

export default angular.module('app.admin.service', [])
    .factory('adminService', adminService);

adminService.$inject = ['messageService', 'monolithService', 'SHARED_TOOLS'];

function adminService(messageService: MessageService, monolithService: MonolithService, SHARED_TOOLS: any) {
    const
        /** Public */
        /** Private */
        BASE_THEME: Theme = {
            id: 'BASE',
            name: 'Default',
            theme: {
                homeName: 'SEMOSS',
                menuColor: variables.defaultMenu,
                menuDarkColor: variables.defaultMenuDark,
                primaryColor: variables.defaultPrimary,
                primaryLightColor: variables.defaultPrimaryLight,
                primaryDarkColor: variables.defaultPrimaryDark,
                backgroundColor: variables.defaultBackground,
                backgroundImage: '',
                backgroundImageOpacity: 0.95,
                homeImage: '',
                visualizationBackgroundColor: '#FFFFFF',
                visualizationColorPalette: 'Semoss',
                highlightColor: variables.defaultHighlight
            }
        },
        _state = {
            base: BASE_THEME,
            active: BASE_THEME,
            list: [BASE_THEME]
        },
        _actions = {
            /**
             * @name updated-theme
             * @desc set the view based on the active them
             */
            'updated-theme': (): void => {
                // don't need to set the theme
                if (!_state.active || _state.active.id === _state.base.id) {
                    document.documentElement.style.setProperty('--color-nav', '');
                    document.documentElement.style.setProperty('--color-nav--hover', '');
                    document.documentElement.style.setProperty('--color-primary', '');
                    document.documentElement.style.setProperty('--color-primary--background', '');
                    document.documentElement.style.setProperty('--color-primary--hover', '');
                    document.documentElement.style.setProperty('--color-background', '');
                    document.documentElement.style.setProperty('--color-highlight', '');

                    return;
                }

                if (_state.active.theme.hasOwnProperty('menuColor')) {
                    document.documentElement.style.setProperty('--color-nav', _state.active.theme.menuColor, 'important');
                }

                if (_state.active.theme.hasOwnProperty('menuDarkColor')) {
                    document.documentElement.style.setProperty('--color-nav--hover', _state.active.theme.menuDarkColor, 'important');
                }

                if (_state.active.theme.hasOwnProperty('highlightColor')) {
                    document.documentElement.style.setProperty('--color-highlight', _state.active.theme.highlightColor, 'important');
                } else {
                    document.documentElement.style.setProperty('--color-highlight', _state.active.theme.primaryColor, 'important');
                }

                if (_state.active.theme.hasOwnProperty('primaryColor')) {
                    document.documentElement.style.setProperty('--color-primary', _state.active.theme.primaryColor, 'important');
                }

                if (_state.active.theme.hasOwnProperty('primaryLightColor')) {
                    document.documentElement.style.setProperty('--color-primary--background', _state.active.theme.primaryLightColor, 'important');
                }

                if (_state.active.theme.hasOwnProperty('primaryDarkColor')) {
                    document.documentElement.style.setProperty('--color-primary--hover', _state.active.theme.primaryDarkColor, 'important');
                }

                if (_state.active.theme.hasOwnProperty('backgroundColor')) {
                    document.documentElement.style.setProperty('--color-background', _state.active.theme.backgroundColor, 'important');
                }

                if (_state.active.theme.hasOwnProperty('visualizationColorPalette')) {
                    SHARED_TOOLS.colorName = _state.active.theme.visualizationColorPalette;
                }
            },
            /**
             * @name set-theme
             * @param payload
             * @desc get a list of all of the available themes (as well as set the active)
             */
            'set-theme': (payload: { id: string, name: string, theme?: any }): void => {
                _state.active = {
                    id: payload.id,
                    name: payload.name,
                    theme: payload.theme || {}
                };

                messageService.emit('updated-theme');
            },
            /**
             * @name get-theme
             * @desc get a list of all of the available themes (as well as set the active)
             */
            'get-theme': (): void => {
                monolithService
                    .getAdminThemes()
                    .then(
                        (output: GetThemesReturn[]): void => {
                            // use the base if it is not present
                            _state.list = [BASE_THEME];
                            _state.active = BASE_THEME;

                            for (let outputIdx = 0, outputLen = output.length; outputIdx < outputLen; outputIdx++) {
                                _state.list.push({
                                    id: output[outputIdx].ID,
                                    theme: JSON.parse(output[outputIdx].THEME_MAP),
                                    name: output[outputIdx].THEME_NAME
                                });

                                if (output[outputIdx].IS_ACTIVE) {
                                    _state.active = _state.list[_state.list.length - 1];
                                }
                            }

                            messageService.emit('updated-theme');
                        }
                    );
            },
            /**
             * @name create-theme
             * @param payload - info to create theme
             * @desc create a new theme
             */
            'create-theme': (payload: { name: string, theme: Theme }): void => {
                monolithService.createAdminTheme({
                    name: payload.name,
                    json: payload.theme,
                    isActive: true
                })
                    .then(
                        (response: any): void => {
                            if (response) {
                                messageService.emit('get-theme');
                            } else {
                                messageService.emit('alert', {
                                    color: 'error',
                                    text: 'Error saving theme'
                                });
                            }
                        },
                        (error: string): void => {
                            messageService.emit('alert', {
                                color: 'error',
                                text: error
                            });
                        }
                    );
            },
            /**
             * @name edit-theme
             * @param payload- theme data
             * @desc modify an existing theme
             */
            'edit-theme': (payload: { id: string, name: string, theme: Theme }): void => {
                monolithService.editAdminTheme({
                    name: payload.name,
                    json: payload.theme,
                    id: payload.id,
                    isActive: true
                })
                    .then(
                        (response: any): void => {
                            if (response) {
                                messageService.emit('get-theme');
                            } else {
                                messageService.emit('alert', {
                                    color: 'error',
                                    text: 'Error saving theme'
                                });
                            }
                        },
                        (error: string): void => {
                            messageService.emit('alert', {
                                color: 'error',
                                text: error
                            });
                        }
                    );
            },
            /**
             * @name delete-theme
             * @param payload - theme data
             * @desc delete an existing theme
             */
            'delete-theme': (payload: Theme): void => {
                monolithService.deleteAdminTheme({
                    id: payload.id
                }).then(function (response) {
                    if (response) {
                        messageService.emit('get-theme');
                    } else {
                        messageService.emit('alert', {
                            color: 'error',
                            text: 'Error saving theme'
                        });
                    }
                }, function (error) {
                    messageService.emit('alert', {
                        color: 'error',
                        text: error
                    });
                });
            },
            /**
             * @name select-theme
             * @param  payload - theme data
             * @desc select an existing theme
             */
            'select-theme': (payload: Theme): void => {
                monolithService.setActiveAdminTheme({
                    id: payload.id
                })
                    .then(
                        (response: any): void => {
                            if (response) {
                                let option;
                                for (let listIdx = 0, listLen = _state.list.length; listIdx < listLen; listIdx++) {
                                    if (_state.list[listIdx].id === payload.id) {
                                        option = _state.list[listIdx];
                                        break;
                                    }
                                }

                                if (!option) {
                                    messageService.emit('get-theme');
                                    return;
                                }

                                messageService.emit('set-theme', option);
                            } else {
                                messageService.emit('alert', {
                                    color: 'error',
                                    text: 'Error saving theme'
                                });
                            }
                        },
                        (error: string): void => {
                            messageService.emit('alert', {
                                color: 'error',
                                text: error
                            });
                        }
                    );
            },
            /**
             * @name reset-theme
             * @param payload - theme data
             * @desc reset the themes
             */
            'reset-theme': (payload: Theme): void => {
                monolithService.setAllAdminThemesInactive()
                    .then(
                        (response: any): void => {
                            if (response) {
                                messageService.emit('set-theme', _state.base);
                            } else {
                                messageService.emit('alert', {
                                    color: 'error',
                                    text: 'Error saving theme'
                                });
                            }
                        },
                        (error: string): void => {
                            messageService.emit('alert', {
                                color: 'error',
                                text: error
                            });
                        }
                    );
            }
        };

    /**
     * @name get
     * @param accessor - string to get to the object. In the form of 'a.b.c'
     * @desc function that gets data from the store
     * @returns value of the requested object
     */
    function get(accessor?: string): any {
        return Utility.getter(_state, accessor);
    }

    /**
     * @name initialize
     * @desc called when the module is loaded
     * @return {void}
     */
    function initialize(): void {
        // register the store to force conformity
        for (let a in _actions) {
            if (_actions.hasOwnProperty(a)) {
                messageService.on(a, _actions[a]);
            }
        }
    }

    return {
        initialize: initialize,
        get: get
    };
}