import angular from 'angular';

import './workspace-save.scss';

import Utility from '../../../utility/utility.js';

export default angular.module('app.workspace.workspace-save', [])
    .directive('workspaceSave', workspaceSaveDirective);

workspaceSaveDirective.$inject = ['$location', '$window', 'monolithService', 'semossCoreService'];

function workspaceSaveDirective($location, $window, monolithService, semossCoreService) {
    workspaceSaveCtrl.$inject = ['$scope'];
    workspaceSaveLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        require: ['^insight'],
        scope: {},
        controller: workspaceSaveCtrl,
        controllerAs: 'workspaceSave',
        bindToController: {
            close: '&?'
        },
        template: require('./workspace-save.directive.html'),
        link: workspaceSaveLink,
        replace: true
    };

    function workspaceSaveCtrl() { }

    function workspaceSaveLink(scope, ele, attrs, ctrl) {
        scope.insightCtrl = ctrl[0];

        scope.workspaceSave.accordion = {
            save: 100,
            embed: 0
        }

        scope.workspaceSave.apps = {
            options: [],
            insights: []
        }
        scope.workspaceSave.newApp = {
            open: false,
            name: ''
        };

        scope.workspaceSave.question = {
            title: '',
            app: '',
            params: [],
            saved: false,
            error: '',
            tags: '',
            description: ''
        };

        scope.workspaceSave.parameters = {
            open: false,
            available: [],
            selected: ''
        }

        scope.workspaceSave.embed = {
            url: '',
            code: '',
            config: {
                resizable: false
            },
            custom: {
                open: false,
                valid: false,
                path: ''
            }
        }

        scope.workspaceSave.overrideInsights = false;
        scope.workspaceSave.override = {
            app: '',
            insight: '',
            insightList: []
        }

        scope.workspaceSave.changeApp = changeApp;
        scope.workspaceSave.changeOverrides = changeOverrides;
        scope.workspaceSave.validateNewApp = validateNewApp;
        scope.workspaceSave.removeParameter = removeParameter;
        scope.workspaceSave.addParameter = addParameter;
        scope.workspaceSave.validateSave = validateSave;
        scope.workspaceSave.executeSave = executeSave;
        scope.workspaceSave.toggleCustomEmbed = toggleCustomEmbed;
        scope.workspaceSave.saveCustomEmbed = saveCustomEmbed;
        scope.workspaceSave.copyEmbed = copyEmbed;
        scope.workspaceSave.updateEmbed = updateEmbed;

        /** Actions */
        /**
         * @name changeApp
         * @param list - from the app-list-dropdown
         * @desc toggle between creating a new app
         */
        function changeApp(list: any[]): void {
            let initialSet = false;

            // clear it out
            scope.workspaceSave.question.error = '';

            // set it once
            if (scope.workspaceSave.apps.options.length === 0) {
                scope.workspaceSave.apps.options = list;
                initialSet = true;
            }

            if (!scope.workspaceSave.newApp.open) {
                // first time, we want to try to guess the app
                if (initialSet) {
                    let appId = scope.insightCtrl.getShared('insight.app_id');

                    // find what database we are using
                    if (!appId) {
                        let recipe = scope.insightCtrl.getShared('steps');
                        for (let i = 0, len = recipe.length; i < len; i++) {
                            // regex is looking for something between Database (  and the next ,
                            // so basically the first argument to Database
                            let regEx = /Database\s*\(\s*database\s*=\s*\[\s*\"([a-zA-Z0-9-]+)+\"\s*]\s*\)/g;
                            let match = recipe[i].expression.match(regEx); // regEx.exec(recipe[i].expression);
                            if (match) {
                                // grab only the id and remove all the rest of the text surrounding it
                                appId = match[0].replace(regEx, '$1');
                                break;
                            }
                        }
                    }

                    if (!appId) {
                        if (list.length > 0) {
                            appId = list[0].value;
                        } else {
                            scope.insightCtrl.alert('error', 'No apps available');
                        }
                    }

                    // validate
                    for (let appIdx = 0, appLen = list.length; appIdx < appLen; appIdx++) {
                        if (list[appIdx].value === appId) {
                            scope.workspaceSave.question.app = list[appIdx];
                            break;
                        }
                    }

                    // still no app, they have to manually select
                    if (!scope.workspaceSave.question.app) {
                        return;
                    }
                }

                monolithService.getSearchInsightsResults({
                    searchString: '',
                    filterData: {
                        'app_id': [
                            scope.workspaceSave.question.app.value
                        ]
                    },
                    limit: 100,
                    offset: 0,
                    sortOrdering: 'desc',
                    sortField: 'last_viewed_on'
                }).then((response) => {
                    scope.workspaceSave.apps.insights = [];

                    if (Array.isArray(response)) {
                        scope.workspaceSave.apps.insights = response.map((insight) => {
                            return insight.name
                        });
                    }
                });
            }
        }
        /**
         * @name changeOverrides
         * @param list - from the app-list-dropdown
         * @desc will update the insight list when the selected app changes
         */
        function changeOverrides(list: any[]): void {
            monolithService.getSearchInsightsResults({
                searchString: '',
                filterData: {
                    'app_id': [
                        scope.workspaceSave.override.app.value
                    ]
                },
                limit: 100,
                offset: 0,
                sortOrdering: 'desc',
                sortField: 'last_viewed_on'
            }).then((response) => {
                scope.workspaceSave.apps.insights = [];

                if (Array.isArray(response)) {
                    scope.workspaceSave.override.insightList = response.map((insight) => {
                        return { name: insight.name, id: insight.app_insight_id };
                    });
                }
            });
        }

        /**
         * @name validateNewApp
         * @desc check if the databasename is valid
         * @returns {void}
         */
        function validateNewApp(): void {
            let name = String(scope.workspaceSave.newApp.name);

            scope.workspaceSave.question.error = '';

            if (!name) {
                scope.workspaceSave.question.error = 'required';
                return;
            }

            // return false if special characters, true otherwise
            if (name.match(/[@.*+?&^$%{}()";|[\]\\]/g)) {
                scope.workspaceSave.question.error = 'special';
                return;
            }

            const cleaned = name.toUpperCase();

            if (scope.workspaceSave.apps.options) { // see if db name exists
                for (let appIdx = 0, appLen = scope.workspaceSave.apps.options.length; appIdx < appLen; appIdx++) {
                    if (scope.workspaceSave.apps.options[appIdx].display.toUpperCase() === cleaned) {
                        scope.workspaceSave.question.error = 'exists';
                        break;
                    }
                }
            }
        }

        /**
         * @name removeParameter
         * @param idx - idx of the parameter to remove
         * @desc removes the param from selectedParameters
         */
        function removeParameter(idx: number): void {
            scope.workspaceSave.question.params.splice(idx, 1);
        }

        /**
         * @name addParameter
         * @desc add a new parameter
         */
        function addParameter(): void {
            let add = true;

            // check if it is there
            if (scope.workspaceSave.parameters.selected) {
                for (let paramIdx = 0, paramLen = scope.workspaceSave.question.params.length; paramIdx < paramLen; paramIdx++) {
                    if (scope.workspaceSave.question.params[paramIdx].value === scope.workspaceSave.parameters.selected.value) {
                        add = false;
                        break;
                    }
                }
            }

            //add the new one
            if (add) {
                scope.workspaceSave.question.params.push(scope.workspaceSave.parameters.selected);
            }

            // clear it
            scope.workspaceSave.parameters.selected = {};
        }

        /**
         * @name validateSave
         * @param type - type of save (save - edit OR save as - new)
         * @desc check if save should be disabled
         * @returns {boolean} - should save be disabled?
         */
        function validateSave(type: string): boolean {
            if (!scope.workspaceSave.question.title) {
                return false;
            }

            if (!scope.workspaceSave.newApp.open && !scope.workspaceSave.question.app.value) {
                return false;
            }

            if (scope.workspaceSave.question.error) {
                return false;
            }

            if (scope.workspaceSave.overrideInsights) {
                if (!scope.workspaceSave.override.app || !scope.workspaceSave.override.insight) {
                    return false;
                }
                // Disables "Save As" button if overriding insights
                if (type === 'new') {
                    return false;
                }
            }
            return true;
        }

        /**
         * @name executeSave
         * @param saveType - type fo the save (new or edit)
         * @desc saves visualization based on save type
         */
        function executeSave(saveType: 'new' | 'edit'): void {
            let components: any[] = [],
                steps = scope.insightCtrl.getShared('steps'),
                params: string[] = [],
                appVar = '',
                config: any;

            // automatically save in presentation mode if it's a new insight
            if (saveType === 'new') {
                scope.insightCtrl.emit('change-presentation', {
                    presentation: true
                });
            }

            // delete the empty keys
            config = semossCoreService.workspace.saveWorkspace(scope.insightCtrl.insightID);

            // add the config
            if (config && Object.keys(config).length !== 0) {
                steps.push({
                    type: '',
                    output: '',
                    expression: `SetInsightConfig(${JSON.stringify(config)});`
                });
            }

            // build parameters
            for (let paramIdx = 0, paramLen = scope.workspaceSave.question.params.length; paramIdx < paramLen; paramIdx++) {
                params.push(scope.workspaceSave.question.params[paramIdx].value);
            }

            if (scope.workspaceSave.newApp.open) {
                components.push({
                    type: 'Pixel',
                    components: [
                        `appVar = GenerateEmptyApp(${String(scope.workspaceSave.newApp.name).replace(/ /g, '_')})`
                    ],
                    terminal: true,
                    meta: true
                });

                appVar = 'appVar';
            }

            if (saveType === 'new') {
                components.push({
                    type: 'saveInsight',
                    components: [
                        appVar ? '' : scope.workspaceSave.question.app.value,
                        scope.workspaceSave.question.title,
                        false,
                        steps,
                        '',
                        '',
                        params,
                        appVar ? appVar : '',
                        semossCoreService.getPipeline('shared', scope.insightCtrl.insightID),
                        scope.workspaceSave.question.tags && Array.isArray(scope.workspaceSave.question.tags) ? scope.workspaceSave.question.tags.map(item => item.trim()) : [],
                        scope.workspaceSave.question.description
                    ],
                    terminal: true,
                    meta: true
                });
            } else {
                let appId: string,
                    insightId: string;

                if (scope.workspaceSave.overrideInsights) {
                    appId = scope.workspaceSave.override.app.value;
                    insightId = scope.workspaceSave.override.insight;
                } else {
                    appId = appVar ? '' : scope.workspaceSave.question.app.value;
                    insightId = scope.insightCtrl.getShared('insight.app_insight_id');
                }
                components.push({
                    type: 'updateInsight',
                    components: [
                        appId,
                        scope.workspaceSave.question.title,
                        false,
                        steps,
                        '',
                        '',
                        insightId,
                        params,
                        appVar ? appVar : '',
                        semossCoreService.getPipeline('shared', scope.insightCtrl.insightID),
                        scope.workspaceSave.question.tags && Array.isArray(scope.workspaceSave.question.tags) ? scope.workspaceSave.question.tags.map(item => item.trim()) : [],
                        scope.workspaceSave.question.description
                    ],
                    terminal: true,
                    meta: true
                });
            }

            let callback = function (response: PixelReturnPayload) {
                let output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') > -1) {
                    return;
                }

                // close the menu
                scope.workspaceSave.close();
            }

            scope.insightCtrl.execute(components, callback);
        }

        /**
         * @name toggleCustomEmbed
         * @desc toggles the custom
         */
        function toggleCustomEmbed(): void {
            scope.workspaceSave.embed.custom.path = '';
            scope.workspaceSave.embed.custom.valid = false;

            if (!scope.workspaceSave.embed.custom.open) {
                updateEmbed();
            }
        }

        /**
         * @name saveCustomEmbed
         * @desc save the custom
         */
        function saveCustomEmbed(): void {
            let callback,
                appId: string,
                appInsightId: string;

            if (!scope.workspaceSave.embed.custom.path.match(/^[a-zA-Z0-9-_]+$/)) {
                semossCoreService.emit('alert', {
                    color: 'error',
                    text: 'Path is not a valid URL'
                });
                return;
            }

            callback = function (response: PixelReturnPayload) {
                let output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') > -1) {
                    return;
                }

                semossCoreService.emit('alert', {
                    color: 'success',
                    text: output
                });

                scope.workspaceSave.embed.custom.valid = true;

                updateEmbed();
            };

            appId = scope.insightCtrl.getShared('insight.app_id');
            appInsightId = scope.insightCtrl.getShared('insight.app_insight_id');

            scope.widgetCtrl.meta([{
                type: 'Pixel',
                components: [
                    `badd(fancy=["${scope.workspaceSave.embed.custom.path}"], embed=["<encode>#!/insight?engine=${appId}&id=${appInsightId}</encode>"] )`
                ],
                meta: true,
                terminal: true
            }], callback);
        }

        /**
         * @name copyEmbed
         * @param {string} content - content to copy
         */
        function copyEmbed(content: string): void {
            // For IE.
            if ($window.clipboardData) {
                $window.clipboardData.setData('Text', content);

                semossCoreService.emit('alert', {
                    color: 'success',
                    text: 'Successfully copied to clipboard'
                });
            } else {
                let exportElement = angular.element("<textarea style='position:fixed;left:-1000px;top:-1000px;'>" + content + '</textarea>');
                ele.append(exportElement);
                (exportElement as any).select();


                if (document.execCommand('copy')) {
                    exportElement.remove();

                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully copied to clipboard'
                    });
                } else {
                    exportElement.remove();
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'Unsuccessfully copied to clipboard'
                    });
                }
            }
        }


        /** Updates */
        /**
        * @name updateSave
        * @desc update the save options
        */
        function updateSave(): void {
            let appId = scope.insightCtrl.getShared('insight.app_id'),
                appInsightId = scope.insightCtrl.getShared('insight.app_insight_id');

            scope.workspaceSave.question.saved = !!appInsightId;

            if (scope.workspaceSave.question.saved) {
                scope.workspaceSave.accordion.save = 70;
                scope.workspaceSave.accordion.embed = 30;
            } else {
                scope.workspaceSave.accordion.save = 100;
                scope.workspaceSave.accordion.embed = 0;
            }

            // Show the current name in the save directive
            scope.workspaceSave.question.title = scope.insightCtrl.getShared('insight.name');

            // clear it if it is called New Insight
            if (!scope.workspaceSave.question.saved && scope.workspaceSave.question.title === 'New Insight') {
                scope.workspaceSave.question.title = ''
            }

            // if saved insight, get the meta data
            if (scope.workspaceSave.question.saved) {
                let callback = function (response: PixelReturnPayload) {
                    let output = response.pixelReturn[0].output,
                        type = response.pixelReturn[0].operationType;

                    if (type.indexOf('ERROR') > -1) {
                        return;
                    }

                    scope.workspaceSave.question.tags = output.tags;
                    scope.workspaceSave.question.description = output.description;
                };

                scope.insightCtrl.query([{
                    meta: true,
                    type: 'getSpecificInsightMeta',
                    components: [appId, appInsightId],
                    terminal: true
                }], callback);
            }

            // update parameters
            updateParameters();

            if (scope.workspaceSave.question.saved) {
                updateEmbed();
            }
        }

        /**
         * @name updateParameters
         * @desc update the parameter information
         */
        function updateParameters(): void {
            let frames: any,
                added = {},
                available: any[] = [],
                panels: { panelId: string, widgetId: string }[] = [],
                param: any;

            frames = scope.insightCtrl.getShared('frames');

            for (let frame in frames) {
                if (frames.hasOwnProperty(frame)) {
                    let headers = frames[frame].headers || [];

                    headers.forEach(function (header: any) {
                        if (header.qs !== 'unknown') {
                            if (!added.hasOwnProperty(header.alias)) {
                                available.push({
                                    display: String(header.alias).replace(/_/g, ' '),
                                    value: header.alias
                                })

                                added[header.alias] = true;
                            }
                        }
                    });
                }
            }

            // sort by title
            Utility.sort(available, 'value')

            scope.workspaceSave.parameters.available = available;

            console.warn('TODO: This is going off of the widget, not the insight. FIX IT.')
            panels = scope.insightCtrl.getShared('panels');
            for (let panelIdx = 0, panelLen = panels.length; panelIdx < panelLen; panelIdx++) {
                if (panels[panelIdx].panelId === '0') {
                    param = semossCoreService.getWidget(panels[panelIdx].widgetId, 'view.param.options.json');
                    break;
                }
            }

            // also this requires the frame to be executed/loaded
            scope.workspaceSave.question.params = [];
            if (param) {
                for (let paramIdx = 0, paramLen = param[0].params.length; paramIdx < paramLen; paramIdx++) {
                    for (let availableIdx = 0, availableLen = scope.workspaceSave.parameters.available.length; availableIdx < availableLen; availableIdx++) {
                        if (scope.workspaceSave.parameters.available[availableIdx].value === param[0].params[paramIdx].paramName) {
                            scope.workspaceSave.question.params.push(scope.workspaceSave.parameters.available[availableIdx]);
                        }
                    }
                }
            }
        }

        /**
        * @name updatePermission
        * @desc update the permission
        */
        function updatePermission(): void {
            // check if it is is saved
            const appId = scope.insightCtrl.getShared('insight.app_id'),
                appInsightId = scope.insightCtrl.getShared('insight.app_insight_id');

            if (appId && appInsightId) {
                // make call to get user's permission access for this insight so overriding the insight (Update) is disabled
                monolithService.getUserInsightPermission(appId, appInsightId)
                    .then(function (response: { data: { permission: string } }) {
                        if (response.data.permission === 'OWNER') {
                            scope.workspaceSave.permission = 'AUTHOR';
                        } else if (response.data.permission === 'EDIT') {
                            scope.workspaceSave.permission = 'EDITOR';
                        } else {
                            scope.workspaceSave.permission = 'READ_ONLY';
                        }
                    });
            }
        }

        /**
         * @name updateEmbed
         * @desc updates the embed string
         */
        function updateEmbed(): void {
            let appId = scope.insightCtrl.getShared('insight.app_id'),
                appInsightId = scope.insightCtrl.getShared('insight.app_insight_id');

            scope.workspaceSave.embed.url = $location.absUrl().split('#')[0];
            if (scope.workspaceSave.embed.custom.open) {
                scope.workspaceSave.embed.url += `#!/r/${scope.workspaceSave.embed.custom.path}`;
            } else {
                scope.workspaceSave.embed.url += `#!/insight?engine=${appId}&id=${appInsightId}`
            }

            scope.workspaceSave.embed.code = `<iframe frameborder="0" width="1000" height="600" style="border: 1px solid #ccc; ${scope.workspaceSave.embed.config.resizable ? 'resize: both;overflow: auto;' : ''}"
                src="${scope.workspaceSave.embed.url}">/<iframe>`
        }

        /** Initialize */
        /**
         * @name initialize
         * @desc initializes the workspaceSave directive
         * @returns {void}
         */
        function initialize(): void {
            let savedInsightListener: () => {};


            // register listeners
            savedInsightListener = scope.insightCtrl.on('saved-insight', updateSave);

            scope.$on('$destroy', function () {
                console.log('destroying workspaceSave....');
                savedInsightListener();
            });


            updateSave();
            updatePermission();
        }

        initialize();
    }
}