'use strict';
import angular from 'angular';

import './workbook.scss';

export default angular.module('app.workbook.directive', [])
    .directive('workbook', workbookDirective);

workbookDirective.$inject = ['$timeout'];

function workbookDirective($timeout) {
    workbookCtrl.$inject = [];
    workbookLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        require: ['^insight'],
        template: require('./workbook.directive.html'),
        scope: {},
        controller: workbookCtrl,
        controllerAs: 'workbook',
        bindToController: {},
        link: workbookLink,
        replace: true
    };

    function workbookCtrl() { }

    function workbookLink(scope, ele, attrs, ctrl) {
        scope.insightCtrl = ctrl[0];

        scope.workbook.sheets = [];
        scope.workbook.sheetId = undefined;
        scope.workbook.settings = {
            open: false
        };


        scope.workbook.newSheet = newSheet;
        scope.workbook.selectSheet = selectSheet;
        scope.workbook.deleteSheet = deleteSheet;
        scope.workbook.getSheetContent = getSheetContent;
        scope.workbook.closeSettings = closeSettings;


        /** Actions */
        /**
         * @name newSheet
         * @desc called when a sheet is added
         */
        function newSheet(): void {
            let counter = scope.insightCtrl.getWorkbook('worksheetCounter');

            // add one
            counter++

            // emit
            scope.insightCtrl.execute([
                {
                    type: 'Pixel',
                    components: [`AddSheet("${counter}")`],
                    terminal: true
                }
            ]);
        }

        /**
         * @name selectSheet
         * @param sheetId - sheetId to select
         * @desc called when a sheet is selected
         */
        function selectSheet(sheetId: string): void {
            scope.insightCtrl.emit('select-worksheet', {
                sheetId: sheetId
            });
        }

        /**
         * @name deleteSheet
         * @param sheetId - sheetId to delete
         * @desc called when a sheet is deleted
         */
        function deleteSheet(sheetId: string): void {
            let components: PixelCommand[] = [];

            const panels = scope.insightCtrl.getWorksheet(sheetId, 'panels');
            for (let panelId in panels) {
                if (panels.hasOwnProperty(panelId)) {
                    components.push({
                        type: 'closePanel',
                        components: [panelId],
                        terminal: true
                    });
                }
            }

            components.push({
                type: 'Pixel',
                components: [`CloseSheet("${sheetId}")`],
                terminal: true
            });

            scope.insightCtrl.execute(components);
        }

        /**Settings */
        /**
         * @name closeSettings
         * @desc close the save
         */
        function closeSettings(): void {
            scope.workbook.settings.open = false;
        }

        /** Getters */

        /**
         * @name getSheetContent
         * @desc get the sheet content to paint
         * @returns html to render
         */
        function getSheetContent(): string {
            if (typeof scope.workbook.sheetId === 'undefined') {
                return '<div style="position: absolute; left: 50%; top: 50%; transform:translate(-50%, -50%);">No Sheets. Please Add One.</div> ';
            }

            return `<worksheet sheet-id="${scope.workbook.sheetId}"></worksheet>`;
        }

        /** Updates */
        /**
         * @name updateWorkbook
         * @desc called when the sheet information changes
         */
        function updateWorkbook(): void {
            const book = scope.insightCtrl.getWorkbook();

            scope.workbook.sheets = [];
            for (let sheetId in book.worksheets) {
                if (book.worksheets.hasOwnProperty(sheetId)) {
                    scope.workbook.sheets.push(book.worksheets[sheetId]);
                }
            }

            // set the selected sheet
            scope.workbook.sheetId = book.worksheet;

            // auto add
            if (typeof scope.workbook.sheetId === 'undefined') {
                newSheet();
            }
        }

        /**
        * @name updatedPresentation
        * @desc called when the presentation information changes
        */
        function updatedPresentation(): void {
            scope.workbook.presentation = scope.insightCtrl.getWorkspace('presentation');
        }

        /** Initialize */
        /**
         * @name initialize
         * @desc initializes the workbook directive
         * @returns {void}
         */
        function initialize(): void {
            let selectedWorksheetListener: () => {},
                updatedWorksheetListener: () => {},
                addedWorksheetListener: () => {},
                closedWorksheetListener: () => {},
                updatedPresentationListener: () => {};

            // register listeners
            selectedWorksheetListener = scope.insightCtrl.on('selected-worksheet', updateWorkbook);
            updatedWorksheetListener = scope.insightCtrl.on('updated-worksheet', updateWorkbook);
            addedWorksheetListener = scope.insightCtrl.on('added-worksheet', updateWorkbook);
            closedWorksheetListener = scope.insightCtrl.on('closed-worksheet', updateWorkbook);
            updatedPresentationListener = scope.insightCtrl.on('updated-presentation', updatedPresentation);

            scope.$on('$destroy', function () {
                console.log('destroying workbook....');
                selectedWorksheetListener();
                updatedWorksheetListener();
                addedWorksheetListener();
                closedWorksheetListener();
                updatedPresentationListener();
            });

            // initialize the workbook
            updateWorkbook();
            updatedPresentation();
        }

        initialize();
    }
}