'use strict';

import angular from 'angular';

import './view.scss';

import './view-layer/view-layer.directive.ts';
import './view-layout/view-layout.directive.ts';
import './view-dimensions/view-dimensions.directive.ts';
import './view-tools/view-tools.directive.ts';

export default angular.module('app.view.directive',
    [
        'app.view.view-layer',
        'app.view.view-layout',
        'app.view.view-dimensions',
        'app.view.view-tools'
    ])
    .directive('view', viewDirective);

viewDirective.$inject = ['semossCoreService', '$timeout'];

function viewDirective(semossCoreService, $timeout) {
    viewCtrl.$inject = ['$scope'];
    viewLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        scope: {},
        restrict: 'EA',
        require: ['^widget'],
        controllerAs: 'view',
        bindToController: {},
        template: require('./view.directive.html'),
        controller: viewCtrl,
        link: viewLink
    };


    function viewCtrl($scope) {
        $scope.view.updatedLayer = updatedLayer; //message to children that layer has been update
        $scope.view.getLayer = getLayer// get the layer. Overwritten in view-layer
        $scope.view.updatedActive = updatedActive; //message to children that layer has been update
        $scope.view.getActive = getActive// get the active. Overwritten in view-layout
        $scope.view.openTool = openTool// open a tool. Overwritten in the link

        /**
         * @name updatedLayer
         * @desc the layer has been updated, tell everyone
         */
        function updatedLayer(): void {
            $scope.$broadcast("view--layer-updated");
        }


        /**
         * @name getLayer
         * @desc the layer information.
         */
        function getLayer(): any {
            return {};
        }

        /**
         * @name updatedActive
         * @desc the view/layer has been updated, tell everyone
         */
        function updatedActive(paint: boolean): void {
            $scope.$broadcast("view--active-updated", paint);
        }

        /**
         * @name getActive
         * @desc the layer information.
         */
        function getActive(): any {
            return {};
        }

        /**
         * @name openTool
         * @desc open a tool
         * @param {string} widget - widget to add
         */
        function openTool(widget: string): any {
            return '';
        }
    }

    function viewLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];

        scope.view.open = false;

        scope.view.openTool = openTool;

        /**
         * @name resetView
         * @desc reset the view
         */
        function resetView(): void {
            // set the new layout
            const active = scope.view.getActive();

            // reset
            scope.view.items = [];
            if (Object.keys(active).length === 0) {
                scope.view.items = [
                    {
                        name: 'Change View',
                        height: 100,
                        content: '<view-layout/>'
                    }
                ];
                return;
            }

            // this is the view
            scope.view.items = [
                {
                    name: 'Change View',
                    height: 30,
                    content: '<view-layout/>'
                },
                {
                    name: 'Update View',
                    height: 70,
                    content: '<view-dimensions/>'
                },
                {
                    name: 'Additional Tools',
                    height: 0,
                    content: '<view-tools/>'
                }
            ];
        }

        /**
         * @name openTool
         * @desc open a tool
         * @param {string} tool - widget to add
         */
        function openTool(widget: string): any {
            const config = semossCoreService.getSpecificConfig(widget);

            if (!config || Object.keys(config).length === 0) {
                return;
            }

            if (
                config.content && config.content.json && config.content.json[0].execute === 'auto' ||
                (config.content && config.content.template && config.content.template.execute === 'auto')
            ) {
                destroyHiddenTool();
                semossCoreService.loadWidget(widget, 'content').then((html: string) => {
                    $timeout(function () {
                        scope.view.hidden = html;
                    });
                });

                return;
            }

            if (config.hasOwnProperty('view') && config.view) {
                const active = scope.widgetCtrl.getWidget('active');
                if (active !== widget) {
                    let viewComponents = [{
                        'type': 'panel',
                        'components': [scope.widgetCtrl.panelId],
                        'meta': false
                    },
                    {
                        'type': 'setPanelView',
                        'components': [widget],
                        'terminal': true
                    }
                    ];

                    scope.widgetCtrl.execute(viewComponents);
                }

                return;
            }

            // check if it is there
            // check if the group is already there;
            let addedIdx = -1;
            for (let itemIdx = 0, itemLen = scope.view.items.length; itemIdx < itemLen; itemIdx++) {
                if (scope.view.items[itemIdx].added) {
                    addedIdx = itemIdx;
                } else {
                    scope.view.items[itemIdx].height = 0;
                }
            }


            if (addedIdx > -1) {
                scope.view.items[addedIdx].name = config.name;
                scope.view.items[addedIdx].height = 100;

                semossCoreService.loadWidget(widget, 'content').then((html: string) => {
                    scope.view.items[addedIdx].content = html;
                });
            } else {
                let added = {
                    name: config.name,
                    height: 100,
                    content: '',
                    added: true
                };

                scope.view.items.push(added);

                semossCoreService.loadWidget(widget, 'content').then((html: string) => {
                    added.content = html;
                });
            }
        }

        /**
         * @name destroyHiddenTool
         * @desc set the hidden content to null
         */
        function destroyHiddenTool(): void {
            scope.view.hidden = '';
        }


        /**
         * @name initialize
         * @desc function that is called on directive load
         */
        function initialize(): void {
            let hiddenWidgetDestroyListener: () => void;

            // register messages
            scope.$on("view--active-updated", resetView);

            hiddenWidgetDestroyListener = scope.widgetCtrl.on('hidden-widget-destroy', destroyHiddenTool);

            scope.$on('$destroy', function () {
                hiddenWidgetDestroyListener();
            });

            // this will auto update the builder as well
            resetView();
        }

        initialize();
    }
}