'use strict';

/**
 * @name Scheduler
 * @desc Schedule backend tasks to happen on a daily, weekly, or monthly basis
 *       When adding a jobType to the scheduler, please seperate it out into its 
 *       own directive inside of jobTemplates 
 *       (don't forget to update components-entry.js, scheduler-entry.js, 
 *       and add new directives to the app in index.js!). 
 *       The scheduler attaches front end meta data to jobs so the user can go back 
 *       and edit them. For job templates, the user's answers to the template form
 *       are stored inside of <job>.jobTypeTemplate. The final pixel query is constructed
 *       inside of the template directive, and is stored inside of 
 *       <job>.jobTypeTemplate.templatePixelQuery
 */
export default angular.module('app.scheduler.directive', [])
    .directive('scheduler', scheduler);

import './scheduler.scss';
import './jobTemplates/sync-app-to-github/sync-app-to-github.directive.js';
import './jobTemplates/backup-app/backup-app.directive.js';
import './jobTemplates/extract-transform-load/extract-transform-load.directive.js';
import './jobTemplates/send-email/send-email.directive.js';

scheduler.$inject = ['semossCoreService'];

function scheduler(semossCoreService) {
    schedulerCtrl.$inject = [];
    schedulerLink.$inject = ['scope'];

    return {
        restrict: 'E',
        scope: {},
        controller: schedulerCtrl,
        link: schedulerLink,
        template: require('./scheduler.directive.html'),
        bindToController: {
            appId: '='
        },
        controllerAs: 'schedule'
    };

    function schedulerCtrl() {
        var schedule = this;
        schedule.jobBeingEditedIdx = -1; // index of job in all jobs to edit
        schedule.jobBeingEdited = {}; // hold data of job being edited
        schedule.frequencyOpts = [{
            computer: 1,
            human: 'Daily'
        }, {
            computer: 2,
            human: 'Weekly'
        }, {
            computer: 3,
            human: 'Monthly'
        }];
        schedule.daysOfWeek = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
        schedule.jobTypes = ['Custom Job', 'Backup App', 'Sync App', 'Extract Transform Load', 'Send Email'];
        schedule.currentJob = {
            jobType: 'Custom Job',
            jobName: '',
            jobGroup: schedule.appId,
            cronExpression: '',
            recipe: '',
            hour: 12,
            minute: '00',
            ampm: 'PM',
            frequency: schedule.frequencyOpts[0],
            dayOfWeek: schedule.daysOfWeek[new Date().getDay()],
            dayOfMonth: new Date().getDate(),
            jobTypeTemplate: {},
            onLoad: true
        };

        schedule.hours = [];
        schedule.minutes = [];
        schedule.daysOfMonth = [];
        schedule.allJobs = [];

        schedule.scheduleJob = scheduleJob;
        schedule.editJob = editJob;
        schedule.deleteJob = deleteJob;
        schedule.isJobInvalid = isJobInvalid;

        /**
         * @name scheduleJob
         * @desc schedules the job
         * @return {void}
         */
        function scheduleJob() {
            var message = semossCoreService.utility.random('query-pixel'),
                job, stringifiedJob;
            if (schedule.jobBeingEditedIdx > -1) {
                job = schedule.jobBeingEdited;
            } else {
                job = schedule.currentJob;
            }

            if (!job.customCron) {
                job.cronExpression = convertTimeToCron(job);
            }

            semossCoreService.once(message, function (response) {
                var jsonResponse;
                if (response.pixelReturn[0].operationType[0] === 'ERROR') {
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'Something went wrong. Job was not scheduled'
                    });
                } else {
                    jsonResponse = JSON.parse(response.pixelReturn[0].output.jsonConfig);
                    schedule.allJobs.push(JSON.parse(jsonResponse.parameters));

                    // if we aren't editing need to clear current job
                    // otherwise we need to delete this old job
                    if (schedule.jobBeingEditedIdx === -1) {
                        semossCoreService.emit('alert', {
                            color: 'success',
                            text: 'Job was scheduled'
                        });
                        resetControllerVariables();
                    } else {
                        schedule.jobBeingEditedIdx = -1;
                    }
                }
            });
            if (uniqueJob(job.jobName)) {
                if (job.jobType && job.jobType !== 'Custom Job') {
                    job.recipe = job.jobTypeTemplate.templatePixelQuery;
                }

                // Encode the recipe
                // Otherwise we have a character escaping dilemma
                job.recipe = encodeURIComponent(job.recipe);

                // Stringify the job so that we can edit it later
                stringifiedJob = JSON.stringify(job).replace(/"/g, '\\"');

                semossCoreService.emit('query-pixel', {
                    commandList: [{
                        type: 'scheduleJob',
                        components: [job.jobName, job.jobGroup, job.cronExpression, job.recipe, job.onLoad, stringifiedJob],
                        terminal: true,
                        meta: true
                    }],
                    response: message
                });

                // Reset the recipe so that it is not encoded on the UI
                job.recipe = decodeURIComponent(job.recipe);
            } else {
                semossCoreService.emit('alert', {
                    color: 'error',
                    text: 'Job name must be unique'
                });
            }
        }

        /**
         * @name uniqueJob
         * @param {object} jobName  name of job to check
         * @desc goes thru all jobs and deterimes if passed job is uniquely named
         * @return {boolean} true if unique, false otherwise
         */
        function uniqueJob(jobName) {
            var i;
            for (i = 0; i < schedule.allJobs.length; i++) {
                if (jobName === schedule.allJobs[i].jobName) {
                    return false;
                }
            }

            return true;
        }

        /**
         * @name resetControllerVariables
         * @desc resets the controllwe variables to their defaults (except all jobs)
         * @return {void}
         */
        function resetControllerVariables() {
            schedule.currentJob = {
                jobType: 'Custom Job',
                jobName: '',
                jobGroup: schedule.appId,
                cronExpression: '',
                recipe: '',
                hour: 12,
                minute: '00',
                ampm: 'PM',
                frequency: schedule.frequencyOpts[0],
                dayOfWeek: schedule.daysOfWeek[new Date().getDay()],
                dayOfMonth: new Date().getDate(),
                jobTypeTemplate: {},
                onLoad: true
            };
        }
        /**
         * @name editJob
         * @param {number} idx - index of job in schedule.allJobs
         * @desc allows user to edit a job
         * @return {void} 
         */
        function editJob(idx) {
            schedule.jobBeingEditedIdx = idx;
            schedule.jobBeingEdited = JSON.parse(JSON.stringify(schedule.allJobs[idx]));
            schedule.jobBeingEdited.recipe = decodeURIComponent(schedule.jobBeingEdited.recipe);
            schedule.jobBeingEdited.originalName = schedule.jobBeingEdited.jobName;

            // set to custom if values not present
            if(!schedule.jobBeingEdited.jobType) {
                schedule.jobBeingEdited.jobType = 'Custom Job';
                schedule.jobBeingEdited.customCron = true;
            }
        }

        /**
         * @name deleteJob
         * @param {number} idx - index if deleted from list
         * @desc allows user to delete a job
         * @return {void} 
         */
        function deleteJob(idx) {
            var job, i, jobToDelete,
                message = semossCoreService.utility.random('query-pixel');

            if (idx || idx === 0) {
                job = schedule.allJobs[idx];
                jobToDelete = job.jobName;
            } else if (schedule.jobBeingEditedIdx > -1) {
                job = schedule.jobBeingEdited;
                jobToDelete = job.originalName;
            } else {
                job = schedule.currentJob;
                jobToDelete = job.jobName;
            }
            semossCoreService.once(message, function (response) {
                if (response.pixelReturn[0].operationType[0] === 'ERROR') {
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'Something went wrong. Job could not be unscheduled'
                    });
                } else {
                    for (i = 0; i < schedule.allJobs.length; i++) {
                        if ((schedule.allJobs[i].jobName === job.jobName) || job.originalName === schedule.allJobs[i].jobName) {
                            schedule.allJobs.splice(i, 1);
                            break;
                        }
                    }

                    if (schedule.jobBeingEditedIdx > -1) {
                        scheduleJob();
                        semossCoreService.emit('alert', {
                            color: 'success',
                            text: 'Job Rescheduled'
                        });
                    } else {
                        // Here only emit job deleted if we are not rescheduling it
                        semossCoreService.emit('alert', {
                            color: 'success',
                            text: 'Job Deleted'
                        });
                    }
                }
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    type: 'deleteJob',
                    components: [jobToDelete, job.jobGroup],
                    terminal: true,
                    meta: true
                }],
                listeners: [],
                response: message
            });
        }

        /**
         * @name convertTimeToCron
         * @param {object} job job to create cron job for
         * @desc converts the time user selected to java compatible cron string
         * @return {void}
         */
        function convertTimeToCron(job) {
            var freq = job.frequency.computer,
                cronPieces = ['00', '*', '*', '*', '*', '?', '*'];

            if (freq >= 1) {
                cronPieces[1] = job.minute;
                if (job.ampm === 'AM') {
                    if (job.hour === 12) {
                        cronPieces[2] = 0;
                    } else {
                        cronPieces[2] = job.hour;
                    }
                } else if (job.hour === 12) {
                    cronPieces[2] = 12;
                } else {
                    cronPieces[2] = job.hour + 12;
                }
                cronPieces[3] = '1/1';
            }

            if (freq === 2) {
                cronPieces[3] = '1/' + job.dayOfWeek.substr(0, 3).toUpperCase();
            }

            if (freq === 3) {
                cronPieces[3] = '1/' + job.dayOfMonth;
            }

            return cronPieces.join(' ');
        }

        /**
         * @name isJobInvalid
         * @desc determines if job is missing information so submit button can be deactivated
         * @return {boolean} true if job is invalid
         */
        function isJobInvalid() {
            var job;
            if (schedule.jobBeingEditedIdx > -1) {
                job = schedule.jobBeingEdited;
            } else {
                job = schedule.currentJob;
            }

            if (!job.jobName && job.jobName !== 0) {
                return true;
            }

            return false;
        }
    }

    function schedulerLink(scope) {
        function initialize() {
            const message = semossCoreService.utility.random('query-pixel');
            let i = 0,
                iAsString;

            // initialize value for date and time info dropdowns
            while (i < 61) {
                // minutes
                if (i === 0) {
                    scope.schedule.minutes.push('00');
                } else {
                    if (i < 10) {
                        iAsString = String(i);
                        scope.schedule.minutes.push(iAsString + iAsString);
                    }
                    scope.schedule.minutes.push(i);
                }

                // hours
                if (i < 13 && i !== 0) {
                    scope.schedule.hours.push(i);
                }

                // day of month
                if (i > 0 && i < 32) {
                    scope.schedule.daysOfMonth.push(i);
                }

                i++;
            }

            semossCoreService.once(message, function (response) {
                if (response.pixelReturn[0].operationType[0] === 'ERROR') {
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'Something went wrong. Jobs could not be retrieved.'
                    });
                } else {
                    // jobs is a map or maps
                    // where the jobId is a unique id of the job inputs
                    var jobs = response.pixelReturn[0].output,
                        jobId;
                    for (jobId in jobs) {
                        var job = jobs[jobId];
                        if (job.hasOwnProperty('parameters')) {
                            // Parse the job back from the stringified version
                            var jobJson = JSON.parse(job.parameters.replace(/\\"/g, '"'));

                            // Also, need to decode the recipe again
                            jobJson.recipe = decodeURIComponent(jobJson.recipe);
                            scope.schedule.allJobs.push(jobJson);
                        } else {
                            scope.schedule.allJobs.push(job);
                        }
                    }
                }
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    type: 'listAllJobs',
                    components: [],
                    meta: true,
                    terminal: true
                }],
                listeners: [],
                response: message
            });
        }

        initialize();
    }
}
