'use strict';

import './preview.scss';
import '../total-instances/total-instances.directive.js';
import { Grid } from 'ag-grid-community';
import '@/widgets/grid-standard/grid-standard.scss';
/**
 * @name preview.directive.js
 * @desc federate view
 */
angular.module('app.preview.directive', [
    'app.total-instances.directive'
]).directive('preview', previewDirective);

previewDirective.$inject = ['$timeout', 'semossCoreService'];

function previewDirective($timeout, semossCoreService) {
    previewCtrl.$inject = [];
    previewLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        template: require('./preview.directive.html'),
        controller: previewCtrl,
        link: previewLink,
        require: ['^widget'],
        scope: {},
        bindToController: {},
        controllerAs: 'preview',
        replace: true
    };

    function previewCtrl() {}

    function previewLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];

        var clickTimeout;

        scope.preview.loading = false;
        scope.preview.grid = undefined;

        /**
         * @name drawPreview
         * @returns {void}
         * @desc draw the preview grid
         */
        function drawPreview() {
            var previewGridEle = ele[0].querySelector('#preview__grid');
            previewGridEle.className += ' ag-theme-balham';
            const options = {
                rowData: [],
                columnDefs: []
            };
            scope.preview.grid = new Grid(previewGridEle, options);
            if (scope.preview.grid.gridOptions.api) {
                scope.preview.grid.gridOptions.api.addEventListener('cellClicked', clickEvent);
            }
        }

        /**
         * @name clearPreview
         * @returns {void}
         * @desc clear the data for preview
         */
        function clearPreview() {
            scope.preview.grid.gridOptions.api.setColumnDefs([]);
            scope.preview.grid.gridOptions.api.setRowData([]);
        }

        /**
         * @name setPreview
         * @desc set the data for preview
         * @param {object} data - data to load into the view
         * @returns {void}
         */
        function setPreview(data) {
            var schema,
                values,
                headerIdx,
                headerLen;

            if (typeof data === 'undefined' || typeof data.headers === 'undefined' || typeof data.values === 'undefined' || data.headers.length === 0) {
                clearPreview();
                return;
            }

            schema = [];
            for (headerIdx = 0, headerLen = data.headers.length; headerIdx < headerLen; headerIdx++) {
                schema.push({
                    headerName: String(data.headers[headerIdx]).replace(/_/g, ' '),
                    field: data.headers[headerIdx]
                });
            }

            values = semossCoreService.visualization.getTableData(data.headers, data.values, data.rawHeaders);


            // Paint
            scope.preview.grid.gridOptions.api.setColumnDefs(schema);
            scope.preview.grid.gridOptions.api.setRowData(values.rawData);
        }

        /**
         * @name formatPreview
         * @desc load the data for preview
         * @param {array} response - pixel response to extra data from
         * @returns {object} data - data to extra
         */
        function formatPreview(response) {
            var data,
                stepIdx;


            // backwards look at task
            for (stepIdx = response.pixelReturn.length - 1; stepIdx >= 0; stepIdx--) {
                if (response.pixelReturn[stepIdx].operationType.indexOf('TASK_DATA') > -1) {
                    data = response.pixelReturn[stepIdx].output.data;
                    break;
                } else if (response.pixelReturn[stepIdx].operationType.indexOf('NEW_EMPTY_INSIGHT') > -1) {
                    data = formatPreview(response.pixelReturn[stepIdx].output.insightData);
                    if (typeof data !== 'undefined') {
                        break;
                    }
                }
            }

            return data;
        }

        /**
         * @name loadPreview
         * @desc load the data for preview
         * @param {array} pixelComponents - pixelComponents to run
         * @param {array} totalCountPixelComponents - run to get the total count
         * @returns {void}
         */
        function loadPreview(pixelComponents, totalCountPixelComponents) {
            var callback,
                totalCountCallback;

            if (!pixelComponents || pixelComponents.length === 0) {
                clearPreview();
                scope.preview.loading = false;
                return;
            }

            // register message to come back to
            callback = function (response) {
                // we look at the last one
                var data = formatPreview(response);

                if (typeof data === 'undefined') {
                    clearPreview();
                    scope.preview.loading = false;
                    return;
                }
                scope.preview.data = data;

                setPreview(data);
                scope.preview.loading = false;
            };

            totalCountCallback = function (response) {
                var data = formatPreview(response);
                if (typeof data !== 'undefined' && data.values) {
                    scope.preview.totalCount = data.values[0][0];
                    scope.widgetCtrl.emit('preview-total-count', {
                        data: scope.preview.totalCount
                    });
                }
            };

            if (pixelComponents.length > 0) {
                scope.preview.loading = true;

                scope.widgetCtrl.meta(pixelComponents, callback, []);
                if (totalCountPixelComponents) {
                    scope.widgetCtrl.meta(totalCountPixelComponents, totalCountCallback, []);
                }
            }
        }

        /**
         * @name clickEvent
         * @desc add a click event
         * @param {event} e - canvas data event
         * @returns {void}
         */
        function clickEvent(e) {
            if (clickTimeout) {
                $timeout.cancel(clickTimeout);
            }

            clickTimeout = $timeout(function (cell) {
                if (cell) {
                    scope.widgetCtrl.emit('select-preview',
                        [{
                            display: String(cell.colDef.field).replace(/_/g, ' ')
                        }]
                    );
                }
            }.bind(null, e), 300);
        }

        /**
         * @name initialize
         * @desc initialize the module
         * @returns {void}
         */
        function initialize() {
            var initListener,
                initComponents,
                loadListener;

            scope.preview.loading = true;

            initListener = scope.widgetCtrl.on('load-preview', function (payload) {
                initComponents = payload.pixelComponents;
            });

            // destroy the initListiner
            initListener();

            drawPreview();
            if (initComponents) {
                loadPreview(initComponents);
            }

            loadListener = scope.widgetCtrl.on('load-preview', function (payload) {
                if (scope.preview.grid) {
                    loadPreview(payload.pixelComponents, payload.totalCountPixelComponents);
                }
            });


            scope.$on('$destroy', function () {
                scope.preview.grid.gridOptions.api.destroy();
                loadListener();
            });

            scope.preview.loading = false;
        }

        initialize();
    }
}
