import angular from 'angular';
import Utility from '../../../utility/utility.js';

export default angular.module('app.home.home-nav', [])
    .directive('homeNav', homeNavDirective);

import './home-nav.scss';

homeNavDirective.$inject = ['ENDPOINT', '$timeout', '$state', '$stateParams', '$transitions', 'semossCoreService'];

function homeNavDirective(ENDPOINT, $timeout, $state, $stateParams, $transitions, semossCoreService) {
    homeNavCtrl.$inject = [];
    homeNavLink.$inject = ['scope', 'ele'];

    return {
        restrict: 'E',
        template: require('./home-nav.directive.html'),
        scope: {},
        controller: homeNavCtrl,
        controllerAs: 'homeNav',
        bindToController: {
            open: '='
        },
        link: homeNavLink
    };

    function homeNavCtrl() { }

    function homeNavLink(scope, ele) {
        scope.homeNav.active = {
            listeners: [],
            search: '',
            open: true,
            rawApps: [],
            renderedApps: [],
            rawInsights: [],
            renderedInsights: []
        }

        scope.homeNav.toggle = toggle;
        scope.homeNav.navigate = navigate;
        scope.homeNav.newInsight = newInsight;
        scope.homeNav.toggleActive = toggleActive;
        scope.homeNav.searchActive = searchActive;
        scope.homeNav.remove = remove;


        /**
         * @name toggle
         * @desc toggle the nav menu open or close
         * @returns {void}
         */
        function toggle(): void {
            scope.homeNav.open = !scope.homeNav.open;

            // when we close it, remove the active
            if (!scope.homeNav.open) {
                closeActive();
            } else {
                openActive();
            }
        }

        /**
         * @name navigate
         * @desc navigate to an active insight or app
         * @param state - selected state
         * @param params - options for the active state
         */
        function navigate(state: string, params: any): void {
            $state.go(state, params || {});
        }

        /**
         * @name newInsight
         * @desc newInsight a new insight
         */
        function newInsight(): void {
            semossCoreService.emit('open', {
                type: 'new',
                options: {}
            });
        }

        /**
         * @name toggleActive
         * @desc toggle the active menu open or close
         * @returns {void}
         */
        function toggleActive(): void {
            if (!scope.homeNav.open) {
                // open the menu
                toggle();

                // open the active menu
                openActive();
            } else if (scope.homeNav.active.open) {
                closeActive();
            } else {
                openActive();
            }
        }

        /**
         * @name openActive
         * @desc open the active menu
         * @returns {void}
         */
        function openActive(): void {
            scope.homeNav.active.open = true;

            scope.homeNav.active.search = '';

            // remove it  (in case)
            for (let listenerIdx = 0, listenerLen = scope.homeNav.active.listeners.length; listenerIdx < listenerLen; listenerIdx++) {
                scope.homeNav.active.listeners[listenerIdx]();
            }

            scope.homeNav.active.listeners = [
                semossCoreService.on('update-app', function () {
                    updateActiveApps();
                    searchActive();
                }),
                semossCoreService.on('new-insight', function () {
                    updateActiveInsights();
                    searchActive();
                }),
                semossCoreService.on('saved-insight', function () {
                    updateActiveInsights();
                    searchActive();
                }),
                semossCoreService.on('closed-insight', function () {
                    updateActiveInsights();
                    searchActive();
                }),
                semossCoreService.on('changed-insight-name', function () {
                    updateActiveInsights();
                    searchActive();
                }),
                semossCoreService.on('close-app', function () {
                    updateActiveApps();
                    searchActive();
                })
            ]

            updateActiveApps();
            updateActiveInsights();
            searchActive();
        }

        /**
         * @name closeActive
         * @desc close the active menu
         * @returns {void}
         */
        function closeActive(): void {
            scope.homeNav.active.open = false;

            // remove it
            for (let listenerIdx = 0, listenerLen = scope.homeNav.active.listeners.length; listenerIdx < listenerLen; listenerIdx++) {
                scope.homeNav.active.listeners[listenerIdx]();
            }
        }


        /**
         * @name searchActive
         * @desc navigate to an active insight or app
         */
        function searchActive(): void {
            scope.homeNav.active.renderedApps = Utility.filter(scope.homeNav.active.rawApps, scope.homeNav.active.search, 'name');
            scope.homeNav.active.renderedInsights = Utility.filter(scope.homeNav.active.rawInsights, scope.homeNav.active.search, 'name');
        }


        /**
         * @name updateActiveApps
         * @desc update the active insights
         */
        function updateActiveApps(): void {
            let apps: any = semossCoreService.app.get('available');

            scope.homeNav.active.rawApps = [];
            for (let app in apps) {
                if (apps.hasOwnProperty(app)) {
                    scope.homeNav.active.rawApps.push({
                        name: String(apps[app].name).replace(/_/g, ' '),
                        image: apps[app].image,
                        appId: apps[app].app_id
                    });
                }
            }

            Utility.sort(scope.homeNav.active.rawApps, 'name');
        }

        /**
         * @name updateActiveInsights
         * @desc update the active insights
         */
        function updateActiveInsights(): void {
            scope.homeNav.active.rawInsights = [];

            const shared = semossCoreService.get('shared');
            for (let insight in shared) {
                if (shared.hasOwnProperty(insight)) {
                    if (!shared[insight]) {
                        continue;
                    }

                    let image = require('images/blue-logo.svg');
                    if (shared[insight].insight && shared[insight].insight.app_id && shared[insight].insight.app_insight_id) {
                        let imageUpdates = semossCoreService.getOptions('imageUpdates'),
                            insightImageKey = shared[insight].insight.app_id + shared[insight].insight.app_insight_id;

                        if (imageUpdates[insightImageKey]) {
                            image = imageUpdates[insightImageKey];
                        } else {
                            image = semossCoreService.app.generateInsightImageURL(shared[insight].insight.app_id, shared[insight].insight.app_insight_id);
                        }
                    }
                    scope.homeNav.active.rawInsights.push({
                        name: shared[insight].insight.name,
                        image: image,
                        insightID: shared[insight].insightID
                    })
                }
            }

            Utility.sort(scope.homeNav.active.rawInsights, 'name');
        }

        /**
         * @name remove
         * @desc close  an active insight or app
         * @param type - type of item to open
         * @param options - options for the active item
         */
        function remove(type: string, options: { appId?: string, insightID?: string }): void {
            if (type === 'app') {
                if (options.appId !== semossCoreService.app.get('newAppId')) {
                    semossCoreService.emit('close-app', {
                        appId: options.appId
                    });
                }
            } else if (type === 'insight') {
                semossCoreService.emit('execute-pixel', {
                    insightID: options.insightID,
                    commandList: [{
                        type: 'dropInsight',
                        components: [],
                        terminal: true
                    }]
                });
            }
        }

        /**
         * @name updateNavigation
         * @desc called when a route changes
         * @returns {void}
         */
        function updateNavigation(): void {
            scope.homeNav.state = {
                name: $state.current.name,
                params: $stateParams
            };
        }


        /**
         * @name initialize
         * @desc function that is called on directive load
         * @returns {void}
         */
        function initialize(): void {
            let navigationListener;

            navigationListener = $transitions.onSuccess({}, updateNavigation);

            scope.$on('$destroy', function () {
                navigationListener();

                // remove the active listeners
                for (let listenerIdx = 0, listenerLen = scope.homeNav.active.listeners.length; listenerIdx < listenerLen; listenerIdx++) {
                    scope.homeNav.active.listeners[listenerIdx]();
                }
            });

            updateNavigation();
        }

        initialize();
    }
}
