'use strict';

import './edit-insight.scss';
import angular from 'angular';

export default angular.module('app.edit-insight.directive', [])
    .directive('editInsight', editInsightDirective);

editInsightDirective.$inject = ['$q', 'monolithService', 'semossCoreService', 'ENDPOINT', 'CONFIG'];

function editInsightDirective($q, monolithService, semossCoreService, ENDPOINT, CONFIG) {
    editInsightCtrl.$inject = [];
    editInsightLink.$inject = ['scope'];

    return {
        restrict: 'E',
        template: require('./edit-insight.directive.html'),
        bindToController: {
            open: '=',
            insight: '='
        },
        controller: editInsightCtrl,
        controllerAs: 'editInsight',
        link: editInsightLink
    };

    function editInsightCtrl() {}

    function editInsightLink(scope) {
        scope.editInsight.CONFIG = CONFIG;

        scope.editInsight.name = {
            new: '',
            old: ''
        };

        scope.editInsight.description = {
            new: '',
            old: ''
        };

        scope.editInsight.tags = {
            new: [],
            old: []
        };

        scope.editInsight.image = {
            flow: undefined,
            src: ''
        };

        scope.editInsight.cache = {
            new: false,
            old: false
        };

        scope.editInsight.public = {
            new: false,
            old: false
        };

        scope.editInsight.users = {
            new: [],
            old: [],
            added: {
                selection: {},
                permission: 'READ_ONLY',
                id: '',
                name: '',
                options: []
            }
        };

        scope.editInsight.save = save;
        scope.editInsight.cancel = cancel;
        scope.editInsight.checkImage = checkImage;
        scope.editInsight.deleteInsightImage = deleteInsightImage;
        scope.editInsight.deleteCache = deleteCache;
        scope.editInsight.deleteUsers = deleteUsers;
        scope.editInsight.searchUsers = searchUsers;
        scope.editInsight.addUsers = addUsers;


        /** General */
        /**
         * @name reset
         * @desc  reset everything
         * @returns {void}
         */
        function reset() {
            let imageUpdates,
                insightImageKey;

            if (!scope.editInsight.open) {
                return;
            }

            // name
            scope.editInsight.name.new = scope.editInsight.insight.name;
            scope.editInsight.name.old = scope.editInsight.insight.name;

            // description
            scope.editInsight.description.new = scope.editInsight.insight.description;
            scope.editInsight.description.old = scope.editInsight.insight.description;

            // tags
            scope.editInsight.tags.new = semossCoreService.utility.freeze(scope.editInsight.insight.tags);
            scope.editInsight.tags.old = semossCoreService.utility.freeze(scope.editInsight.insight.tags);

            // image
            imageUpdates = semossCoreService.getOptions('imageUpdates');
            insightImageKey = scope.editInsight.insight.app_id + scope.editInsight.insight.app_insight_id;

            if (imageUpdates[insightImageKey]) {
                scope.editInsight.image.src = imageUpdates[insightImageKey];
            } else {
                scope.editInsight.image.src = semossCoreService.app.generateInsightImageURL(scope.editInsight.insight.app_id, scope.editInsight.insight.app_insight_id);
            }

            scope.editInsight.cache.new = scope.editInsight.insight.cacheable;
            scope.editInsight.cache.old = scope.editInsight.insight.cacheable;


            scope.editInsight.permission = 'READ_ONLY';

            // make call to get user's permission access for this insight so 'Public' can be disabled
            monolithService.getUserInsightPermission(scope.editInsight.insight.app_id, scope.editInsight.insight.app_insight_id)
                .then(function (response) {
                    if (response && response.data) {
                        if (response.data.permission === 'OWNER') {
                            scope.editInsight.permission = 'AUTHOR';
                        } else if (response.data.permission === 'EDIT') {
                            scope.editInsight.permission = 'EDITOR';
                        } else {
                            scope.editInsight.permission = 'READ_ONLY';
                        }
                    }
                });

            scope.editInsight.public.new = scope.editInsight.insight.insight_global;
            scope.editInsight.public.old = scope.editInsight.insight.insight_global;

            scope.editInsight.users.new = [];
            scope.editInsight.users.old = [];

            monolithService.getInsightUsers(scope.editInsight.insight.app_id, scope.editInsight.insight.app_insight_id)
                .then(function (response) {
                    let users = response.data || [];
                    scope.editInsight.users.new = JSON.parse(JSON.stringify(users));
                    scope.editInsight.users.old = JSON.parse(JSON.stringify(users));
                });

            searchUsers('');
        }

        /**
         * @name save
         * @desc actually save everything
         * @returns {void}
         */
        function save() {
            let promises: any = [],
                pixelComponents: any = [],
                file,
                users,
                mapped,
                added,
                edited,
                deleted;

            // name
            if (scope.editInsight.name.new !== scope.editInsight.name.old) {
                pixelComponents.push({
                    type: 'Pixel',
                    components: [
                        `SetInsightName(app=["${scope.editInsight.insight.app_id}"], id=["${scope.editInsight.insight.app_insight_id}"], insightName=["${scope.editInsight.name.new}"])`
                    ],
                    terminal: true
                });
            }

            // description
            if (scope.editInsight.description.new !== scope.editInsight.description.old) {
                pixelComponents.push({
                    type: 'Pixel',
                    components: [
                        `SetInsightDescription(app=["${scope.editInsight.insight.app_id}"], id=["${scope.editInsight.insight.app_insight_id}"], description=["${scope.editInsight.description.new}"])`
                    ],
                    terminal: true
                });
            }

            // tags
            if (JSON.stringify(scope.editInsight.tags.new) !== JSON.stringify(scope.editInsight.tags.old)) {
                let tags = scope.editInsight.tags.new;

                pixelComponents.push({
                    type: 'Pixel',
                    components: [
                        `SetInsightTags(app=["${scope.editInsight.insight.app_id}"], id=["${scope.editInsight.insight.app_insight_id}"], tags=${JSON.stringify(tags)})`
                    ],
                    terminal: true
                });
            }

            // file
            file = scope.editInsight.image.flow && scope.editInsight.image.flow.files && scope.editInsight.image.flow.files.length > 0 ? scope.editInsight.image.flow.files[0].file : undefined;
            if (file) {
                promises.push(monolithService.uploadInsightImage(scope.editInsight.insight.app_id, scope.editInsight.insight.app_insight_id, file));
            }

            // cache
            if (scope.editInsight.cache.new !== scope.editInsight.cache.old) {
                pixelComponents.push({
                    type: 'Pixel',
                    components: [
                        `SetInsightCacheable(app=["${scope.editInsight.insight.app_id}"], id=["${scope.editInsight.insight.app_insight_id}"], cache=[${scope.editInsight.cache.new}])`
                    ],
                    terminal: true
                });
            }

            // permissions
            if (scope.editInsight.public.new !== scope.editInsight.public.old) {
                promises.push(monolithService.setInsightGlobal(scope.editInsight.insight.app_id, scope.editInsight.insight.app_insight_id, scope.editInsight.public.new));
            }

            // users
            users = {};
            mapped = {};
            added = [];
            edited = [];
            deleted = [];

            // mapp all of the ids to the index in the array
            users = {};
            scope.editInsight.users.old.forEach((value, idx) => {
                users[value.id] = idx;
            });

            for (let newIdx = 0, newLen = scope.editInsight.users.new.length; newIdx < newLen; newIdx++) {
                // i keep track of what was already mapped, incase there are duplicates (there shouldn't be)
                if (mapped.hasOwnProperty(scope.editInsight.users.new[newIdx].id)) {
                    continue;
                }
                // is it already added?
                if (users.hasOwnProperty(scope.editInsight.users.new[newIdx].id)) {
                    // check if its been edited
                    if (scope.editInsight.users.new[newIdx].permission !== scope.editInsight.users.old[users[scope.editInsight.users.new[newIdx].id]].permission) {
                        edited.push(scope.editInsight.users.new[newIdx]);
                    }


                    // remove from mapping
                    delete users[scope.editInsight.users.new[newIdx].id];
                } else {
                    added.push(scope.editInsight.users.new[newIdx]);
                }


                mapped[scope.editInsight.users.new[newIdx].id] = true;
            }

            // these are all the removed one (leftover)
            for (let user in users) {
                if (users.hasOwnProperty(user)) {
                    deleted.push(scope.editInsight.users.old[users[user]]);
                }
            }

            if (added.length !== 0 || edited.length !== 0 || deleted.length !== 0) {
                added.forEach((user) => {
                    promises.push(monolithService.addInsightUserPermission(scope.editInsight.insight.app_id, scope.editInsight.insight.app_insight_id, user.id, user.permission));
                });

                edited.forEach((user) => {
                    promises.push(monolithService.editInsightUserPermission(scope.editInsight.insight.app_id, scope.editInsight.insight.app_insight_id, user.id, user.permission));
                });

                deleted.forEach((user) => {
                    promises.push(monolithService.removeInsightUserPermission(scope.editInsight.insight.app_id, scope.editInsight.insight.app_insight_id, user.id));
                });
            }


            // message pixel
            if (pixelComponents.length > 0) {
                let message = semossCoreService.utility.random('execute-pixel'),
                    deffered = $q.defer();

                semossCoreService.once(message, function (response) {
                    let type = response.pixelReturn[0].operationType;

                    if (type.indexOf('ERROR') > -1) {
                        // because it is pixel, it will resolve
                        deffered.reject();
                        return;
                    }

                    deffered.resolve();
                });

                promises.push(deffered.promise);

                semossCoreService.emit('query-pixel', {
                    commandList: pixelComponents,
                    response: message
                });
            }

            if (promises.length === 0) {
                semossCoreService.emit('alert', {
                    color: 'warn',
                    text: 'Nothing to update.'
                });
            } else {
                $q.all(promises).then(function () {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully updated insight.'
                    });

                    // update with the new information
                    scope.editInsight.insight.name = scope.editInsight.name.new;

                    // description
                    scope.editInsight.insight.description = scope.editInsight.description.new;

                    // tags
                    scope.editInsight.insight.tags = scope.editInsight.tags.new;

                    // image
                    if (file) {
                        // tag the insight as updated image so we can refresh it next time we get it
                        semossCoreService.setOptions('imageUpdates', scope.editInsight.insight.app_id + scope.editInsight.insight.app_insight_id, semossCoreService.app.generateInsightImageURL(scope.editInsight.insight.app_id, scope.editInsight.insight.app_insight_id) + '&time=' + new Date().getTime());
                        scope.editInsight.image.src = semossCoreService.getOptions('imageUpdates')[scope.editInsight.insight.app_id + scope.editInsight.insight.app_insight_id];
                        semossCoreService.emit('update-insight-image');
                    }

                    // cache
                    scope.editInsight.insight.cacheable = scope.editInsight.cache.new;

                    // permissions
                    scope.editInsight.insight.insight_global = scope.editInsight.public.new;

                    reset();
                }, function (error) {
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: error || 'Error updating insight.'
                    });
                });
            }

            // close the model
            scope.editInsight.open = false;
        }

        /**
         * @name cancel
         * @desc do not save things
         * @returns {void}
         */
        function cancel() {
            scope.editInsight.open = false;
        }


        /** Image */
        /**
         * @name checkImage
         * @desc update and save the image
         * @param {file} file - flow file
         * @return {void}
         */
        function checkImage(file) {
            const fileExtension = file.getExtension();
            if (fileExtension !== 'png' && fileExtension !== 'svg' && fileExtension !== 'jpg' && fileExtension !== 'jpeg') {
                semossCoreService.emit('alert', {
                    color: 'error',
                    text: 'Image must be .png, .svg, .jpg, .jpeg'
                });

                return false;
            }


            return true;
        }

        /**
         * @name deleteInsightImage
         * @desc delete the app image
         * @returns {void}
         */
        function deleteInsightImage(): void {
            monolithService.deleteInsightImage(scope.editInsight.insight.app_id, scope.editInsight.insight.app_insight_id).then(response => {
                // scope.editApp.app.last_app_image_timestamp = response.last_app_image_timestamp;
                semossCoreService.emit('alert', {
                    color: 'success',
                    text: 'Insight image has been deleted.'
                });
                scope.editInsight.image.flow.files = [];
                // tag the insight as updated image so we can refresh it next time we get it
                semossCoreService.setOptions('imageUpdates', scope.editInsight.insight.app_id + scope.editInsight.insight.app_insight_id, semossCoreService.app.generateInsightImageURL(scope.editInsight.insight.app_id, scope.editInsight.insight.app_insight_id) + '&time=' + new Date().getTime());
                scope.editInsight.image.src = semossCoreService.getOptions('imageUpdates')[scope.editInsight.insight.app_id + scope.editInsight.insight.app_insight_id];
                semossCoreService.emit('update-insight-image');
            });
        }

        /**
         * @name deleteCache
         * @desc delete the cache so that it can recache once the insight runs again
         * @returns {void} 
         */
        function deleteCache() {
            let message = semossCoreService.utility.random('query-pixel');

            semossCoreService.once(message, function (response) {
                let type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') > -1) {
                    return;
                }

                semossCoreService.emit('alert', {
                    color: 'success',
                    text: 'Successfully deleted insight cache'
                });
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    type: 'deleteInsightCache',
                    components: [scope.editInsight.insight.app_id, scope.editInsight.insight.app_insight_id],
                    terminal: true
                }],
                response: message
            });
        }

        /** Users */
        /**
         * @name deleteUsers
         * @desc delete a newly added user
         * @param {string} deletedUser - user to delete
         * @returns {void}
         */
        function deleteUsers(deletedUser: string) {
            var index = scope.editInsight.users.new.indexOf(deletedUser);
            scope.editInsight.users.new.splice(index, 1);
        }

        /**
         * @name searchUsers
         * @param {string} search - search term
         * @desc search the users
         * @returns {void}
         */
        function searchUsers(search) {
            scope.editInsight.users.added.options = [];

            // get list of users that meet the search input
            monolithService.getUserInformation(search)
                .then(function (data) {
                    scope.editInsight.users.added.options = data.map((user) => {
                        user.display = user.name + ' | ' + user.email;
                        user.selection = {'name':user.name, 'id':user.id, 'email':user.email};
                        return user;
                    });
                });
        }

        /**
         * @name addUsers
         * @desc add the users to the list
         * @returns {void}
         */
        function addUsers() {
            let exists;
            if (!scope.editInsight.users.added.selection) {
                semossCoreService.emit('alert', {
                    color: 'warn',
                    text: 'Please select a user.'
                });
                return;
            }
            // set both the id and name for the display
            scope.editInsight.users.added.id = scope.editInsight.users.added.selection.id;
            if (!scope.editInsight.users.added.id) {
                semossCoreService.emit('alert', {
                    color: 'warn',
                    text: 'Please select a user.'
                });
                return;
            }
            scope.editInsight.users.added.name = scope.editInsight.users.added.selection.name;

            exists = false;
            for (let newIdx, newLen = scope.editInsight.users.new.length; newIdx < newLen; newIdx++) {
                if (scope.editInsight.users.new[newIdx] === scope.editInsight.users.added.id) {
                    // update the permission
                    scope.editInsight.users.new[newIdx].permission = scope.editInsight.users.added.permission;
                    exists = true;
                    break;
                }
            }

            if (!exists) {
                scope.editInsight.users.new.push({
                    permission: scope.editInsight.users.added.permission,
                    id: scope.editInsight.users.added.id,
                    name : scope.editInsight.users.added.name
                });
            }

            scope.editInsight.users.added.permission = 'READ_ONLY';
            scope.editInsight.users.added.id = '';
        }

        /** Utility */
        /**
         * @name initialize
         * @desc initialize the module
         * @returns {void}
         */
        function initialize() {
            reset();

            scope.$watch(function () {
                return JSON.stringify(scope.editInsight.insight) + '_' + scope.editInsight.open;
            }, function (newValue, oldValue) {
                if (newValue !== oldValue) {
                    reset();
                }
            });

            // listeners
            scope.$on('$destroy', function () {
                console.log('destroying editInsight....');
            });
        }

        initialize();
    }
}
