'use strict';

export default angular.module('app.database-admin.directive', [])
    .directive('databaseAdmin', databaseAdminDirective);

import './database-admin.scss';

databaseAdminDirective.$inject = ['$location', 'monolithService', 'semossCoreService', 'ENDPOINT', 'CONFIG'];

function databaseAdminDirective($location, monolithService, semossCoreService, ENDPOINT, CONFIG) {
    databaseAdminCtrl.$inject = [];
    databaseAdminLink.$inject = ['scope'];

    return {
        restrict: 'E',
        template: require('./database-admin.directive.html'),
        bindToController: {
            appId: '='
        },
        controller: databaseAdminCtrl,
        controllerAs: 'databaseAdmin',
        link: databaseAdminLink
    };

    function databaseAdminCtrl() {}

    function databaseAdminLink(scope) {
        scope.databaseAdmin.loading = false;
        scope.databaseAdmin.insights = {
            canCollect: false,
            input: '',
            offset: 0,
            limit: 50,
            results: [],
            selected: []
        };

        scope.databaseAdmin.edit = {
            open: false,
            insight: undefined
        };


        scope.databaseAdmin.delete = {
            open: false
        };

        scope.databaseAdmin.optimized = false;

        scope.databaseAdmin.CONFIG = CONFIG;

        scope.databaseAdmin.getInsights = getInsights;
        scope.databaseAdmin.searchAppInsights = searchAppInsights;
        scope.databaseAdmin.getMoreInsights = getMoreInsights;
        scope.databaseAdmin.multiSaveImg = multiSaveImg;
        scope.databaseAdmin.spliceInsight = spliceInsight;
        scope.databaseAdmin.deleteInsights = deleteInsights;
        scope.databaseAdmin.openDelete = openDelete;
        scope.databaseAdmin.closeDelete = closeDelete;
        scope.databaseAdmin.openEdit = openEdit;
        scope.databaseAdmin.optimize = optimize;
        scope.databaseAdmin.deleteCache = deleteCache;

        // Default Insights
        scope.databaseAdmin.addDefaultInsights = addDefaultInsights;
        scope.databaseAdmin.defaultInsightOptions = [
        {
            'display': 'Explore an Instance',
            'value': 'explore'
        },
        {
            'display': 'Grid Delta',
            'value': 'grid-delta'
        },
        {
            'display': 'What are the modifications made to specific column(s)?',
            'value': 'audit-modification'
        },
        {
            'display': 'What are the modifications made to the specific column(s) over time?',
            'value': 'audit-timeline'
        },
        {
            'display': 'View insight usage stats',
            'value': 'insight-stats'
        }
        ];
        scope.databaseAdmin.selectedDefaultInsights = [];
        scope.databaseAdmin.defaultOverlay = false;

        /** Insight */
        /**
         * @name getInsights
         * @desc gets the insights for the current selected app
         * @param {boolean} clear - reset the search
         * @returns {void}
         */
        function getInsights(clear) {
            var message = semossCoreService.utility.random('query-pixel');

            scope.databaseAdmin.loading = true;

            if (clear) {
                scope.databaseAdmin.insights.canCollect = true;
                scope.databaseAdmin.insights.offset = 0;
            }

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var output = response.pixelReturn[0].output,
                    i,
                    len;

                if (clear) {
                    scope.databaseAdmin.insights.selected = [];
                    scope.databaseAdmin.insights.results = [];
                }

                for (i = 0, len = output.length; i < len; i++) {
                    output[i].displayDate = getDateFormat(new Date(output[i].last_modified_on));
                }

                scope.databaseAdmin.insights.results = scope.databaseAdmin.insights.results.concat(output);
                scope.databaseAdmin.insights.canCollect = (output.length === scope.databaseAdmin.insights.limit);
                scope.databaseAdmin.loading = false;
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'getInsights',
                    components: [
                        scope.databaseAdmin.appId,
                        scope.databaseAdmin.insights.limit,
                        scope.databaseAdmin.insights.offset,
                        scope.databaseAdmin.insights.input,
                        []
                    ],
                    terminal: true
                }],
                listeners: [], // taken care of in the directive
                response: message
            });
        }
        /**
         * @name addDefaultInsights
         * @desc Adds deleted default insights to the app
         * @returns {void}
         */
        function addDefaultInsights() {
            var message = semossCoreService.utility.random('query-pixel'),
                selectedOptions = scope.databaseAdmin.selectedDefaultInsights,
                insightOptions = JSON.stringify(selectedOptions),
                appId = JSON.stringify(scope.databaseAdmin.appId);

            semossCoreService.once(message, function (response) {
                if (response.pixelReturn[0].operationType.indexOf('ERROR') === -1) {
                    // Refresh insights and close overlay
                    getInsights(true);
                    scope.databaseAdmin.defaultOverlay = false;
                }
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'Pixel',
                    components: [
                        'AddDefaultInsights(app = [' + appId + '], insights = ' + insightOptions + ');'
                    ],
                    terminal: true
                }],
                listeners: [],
                response: message
            });
        }

        /**
         * @name getDateFormat
         * @desc format a date into the wanted format
         * @param {date} date - date the date to format
         * @returns {string} formatted date
         */
        function getDateFormat(date) {
            return date.toLocaleDateString([], {
                year: 'numeric',
                month: 'short',
                day: 'numeric'
            }) + ' ' + date.toLocaleTimeString();
        }

        /**
         * @name getMoreInsights
         * @desc gets the insights for the current selected app
         * @returns {void}
         */
        function getMoreInsights() {
            if (!scope.databaseAdmin.insights.canCollect) {
                return;
            }

            // increment the offset to get more
            scope.databaseAdmin.insights.offset += scope.databaseAdmin.insights.limit;
            getInsights();
        }

        /**
         * @name searchAppInsights
         * @desc searchs the app insights
         * @param {string} search - search string
         * @returns {void}
         */
        function searchAppInsights(search) {
            scope.databaseAdmin.insights.input = search;
            getInsights(true);
        }


        /**
         * @name multiSaveImg
         * @desc Takes in the list of insights the user has checked in the insights panel and
         *       updates the images for those insights on feed. If the insight already has an image
         *       it will be replaced with the same image. This also updates the images of dashboards. Note,
         *       sometimes the images don't come back right away (assuming this is back-end sync related).
         * @returns {void}
         */
        function multiSaveImg() {
            var i,
                len,
                loc,
                pixelArray = [],
                message = semossCoreService.utility.random('execute-pixel');

            loc = $location.absUrl().split('#!')[0];

            // Go through the list of insights
            for (i = 0, len = scope.databaseAdmin.insights.selected.length; i < len; i++) {
                pixelArray.push({
                    type: 'updateInsightImage',
                    components: [scope.databaseAdmin.insights.selected[i].app_id, scope.databaseAdmin.insights.selected[i].app_insight_id, loc]
                });
            }

            // Add in terminal statement to final pixel Object and execute
            pixelArray[pixelArray.length - 1].terminal = true;
            semossCoreService.once(message, function (response) {
                if (response.pixelReturn[0].operationType.indexOf('ERROR') === -1) {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Insight image(s) has been created.'
                    });

                    // refresh the insight images
                    semossCoreService.emit('reset-insights-list');
                }
            });
            semossCoreService.emit('query-pixel', {
                commandList: pixelArray,
                response: message
            });

            // Uncheck checked insights and hide buttons
            scope.databaseAdmin.insights.selected = [];
        }

        /**
         * @name spliceInsight
         * @desc remove new value
         * @param {number} idx - value's idx to remove
         * @returns {void}
         */
        function spliceInsight(idx) {
            scope.databaseAdmin.insights.selected.splice(idx, 1);
        }

        /**
         * @name deleteInsights
         * @desc delete the selected insights
         * @returns {void}
         */
        function deleteInsights() {
            var insights = [];

            for (let insightIdx = 0, insightLen = scope.databaseAdmin.insights.selected.length; insightIdx < insightLen; insightIdx++) {
                insights.push(scope.databaseAdmin.insights.selected[insightIdx].app_insight_id);
            }

            closeDelete();

            if (insights.length > 0) {
                scope.databaseAdmin.loading = true;

                semossCoreService.once('delete-app-insights-end', function (data) {
                    if (!data.success) {
                        // error occured
                        // remove the loading screen
                        scope.databaseAdmin.loading = false;
                    } else {
                        // deletion occured successfully
                        semossCoreService.emit('alert', {
                            color: 'success',
                            text: 'Insights successfully deleted.'
                        });

                        scope.databaseAdmin.loading = false;

                        getInsights(true);
                    }
                });

                semossCoreService.emit('delete-app-insights', {
                    appId: scope.databaseAdmin.appId,
                    insights: insights
                });
            }
        }

        /**
         * @name deleteCache
         * @param {*} insight the insight info
         * @desc delete the cache so that it can recache once the insight runs again
         * @returns {void} 
         */
        function deleteCache(insight) {
            var message = semossCoreService.utility.random('query-pixel');
            semossCoreService.emit('query-pixel', {
                commandList: [{
                    type: 'deleteInsightCache',
                    components: [insight.app_id, insight.app_insight_id],
                    terminal: true
                }],
                response: message
            });

            semossCoreService.once(message, function (response) {
                try {
                    if (response.pixelReturn[0].operationType[0] !== 'ERROR') {
                        semossCoreService.emit('alert', {
                            color: 'success',
                            text: 'Successfully deleted insight cache'
                        });
                    }
                } catch (err) {
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'An error occurred.'
                    });
                }
            });
        }


        /** Generic */
        /**
         * @name openDelete
         * @desc open the overlay
         * @returns {void}
         */
        function openDelete() {
            scope.databaseAdmin.delete.open = true;
        }

        /**
         * @name closeDelete
         * @desc close the overlay
         * @returns {void}
         */
        function closeDelete() {
            scope.databaseAdmin.delete.open = false;
        }

        /**
         * @name openEdit
         * @desc open the overlay
         * @param {object} insight - insight to select
         * @returns {void}
         */
        function openEdit(insight) {
            scope.databaseAdmin.edit.open = true;
            scope.databaseAdmin.edit.insight = insight;
        }

        /**
         * @name optimize
         * @description runs optimize routine
         * @return {void}
         */
        function optimize() {
            var message = semossCoreService.utility.random('meta');

            semossCoreService.emit('meta-pixel', {
                commandList: [{
                    type: 'extractAppMeta',
                    components: [scope.databaseAdmin.appId, true],
                    terminal: true
                }],
                meta: true,
                response: message
            });

            semossCoreService.once(message, function (response) {
                if (response.pixelReturn[0].operationType[0] === 'ERROR') {
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'App could not be optimized'
                    });
                }
            });
        }

        /**
         * @name initialize
         * @desc initialize the module
         * @returns {void}
         */
        function initialize() {
            getInsights(true);

            // listeners
            scope.$on('$destroy', function () {
                console.log('destroying databaseAdmin....');
            });
        }

        initialize();
    }
}
