'use strict';

import Utility from '@/core/utility/utility.js';

export default angular.module('app.app-home-insights.directive', [])
    .directive('appHomeInsights', appHomeInsightsDirective);

import './app-home-insights.scss';

appHomeInsightsDirective.$inject = ['ENDPOINT', 'semossCoreService', '$timeout'];


function appHomeInsightsDirective(ENDPOINT, semossCoreService, $timeout) {
    appHomeInsightsCtrl.$inject = [];
    appHomeInsightsLink.$inject = ['scope', 'ele'];

    return {
        restrict: 'E',
        template: require('./app-home-insights.directive.html'),
        controller: appHomeInsightsCtrl,
        link: appHomeInsightsLink,
        scope: {},
        bindToController: {
            appId: '='
        },
        controllerAs: 'appHomeInsights'
    };

    function appHomeInsightsCtrl() {}

    function appHomeInsightsLink(scope, ele) {
        let scrollEle,
            scrollTimeout,
            searchTimeout;

        scope.appHomeInsights.search = {
            canCollect: false,
            loading: true,
            input: '',
            offset: 0,
            limit: 41,
            results: []
        };

        scope.appHomeInsights.tags = {
            options: [],
            selected: []
        };

        scope.appHomeInsights.edit = {
            open: false,
            insight: undefined
        };

        scope.appHomeInsights.showDelete = false;
        scope.appHomeInsights.appToDelete = null;

        scope.appHomeInsights.searchInsights = searchInsights;
        scope.appHomeInsights.filterInsights = filterInsights;
        scope.appHomeInsights.createViz = createViz;
        scope.appHomeInsights.getNLPSuggestedInsights = getNLPSuggestedInsights;
        scope.appHomeInsights.openEdit = openEdit;
        scope.appHomeInsights.changeMode = changeMode;
        scope.appHomeInsights.openDelete = openDelete;
        scope.appHomeInsights.closeDelete = closeDelete;
        scope.appHomeInsights.onDelete = onDelete;

        /** Insights */
        /**
         * @name getInsights
         * @desc gets the insights for the current selected app
         * @param {boolean} clear - reset the search
         * @returns {void}
         */
        function getInsights(clear) {
            let message = semossCoreService.utility.random('query-pixel');

            scope.appHomeInsights.search.loading = true;

            // register message to come back to
            semossCoreService.once(message, function (response) {
                let output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType[0];

                scope.appHomeInsights.search.loading = false;

                if (type.indexOf('ERROR') > -1) {
                    return;
                }

                if (clear) {
                    scope.appHomeInsights.search.canCollect = true;
                    scope.appHomeInsights.search.offset = 0;
                    scope.appHomeInsights.search.results = [];
                }

                for (let insightIdx = 0, insightLen = output.length; insightIdx < insightLen; insightIdx++) {
                    output[insightIdx].image = getImage(output[insightIdx].app_insight_id);
                    output[insightIdx].last_modified_on = new Date(output[insightIdx].last_modified_on);
                    output[insightIdx].last_modified_onDisplay = getDateFormat(output[insightIdx].last_modified_on);
                    output[insightIdx].created_on = new Date(output[insightIdx].created_on);
                    output[insightIdx].created_onDisplay = getDateFormat(output[insightIdx].created_on);

                    scope.appHomeInsights.search.results.push(output[insightIdx]);
                }

                scope.appHomeInsights.search.canCollect = (output.length === scope.appHomeInsights.search.limit);
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'getInsights',
                    components: [
                        scope.appHomeInsights.appId,
                        scope.appHomeInsights.search.limit,
                        scope.appHomeInsights.search.offset,
                        scope.appHomeInsights.search.input,
                        scope.appHomeInsights.tags.selected ? scope.appHomeInsights.tags.selected : []
                    ],
                    terminal: true
                }],
                listeners: [], // taken care of in the directive
                response: message
            });
        }

        /**
         * @name updateInsightImage
         * @desc update the insight image urls
         * @returns {void}
         */
        function updateInsightImage() {
            // loop through scope.appHomeInsights.search.results and check imageUpdates to see if image needs to be updated
            let insightIdx;
            for (insightIdx = 0; insightIdx < scope.appHomeInsights.search.results.length; insightIdx++) {
                scope.appHomeInsights.search.results[insightIdx].image = getImage(scope.appHomeInsights.search.results[insightIdx].app_insight_id);
            }
        }

        /**
         * @name getTags
         * @desc gets the list of available tags
         * @returns {void}
         */
        function getTags() {
            let message = semossCoreService.utility.random('query-pixel');

            // register message to come back to
            semossCoreService.once(message, function (response) {
                let output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType[0];

                if (type.indexOf('ERROR') > -1) {
                    return;
                }

                scope.appHomeInsights.tags.options = [];
                scope.appHomeInsights.tags.selected = [];
                for (let tagIdx = 0, tagLen = output.length; tagIdx < tagLen; tagIdx++) {
                    scope.appHomeInsights.tags.options.push(output[tagIdx].tag);
                }

                Utility.sort(scope.appHomeInsights.tags.options);
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'getAvailableTags',
                    components: [
                        scope.appHomeInsights.appId
                    ],
                    terminal: true
                }],
                listeners: [], // taken care of in the directive
                response: message
            });
        }

        /**
         * @name getMoreInsights
         * @desc gets the insights for the current selected app
         * @returns {void}
         */
        function getMoreInsights() {
            if (!scope.appHomeInsights.search.canCollect) {
                return;
            }

            if (scrollTimeout) {
                $timeout.cancel(scrollTimeout);
            }

            // debounce
            scrollTimeout = $timeout(function () {
                // check if it is at the bottom and going downwards
                if (((scrollEle.scrollTop + scrollEle.clientHeight) >= (scrollEle.scrollHeight * 0.75)) && !scope.appHomeInsights.search.loading) {
                    // increment the offset to get more
                    scope.appHomeInsights.search.offset += scope.appHomeInsights.search.limit;
                    getInsights(false);
                }
            }, 300);
        }

        /**
         * @name searchInsights
         * @desc searchs the app insights
         * @returns {void}
         */
        function searchInsights() {
            if (searchTimeout) {
                $timeout.cancel(searchTimeout);
            }

            // debounce
            searchTimeout = $timeout(function () {
                scope.appHomeInsights.search.offset = 0;
                getInsights(true);
                $timeout.cancel(searchTimeout);
            }, 300);
        }

        /**
         * @name filterInsights
         * @desc filter insight by a tag
         * @param {string} tag - tag to filter on
         * @returns {void}
         */
        function filterInsights(tag) {
            if (scope.appHomeInsights.tags.selected.indexOf(tag) === -1) {
                scope.appHomeInsights.tags.selected.push(tag);
                searchInsights();
            }
        }

        /**
         * @name clearInsights
         * @desc clears out the current filters and gets the app
         * @returns {void}
         */
        function clearInsights() {
            scope.appHomeInsights.search.input = '';
            scope.appHomeInsights.tags.selected = [];
            getInsights(true);
        }

        /**
         * @name getImage
         * @desc gets the image for the app
         * @param {string} rdbmsId - rdbmsId of the image
         * @returns {void}
         */
        function getImage(rdbmsId) {
            let imageUpdates = semossCoreService.getOptions('imageUpdates'),
                insightImageKey = scope.appHomeInsights.appId + rdbmsId;

            if (imageUpdates[insightImageKey]) {
                return imageUpdates[insightImageKey];
            }

            return semossCoreService.app.generateInsightImageURL(scope.appHomeInsights.appId, rdbmsId);
        }

        /**
         * @name getDateFormat
         * @desc format a date into the wanted format
         * @param {date} date - date the date to format
         * @returns {string} formatted date
         */
        function getDateFormat(date) {
            return date.toLocaleDateString([], {
                year: 'numeric',
                month: 'short',
                day: 'numeric'
            }) + ' ' + date.toLocaleTimeString();
        }

        /** NLP */


        /**
         * @name createViz
         * @param {event} $event - DOM event
         * @param {string} insight - type of widget to open
         * @desc function that adds in a new upload panel
         * @return {void}
         */
        function createViz($event, insight) {
            let newSheet = true;

            // close the insight
            insight.open = false;

            if ($event && ($event.metaKey || $event.ctrlKey || $event.keyCode === 17 || $event.keyCode === 224)) {
                newSheet = false;
            }

            semossCoreService.emit('open', {
                type: 'insight',
                options: insight,
                newSheet: newSheet
            });
        }

        /**
         * @name getNLPSuggestedInsights
         * @desc execute query using NaturalLanguageSearch to autocreate a visualization
         * @returns {void}
         */
        function getNLPSuggestedInsights() {
            let message = semossCoreService.utility.random('query-pixel');

            scope.appHomeInsights.search.loading = true;

            semossCoreService.once(message, function (response) {
                let output = response.pixelReturn[0].output[0],
                    type = response.pixelReturn[0].operationType[0];

                scope.appHomeInsights.search.loading = false;

                if (type.indexOf('ERROR') > -1) {
                    return;
                }

                if (output.hasOwnProperty('pixel')) {
                    createNLPSuggestedInsight(output);
                }
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'naturalLanguageSearch',
                    components: [
                        [scope.appHomeInsights.appId],
                        scope.appHomeInsights.search.input
                    ],
                    terminal: true
                }],
                listeners: [],
                response: message
            });
        }

        /**
         * @name createNLPSuggestedInsight
         * @param {object} insight - selected recommended insight to create
         * @desc begins the process to create a recommended insight
         * @returns {void}
         */
        function createNLPSuggestedInsight(insight) {
            semossCoreService.emit('open', {
                type: 'pixel',
                options: insight,
                newSheet: true
            });
        }


        /**
         * @name openEdit
         * @desc open the overlay
         * @param {object} insight - insight to select
         * @returns {void}
         */
        function openEdit(insight) {
            // close the insight
            insight.open = false;

            scope.appHomeInsights.edit.open = true;
            scope.appHomeInsights.edit.insight = insight;
        }

        /**
         * @name changeMode
         * @desc change the layout mode
         * @param {string} mode - the layout mode to change to
         * @returns {void}
         */
        function changeMode(mode) {
            scope.appHomeInsights.mode = mode;
            window.localStorage.setItem('smss-insights-mode', mode);
        }

        /**
         * @name openDelete
         * @desc Opens the delete overlay and sets the insight to delete
         * @param {*} insight - the insight to delete
         * @returns {void}
         */
        function openDelete(insight) {
            scope.appHomeInsights.showDelete = true;
            scope.appHomeInsights.insightToDelete = insight;
        }

        /**
         * @name closeDelete
         * @desc Closes the delete overlay and resets the insight to delete
         * @returns {void}
         */
        function closeDelete() {
            scope.appHomeInsights.showDelete = false;
            scope.appHomeInsights.insightToDelete = null;
        }

        /**
         * @name onDelete
         * @desc Deletes the insight
         * @returns {void}
         */
        function onDelete() {
            const insightId = scope.appHomeInsights.insightToDelete.app_insight_id;
            if (insightId && insightId.length) {
                semossCoreService.once('delete-app-insights-end', function (response) {
                    if (response.success) {
                        scope.appHomeInsights.search.results = scope.appHomeInsights.search.results.filter((app) => app.app_insight_id !== insightId);
                        semossCoreService.emit('alert', {
                            color: 'success',
                            text: 'Insight successfully deleted'
                        });
                    }
                });
                semossCoreService.emit('delete-app-insights', {
                    appId: scope.appHomeInsights.insightToDelete.app_id,
                    insights: [insightId]
                });
            } else {
                semossCoreService.emit('alert', {
                    color: 'error',
                    text: 'Error deleting app.'
                });
            }
            scope.appHomeInsights.insightToDelete = null;
            scope.appHomeInsights.showDelete = false;
        }

        /**
         * @name initialize
         * @desc initialize the module
         * @returns {void}
         */
        function initialize() {
            let resetInsightsListener, updateImageURLListener;

            scrollEle = ele[0].querySelector('#app-home-insights__scroll')
                .querySelector('#smss-scroll-scroller');

            scrollEle.addEventListener('scroll', getMoreInsights);

            resetInsightsListener = semossCoreService.on('reset-insights-list', function () {
                clearInsights();
            });

            updateImageURLListener = semossCoreService.on('update-insight-image', updateInsightImage);

            if (window.localStorage.getItem('smss-insights-mode')) {
                scope.appHomeInsights.mode = window.localStorage.getItem('smss-insights-mode');
            } else {
                scope.appHomeInsights.mode = 'CARD';
                window.localStorage.setItem('smss-insights-mode', 'CARD');
            }

            // add listeners
            scope.$on('$destroy', function () {
                resetInsightsListener();
                updateImageURLListener();
                scrollEle.removeEventListener('scroll', getMoreInsights);
            });

            getInsights(true);
            getTags();
        }

        initialize();
    }
}
