'use strict';

/**
 * @name appHomeCollab
 * @desc appHomeCollab - accessible as the context menu widget
 */
export default angular.module('app.app-home-collab.directive', [])
    .directive('appHomeCollab', appHomeCollabDirective);

import './app-home-collab.scss';

appHomeCollabDirective.$inject = ['$timeout', 'semossCoreService'];


function appHomeCollabDirective($timeout, semossCoreService) {
    appHomeCollabCtrl.$inject = ['$scope'];

    return {
        restrict: 'E',
        template: require('./app-home-collab.directive.html'),
        controller: appHomeCollabCtrl,
        scope: {},
        bindToController: {
            appId: '='
        },
        controllerAs: 'appHomeCollab'
    };

    function appHomeCollabCtrl($scope) {
        var appHomeCollab = this,
            logoutListener,
            loginListener;

        // variables
        appHomeCollab.accordionSettings = {
            repo: {
                height: 100,
                disabled: true
            },
            settings: {
                height: 0,
                disabled: true
            },
            sync: {
                height: 0,
                disabled: true
            }
        };

        appHomeCollab.engines = {
            options: [],
            selected: '',
            isGit: false
        };

        appHomeCollab.repos = {
            valid: false,
            options: [],
            selected: '',
            custom: false,
            user: appHomeCollab.credentials && appHomeCollab.credentials.username ? appHomeCollab.credentials.username : '',
            name: ''
        };

        appHomeCollab.members = [];
        appHomeCollab.member = {
            selected: '',
            options: []
        };

        appHomeCollab.files = {
            options: [],
            selected: []
        };

        appHomeCollab.databaseSync = false;

        appHomeCollab.validation = {
            open: false,
            valid: false,
            type: '',
            name: '',
            title: ''
        };


        // functions
        appHomeCollab.login = login;
        appHomeCollab.toggleRepo = toggleRepo;
        appHomeCollab.selectRepo = selectRepo;
        appHomeCollab.validateRepo = validateRepo;
        appHomeCollab.createRepo = createRepo;
        appHomeCollab.deleteRepo = deleteRepo;
        appHomeCollab.dropRepo = dropRepo;
        appHomeCollab.getTeam = getTeam;
        appHomeCollab.searchMember = searchMember;
        appHomeCollab.addMember = addMember;
        appHomeCollab.removeMember = removeMember;
        appHomeCollab.getStatus = getStatus;
        appHomeCollab.pull = pull;
        appHomeCollab.syncAll = syncAll;
        appHomeCollab.syncSpecific = syncSpecific;
        appHomeCollab.checkValidation = checkValidation;
        appHomeCollab.confirmValidation = confirmValidation;

        /** Default */
        /**
         * @name resetCollaboration
         * @desc checks if collaboration is enabled
         * @returns {void}
         */

        function resetCollaboration() {
            semossCoreService.getCredentials('github').then(function (response) {
                // check if the user is logged in
                if (!response.name) {
                    setAuth();
                    return;
                }

                // logged in
                $timeout(function () {
                    appHomeCollab.credentials = {
                        username: response.name
                    };
                    appHomeCollab.authenticated = true;

                    resetEngines();
                });
            });
        }

        /** Auth */
        /**
         * @name setAuth
         * @desc set the accordion for an unauthenticated
         * @returns {void}
         */
        function setAuth() {
            appHomeCollab.authenticated = false;
        }


        /**
         * @name login
         * @desc function that is called to login
         * @returns {void}
         */
        function login() {
            var message = semossCoreService.utility.random('login');
            semossCoreService.once(message, function (response) {
                if (response.success) {
                    resetCollaboration();
                }
            });
            semossCoreService.emit('oauth-login', {
                provider: 'github',
                message: message
            });
        }

        /** Engine */
        /**
         * @name resetEngines
         * @desc function gets the list of engine options (and sets the default)
         * @returns {void}
         */
        function resetEngines() {
            var currentAppName = semossCoreService.app.get(`available.${appHomeCollab.appId}.name`) || '';

            if (appHomeCollab.appId) {
                appHomeCollab.engines = {
                    selected: appHomeCollab.appId,
                    selectedName: currentAppName,
                    isGit: false
                };

                getGit();
            }
        }
        /**
         * @name getGit
         * @desc function gets the git status based on the selected engine
         * @returns {void}
         */
        function getGit() {
            var message = semossCoreService.utility.random('query-pixel');

            // clear
            appHomeCollab.repos = {
                valid: false,
                options: [],
                selected: '',
                custom: false,
                user: appHomeCollab.credentials && appHomeCollab.credentials.username ? appHomeCollab.credentials.username : '',
                name: ''
            };

            appHomeCollab.members = [];
            appHomeCollab.member = {
                selected: '',
                options: []
            };

            appHomeCollab.files = {
                options: [],
                selected: []
            };

            appHomeCollab.databaseSync = false;

            appHomeCollab.validation = {
                open: false,
                valid: false,
                type: '',
                name: '',
                title: ''
            };

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    appHomeCollab.engines.isGit = output;
                    if (appHomeCollab.engines.isGit) {
                        getRepos(appHomeCollab.engines.selected, false);
                    } else {
                        customRepo();
                    }
                }
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'isGit',
                    components: [
                        appHomeCollab.engines.selected
                    ],
                    terminal: true
                }],
                response: message
            });
        }

        /** Repos */
        /**
         * @name toggleRepo
         * @desc toggle the mode for repo creation
         * @returns {void}
         */
        function toggleRepo() {
            appHomeCollab.repos.custom = !appHomeCollab.repos.custom;

            if (appHomeCollab.repos.custom) {
                customRepo();
            } else {
                getRepos(appHomeCollab.engines.selected, appHomeCollab.repos.selected);
            }
        }

        /**
         * @name customRepo
         * @desc customRepo
         * @returns {void}
         */
        function customRepo() {
            appHomeCollab.repos.custom = true;

            appHomeCollab.accordionSettings = {
                repo: {
                    height: 100,
                    disabled: false
                },
                settings: {
                    height: 0,
                    disabled: true
                },
                sync: {
                    height: 0,
                    disabled: true
                }
            };
        }

        /**
         * @name getRepos
         * @desc function gets the list of repos options based on the selected engine
         * @param {string} engine - selected engine
         * @param {string} selected - use a selected repo?
         * @returns {void}
         */
        function getRepos(engine, selected) {
            var message = semossCoreService.utility.random('query-pixel');
            appHomeCollab.engines.selected = engine;

            // clear
            appHomeCollab.repos = {
                valid: false,
                options: [],
                selected: '',
                custom: false,
                user: appHomeCollab.credentials && appHomeCollab.credentials.username ? appHomeCollab.credentials.username : '',
                name: ''
            };

            appHomeCollab.members = [];
            appHomeCollab.member = {
                selected: '',
                options: []
            };

            appHomeCollab.files = {
                options: [],
                selected: []
            };

            appHomeCollab.databaseSync = false;

            appHomeCollab.validation = {
                open: false,
                valid: false,
                type: '',
                name: '',
                title: ''
            };

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    appHomeCollab.repos.options = output;

                    if (selected) {
                        selectRepo(selected);
                    } else if (appHomeCollab.repos.options.length > 0) {
                        selectRepo(appHomeCollab.repos.options[0].name);
                    } else {
                        customRepo();
                    }
                }
            });

            // get remote repos
            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'listAppRemotes',
                    components: [
                        appHomeCollab.engines.selected
                    ],
                    terminal: true
                }],
                response: message
            });
        }

        /**
         * @name selectRepo
         * @desc select a repo and get the associated information
         * @param {string} repo - location of the remote repo
         * @returns {void}
         */
        function selectRepo(repo) {
            appHomeCollab.repos.selected = repo;

            getTeam();
            getStatus();

            appHomeCollab.accordionSettings = {
                repo: {
                    height: 20,
                    disabled: false
                },
                settings: {
                    height: 40,
                    disabled: false
                },
                sync: {
                    height: 40,
                    disabled: false
                }
            };
        }


        /**
         * @name validateRepo
         * @desc is the repo valid?
         * @returns {void}
         */
        function validateRepo() {
            var newRepo = appHomeCollab.repos.user + '/' + appHomeCollab.repos.name,
                i, len;

            if (appHomeCollab.repos.user.length === 0) {
                appHomeCollab.repos.valid = false;
                return;
            }

            if (appHomeCollab.repos.name.length === 0) {
                appHomeCollab.repos.valid = false;
                return;
            }

            for (i = 0, len = appHomeCollab.repos.options.length; i < len; i++) {
                if (appHomeCollab.repos.options[i].name === newRepo) {
                    appHomeCollab.repos.valid = false;
                    return;
                }
            }
            appHomeCollab.repos.valid = true;
        }
        /**
         * @name createRepo
         * @desc create a new repo
         * @param {boolean} valid - able to proceed with irreversible actions?
         * @returns {void}
         */
        function createRepo(valid) {
            var newRepo, message = semossCoreService.utility.random('query-pixel');

            validateRepo();

            if (!appHomeCollab.repos.valid) {
                return;
            }

            newRepo = appHomeCollab.repos.user + '/' + appHomeCollab.repos.name;

            // list remote
            if (appHomeCollab.databaseSync && !valid) {
                appHomeCollab.validation.type = 'createRepo';
                appHomeCollab.validation.name = '';
                appHomeCollab.validation.title = 'Enter app name (' + appHomeCollab.engines.selectedName + ') to confirm';
                appHomeCollab.validation.valid = false;
                appHomeCollab.validation.open = true;
                return;
            }

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var type = response.pixelReturn[0].operationType;
                if (type.indexOf('ERROR') === -1) {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully created ' + newRepo
                    });
                    appHomeCollab.engines.isGit = true;
                    getRepos(appHomeCollab.engines.selected, newRepo);
                }
            });

            // list remote
            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'initAppRepo',
                    components: [
                        appHomeCollab.engines.selected,
                        newRepo,
                        valid
                    ],
                    terminal: true
                }],
                response: message
            });
        }

        /**
         * @name deleteRepo
         * @desc delete a repo
         * @param {boolean} valid - able to proceed with irreversible actions?
         * @returns {void}
         */
        function deleteRepo(valid) {
            var message = semossCoreService.utility.random('query-pixel');
            // list remote
            if (!valid) {
                appHomeCollab.validation.type = 'deleteRepo';
                appHomeCollab.validation.name = '';
                appHomeCollab.validation.title = 'Enter app name (' + appHomeCollab.engines.selectedName + ') to confirm';
                appHomeCollab.validation.valid = false;
                appHomeCollab.validation.open = true;
                return;
            }

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var type = response.pixelReturn[0].operationType;
                if (type.indexOf('ERROR') === -1) {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully deleted ' + appHomeCollab.repos.selected
                    });
                    getGit();
                }
            });

            // list remote
            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'deleteAppRepo',
                    components: [
                        appHomeCollab.engines.selected,
                        appHomeCollab.repos.selected
                    ],
                    terminal: true
                }],
                response: message
            });
        }

        /**
         * @name dropRepo
         * @desc drop a remote repo
         * @param {boolean} valid - able to proceed with irreversible actions?
         * @returns {void}
         */
        function dropRepo(valid) {
            var message = semossCoreService.utility.random('query-pixel');
            // list remote
            if (!valid) {
                appHomeCollab.validation.type = 'dropRepo';
                appHomeCollab.validation.name = '';
                appHomeCollab.validation.title = 'Enter app name (' + appHomeCollab.engines.selectedName + ') to confirm';
                appHomeCollab.validation.valid = false;
                appHomeCollab.validation.open = true;
                return;
            }

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var type = response.pixelReturn[0].operationType;
                if (type.indexOf('ERROR') === -1) {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully dropped ' + appHomeCollab.repos.selected
                    });
                    getGit();
                }
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'dropAppRepo',
                    components: [
                        appHomeCollab.engines.selected,
                        appHomeCollab.repos.selected
                    ],
                    terminal: true
                }],
                response: message
            });
        }


        /** Team */
        /**
         * @name getTeam
         * @desc get a team associated with a remote repo
         * @returns {void}
         */
        function getTeam() {
            var message = semossCoreService.utility.random('query-pixel');
            // clear
            appHomeCollab.members = [];
            appHomeCollab.member = {
                selected: '',
                options: []
            };

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;
                if (type.indexOf('ERROR') === -1) {
                    appHomeCollab.members = output;
                }
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'listAppCollaborators',
                    components: [
                        appHomeCollab.repos.selected
                    ],
                    terminal: true
                }],
                response: message
            });
        }

        /**
         * @name searchMember
         * @desc search a team member to your app
         * @param {string} search - search ter
         * @returns {void}
         */
        function searchMember(search) {
            var message = semossCoreService.utility.random('query-pixel');
            // clear
            appHomeCollab.member.options = [];
            if (search) {
                appHomeCollab.member.loading = true;
            } else {
                appHomeCollab.member.loading = false;
            }

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                appHomeCollab.member.loading = false;
                if (type.indexOf('ERROR') === -1) {
                    appHomeCollab.member.options = output;
                }
            });

            // list remote
            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'searchAppCollaborator',
                    components: [
                        search
                    ],
                    terminal: true
                }],
                listeners: [],
                response: message
            });
        }

        /**
         * @name addMember
         * @desc add a team member to your app
         * @returns {void}
         */
        function addMember() {
            var message = semossCoreService.utility.random('query-pixel');

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully invited ' + appHomeCollab.member.selected[0] + ' to ' + appHomeCollab.repos.selected
                    });
                    getTeam();
                }
            });

            // list remote
            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'addAppCollaborator',
                    components: [
                        appHomeCollab.repos.selected,
                        appHomeCollab.member.selected[0]
                    ],
                    terminal: true
                }],
                response: message
            });
        }

        /**
         * @name removeMember
         * @desc removes a team member from your app
         * @param {string} removedMember - member to remove from your app
         * @returns {void}
         */
        function removeMember(removedMember) {
            var message = semossCoreService.utility.random('query-pixel');

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully removed ' + removedMember + ' from ' + appHomeCollab.repos.selected
                    });
                    getTeam();
                }
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'removeAppCollaborator',
                    components: [
                        appHomeCollab.repos.selected,
                        removedMember
                    ],
                    terminal: true
                }],
                response: message
            });
        }

        /** Sync Functions */
        /**
         * @name getStatus
         * @desc gets the current status of the app
         * @returns {void}
         */
        function getStatus() {
            var message = semossCoreService.utility.random('query-pixel');
            // clear
            appHomeCollab.files = {
                options: [],
                selected: []
            };

            appHomeCollab.databaseSync = false;

            appHomeCollab.validation = {
                open: false,
                valid: false,
                type: '',
                name: '',
                title: ''
            };

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    appHomeCollab.files = {
                        options: JSON.parse(JSON.stringify(output)),
                        selected: JSON.parse(JSON.stringify(output))
                    };
                }
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'gitStatus',
                    components: [
                        appHomeCollab.engines.selected
                    ],
                    terminal: true
                }],
                response: message
            });
        }

        /**
         * @name pull
         * @desc pull your app
         * @returns {void}
         */
        function pull() {
            var message = semossCoreService.utility.random('query-pixel');

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully pulled from ' + appHomeCollab.repos.selected
                    });
                    getStatus();
                }
            });

            // list remote
            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'syncApp',
                    components: [
                        appHomeCollab.engines.selected,
                        appHomeCollab.repos.selected,
                        false,
                        appHomeCollab.databaseSync
                    ],
                    terminal: true
                }],
                response: message
            });
        }

        /**
         * @name syncAll
         * @desc sync your app
         * @param {boolean} valid - able to proceed with irreversible actions?
         * @returns {void}
         */
        function syncAll(valid) {
            var message = semossCoreService.utility.random('query-pixel');
            // list remote
            if (appHomeCollab.databaseSync && !valid) {
                appHomeCollab.validation.type = 'syncAll';
                appHomeCollab.validation.name = '';
                appHomeCollab.validation.title = 'Enter App name (' + appHomeCollab.engines.selected + ') to confirm';
                appHomeCollab.validation.valid = false;
                appHomeCollab.validation.open = true;
                return;
            }

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully synced with ' + appHomeCollab.repos.selected
                    });
                    getStatus();
                }
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'syncApp',
                    components: [
                        appHomeCollab.engines.selected,
                        appHomeCollab.repos.selected,
                        true,
                        valid
                    ],
                    terminal: true
                }],
                response: message
            });
        }

        /**
         * @name syncSpecific
         * @desc sync specific portions of your app
         * @param {boolean} valid - able to proceed with irreversible actions?
         * @returns {void}
         */
        function syncSpecific(valid) {
            var files = [],
                i, len, message = semossCoreService.utility.random('query-pixel');

            if (appHomeCollab.databaseSync && !valid) {
                appHomeCollab.validation.type = 'syncSpecific';
                appHomeCollab.validation.name = '';
                appHomeCollab.validation.title = 'Enter app name ( ' + appHomeCollab.engines.selectedName + ' ) to confirm';
                appHomeCollab.validation.valid = false;
                appHomeCollab.validation.open = true;
                return;
            }

            for (i = 0, len = appHomeCollab.files.selected.length; i < len; i++) {
                files.push(appHomeCollab.files.selected[i].fileLoc);
            }

            if (files.length === 0) {
                return;
            }

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: output
                    });
                    getStatus();
                }
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'syncAppFilesO',
                    components: [
                        appHomeCollab.engines.selected,
                        appHomeCollab.repos.selected,
                        true,
                        files,
                        valid
                    ],
                    terminal: true
                }],
                response: message
            });
        }

        /**
         * @name checkValidation
         * @desc sync with a specific database
         * @returns {void}
         */
        function checkValidation() {
            var cleanedName = appHomeCollab.validation.name.replace(/ /g, '_');
            appHomeCollab.validation.valid = (cleanedName === appHomeCollab.engines.selectedName.replace(/ /g, '_'));
        }

        /**
         * @name confirmValidation
         * @desc confirm validation and trigger appropriate function
         * @returns {void}
         */
        function confirmValidation() {
            checkValidation();

            if (!appHomeCollab.validation.valid) {
                return;
            }

            if (appHomeCollab.validation.type === 'syncAll') {
                syncAll(true);
            } else if (appHomeCollab.validation.type === 'syncSpecific') {
                syncSpecific(true);
            } else if (appHomeCollab.validation.type === 'createRepo') {
                createRepo(true);
            } else if (appHomeCollab.validation.type === 'dropRepo') {
                dropRepo(true);
            } else if (appHomeCollab.validation.type === 'deleteRepo') {
                deleteRepo(true);
            }
        }


        /**
         * @name initialize
         * @desc function that is called on directive load
         * @returns {void}
         */
        function initialize() {
            logoutListener = semossCoreService.on('oauth-logout-success', resetCollaboration);
            loginListener = semossCoreService.on('oauth-login-success', resetCollaboration);

            // cleanup
            $scope.$on('$destroy', function () {
                logoutListener();
                loginListener();
                console.log('DESTROY');
            });

            resetCollaboration();
        }

        initialize();
    }
}
