'use strict';

import './admin-insights.scss';

export default angular.module('app.admin.admin-insights', [])
    .directive('adminInsights', adminInsightsDirective);

adminInsightsDirective.$inject = ['monolithService', 'semossCoreService'];

function adminInsightsDirective(monolithService, semossCoreService) {
    adminInsightsCtrl.$inject = ['$scope'];
    adminInsightsLink.$inject = [];

    return {
        restrict: 'E',
        template: require('./admin-insights.directive.html'),
        controller: adminInsightsCtrl,
        link: adminInsightsLink,
        require: ['^admin'],
        scope: {},
        bindToController: {},
        controllerAs: 'adminInsights'
    };

    function adminInsightsCtrl() {
        var adminInsights = this;

        adminInsights.deleteInsightOverlay = {
            show: false
        };
        adminInsights.editInsightOverlay = {
            show: false
        };
        adminInsights.insights = {};
        adminInsights.appId = '';
        adminInsights.users = {
            list: []
        };
        adminInsights.newUser = {
            id: '',
            permission: 'READ_ONLY'
        };

        adminInsights.getInsights = getInsights;
        adminInsights.setInsightGlobal = setInsightGlobal;
        adminInsights.openEditInsightOverlay = openEditInsightOverlay;
        adminInsights.openDeleteInsightOverlay = openDeleteInsightOverlay;
        adminInsights.deleteInsight = deleteInsight;
        adminInsights.addInsightUser = addInsightUser;
        adminInsights.editInsightUser = editInsightUser;
        adminInsights.deleteInsightUser = deleteInsightUser;

        /**
         * @name addInsightUser
         * @param {object} insight the insight to add user to
         * @param {string} userId the user to add to insight
         * @param {string} permission the access level to give user
         * @desc add user to insight
         * @returns {void}
         */
        function addInsightUser(insight, userId, permission) {
            monolithService.addAdminInsightUserPermission(insight.app_id, insight.app_insight_id, userId, permission)
                .then(function (response) {
                    if (response.success) {
                        // refresh current users list
                        getCurrentUsers(insight);

                        semossCoreService.emit('alert', {
                            color: 'success',
                            text: 'Successfully added user to insight.'
                        });
                    }
                }, function (response) {
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: response.data.errorMessage
                    });
                });
        }

        /**
         * @name editInsightUser
         * @param {object} insight the insight to edit
         * @param {string} userId the user to edit
         * @param {string} permission the access level to edit
         * @desc edit user for insight
         * @returns {void}
         */
        function editInsightUser(insight, userId, permission) {
            monolithService.editAdminInsightUserPermission(insight.app_id, insight.app_insight_id, userId, permission)
                .then(function (response) {
                    if (response.success) {
                        // refresh current users list
                        getCurrentUsers(insight);

                        semossCoreService.emit('alert', {
                            color: 'success',
                            text: 'Successfully edited user permission.'
                        });
                    }
                }, function (response) {
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: response.data.errorMessage
                    });
                });
        }

        /**
         * @name deleteInsightUser
         * @param {*} insight the insight to delete user from
         * @param {*} userId the user to delete
         * @desc delete the user from the insight
         * @returns {void}
         */
        function deleteInsightUser(insight, userId) {
            monolithService.removeAdminInsightUserPermission(insight.app_id, insight.app_insight_id, userId)
                .then(function (response) {
                    if (response.success) {
                        // refresh current users list
                        getCurrentUsers(insight);

                        semossCoreService.emit('alert', {
                            color: 'success',
                            text: 'Successfully removed user permission.'
                        });
                    }
                }, function (response) {
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: response.data.errorMessage
                    });
                });
        }

        /**
         * @name deleteInsight
         * @param {*} insight insight to delete
         * @desc delete the insight
         * @returns {void}
         */
        function deleteInsight(insight) {
            monolithService.deleteAdminInsight(insight.app_id, [insight.app_insight_id])
                .then(function (response) {
                    if (response.success) {
                        adminInsights.deleteInsightOverlay.show = false;

                        // refresh current users list
                        getInsights(insight.app_id, true);

                        semossCoreService.emit('alert', {
                            color: 'success',
                            text: 'Successfully deleted the insight.'
                        });
                    }
                }, function (response) {
                    adminInsights.deleteInsightOverlay.show = false;
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: response.data.errorMessage
                    });
                });
        }

        /**
         * @name searchUsers
         * @param {string} searchInput search term
         * @desc search the users
         * @returns {void}
         */
        function searchUsers(searchInput) {
            // get list of users that meet the search input
            monolithService.getUserInformation(searchInput)
                .then(function (data) {
                    var userIdx, display = '',
                        tempUser;
                    adminInsights.users.list = [];

                    for (userIdx = 0; userIdx < data.length; userIdx++) {
                        tempUser = semossCoreService.utility.freeze(data[userIdx]);
                        display = data[userIdx].name + ' | ' + data[userIdx].email;
                        tempUser.display = display;
                        adminInsights.users.list.push(tempUser);
                    }
                });
        }

        /**
         * @name getCurrentUsers
         * @param {object} insight the insight to get users for
         * @desc get the list of current users for this insight
         * @returns {void}
         */
        function getCurrentUsers(insight) {
            monolithService.getInsightUsers(insight.app_id, insight.app_insight_id)
                .then(function (response) {
                    adminInsights.editInsightOverlay.currentUsers = response.data;
                });
        }

        /**
         * @name openEditInsightOverlay
         * @param {object} insight the insight to edit
         * @desc open the overlay to edit the insight
         * @returns {void}
         */
        function openEditInsightOverlay(insight) {
            getCurrentUsers(insight);
            searchUsers('');
            adminInsights.editInsightOverlay.show = true;
            adminInsights.editInsightOverlay.insight = insight;
        }

        /**
         * @name openDeleteInsightOverlay
         * @param {object} insight the insight to delete
         * @desc open the overlay to confirm deletion of insight
         * @returns {void}
         */
        function openDeleteInsightOverlay(insight) {
            adminInsights.deleteInsightOverlay.show = true;
            adminInsights.deleteInsightOverlay = {
                show: true,
                insight: insight
            };
        }

        /**
         * @name getInsights
         * @param {*} appId app id to get insights for
         * @param {boolean} refresh get new insights from app
         * @desc get the insights for this app
         * @returns {void}
         */
        function getInsights(appId, refresh) {
            if (!adminInsights.insights[appId] || refresh) {
                monolithService.getAdminAppInsights(appId)
                    .then(function (response) {
                        adminInsights.insights[appId] = response.data;
                    });
            }
        }

        /**
         * @name setInsightGlobal
         * @param {string} app the id of the app
         * @param {string} id the insight id
         * @param {*} bool toggle the boolean for public
         * @desc toggle the insight public
         * @returns {void}
         */
        function setInsightGlobal(app, id, bool) {
            monolithService.setAdminInsightGlobal(app, id, bool)
                .then(function (response) {
                    if (response.success) {
                        semossCoreService.emit('alert', {
                            color: 'success',
                            text: 'Successfully set insight public to ' + bool + '.'
                        });
                    }
                }, function (response) {
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: response.data.errorMessage
                    });
                });
        }
    }

    function adminInsightsLink(scope, ele, attrs, ctrl) {
        scope.adminCtrl = ctrl[0];
        /**
         * @name initialize
         * @desc function that is called on directive load
         * @returns {void}
         */
        function initialize() {}

        initialize();

        // cleanup
        scope.$on('$destroy', function () {
            console.log('destroying admin insights...');
        });
    }
}
