package org.openqa.selenium.devtools.v122.debugger;

import org.openqa.selenium.Beta;
import org.openqa.selenium.devtools.Command;
import org.openqa.selenium.devtools.Event;
import org.openqa.selenium.devtools.ConverterFunctions;
import java.util.Map;
import java.util.LinkedHashMap;
import org.openqa.selenium.json.JsonInput;

/**
 * Debugger domain exposes JavaScript debugging capabilities. It allows setting and removing
 * breakpoints, stepping through execution, exploring stack traces, etc.
 */
public class Debugger {

    public enum ContinueToLocationTargetCallFrames {

        ANY("any"), CURRENT("current");

        private String value;

        ContinueToLocationTargetCallFrames(String value) {
            this.value = value;
        }

        public static ContinueToLocationTargetCallFrames fromString(String s) {
            return java.util.Arrays.stream(ContinueToLocationTargetCallFrames.values()).filter(rs -> rs.value.equalsIgnoreCase(s)).findFirst().orElseThrow(() -> new org.openqa.selenium.devtools.DevToolsException("Given value " + s + " is not found within ContinueToLocationTargetCallFrames "));
        }

        public String toString() {
            return value;
        }

        public String toJson() {
            return value;
        }

        private static ContinueToLocationTargetCallFrames fromJson(JsonInput input) {
            return fromString(input.nextString());
        }
    }

    /**
     * Continues execution until specific location is reached.
     */
    public static Command<Void> continueToLocation(org.openqa.selenium.devtools.v122.debugger.model.Location location, java.util.Optional<ContinueToLocationTargetCallFrames> targetCallFrames) {
        java.util.Objects.requireNonNull(location, "location is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("location", location);
        targetCallFrames.ifPresent(p -> params.put("targetCallFrames", p));
        return new Command<>("Debugger.continueToLocation", Map.copyOf(params));
    }

    /**
     * Disables debugger for given page.
     */
    public static Command<Void> disable() {
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        return new Command<>("Debugger.disable", Map.copyOf(params));
    }

    /**
     * Enables debugger for the given page. Clients should not assume that the debugging has been
     * enabled until the result for this command is received.
     */
    public static Command<org.openqa.selenium.devtools.v122.runtime.model.UniqueDebuggerId> enable(java.util.Optional<java.lang.Number> maxScriptsCacheSize) {
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        maxScriptsCacheSize.ifPresent(p -> params.put("maxScriptsCacheSize", p));
        return new Command<>("Debugger.enable", Map.copyOf(params), ConverterFunctions.map("debuggerId", org.openqa.selenium.devtools.v122.runtime.model.UniqueDebuggerId.class));
    }

    public static class EvaluateOnCallFrameResponse {

        private final org.openqa.selenium.devtools.v122.runtime.model.RemoteObject result;

        private final java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.ExceptionDetails> exceptionDetails;

        public EvaluateOnCallFrameResponse(org.openqa.selenium.devtools.v122.runtime.model.RemoteObject result, java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.ExceptionDetails> exceptionDetails) {
            this.result = java.util.Objects.requireNonNull(result, "result is required");
            this.exceptionDetails = exceptionDetails;
        }

        /**
         * Object wrapper for the evaluation result.
         */
        public org.openqa.selenium.devtools.v122.runtime.model.RemoteObject getResult() {
            return result;
        }

        /**
         * Exception details.
         */
        public java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.ExceptionDetails> getExceptionDetails() {
            return exceptionDetails;
        }

        private static EvaluateOnCallFrameResponse fromJson(JsonInput input) {
            org.openqa.selenium.devtools.v122.runtime.model.RemoteObject result = null;
            java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.ExceptionDetails> exceptionDetails = java.util.Optional.empty();
            input.beginObject();
            while (input.hasNext()) {
                switch(input.nextName()) {
                    case "result":
                        result = input.read(org.openqa.selenium.devtools.v122.runtime.model.RemoteObject.class);
                        break;
                    case "exceptionDetails":
                        exceptionDetails = java.util.Optional.ofNullable(input.read(org.openqa.selenium.devtools.v122.runtime.model.ExceptionDetails.class));
                        break;
                    default:
                        input.skipValue();
                        break;
                }
            }
            input.endObject();
            return new EvaluateOnCallFrameResponse(result, exceptionDetails);
        }
    }

    /**
     * Evaluates expression on a given call frame.
     */
    public static Command<org.openqa.selenium.devtools.v122.debugger.Debugger.EvaluateOnCallFrameResponse> evaluateOnCallFrame(org.openqa.selenium.devtools.v122.debugger.model.CallFrameId callFrameId, java.lang.String expression, java.util.Optional<java.lang.String> objectGroup, java.util.Optional<java.lang.Boolean> includeCommandLineAPI, java.util.Optional<java.lang.Boolean> silent, java.util.Optional<java.lang.Boolean> returnByValue, java.util.Optional<java.lang.Boolean> generatePreview, java.util.Optional<java.lang.Boolean> throwOnSideEffect, java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.TimeDelta> timeout) {
        java.util.Objects.requireNonNull(callFrameId, "callFrameId is required");
        java.util.Objects.requireNonNull(expression, "expression is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("callFrameId", callFrameId);
        params.put("expression", expression);
        objectGroup.ifPresent(p -> params.put("objectGroup", p));
        includeCommandLineAPI.ifPresent(p -> params.put("includeCommandLineAPI", p));
        silent.ifPresent(p -> params.put("silent", p));
        returnByValue.ifPresent(p -> params.put("returnByValue", p));
        generatePreview.ifPresent(p -> params.put("generatePreview", p));
        throwOnSideEffect.ifPresent(p -> params.put("throwOnSideEffect", p));
        timeout.ifPresent(p -> params.put("timeout", p));
        return new Command<>("Debugger.evaluateOnCallFrame", Map.copyOf(params), input -> input.read(org.openqa.selenium.devtools.v122.debugger.Debugger.EvaluateOnCallFrameResponse.class));
    }

    /**
     * Returns possible locations for breakpoint. scriptId in start and end range locations should be
     * the same.
     */
    public static Command<java.util.List<org.openqa.selenium.devtools.v122.debugger.model.BreakLocation>> getPossibleBreakpoints(org.openqa.selenium.devtools.v122.debugger.model.Location start, java.util.Optional<org.openqa.selenium.devtools.v122.debugger.model.Location> end, java.util.Optional<java.lang.Boolean> restrictToFunction) {
        java.util.Objects.requireNonNull(start, "start is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("start", start);
        end.ifPresent(p -> params.put("end", p));
        restrictToFunction.ifPresent(p -> params.put("restrictToFunction", p));
        return new Command<>("Debugger.getPossibleBreakpoints", Map.copyOf(params), ConverterFunctions.map("locations", input -> input.readArray(org.openqa.selenium.devtools.v122.debugger.model.BreakLocation.class)));
    }

    public static class GetScriptSourceResponse {

        private final java.lang.String scriptSource;

        private final java.util.Optional<java.lang.String> bytecode;

        public GetScriptSourceResponse(java.lang.String scriptSource, java.util.Optional<java.lang.String> bytecode) {
            this.scriptSource = java.util.Objects.requireNonNull(scriptSource, "scriptSource is required");
            this.bytecode = bytecode;
        }

        /**
         * Script source (empty in case of Wasm bytecode).
         */
        public java.lang.String getScriptSource() {
            return scriptSource;
        }

        /**
         * Wasm bytecode.
         */
        public java.util.Optional<java.lang.String> getBytecode() {
            return bytecode;
        }

        private static GetScriptSourceResponse fromJson(JsonInput input) {
            java.lang.String scriptSource = null;
            java.util.Optional<java.lang.String> bytecode = java.util.Optional.empty();
            input.beginObject();
            while (input.hasNext()) {
                switch(input.nextName()) {
                    case "scriptSource":
                        scriptSource = input.nextString();
                        break;
                    case "bytecode":
                        bytecode = java.util.Optional.ofNullable(input.nextString());
                        break;
                    default:
                        input.skipValue();
                        break;
                }
            }
            input.endObject();
            return new GetScriptSourceResponse(scriptSource, bytecode);
        }
    }

    /**
     * Returns source for the script with given id.
     */
    public static Command<org.openqa.selenium.devtools.v122.debugger.Debugger.GetScriptSourceResponse> getScriptSource(org.openqa.selenium.devtools.v122.runtime.model.ScriptId scriptId) {
        java.util.Objects.requireNonNull(scriptId, "scriptId is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("scriptId", scriptId);
        return new Command<>("Debugger.getScriptSource", Map.copyOf(params), input -> input.read(org.openqa.selenium.devtools.v122.debugger.Debugger.GetScriptSourceResponse.class));
    }

    public static class DisassembleWasmModuleResponse {

        private final java.util.Optional<java.lang.String> streamId;

        private final java.lang.Integer totalNumberOfLines;

        private final java.util.List<java.lang.Integer> functionBodyOffsets;

        private final org.openqa.selenium.devtools.v122.debugger.model.WasmDisassemblyChunk chunk;

        public DisassembleWasmModuleResponse(java.util.Optional<java.lang.String> streamId, java.lang.Integer totalNumberOfLines, java.util.List<java.lang.Integer> functionBodyOffsets, org.openqa.selenium.devtools.v122.debugger.model.WasmDisassemblyChunk chunk) {
            this.streamId = streamId;
            this.totalNumberOfLines = java.util.Objects.requireNonNull(totalNumberOfLines, "totalNumberOfLines is required");
            this.functionBodyOffsets = java.util.Objects.requireNonNull(functionBodyOffsets, "functionBodyOffsets is required");
            this.chunk = java.util.Objects.requireNonNull(chunk, "chunk is required");
        }

        /**
         * For large modules, return a stream from which additional chunks of
         * disassembly can be read successively.
         */
        public java.util.Optional<java.lang.String> getStreamId() {
            return streamId;
        }

        /**
         * The total number of lines in the disassembly text.
         */
        public java.lang.Integer getTotalNumberOfLines() {
            return totalNumberOfLines;
        }

        /**
         * The offsets of all function bodies, in the format [start1, end1,
         * start2, end2, ...] where all ends are exclusive.
         */
        public java.util.List<java.lang.Integer> getFunctionBodyOffsets() {
            return functionBodyOffsets;
        }

        /**
         * The first chunk of disassembly.
         */
        public org.openqa.selenium.devtools.v122.debugger.model.WasmDisassemblyChunk getChunk() {
            return chunk;
        }

        private static DisassembleWasmModuleResponse fromJson(JsonInput input) {
            java.util.Optional<java.lang.String> streamId = java.util.Optional.empty();
            java.lang.Integer totalNumberOfLines = 0;
            java.util.List<java.lang.Integer> functionBodyOffsets = null;
            org.openqa.selenium.devtools.v122.debugger.model.WasmDisassemblyChunk chunk = null;
            input.beginObject();
            while (input.hasNext()) {
                switch(input.nextName()) {
                    case "streamId":
                        streamId = java.util.Optional.ofNullable(input.nextString());
                        break;
                    case "totalNumberOfLines":
                        totalNumberOfLines = input.nextNumber().intValue();
                        break;
                    case "functionBodyOffsets":
                        functionBodyOffsets = input.readArray(java.lang.Integer.class);
                        break;
                    case "chunk":
                        chunk = input.read(org.openqa.selenium.devtools.v122.debugger.model.WasmDisassemblyChunk.class);
                        break;
                    default:
                        input.skipValue();
                        break;
                }
            }
            input.endObject();
            return new DisassembleWasmModuleResponse(streamId, totalNumberOfLines, functionBodyOffsets, chunk);
        }
    }

    @Beta()
    public static Command<org.openqa.selenium.devtools.v122.debugger.Debugger.DisassembleWasmModuleResponse> disassembleWasmModule(org.openqa.selenium.devtools.v122.runtime.model.ScriptId scriptId) {
        java.util.Objects.requireNonNull(scriptId, "scriptId is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("scriptId", scriptId);
        return new Command<>("Debugger.disassembleWasmModule", Map.copyOf(params), input -> input.read(org.openqa.selenium.devtools.v122.debugger.Debugger.DisassembleWasmModuleResponse.class));
    }

    /**
     * Disassemble the next chunk of lines for the module corresponding to the
     * stream. If disassembly is complete, this API will invalidate the streamId
     * and return an empty chunk. Any subsequent calls for the now invalid stream
     * will return errors.
     */
    @Beta()
    public static Command<org.openqa.selenium.devtools.v122.debugger.model.WasmDisassemblyChunk> nextWasmDisassemblyChunk(java.lang.String streamId) {
        java.util.Objects.requireNonNull(streamId, "streamId is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("streamId", streamId);
        return new Command<>("Debugger.nextWasmDisassemblyChunk", Map.copyOf(params), ConverterFunctions.map("chunk", org.openqa.selenium.devtools.v122.debugger.model.WasmDisassemblyChunk.class));
    }

    /**
     * This command is deprecated. Use getScriptSource instead.
     */
    @Deprecated()
    public static Command<java.lang.String> getWasmBytecode(org.openqa.selenium.devtools.v122.runtime.model.ScriptId scriptId) {
        java.util.Objects.requireNonNull(scriptId, "scriptId is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("scriptId", scriptId);
        return new Command<>("Debugger.getWasmBytecode", Map.copyOf(params), ConverterFunctions.map("bytecode", java.lang.String.class));
    }

    /**
     * Returns stack trace with given `stackTraceId`.
     */
    @Beta()
    public static Command<org.openqa.selenium.devtools.v122.runtime.model.StackTrace> getStackTrace(org.openqa.selenium.devtools.v122.runtime.model.StackTraceId stackTraceId) {
        java.util.Objects.requireNonNull(stackTraceId, "stackTraceId is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("stackTraceId", stackTraceId);
        return new Command<>("Debugger.getStackTrace", Map.copyOf(params), ConverterFunctions.map("stackTrace", org.openqa.selenium.devtools.v122.runtime.model.StackTrace.class));
    }

    /**
     * Stops on the next JavaScript statement.
     */
    public static Command<Void> pause() {
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        return new Command<>("Debugger.pause", Map.copyOf(params));
    }

    @Beta()
    @Deprecated()
    public static Command<Void> pauseOnAsyncCall(org.openqa.selenium.devtools.v122.runtime.model.StackTraceId parentStackTraceId) {
        java.util.Objects.requireNonNull(parentStackTraceId, "parentStackTraceId is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("parentStackTraceId", parentStackTraceId);
        return new Command<>("Debugger.pauseOnAsyncCall", Map.copyOf(params));
    }

    /**
     * Removes JavaScript breakpoint.
     */
    public static Command<Void> removeBreakpoint(org.openqa.selenium.devtools.v122.debugger.model.BreakpointId breakpointId) {
        java.util.Objects.requireNonNull(breakpointId, "breakpointId is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("breakpointId", breakpointId);
        return new Command<>("Debugger.removeBreakpoint", Map.copyOf(params));
    }

    public static class RestartFrameResponse {

        private final java.util.List<org.openqa.selenium.devtools.v122.debugger.model.CallFrame> callFrames;

        private final java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTrace> asyncStackTrace;

        private final java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTraceId> asyncStackTraceId;

        public RestartFrameResponse(java.util.List<org.openqa.selenium.devtools.v122.debugger.model.CallFrame> callFrames, java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTrace> asyncStackTrace, java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTraceId> asyncStackTraceId) {
            this.callFrames = java.util.Objects.requireNonNull(callFrames, "callFrames is required");
            this.asyncStackTrace = asyncStackTrace;
            this.asyncStackTraceId = asyncStackTraceId;
        }

        /**
         * New stack trace.
         */
        public java.util.List<org.openqa.selenium.devtools.v122.debugger.model.CallFrame> getCallFrames() {
            return callFrames;
        }

        /**
         * Async stack trace, if any.
         */
        public java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTrace> getAsyncStackTrace() {
            return asyncStackTrace;
        }

        /**
         * Async stack trace, if any.
         */
        public java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTraceId> getAsyncStackTraceId() {
            return asyncStackTraceId;
        }

        private static RestartFrameResponse fromJson(JsonInput input) {
            java.util.List<org.openqa.selenium.devtools.v122.debugger.model.CallFrame> callFrames = null;
            java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTrace> asyncStackTrace = java.util.Optional.empty();
            java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTraceId> asyncStackTraceId = java.util.Optional.empty();
            input.beginObject();
            while (input.hasNext()) {
                switch(input.nextName()) {
                    case "callFrames":
                        callFrames = input.readArray(org.openqa.selenium.devtools.v122.debugger.model.CallFrame.class);
                        break;
                    case "asyncStackTrace":
                        asyncStackTrace = java.util.Optional.ofNullable(input.read(org.openqa.selenium.devtools.v122.runtime.model.StackTrace.class));
                        break;
                    case "asyncStackTraceId":
                        asyncStackTraceId = java.util.Optional.ofNullable(input.read(org.openqa.selenium.devtools.v122.runtime.model.StackTraceId.class));
                        break;
                    default:
                        input.skipValue();
                        break;
                }
            }
            input.endObject();
            return new RestartFrameResponse(callFrames, asyncStackTrace, asyncStackTraceId);
        }
    }

    public enum RestartFrameMode {

        STEPINTO("StepInto");

        private String value;

        RestartFrameMode(String value) {
            this.value = value;
        }

        public static RestartFrameMode fromString(String s) {
            return java.util.Arrays.stream(RestartFrameMode.values()).filter(rs -> rs.value.equalsIgnoreCase(s)).findFirst().orElseThrow(() -> new org.openqa.selenium.devtools.DevToolsException("Given value " + s + " is not found within RestartFrameMode "));
        }

        public String toString() {
            return value;
        }

        public String toJson() {
            return value;
        }

        private static RestartFrameMode fromJson(JsonInput input) {
            return fromString(input.nextString());
        }
    }

    /**
     * Restarts particular call frame from the beginning. The old, deprecated
     * behavior of `restartFrame` is to stay paused and allow further CDP commands
     * after a restart was scheduled. This can cause problems with restarting, so
     * we now continue execution immediatly after it has been scheduled until we
     * reach the beginning of the restarted frame.
     *
     * To stay back-wards compatible, `restartFrame` now expects a `mode`
     * parameter to be present. If the `mode` parameter is missing, `restartFrame`
     * errors out.
     *
     * The various return values are deprecated and `callFrames` is always empty.
     * Use the call frames from the `Debugger#paused` events instead, that fires
     * once V8 pauses at the beginning of the restarted function.
     */
    public static Command<org.openqa.selenium.devtools.v122.debugger.Debugger.RestartFrameResponse> restartFrame(org.openqa.selenium.devtools.v122.debugger.model.CallFrameId callFrameId, java.util.Optional<RestartFrameMode> mode) {
        java.util.Objects.requireNonNull(callFrameId, "callFrameId is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("callFrameId", callFrameId);
        mode.ifPresent(p -> params.put("mode", p));
        return new Command<>("Debugger.restartFrame", Map.copyOf(params), input -> input.read(org.openqa.selenium.devtools.v122.debugger.Debugger.RestartFrameResponse.class));
    }

    /**
     * Resumes JavaScript execution.
     */
    public static Command<Void> resume(java.util.Optional<java.lang.Boolean> terminateOnResume) {
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        terminateOnResume.ifPresent(p -> params.put("terminateOnResume", p));
        return new Command<>("Debugger.resume", Map.copyOf(params));
    }

    /**
     * Searches for given string in script content.
     */
    public static Command<java.util.List<org.openqa.selenium.devtools.v122.debugger.model.SearchMatch>> searchInContent(org.openqa.selenium.devtools.v122.runtime.model.ScriptId scriptId, java.lang.String query, java.util.Optional<java.lang.Boolean> caseSensitive, java.util.Optional<java.lang.Boolean> isRegex) {
        java.util.Objects.requireNonNull(scriptId, "scriptId is required");
        java.util.Objects.requireNonNull(query, "query is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("scriptId", scriptId);
        params.put("query", query);
        caseSensitive.ifPresent(p -> params.put("caseSensitive", p));
        isRegex.ifPresent(p -> params.put("isRegex", p));
        return new Command<>("Debugger.searchInContent", Map.copyOf(params), ConverterFunctions.map("result", input -> input.readArray(org.openqa.selenium.devtools.v122.debugger.model.SearchMatch.class)));
    }

    /**
     * Enables or disables async call stacks tracking.
     */
    public static Command<Void> setAsyncCallStackDepth(java.lang.Integer maxDepth) {
        java.util.Objects.requireNonNull(maxDepth, "maxDepth is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("maxDepth", maxDepth);
        return new Command<>("Debugger.setAsyncCallStackDepth", Map.copyOf(params));
    }

    /**
     * Replace previous blackbox patterns with passed ones. Forces backend to skip stepping/pausing in
     * scripts with url matching one of the patterns. VM will try to leave blackboxed script by
     * performing 'step in' several times, finally resorting to 'step out' if unsuccessful.
     */
    @Beta()
    public static Command<Void> setBlackboxPatterns(java.util.List<java.lang.String> patterns) {
        java.util.Objects.requireNonNull(patterns, "patterns is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("patterns", patterns);
        return new Command<>("Debugger.setBlackboxPatterns", Map.copyOf(params));
    }

    /**
     * Makes backend skip steps in the script in blackboxed ranges. VM will try leave blacklisted
     * scripts by performing 'step in' several times, finally resorting to 'step out' if unsuccessful.
     * Positions array contains positions where blackbox state is changed. First interval isn't
     * blackboxed. Array should be sorted.
     */
    @Beta()
    public static Command<Void> setBlackboxedRanges(org.openqa.selenium.devtools.v122.runtime.model.ScriptId scriptId, java.util.List<org.openqa.selenium.devtools.v122.debugger.model.ScriptPosition> positions) {
        java.util.Objects.requireNonNull(scriptId, "scriptId is required");
        java.util.Objects.requireNonNull(positions, "positions is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("scriptId", scriptId);
        params.put("positions", positions);
        return new Command<>("Debugger.setBlackboxedRanges", Map.copyOf(params));
    }

    public static class SetBreakpointResponse {

        private final org.openqa.selenium.devtools.v122.debugger.model.BreakpointId breakpointId;

        private final org.openqa.selenium.devtools.v122.debugger.model.Location actualLocation;

        public SetBreakpointResponse(org.openqa.selenium.devtools.v122.debugger.model.BreakpointId breakpointId, org.openqa.selenium.devtools.v122.debugger.model.Location actualLocation) {
            this.breakpointId = java.util.Objects.requireNonNull(breakpointId, "breakpointId is required");
            this.actualLocation = java.util.Objects.requireNonNull(actualLocation, "actualLocation is required");
        }

        /**
         * Id of the created breakpoint for further reference.
         */
        public org.openqa.selenium.devtools.v122.debugger.model.BreakpointId getBreakpointId() {
            return breakpointId;
        }

        /**
         * Location this breakpoint resolved into.
         */
        public org.openqa.selenium.devtools.v122.debugger.model.Location getActualLocation() {
            return actualLocation;
        }

        private static SetBreakpointResponse fromJson(JsonInput input) {
            org.openqa.selenium.devtools.v122.debugger.model.BreakpointId breakpointId = null;
            org.openqa.selenium.devtools.v122.debugger.model.Location actualLocation = null;
            input.beginObject();
            while (input.hasNext()) {
                switch(input.nextName()) {
                    case "breakpointId":
                        breakpointId = input.read(org.openqa.selenium.devtools.v122.debugger.model.BreakpointId.class);
                        break;
                    case "actualLocation":
                        actualLocation = input.read(org.openqa.selenium.devtools.v122.debugger.model.Location.class);
                        break;
                    default:
                        input.skipValue();
                        break;
                }
            }
            input.endObject();
            return new SetBreakpointResponse(breakpointId, actualLocation);
        }
    }

    /**
     * Sets JavaScript breakpoint at a given location.
     */
    public static Command<org.openqa.selenium.devtools.v122.debugger.Debugger.SetBreakpointResponse> setBreakpoint(org.openqa.selenium.devtools.v122.debugger.model.Location location, java.util.Optional<java.lang.String> condition) {
        java.util.Objects.requireNonNull(location, "location is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("location", location);
        condition.ifPresent(p -> params.put("condition", p));
        return new Command<>("Debugger.setBreakpoint", Map.copyOf(params), input -> input.read(org.openqa.selenium.devtools.v122.debugger.Debugger.SetBreakpointResponse.class));
    }

    public enum SetInstrumentationBreakpointInstrumentation {

        BEFORESCRIPTEXECUTION("beforeScriptExecution"), BEFORESCRIPTWITHSOURCEMAPEXECUTION("beforeScriptWithSourceMapExecution");

        private String value;

        SetInstrumentationBreakpointInstrumentation(String value) {
            this.value = value;
        }

        public static SetInstrumentationBreakpointInstrumentation fromString(String s) {
            return java.util.Arrays.stream(SetInstrumentationBreakpointInstrumentation.values()).filter(rs -> rs.value.equalsIgnoreCase(s)).findFirst().orElseThrow(() -> new org.openqa.selenium.devtools.DevToolsException("Given value " + s + " is not found within SetInstrumentationBreakpointInstrumentation "));
        }

        public String toString() {
            return value;
        }

        public String toJson() {
            return value;
        }

        private static SetInstrumentationBreakpointInstrumentation fromJson(JsonInput input) {
            return fromString(input.nextString());
        }
    }

    /**
     * Sets instrumentation breakpoint.
     */
    public static Command<org.openqa.selenium.devtools.v122.debugger.model.BreakpointId> setInstrumentationBreakpoint(SetInstrumentationBreakpointInstrumentation instrumentation) {
        java.util.Objects.requireNonNull(instrumentation, "instrumentation is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("instrumentation", instrumentation);
        return new Command<>("Debugger.setInstrumentationBreakpoint", Map.copyOf(params), ConverterFunctions.map("breakpointId", org.openqa.selenium.devtools.v122.debugger.model.BreakpointId.class));
    }

    public static class SetBreakpointByUrlResponse {

        private final org.openqa.selenium.devtools.v122.debugger.model.BreakpointId breakpointId;

        private final java.util.List<org.openqa.selenium.devtools.v122.debugger.model.Location> locations;

        public SetBreakpointByUrlResponse(org.openqa.selenium.devtools.v122.debugger.model.BreakpointId breakpointId, java.util.List<org.openqa.selenium.devtools.v122.debugger.model.Location> locations) {
            this.breakpointId = java.util.Objects.requireNonNull(breakpointId, "breakpointId is required");
            this.locations = java.util.Objects.requireNonNull(locations, "locations is required");
        }

        /**
         * Id of the created breakpoint for further reference.
         */
        public org.openqa.selenium.devtools.v122.debugger.model.BreakpointId getBreakpointId() {
            return breakpointId;
        }

        /**
         * List of the locations this breakpoint resolved into upon addition.
         */
        public java.util.List<org.openqa.selenium.devtools.v122.debugger.model.Location> getLocations() {
            return locations;
        }

        private static SetBreakpointByUrlResponse fromJson(JsonInput input) {
            org.openqa.selenium.devtools.v122.debugger.model.BreakpointId breakpointId = null;
            java.util.List<org.openqa.selenium.devtools.v122.debugger.model.Location> locations = null;
            input.beginObject();
            while (input.hasNext()) {
                switch(input.nextName()) {
                    case "breakpointId":
                        breakpointId = input.read(org.openqa.selenium.devtools.v122.debugger.model.BreakpointId.class);
                        break;
                    case "locations":
                        locations = input.readArray(org.openqa.selenium.devtools.v122.debugger.model.Location.class);
                        break;
                    default:
                        input.skipValue();
                        break;
                }
            }
            input.endObject();
            return new SetBreakpointByUrlResponse(breakpointId, locations);
        }
    }

    /**
     * Sets JavaScript breakpoint at given location specified either by URL or URL regex. Once this
     * command is issued, all existing parsed scripts will have breakpoints resolved and returned in
     * `locations` property. Further matching script parsing will result in subsequent
     * `breakpointResolved` events issued. This logical breakpoint will survive page reloads.
     */
    public static Command<org.openqa.selenium.devtools.v122.debugger.Debugger.SetBreakpointByUrlResponse> setBreakpointByUrl(java.lang.Integer lineNumber, java.util.Optional<java.lang.String> url, java.util.Optional<java.lang.String> urlRegex, java.util.Optional<java.lang.String> scriptHash, java.util.Optional<java.lang.Integer> columnNumber, java.util.Optional<java.lang.String> condition) {
        java.util.Objects.requireNonNull(lineNumber, "lineNumber is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("lineNumber", lineNumber);
        url.ifPresent(p -> params.put("url", p));
        urlRegex.ifPresent(p -> params.put("urlRegex", p));
        scriptHash.ifPresent(p -> params.put("scriptHash", p));
        columnNumber.ifPresent(p -> params.put("columnNumber", p));
        condition.ifPresent(p -> params.put("condition", p));
        return new Command<>("Debugger.setBreakpointByUrl", Map.copyOf(params), input -> input.read(org.openqa.selenium.devtools.v122.debugger.Debugger.SetBreakpointByUrlResponse.class));
    }

    /**
     * Sets JavaScript breakpoint before each call to the given function.
     * If another function was created from the same source as a given one,
     * calling it will also trigger the breakpoint.
     */
    @Beta()
    public static Command<org.openqa.selenium.devtools.v122.debugger.model.BreakpointId> setBreakpointOnFunctionCall(org.openqa.selenium.devtools.v122.runtime.model.RemoteObjectId objectId, java.util.Optional<java.lang.String> condition) {
        java.util.Objects.requireNonNull(objectId, "objectId is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("objectId", objectId);
        condition.ifPresent(p -> params.put("condition", p));
        return new Command<>("Debugger.setBreakpointOnFunctionCall", Map.copyOf(params), ConverterFunctions.map("breakpointId", org.openqa.selenium.devtools.v122.debugger.model.BreakpointId.class));
    }

    /**
     * Activates / deactivates all breakpoints on the page.
     */
    public static Command<Void> setBreakpointsActive(java.lang.Boolean active) {
        java.util.Objects.requireNonNull(active, "active is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("active", active);
        return new Command<>("Debugger.setBreakpointsActive", Map.copyOf(params));
    }

    public enum SetPauseOnExceptionsState {

        NONE("none"), CAUGHT("caught"), UNCAUGHT("uncaught"), ALL("all");

        private String value;

        SetPauseOnExceptionsState(String value) {
            this.value = value;
        }

        public static SetPauseOnExceptionsState fromString(String s) {
            return java.util.Arrays.stream(SetPauseOnExceptionsState.values()).filter(rs -> rs.value.equalsIgnoreCase(s)).findFirst().orElseThrow(() -> new org.openqa.selenium.devtools.DevToolsException("Given value " + s + " is not found within SetPauseOnExceptionsState "));
        }

        public String toString() {
            return value;
        }

        public String toJson() {
            return value;
        }

        private static SetPauseOnExceptionsState fromJson(JsonInput input) {
            return fromString(input.nextString());
        }
    }

    /**
     * Defines pause on exceptions state. Can be set to stop on all exceptions, uncaught exceptions,
     * or caught exceptions, no exceptions. Initial pause on exceptions state is `none`.
     */
    public static Command<Void> setPauseOnExceptions(SetPauseOnExceptionsState state) {
        java.util.Objects.requireNonNull(state, "state is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("state", state);
        return new Command<>("Debugger.setPauseOnExceptions", Map.copyOf(params));
    }

    /**
     * Changes return value in top frame. Available only at return break position.
     */
    @Beta()
    public static Command<Void> setReturnValue(org.openqa.selenium.devtools.v122.runtime.model.CallArgument newValue) {
        java.util.Objects.requireNonNull(newValue, "newValue is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("newValue", newValue);
        return new Command<>("Debugger.setReturnValue", Map.copyOf(params));
    }

    public static class SetScriptSourceResponse {

        public enum Status {

            OK("Ok"), COMPILEERROR("CompileError"), BLOCKEDBYACTIVEGENERATOR("BlockedByActiveGenerator"), BLOCKEDBYACTIVEFUNCTION("BlockedByActiveFunction"), BLOCKEDBYTOPLEVELESMODULECHANGE("BlockedByTopLevelEsModuleChange");

            private String value;

            Status(String value) {
                this.value = value;
            }

            public static Status fromString(String s) {
                return java.util.Arrays.stream(Status.values()).filter(rs -> rs.value.equalsIgnoreCase(s)).findFirst().orElseThrow(() -> new org.openqa.selenium.devtools.DevToolsException("Given value " + s + " is not found within Status "));
            }

            public String toString() {
                return value;
            }

            public String toJson() {
                return value;
            }

            private static Status fromJson(JsonInput input) {
                return fromString(input.nextString());
            }
        }

        private final java.util.Optional<java.util.List<org.openqa.selenium.devtools.v122.debugger.model.CallFrame>> callFrames;

        private final java.util.Optional<java.lang.Boolean> stackChanged;

        private final java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTrace> asyncStackTrace;

        private final java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTraceId> asyncStackTraceId;

        private final Status status;

        private final java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.ExceptionDetails> exceptionDetails;

        public SetScriptSourceResponse(java.util.Optional<java.util.List<org.openqa.selenium.devtools.v122.debugger.model.CallFrame>> callFrames, java.util.Optional<java.lang.Boolean> stackChanged, java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTrace> asyncStackTrace, java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTraceId> asyncStackTraceId, Status status, java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.ExceptionDetails> exceptionDetails) {
            this.callFrames = callFrames;
            this.stackChanged = stackChanged;
            this.asyncStackTrace = asyncStackTrace;
            this.asyncStackTraceId = asyncStackTraceId;
            this.status = java.util.Objects.requireNonNull(status, "status is required");
            this.exceptionDetails = exceptionDetails;
        }

        /**
         * New stack trace in case editing has happened while VM was stopped.
         */
        public java.util.Optional<java.util.List<org.openqa.selenium.devtools.v122.debugger.model.CallFrame>> getCallFrames() {
            return callFrames;
        }

        /**
         * Whether current call stack  was modified after applying the changes.
         */
        public java.util.Optional<java.lang.Boolean> getStackChanged() {
            return stackChanged;
        }

        /**
         * Async stack trace, if any.
         */
        public java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTrace> getAsyncStackTrace() {
            return asyncStackTrace;
        }

        /**
         * Async stack trace, if any.
         */
        public java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTraceId> getAsyncStackTraceId() {
            return asyncStackTraceId;
        }

        /**
         * Whether the operation was successful or not. Only `Ok` denotes a
         * successful live edit while the other enum variants denote why
         * the live edit failed.
         */
        public Status getStatus() {
            return status;
        }

        /**
         * Exception details if any. Only present when `status` is `CompileError`.
         */
        public java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.ExceptionDetails> getExceptionDetails() {
            return exceptionDetails;
        }

        private static SetScriptSourceResponse fromJson(JsonInput input) {
            java.util.Optional<java.util.List<org.openqa.selenium.devtools.v122.debugger.model.CallFrame>> callFrames = java.util.Optional.empty();
            java.util.Optional<java.lang.Boolean> stackChanged = java.util.Optional.empty();
            java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTrace> asyncStackTrace = java.util.Optional.empty();
            java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.StackTraceId> asyncStackTraceId = java.util.Optional.empty();
            Status status = null;
            java.util.Optional<org.openqa.selenium.devtools.v122.runtime.model.ExceptionDetails> exceptionDetails = java.util.Optional.empty();
            input.beginObject();
            while (input.hasNext()) {
                switch(input.nextName()) {
                    case "callFrames":
                        callFrames = java.util.Optional.ofNullable(input.readArray(org.openqa.selenium.devtools.v122.debugger.model.CallFrame.class));
                        break;
                    case "stackChanged":
                        stackChanged = java.util.Optional.ofNullable(input.nextBoolean());
                        break;
                    case "asyncStackTrace":
                        asyncStackTrace = java.util.Optional.ofNullable(input.read(org.openqa.selenium.devtools.v122.runtime.model.StackTrace.class));
                        break;
                    case "asyncStackTraceId":
                        asyncStackTraceId = java.util.Optional.ofNullable(input.read(org.openqa.selenium.devtools.v122.runtime.model.StackTraceId.class));
                        break;
                    case "status":
                        status = Status.fromString(input.nextString());
                        break;
                    case "exceptionDetails":
                        exceptionDetails = java.util.Optional.ofNullable(input.read(org.openqa.selenium.devtools.v122.runtime.model.ExceptionDetails.class));
                        break;
                    default:
                        input.skipValue();
                        break;
                }
            }
            input.endObject();
            return new SetScriptSourceResponse(callFrames, stackChanged, asyncStackTrace, asyncStackTraceId, status, exceptionDetails);
        }
    }

    /**
     * Edits JavaScript source live.
     *
     * In general, functions that are currently on the stack can not be edited with
     * a single exception: If the edited function is the top-most stack frame and
     * that is the only activation of that function on the stack. In this case
     * the live edit will be successful and a `Debugger.restartFrame` for the
     * top-most function is automatically triggered.
     */
    public static Command<org.openqa.selenium.devtools.v122.debugger.Debugger.SetScriptSourceResponse> setScriptSource(org.openqa.selenium.devtools.v122.runtime.model.ScriptId scriptId, java.lang.String scriptSource, java.util.Optional<java.lang.Boolean> dryRun, java.util.Optional<java.lang.Boolean> allowTopFrameEditing) {
        java.util.Objects.requireNonNull(scriptId, "scriptId is required");
        java.util.Objects.requireNonNull(scriptSource, "scriptSource is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("scriptId", scriptId);
        params.put("scriptSource", scriptSource);
        dryRun.ifPresent(p -> params.put("dryRun", p));
        allowTopFrameEditing.ifPresent(p -> params.put("allowTopFrameEditing", p));
        return new Command<>("Debugger.setScriptSource", Map.copyOf(params), input -> input.read(org.openqa.selenium.devtools.v122.debugger.Debugger.SetScriptSourceResponse.class));
    }

    /**
     * Makes page not interrupt on any pauses (breakpoint, exception, dom exception etc).
     */
    public static Command<Void> setSkipAllPauses(java.lang.Boolean skip) {
        java.util.Objects.requireNonNull(skip, "skip is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("skip", skip);
        return new Command<>("Debugger.setSkipAllPauses", Map.copyOf(params));
    }

    /**
     * Changes value of variable in a callframe. Object-based scopes are not supported and must be
     * mutated manually.
     */
    public static Command<Void> setVariableValue(java.lang.Integer scopeNumber, java.lang.String variableName, org.openqa.selenium.devtools.v122.runtime.model.CallArgument newValue, org.openqa.selenium.devtools.v122.debugger.model.CallFrameId callFrameId) {
        java.util.Objects.requireNonNull(scopeNumber, "scopeNumber is required");
        java.util.Objects.requireNonNull(variableName, "variableName is required");
        java.util.Objects.requireNonNull(newValue, "newValue is required");
        java.util.Objects.requireNonNull(callFrameId, "callFrameId is required");
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        params.put("scopeNumber", scopeNumber);
        params.put("variableName", variableName);
        params.put("newValue", newValue);
        params.put("callFrameId", callFrameId);
        return new Command<>("Debugger.setVariableValue", Map.copyOf(params));
    }

    /**
     * Steps into the function call.
     */
    public static Command<Void> stepInto(java.util.Optional<java.lang.Boolean> breakOnAsyncCall, java.util.Optional<java.util.List<org.openqa.selenium.devtools.v122.debugger.model.LocationRange>> skipList) {
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        breakOnAsyncCall.ifPresent(p -> params.put("breakOnAsyncCall", p));
        skipList.ifPresent(p -> params.put("skipList", p));
        return new Command<>("Debugger.stepInto", Map.copyOf(params));
    }

    /**
     * Steps out of the function call.
     */
    public static Command<Void> stepOut() {
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        return new Command<>("Debugger.stepOut", Map.copyOf(params));
    }

    /**
     * Steps over the statement.
     */
    public static Command<Void> stepOver(java.util.Optional<java.util.List<org.openqa.selenium.devtools.v122.debugger.model.LocationRange>> skipList) {
        LinkedHashMap<String, Object> params = new LinkedHashMap<>();
        skipList.ifPresent(p -> params.put("skipList", p));
        return new Command<>("Debugger.stepOver", Map.copyOf(params));
    }

    public static Event<org.openqa.selenium.devtools.v122.debugger.model.BreakpointResolved> breakpointResolved() {
        return new Event<>("Debugger.breakpointResolved", input -> input.read(org.openqa.selenium.devtools.v122.debugger.model.BreakpointResolved.class));
    }

    public static Event<org.openqa.selenium.devtools.v122.debugger.model.Paused> paused() {
        return new Event<>("Debugger.paused", input -> input.read(org.openqa.selenium.devtools.v122.debugger.model.Paused.class));
    }

    public static Event<Void> resumed() {
        return new Event<>("Debugger.resumed", ConverterFunctions.empty());
    }

    public static Event<org.openqa.selenium.devtools.v122.debugger.model.ScriptFailedToParse> scriptFailedToParse() {
        return new Event<>("Debugger.scriptFailedToParse", input -> input.read(org.openqa.selenium.devtools.v122.debugger.model.ScriptFailedToParse.class));
    }

    public static Event<org.openqa.selenium.devtools.v122.debugger.model.ScriptParsed> scriptParsed() {
        return new Event<>("Debugger.scriptParsed", input -> input.read(org.openqa.selenium.devtools.v122.debugger.model.ScriptParsed.class));
    }
}
