// Licensed to the Software Freedom Conservancy (SFC) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The SFC licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.openqa.selenium.grid.web;

import com.google.common.collect.ImmutableSet;

import org.openqa.selenium.remote.http.HttpClient;
import org.openqa.selenium.remote.http.HttpRequest;
import org.openqa.selenium.remote.http.HttpResponse;

import java.io.IOException;
import java.util.Objects;
import java.util.logging.Logger;

public class ReverseProxyHandler implements CommandHandler {

  private final static Logger LOG = Logger.getLogger(ReverseProxyHandler.class.getName());

  private final static ImmutableSet<String> IGNORED_REQ_HEADERS = ImmutableSet.<String>builder()
      .add("connection")
      .add("keep-alive")
      .add("proxy-authorization")
      .add("proxy-authenticate")
      .add("proxy-connection")
      .add("te")
      .add("trailer")
      .add("transfer-encoding")
      .add("upgrade")
      .build();

  private final HttpClient upstream;

  public ReverseProxyHandler(HttpClient httpClient) {
    this.upstream = Objects.requireNonNull(httpClient, "HTTP client to use must be set.");
  }

  @Override
  public void execute(HttpRequest req, HttpResponse resp) throws IOException {
    HttpRequest toUpstream = new HttpRequest(req.getMethod(), req.getUri());

    for (String name : req.getQueryParameterNames()) {
      for (String value : req.getQueryParameters(name)) {
        toUpstream.addQueryParameter(name, value);
      }
    }

    for (String name : req.getHeaderNames()) {
      if (IGNORED_REQ_HEADERS.contains(name.toLowerCase())) {
        continue;
      }

      for (String value : req.getHeaders(name)) {
        toUpstream.addHeader(name, value);
      }
    }
    // None of this "keep alive" nonsense.
    toUpstream.setHeader("Connection", "keep-alive");

    HttpResponse fromUpstream;
    toUpstream.setContent(req.getContent());
    fromUpstream = upstream.execute(toUpstream);

    resp.setStatus(fromUpstream.getStatus());
    // clear response defaults.
    resp.setHeader("Date",null);
    resp.setHeader("Server",null);

    for (String name : fromUpstream.getHeaderNames()) {
      if (IGNORED_REQ_HEADERS.contains(name)) {
        continue;
      }

      for (String value : fromUpstream.getHeaders(name)) {
        if (value == null) {
          continue;
        }

        resp.addHeader(name, value);
      }
    }
    resp.setContent(fromUpstream.getContent());
  }
}
