/*
 * Copyright (C) from 2022 The Play Framework Contributors <https://github.com/playframework>, 2011-2021 Lightbend Inc. <https://www.lightbend.com>
 */

package javaguide.http;

import static javaguide.testhelpers.MockJavaActionHelper.call;
import static org.assertj.core.api.Assertions.assertThat;
import static play.test.Helpers.*;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import javaguide.testhelpers.MockJavaAction;
import org.junit.Test;
import play.core.j.JavaHandlerComponents;
import play.mvc.Controller;
import play.mvc.Http;
import play.mvc.Result;
import play.test.WithApplication;

public class JavaActions extends WithApplication {
  @Test
  public void simpleAction() {
    assertThat(
            call(
                    new MockJavaAction(instanceOf(JavaHandlerComponents.class)) {
                      // #simple-action
                      public Result index(Http.Request request) {
                        return ok("Got request " + request + "!");
                      }
                      // #simple-action
                    },
                    fakeRequest(),
                    mat)
                .status())
        .isEqualTo(200);
  }

  @Test
  public void fullController() {
    assertThat(
            call(
                    new MockJavaAction(instanceOf(JavaHandlerComponents.class)) {
                      public Result index() {
                        return new javaguide.http.full.Application().index();
                      }
                    },
                    fakeRequest(),
                    mat)
                .status())
        .isEqualTo(200);
  }

  @Test
  public void withParams() {
    Result result =
        call(
            new MockJavaAction(instanceOf(JavaHandlerComponents.class)) {
              // #params-action
              public Result index(String name) {
                return ok("Hello " + name);
              }
              // #params-action

              public CompletionStage<Result> invocation() {
                return CompletableFuture.completedFuture(index("world"));
              }
            },
            fakeRequest(),
            mat);
    assertThat(result.status()).isEqualTo(200);
    assertThat(contentAsString(result)).isEqualTo("Hello world");
  }

  @Test
  public void simpleResult() {
    assertThat(
            call(
                    new MockJavaAction(instanceOf(JavaHandlerComponents.class)) {
                      // #simple-result
                      public Result index() {
                        return ok("Hello world!");
                      }
                      // #simple-result
                    },
                    fakeRequest(),
                    mat)
                .status())
        .isEqualTo(200);
  }

  @Test
  public void otherResults() {

    class Controller5 extends Controller {
      void run() {
        Object formWithErrors = null;

        // #other-results
        Result ok = ok("Hello world!");
        Result notFound = notFound();
        Result pageNotFound = notFound("<h1>Page not found</h1>").as("text/html");
        Result badRequest = badRequest(views.html.form.render(formWithErrors));
        Result oops = internalServerError("Oops");
        Result anyStatus = status(488, "Strange response type");
        // #other-results

        assertThat(anyStatus.status()).isEqualTo(488);
      }
    }

    new Controller5().run();
  }

  // Mock the existence of a view...
  static class views {
    static class html {
      static class form {
        static String render(Object o) {
          return "";
        }
      }
    }
  }

  @Test
  public void redirectAction() {
    Result result =
        call(
            new MockJavaAction(instanceOf(JavaHandlerComponents.class)) {
              // #redirect-action
              public Result index() {
                return redirect("/user/home");
              }
              // #redirect-action
            },
            fakeRequest(),
            mat);
    assertThat(result.status()).isEqualTo(SEE_OTHER);
    assertThat(result.header(LOCATION)).isEqualTo(Optional.of("/user/home"));
  }

  @Test
  public void temporaryRedirectAction() {
    Result result =
        call(
            new MockJavaAction(instanceOf(JavaHandlerComponents.class)) {
              // #temporary-redirect-action
              public Result index() {
                return temporaryRedirect("/user/home");
              }
              // #temporary-redirect-action
            },
            fakeRequest(),
            mat);
    assertThat(result.status()).isEqualTo(TEMPORARY_REDIRECT);
    assertThat(result.header(LOCATION)).hasValue("/user/home");
  }
}
