/**
 * Copyright 2007-2012 Bull S.A.S.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.util.ant.archive;

import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.types.ZipFileSet;
import org.ow2.util.ant.archive.api.IWar;
import org.ow2.util.ant.archive.exploded.WarExploded;
import org.ow2.util.ant.archive.file.WarFile;
import org.ow2.util.ant.archive.info.WarInfo;

/**
 * Task that creates a War archive (.war file or .war directory).
 * @author Florent Benoit
 */
public class War extends AbsArchive {

    /**
     * Path of the WEB-INF folder.
     */
    private static final String WEBINF_FOLDER = "WEB-INF/";

    /**
     * WEB-INF/lib folder.
     */
    private static final String LIB_FOLDER = WEBINF_FOLDER + "lib/";

    /**
     * WEB-INF/classes folder.
     */
    private static final String CLASSES_FOLDER = WEBINF_FOLDER + "classes/";

    /**
     * Default constructor.
     */
    public War() {
        super();
    }

    /**
     * Add files in WEB-INF/classes folder.
     * @param zipFileSet the fileset that contains the files.
     */
    public void addClasses(final ZipFileSet zipFileSet) {
        zipFileSet.setPrefix(CLASSES_FOLDER);
        addFileSet(zipFileSet);
    }

    /**
     * Add files in WEB-INF/lib folder.
     * @param zipFileSet the fileset that contains the files.
     */
    public void addLib(final ZipFileSet zipFileSet) {
        zipFileSet.setPrefix(LIB_FOLDER);
        addFileSet(zipFileSet);
    }

    /**
     * Add files in WEB-INF folder.
     * @param zipFileSet the fileset that contains the files.
     */
    public void addWebinf(final ZipFileSet zipFileSet) {
        zipFileSet.setPrefix(WEBINF_FOLDER);
        addFileSet(zipFileSet);
    }


    /**
     * Execute the task by using either exploded or file mode.
     */
    @Override
    public void execute() {

        log("Building War in '" + getDest() + "'.", Project.MSG_INFO);

        IWar war = null;

        // 2 cases, exploded mode or not
        if (isExploded()) {
            war = new WarExploded(getProject());
        } else {
            war = new WarFile(getProject());
        }

        // Set the name of the task
        ((Task) war).setTaskName(getTaskName());

        // Build the info object
        WarInfo warInfo = new WarInfo();
        war.setWarInfo(warInfo);

        // Fill archive properties
        updateArchiveInfo(warInfo);

        // Execute the task
        war.execute();

    }

}
