package org.ow2.easywsdl.tooling.java2wsdl;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import javax.ejb.Remote;

import org.ow2.easywsdl.tooling.java2wsdl.util.XMLSorter;
import org.ow2.easywsdl.wsdl.WSDLFactory;
import org.ow2.easywsdl.wsdl.api.Description;
import org.ow2.easywsdl.wsdl.api.WSDLException;
import org.ow2.easywsdl.wsdl.api.WSDLWriter;
import org.w3c.dom.Document;

public class Main {

	private static final String TOOL_NAME = "EJB2WSDL";
	private static final String TARGET_WSDL_FLAG = "-o";
	private static final String NS_PREFIX_MAPPING = "-m";
	private static final String VERBOSE_FLAG = "-verbose";
	private static final String VERSION_FLAG = "-version";
	private static final String CONSOLE_FLAG = "-console";
	private static final String POLYMORPH_PARAMETERS = "-poly";	
	private static final String POLYMORPH_CLASSES = "-polyClasses";		
	private String targetWSDLFile;
	private boolean console;
	private boolean verbose;
	private boolean version;
	private List<String> files = new ArrayList<String>();
	private List<String> polymorphClasses = new ArrayList<String>();
	private String targetMapping;
	private boolean customNS = false;
	private boolean polymorph;

	public void setPolymorphClasses(List<String> polymorphClasses) {
		this.polymorphClasses = polymorphClasses;
	}

	public void setPolymorph(boolean polymorph) {
		this.polymorph = polymorph;
	}

	public void setVerbose(boolean verbose) {
		this.verbose = verbose;
	}

	public void setWSDL(String wsdl) {
		this.targetWSDLFile = wsdl;
	}

	public void setServiceClass(String serviceClassName) {
		files.add(serviceClassName);
	}

	public void setCustomNS(boolean customNS) {
		this.customNS = customNS;
	}
	public void setTargetMapping(String targetMapping) {
		this.targetMapping = targetMapping;
	}
	
	public void generateWSDL() {
		process();
	}

	public static void main(String[] args) {
		Main main = new Main();
		
		if (System.getProperty("java.vendor").indexOf("Sun") != -1){
			System.setProperty("javax.xml.transform.TransformerFactory",
			"com.sun.org.apache.xalan.internal.xsltc.trax.TransformerFactoryImpl");
		}else if (System.getProperty("java.vendor").indexOf("IBM") != -1){
			System.setProperty("javax.xml.transform.TransformerFactory",
			"org.apache.xalan.processor.TransformerFactoryImpl");
		}
		
		try {
			if (args.length == 0) {
				main.printUsage();
				System.exit(-1);
			}

			main.processFlags(args);

			main.process();

		} catch (BadParameterException bpe) {
			System.out.println("Unkown or malformed option : "
					+ bpe.getMessage());
			main.printUsage();
		}
	}

	private void process() {
		if (version) {
			printVersion();
		}
		if (files.size() == 0) {
			if (verbose) {
				System.out.println("No class to process");
			}
		} else {
			JavaToEasyWSDL transformer = new JavaToEasyWSDL(verbose);

			List<Class<?>> classes = computeClassList(files);

			try {
				WSDLGenerationContext context = transformer
						.generateWSDL(classes, polymorph, polymorphClasses);
				Description desc = context.processWSDL();
				
				WSDLWriter writer = WSDLFactory.newInstance().newWSDLWriter();

				if (this.customNS)
				{
					StringTokenizer tok = new StringTokenizer(this.targetMapping,",");
					String[] cNS = new String[tok.countTokens()];
					int cpt =0;
					while (tok.hasMoreTokens())
					{
						cNS[cpt++]=tok.nextToken();
					}
					writer.useCustomNamespacesPrefixes(cNS);
				}else
				{
					writer.useNormalizedNamespacesPrefixes();
				}
				
				Document doc = writer.getDocument(desc);
				
				//TODO : Find a better way to sort elements in the XML than
				// this xsl transformation...
		        Document doc2 = XMLSorter.sortNodes(doc);				
				
				String res = org.ow2.easywsdl.schema.util.XMLPrettyPrinter
						.prettyPrint(doc2,"utf8");

				if (this.targetWSDLFile != null
						&& this.targetWSDLFile.length() > 0) {
					File file = new File(targetWSDLFile);
					if (verbose) {
						System.out.println("Writing result to file "
								+ file.getAbsolutePath());
					}
					FileOutputStream fos = new FileOutputStream(file);
					fos.write(res.getBytes());
					fos.close();
				}

				if (this.console) {
					if (verbose) {
						System.out.println("WSDL-----------");
					}
					System.out.println(res);
				}
			} catch (FileNotFoundException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (WSDLException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
	}

	private List<Class<?>> computeClassList(List<String> files2) {
		List<Class<?>> res = new ArrayList<Class<?>>();
		for (String string : files2) {
			Class<?> clazz = null;
			try {
				clazz = Class.forName(string);
			} catch (ClassNotFoundException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
			res.add(clazz);
			if (verbose) {
				if (clazz.getAnnotation(Remote.class) != null) {
					System.out
							.println("Found remote EJB3 Interface ==> Generate WSDL for "
									+ clazz.getName());
				} else {
					System.out.println("Generate WSDL for POJO "
							+ clazz.getName());
				}
			}
		}
		return res;
	}

	private void printVersion() {
		// TODO Not Implemented : Find a way to automatically fill the
		// version/build number from maven, ...
		System.out.println("VERSION NOT IMPLEMENTED");
	}

	private void processFlags(String[] args) throws BadParameterException {
		for (int i = 0; i < args.length; i++) {
			String flag = args[i];
			if (flag.startsWith("-")) {
				if (TARGET_WSDL_FLAG.equals(flag)) {
					i++;
					if (i < args.length) {
						this.targetWSDLFile = args[i];
					} else
						throw new BadParameterException(flag);
				} else if (VERBOSE_FLAG.equals(flag)) {
					this.verbose = true;
				} else if (VERSION_FLAG.equals(flag)) {
					this.version = true;
				} else if (CONSOLE_FLAG.equals(flag)) {
					this.console = true;
				} else if (POLYMORPH_PARAMETERS.equals(flag)) {
					this.polymorph = true;
				} else if (POLYMORPH_CLASSES.equals(flag)) {
					this.polymorph = true;
					i++;
					if (i < args.length) {
						this.polymorphClasses = csvAsList(args[i]);
					} else
						throw new BadParameterException(flag);
				} else if (NS_PREFIX_MAPPING.equals(flag)) {
					this.customNS  = true;
					i++;
					if (i < args.length) {
						this.targetMapping = args[i];
					}else 
						throw new BadParameterException(flag);
				} else 
					throw new BadParameterException(flag);
			} else {
				// it is the name of the file to process
				files.add(flag);
			}
		}
	}

	private List<String> csvAsList(String classList) {
		List<String> res = new ArrayList<String>();
		for (StringTokenizer tok = new StringTokenizer(classList,",;"); tok.hasMoreTokens();){
			res.add(tok.nextToken());
		}
		return res;
	}

	/**
	 * Print the tool usage
	 */
	protected void printUsage() {
		System.out.println(TOOL_NAME + " usage:");
		System.out
				.println(" -?         Display the online help for this utility.");
		System.out.println(" -help");
		System.out.println(" -h");
		System.out
				.println(" -o         Specifies the name of the generated WSDL file");
		System.out
		.println(" -poly Specifies wether or not parameter and return types must be processed as an object tree in the Schema");
		System.out
				.println(" -polyClasses [comma seperated list of additional polymorphic classes] Specifies a list of arbitrary classes to add in the schemas to handle polymorphism");
		System.out
				.println(" -verbose   Displays information during processing");
		System.out.println(" -version   Displays " + TOOL_NAME + " version");
		System.out.println(" -console   Prints results to standard console");
		System.out
		.println(" -m prefix1,ns1,prefix2,ns2...     Force the generation of the namespace mapping " +
						"between prefix1 and ns1, prefix2 and ns2, and so on...");
	}
}
