/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.shutdown;

import java.util.Collections;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.opensearch.client.opensearch._types.ErrorResponse;
import org.opensearch.client.opensearch._types.RequestBase;
import org.opensearch.client.transport.Endpoint;
import org.opensearch.client.transport.endpoints.SimpleEndpoint;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: shutdown.get_node.Request

/**
 * Retrieve status of a node or nodes that are currently marked as shutting
 * down. Designed for indirect use by ECE/ESS and ECK. Direct use is not
 * supported.
 *
 */

public class GetNodeRequest extends RequestBase {
    private final List<String> nodeId;

    // ---------------------------------------------------------------------------------------------

    private GetNodeRequest(Builder builder) {

        this.nodeId = ApiTypeHelper.unmodifiable(builder.nodeId);

    }

    public static GetNodeRequest of(Function<Builder, ObjectBuilder<GetNodeRequest>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Which node for which to retrieve the shutdown status
     * <p>
     * API name: {@code node_id}
     */
    public final List<String> nodeId() {
        return this.nodeId;
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link GetNodeRequest}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<GetNodeRequest> {
        @Nullable
        private List<String> nodeId;

        /**
         * Which node for which to retrieve the shutdown status
         * <p>
         * API name: {@code node_id}
         * <p>
         * Adds all elements of <code>list</code> to <code>nodeId</code>.
         */
        public final Builder nodeId(List<String> list) {
            this.nodeId = _listAddAll(this.nodeId, list);
            return this;
        }

        /**
         * Which node for which to retrieve the shutdown status
         * <p>
         * API name: {@code node_id}
         * <p>
         * Adds one or more values to <code>nodeId</code>.
         */
        public final Builder nodeId(String value, String... values) {
            this.nodeId = _listAdd(this.nodeId, value, values);
            return this;
        }

        /**
         * Builds a {@link GetNodeRequest}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public GetNodeRequest build() {
            _checkSingleUse();

            return new GetNodeRequest(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Endpoint "{@code shutdown.get_node}".
     */
    public static final Endpoint<GetNodeRequest, GetNodeResponse, ErrorResponse> _ENDPOINT = new SimpleEndpoint<>(

        // Request method
        request -> {
            return "GET";

        },

        // Request path
        request -> {
            final int _nodeId = 1 << 0;

            int propsSet = 0;

            if (ApiTypeHelper.isDefined(request.nodeId())) propsSet |= _nodeId;

            if (propsSet == 0) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_nodes");
                buf.append("/shutdown");
                return buf.toString();
            }
            if (propsSet == (_nodeId)) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_nodes");
                buf.append("/");
                SimpleEndpoint.pathEncode(request.nodeId.stream().map(v -> v).collect(Collectors.joining(",")), buf);
                buf.append("/shutdown");
                return buf.toString();
            }
            throw SimpleEndpoint.noPathTemplateFound("path");

        },

        // Request parameters
        request -> {
            return Collections.emptyMap();

        },
        SimpleEndpoint.emptyMap(),
        false,
        GetNodeResponse._DESERIALIZER
    );
}
