/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.nodes;

import jakarta.json.stream.JsonGenerator;
import java.util.function.Function;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.JsonpSerializable;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: nodes._types.DataPathStats

@JsonpDeserializable
public class DataPathStats implements JsonpSerializable {
    private final long availableInBytes;

    private final long freeInBytes;

    private final String mount;

    private final String path;

    private final long totalInBytes;

    private final String type;

    // ---------------------------------------------------------------------------------------------

    private DataPathStats(Builder builder) {

        this.availableInBytes = ApiTypeHelper.requireNonNull(builder.availableInBytes, this, "availableInBytes");
        this.freeInBytes = ApiTypeHelper.requireNonNull(builder.freeInBytes, this, "freeInBytes");
        this.mount = ApiTypeHelper.requireNonNull(builder.mount, this, "mount");
        this.path = ApiTypeHelper.requireNonNull(builder.path, this, "path");
        this.totalInBytes = ApiTypeHelper.requireNonNull(builder.totalInBytes, this, "totalInBytes");
        this.type = ApiTypeHelper.requireNonNull(builder.type, this, "type");

    }

    public static DataPathStats of(Function<Builder, ObjectBuilder<DataPathStats>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Required - API name: {@code available_in_bytes}
     */
    public final long availableInBytes() {
        return this.availableInBytes;
    }

    /**
     * Required - API name: {@code free_in_bytes}
     */
    public final long freeInBytes() {
        return this.freeInBytes;
    }

    /**
     * Required - API name: {@code mount}
     */
    public final String mount() {
        return this.mount;
    }

    /**
     * Required - API name: {@code path}
     */
    public final String path() {
        return this.path;
    }

    /**
     * Required - API name: {@code total_in_bytes}
     */
    public final long totalInBytes() {
        return this.totalInBytes;
    }

    /**
     * Required - API name: {@code type}
     */
    public final String type() {
        return this.type;
    }

    /**
     * Serialize this object to JSON.
     */
    public void serialize(JsonGenerator generator, JsonpMapper mapper) {
        generator.writeStartObject();
        serializeInternal(generator, mapper);
        generator.writeEnd();
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        generator.writeKey("available_in_bytes");
        generator.write(this.availableInBytes);

        generator.writeKey("free_in_bytes");
        generator.write(this.freeInBytes);

        generator.writeKey("mount");
        generator.write(this.mount);

        generator.writeKey("path");
        generator.write(this.path);

        generator.writeKey("total_in_bytes");
        generator.write(this.totalInBytes);

        generator.writeKey("type");
        generator.write(this.type);

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link DataPathStats}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<DataPathStats> {

        private Long availableInBytes;

        private Long freeInBytes;

        private String mount;

        private String path;

        private Long totalInBytes;

        private String type;

        /**
         * Required - API name: {@code available_in_bytes}
         */
        public final Builder availableInBytes(long value) {
            this.availableInBytes = value;
            return this;
        }

        /**
         * Required - API name: {@code free_in_bytes}
         */
        public final Builder freeInBytes(long value) {
            this.freeInBytes = value;
            return this;
        }

        /**
         * Required - API name: {@code mount}
         */
        public final Builder mount(String value) {
            this.mount = value;
            return this;
        }

        /**
         * Required - API name: {@code path}
         */
        public final Builder path(String value) {
            this.path = value;
            return this;
        }

        /**
         * Required - API name: {@code total_in_bytes}
         */
        public final Builder totalInBytes(long value) {
            this.totalInBytes = value;
            return this;
        }

        /**
         * Required - API name: {@code type}
         */
        public final Builder type(String value) {
            this.type = value;
            return this;
        }

        /**
         * Builds a {@link DataPathStats}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public DataPathStats build() {
            _checkSingleUse();

            return new DataPathStats(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link DataPathStats}
     */
    public static final JsonpDeserializer<DataPathStats> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        DataPathStats::setupDataPathStatsDeserializer
    );

    protected static void setupDataPathStatsDeserializer(ObjectDeserializer<DataPathStats.Builder> op) {

        op.add(Builder::availableInBytes, JsonpDeserializer.longDeserializer(), "available_in_bytes");
        op.add(Builder::freeInBytes, JsonpDeserializer.longDeserializer(), "free_in_bytes");
        op.add(Builder::mount, JsonpDeserializer.stringDeserializer(), "mount");
        op.add(Builder::path, JsonpDeserializer.stringDeserializer(), "path");
        op.add(Builder::totalInBytes, JsonpDeserializer.longDeserializer(), "total_in_bytes");
        op.add(Builder::type, JsonpDeserializer.stringDeserializer(), "type");

    }

}
