/*
 * This work is part of OpenLDAP Software <http://www.openldap.org/>.
 *
 *  Copyright 1998-2014 The OpenLDAP Foundation.
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted only as authorized by the OpenLDAP
 *  Public License.
 *
 *  A copy of this license is available in the file LICENSE in the
 *  top-level directory of the distribution or, alternatively, at
 *  <http://www.OpenLDAP.org/license.html>.
 */
package org.openldap.accelerator.api.createSession;


import org.apache.directory.api.ldap.model.message.ExtendedResponseImpl;
import org.apache.directory.api.ldap.model.message.ResultCodeEnum;


/**
 * RbacCreateSessionResponse extended operation implementation.
 * 
 * It follows the ASN.1 description for this operation :
 * <pre>
 * RBACCreateSession ::= SEQUENCE {
 *       sessionId [0] OCTET STRING OPTIONNAL,
 * }
 * </pre>
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class RbacCreateSessionResponseImpl extends ExtendedResponseImpl implements RbacCreateSessionResponse
{
    /** The session ID */
    private String sessionId;

    
    /**
     * Create a new instance for the Rbac Create Session response
     * @param messageId The Message ID
     * @param rcode The result code
     * @param diagnosticMessage The diagnostic message
     */
    public RbacCreateSessionResponseImpl( int messageId, ResultCodeEnum rcode, String diagnosticMessage )
    {
        super( messageId, EXTENSION_OID );

        super.getLdapResult().setMatchedDn( null );
        super.getLdapResult().setResultCode( rcode );
        super.getLdapResult().setDiagnosticMessage( diagnosticMessage );
    }


    /**
     * Create a new instance for the Rbac Create Session response
     * @param messageId The Message ID
     * @param rcode The result code
     */
    public RbacCreateSessionResponseImpl( int messageId, ResultCodeEnum rcode )
    {
        super( messageId, EXTENSION_OID );

        super.getLdapResult().setMatchedDn( null );
        super.getLdapResult().setResultCode( rcode );
    }


    /**
     * Instantiates a new Rbac Create Session response.
     *
     * @param messageId the message id
     */
    public RbacCreateSessionResponseImpl( int messageId )
    {
        super( messageId, EXTENSION_OID );
        super.getLdapResult().setMatchedDn( null );
        super.getLdapResult().setResultCode( ResultCodeEnum.SUCCESS );
    }


    /**
     * Instantiates a new Rbac Create Session response.
     */
    public RbacCreateSessionResponseImpl()
    {
        super( EXTENSION_OID );
        super.getLdapResult().setMatchedDn( null );
        super.getLdapResult().setResultCode( ResultCodeEnum.SUCCESS );
    }


    /**
     * @return The session ID
     */
    public String getSessionId()
    {
        return sessionId;
    }


    /**
     * Sets the session ID
     * 
     * @param sessionID the session ID to set
     */
    public void setSessionId( String sessionID )
    {
        this.sessionId = sessionID;
    }


    /**
     * @see Object#toString()
     */
    public String toString()
    {
        StringBuilder sb = new StringBuilder();

        sb.append( "RbacCreateSessionResponse :" );
        sb.append( "\n    sessionId : " ).append( sessionId );
        
        return sb.toString();
    }
}
