/*
 * This work is part of OpenLDAP Software <http://www.openldap.org/>.
 *
 *  Copyright 1998-2014 The OpenLDAP Foundation.
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted only as authorized by the OpenLDAP
 *  Public License.
 *
 *  A copy of this license is available in the file LICENSE in the
 *  top-level directory of the distribution or, alternatively, at
 *  <http://www.OpenLDAP.org/license.html>.
 */
package org.openldap.accelerator.api.createSession;


import java.util.ArrayList;
import java.util.List;

import org.apache.directory.api.ldap.model.message.AbstractExtendedRequest;


/**
 * RbacCreateSessionRequest extended operation implementation.
 * 
 * It follows the ASN.1 description for this operation :
 * <pre>
 * RbacCreateSession ::= SEQUENCE {
 *       sessionId [0] OCTET STRING OPTIONNAL,
 *       tenantId  [1] OCTET STRING OPTIONNAL,
 *       userId    [2] OCTET STRING OPTIONNAL,
 *       password  [3] OCTET STRING OPTIONNAL,
 *       roles     [4] Roles OPTIONNAL
 * }
 *   
 * Roles ::= SEQUENCE {
 *       role OCTET STRING OPTIONNAL
 * }
 * </pre>
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class RbacCreateSessionRequestImpl extends AbstractExtendedRequest implements RbacCreateSessionRequest
{
    /** The session ID */
    private String sessionId;

    /** The tenant ID */
    private String tenantId;

    /** The user's identity */
    private String userIdentity;

    /** The user's password */
    private String password;

    /** The list of requested roles */
    private List<String> roles;


    /**
     * Create a new instance of the RbacCreateSessionRequest extended operation
     */
    public RbacCreateSessionRequestImpl()
    {
        setRequestName( EXTENSION_OID );
    }


    /**
     * Create a new instance of the RbacCreateSessionRequest extended operation
     * 
     * @param messageId The message ID
     */
    public RbacCreateSessionRequestImpl( int messageId )
    {
        super( messageId );
        setRequestName( EXTENSION_OID );
    }


    /**
     * @return The tenant ID
     */
    public String getTenantId()
    {
        return tenantId;
    }


    /**
     * Sets the tenant ID
     * 
     * @param tenantID The tenant ID to set
     */
    public void setTenantId( String tenantID )
    {
        this.tenantId = tenantID;
    }


    /**
     * Sets the session ID
     * 
     * @param sessionID the session ID to set
     */
    public void setSessionId( String sessionID )
    {
        this.sessionId = sessionID;
    }


    /**
     * @return The session ID
     */
    public String getSessionId()
    {
        return sessionId;
    }


    /**
     * @return the userIdentity
     */
    public String getUserIdentity()
    {
        return userIdentity;
    }


    /**
     * @param userIdentity the userIdentity to set
     */
    public void setUserIdentity( String userIdentity )
    {
        this.userIdentity = userIdentity;
    }


    /**
     * @return the password
     */
    public String getPassword()
    {
        return password;
    }


    /**
     * @param password the password to set
     */
    public void setPassword( String password )
    {
        this.password = password;
    }


    /**
     * @return the roles
     */
    public List<String> getRoles()
    {
        return roles;
    }


    /**
     * @param roles the roles to set
     */
    public void setRoles( List<String> roles )
    {
        if ( roles != null )
        {
            this.roles = new ArrayList<String>( roles.size() );

            this.roles.addAll( roles );
        }
    }


    /**
     * @param role the role to add
     */
    public void addRole( String role )
    {
        if ( roles == null )
        {
            roles = new ArrayList<String>();
        }

        roles.add( role );
    }
    


    @Override
    public RbacCreateSessionResponse getResultResponse()
    {
        if ( getResponse() == null )
        {
            setResponse( new RbacCreateSessionResponseImpl() );
        }

        return null;
    }

    
    /**
     * @see Object#toString()
     */
    public String toString()
    {
        StringBuilder sb = new StringBuilder();

        sb.append( "RbacCreateSessionRequest :" );

        if ( sessionId != null )
        {
            sb.append( "\n    sessionId : " ).append( sessionId );
        }

        if ( tenantId != null )
        {
            sb.append( "\n    tenantId : " ).append( tenantId );
        }

        if ( userIdentity != null )
        {
            sb.append( "\n    userIdentity : " ).append( userIdentity );
        }

        if ( password != null )
        {
            sb.append( "\n    password : " ).append( password );
        }
        
        if ( roles != null )
        {
            sb.append( "\n    roles : {" );
            
            if ( roles.size() != 0 )
            {
                boolean isFirst = true;
                
                for ( String role : roles )
                {
                    if ( isFirst )
                    {
                        isFirst = false;
                    }
                    else
                    {
                        sb.append( ", " );
                    }
                    
                    sb.append( role );
                }
            }

            sb.append( "}" );
        }
        
        return sb.toString();
    }
}
