/*
 * This work is part of OpenLDAP Software <http://www.openldap.org/>.
 *
 *  Copyright 1998-2014 The OpenLDAP Foundation.
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted only as authorized by the OpenLDAP
 *  Public License.
 *
 *  A copy of this license is available in the file LICENSE in the
 *  top-level directory of the distribution or, alternatively, at
 *  <http://www.OpenLDAP.org/license.html>.
 */
package org.openldap.accelerator.api.checkAccess;


import org.apache.directory.api.ldap.model.message.ExtendedResponseImpl;
import org.apache.directory.api.ldap.model.message.ResultCodeEnum;


/**
 * The RbacCheckAccess response
 * 
 * It follows the ASN.1 description for this operation :
 * <pre>
 * RbacCheckAccessResponse ::= Boolean;
 * </pre>
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class RbacCheckAccessResponseImpl extends ExtendedResponseImpl implements RbacCheckAccessResponse
{
    /** The flag telling if the check has granted access to the object */
    private boolean granted;
    
    /**
     * Create a new instance for the RbacCheckAccess response
     * @param messageId The Message ID
     * @param rcode The result code
     * @param diagnosticMessage The diagnostic message
     */
    public RbacCheckAccessResponseImpl( int messageId, ResultCodeEnum rcode, String diagnosticMessage )
    {
        super( messageId, EXTENSION_OID );

        super.getLdapResult().setMatchedDn( null );
        super.getLdapResult().setResultCode( rcode );
        super.getLdapResult().setDiagnosticMessage( diagnosticMessage );
    }


    /**
     * Create a new instance for the RbacCheckAccess response
     * @param messageId The Message ID
     * @param rcode The result code
     */
    public RbacCheckAccessResponseImpl( int messageId, ResultCodeEnum rcode )
    {
        super( messageId, EXTENSION_OID );

        super.getLdapResult().setMatchedDn( null );
        super.getLdapResult().setResultCode( rcode );
    }


    /**
     * Instantiates a new RbacCheckAccess response.
     *
     * @param messageId the message id
     */
    public RbacCheckAccessResponseImpl( int messageId )
    {
        super( messageId, EXTENSION_OID );
        super.getLdapResult().setMatchedDn( null );
        super.getLdapResult().setResultCode( ResultCodeEnum.SUCCESS );
    }


    /**
     * Instantiates a new RbacCheckAccess response.
     */
    public RbacCheckAccessResponseImpl()
    {
        super( EXTENSION_OID );
        super.getLdapResult().setMatchedDn( null );
        super.getLdapResult().setResultCode( ResultCodeEnum.SUCCESS );
    }


    public boolean isGranted()
    {
        return granted;
    }


    public void setGranted( boolean granted )
    {
        this.granted = granted;
    }


    /**
     * @see Object#toString()
     */
    public String toString()
    {
        StringBuilder sb = new StringBuilder();

        sb.append( "RbacCheckAccess response : " );
        
        if ( granted )
        {
            sb.append( "granted" );
        }
        else
        {
            sb.append( "denied" );
        }
            
        sb.append( super.toString() );

        return sb.toString();
    }
}
