/*
 * This work is part of OpenLDAP Software <http://www.openldap.org/>.
 *
 *  Copyright 1998-2014 The OpenLDAP Foundation.
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted only as authorized by the OpenLDAP
 *  Public License.
 *
 *  A copy of this license is available in the file LICENSE in the
 *  top-level directory of the distribution or, alternatively, at
 *  <http://www.OpenLDAP.org/license.html>.
 */
package org.openldap.accelerator.api.checkAccess;


import org.apache.directory.api.ldap.model.message.AbstractExtendedRequest;


/**
 * RbacCheckAccessRequest extended operation implementation.
 * 
 * It follows the ASN.1 description for this operation :
 * <pre>
 * RbacCheckAccessRequest ::= SEQUENCE {
 *       sessionId     [0] OCTET STRING,
 *       operation     [1] OCTET STRING
 *       object        [2] OCTET STRING
 *       objectId      [3] OCTET STRING OPTIONAL
 * }
 * </pre>
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class RbacCheckAccessRequestImpl extends AbstractExtendedRequest implements RbacCheckAccessRequest
{
    /** The session ID */
    private String sessionId;

    /** The operation */
    private String operation;

    /** The object */
    private String object;

    /** The object id is optional*/
    private String objectId;

    /**
     * Create a new instance of the RbacCheckAccessRequestImpl extended operation
     */
    public RbacCheckAccessRequestImpl()
    {
        setRequestName( EXTENSION_OID );
    }


    /**
     * Create a new instance of the RbacCheckAccessRequestImpl extended operation
     * 
     * @param messageId The message ID
     */
    public RbacCheckAccessRequestImpl( int messageId )
    {
        super( messageId );
        setRequestName( EXTENSION_OID );
    }


    /**
     * @return The session ID
     */
    public String getSessionId()
    {
        return sessionId;
    }


    /**
     * Sets the session ID
     * 
     * @param sessionID the session ID to set
     */
    public void setSessionId( String sessionID )
    {
        this.sessionId = sessionID;
    }


    /**
     * @return the operation
     */
    public String getOperation()
    {
        return operation;
    }


    /**
     * @param operation the operation to set
     */
    public void setOperation( String operation )
    {
        this.operation = operation;
    }


    /**
     * @return the object.  This attribute is optional on the request.
     */
    public String getObject()
    {
        return object;
    }


    /**
     * @param object the object to set.  This attribute is optional on the request.
     */
    public void setObject( String object )
    {
        this.object = object;
    }


    /**
     * @return the objectId
     */
    public String getObjectId()
    {
        return objectId;
    }


    /**
     * @param objectId the objectId to set
     */
    public void setObjectId( String objectId )
    {
        this.objectId = objectId;
    }


    @Override
    public RbacCheckAccessResponse getResultResponse()
    {
        if ( getResponse() == null )
        {
            setResponse( new RbacCheckAccessResponseImpl() );
        }

        return null;
    }

    
    /**
     * @see Object#toString()
     */
    public String toString()
    {
        StringBuilder sb = new StringBuilder();
        sb.append( "RbacCheckAccessRequest :" );
        sb.append( "\n    sessionId : " ).append( sessionId );
        sb.append( "\n    operation : " ).append( operation );
        sb.append( "\n    object : " ).append( object );
        if(objectId != null && objectId.length() > 0)
            sb.append( "\n    objectId : " ).append( objectId );
        return sb.toString();
    }
}
