/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package net.java.html.json;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/** Can be used in classes annotated with {@link Model} annotation to
 * define a derived property. Value of derived property is based on values
 * of {@link Property} as enumerated by {@link Model#properties()}.
 * <p>
 * The name of the derived property is the name of the method. The arguments
 * of the method must match names and types of some of the properties 
 * from {@link Model#properties()} list. As soon as one of these properties
 * changes, the method is called to recompute its new value. 
 * This applies to inner changes in derived properties as well - e.g.
 * if the dependant property is another type generated by {@link Model @Model} annotation -
 * changes in its own properties trigger recomputation of this derived 
 * property as well (since version 0.9).
 * <p>
 * Method's return type defines the type of the derived property. It may be
 * any primitive type, {@link String}, {@link Enum enum type} or a 
 * type generated by {@link Model @Model} annotation. One may 
 * also return an array by returning a list of such (boxed) type
 * (for example {@link java.util.List List}&lt;{@link String}&gt; or {@link java.util.List List}&lt;{@link Integer}&gt;).
 * <p>
 * An example testing <a target="_blank" href="http://dew.apidesign.org/dew/#7545568">
 * whether a number is a prime</a> using a {@link ComputedProperty} is available
 * on-line.
 *
 * @author Jaroslav Tulach
 */
@Retention(RetentionPolicy.SOURCE)
@Target(ElementType.METHOD)
public @interface ComputedProperty {
    /** Name of a method to handle changes to the computed property.
     * By default the computed properties are read-only, however one can
     * make them mutable by defining a static method that takes
     * two parameters:
     * <ol>
     * <li>the model class</li>
     * <li>the value - either exactly the return the method annotated
     *   by this property or a superclass (like {@link Object})</li>
     * </ol>
     * Sample code snippet using the <b>write</b> feature of {@link ComputedProperty}
     * could look like this (assuming the {@link Model model class} named
     * <em>DataModel</em> has <b>int</b> property <em>value</em>):
     * <pre>
     * {@link ComputedProperty @ComputedProperty}(write="setPowerValue")
     * <b>static int</b> powerValue(<b>int</b> value) {
     *   <b>return</b> value * value;
     * }
     * <b>static void</b> setPowerValue(DataModel m, <b>int</b> value) {
     *   m.setValue((<b>int</b>){@link Math}.sqrt(value));
     * }
     * </pre>
     * 
     * @return the name of a method to handle changes to the computed
     *   property
     * @since 1.2
     */
    public String write() default "";
}
