/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.cluster;

import java.net.UnknownHostException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.TimeUnit;
import org.neo4j.driver.Bookmark;
import org.neo4j.driver.Logger;
import org.neo4j.driver.Logging;
import org.neo4j.driver.exceptions.DiscoveryException;
import org.neo4j.driver.exceptions.FatalDiscoveryException;
import org.neo4j.driver.exceptions.SecurityException;
import org.neo4j.driver.exceptions.ServiceUnavailableException;
import org.neo4j.driver.internal.BoltServerAddress;
import org.neo4j.driver.internal.DomainNameResolver;
import org.neo4j.driver.internal.ImpersonationUtil;
import org.neo4j.driver.internal.ResolvedBoltServerAddress;
import org.neo4j.driver.internal.cluster.ClusterComposition;
import org.neo4j.driver.internal.cluster.ClusterCompositionLookupResult;
import org.neo4j.driver.internal.cluster.ClusterCompositionProvider;
import org.neo4j.driver.internal.cluster.Rediscovery;
import org.neo4j.driver.internal.cluster.RoutingSettings;
import org.neo4j.driver.internal.cluster.RoutingTable;
import org.neo4j.driver.internal.shaded.io.netty.util.concurrent.EventExecutorGroup;
import org.neo4j.driver.internal.spi.Connection;
import org.neo4j.driver.internal.spi.ConnectionPool;
import org.neo4j.driver.internal.util.Futures;
import org.neo4j.driver.net.ServerAddress;
import org.neo4j.driver.net.ServerAddressResolver;

public class RediscoveryImpl
implements Rediscovery {
    private static final String NO_ROUTERS_AVAILABLE = "Could not perform discovery for database '%s'. No routing server available.";
    private static final String RECOVERABLE_ROUTING_ERROR = "Failed to update routing table with server '%s'.";
    private static final String RECOVERABLE_DISCOVERY_ERROR_WITH_SERVER = "Received a recoverable discovery error with server '%s', will continue discovery with other routing servers if available. Complete failure is reported separately from this entry.";
    private final BoltServerAddress initialRouter;
    private final RoutingSettings settings;
    private final Logger log;
    private final ClusterCompositionProvider provider;
    private final ServerAddressResolver resolver;
    private final EventExecutorGroup eventExecutorGroup;
    private final DomainNameResolver domainNameResolver;

    public RediscoveryImpl(BoltServerAddress initialRouter, RoutingSettings settings, ClusterCompositionProvider provider, EventExecutorGroup eventExecutorGroup, ServerAddressResolver resolver, Logging logging, DomainNameResolver domainNameResolver) {
        this.initialRouter = initialRouter;
        this.settings = settings;
        this.log = logging.getLog(this.getClass());
        this.provider = provider;
        this.resolver = resolver;
        this.eventExecutorGroup = eventExecutorGroup;
        this.domainNameResolver = Objects.requireNonNull(domainNameResolver);
    }

    @Override
    public CompletionStage<ClusterCompositionLookupResult> lookupClusterComposition(RoutingTable routingTable, ConnectionPool connectionPool, Bookmark bookmark, String impersonatedUser) {
        CompletableFuture<ClusterCompositionLookupResult> result = new CompletableFuture<ClusterCompositionLookupResult>();
        ServiceUnavailableException baseError = new ServiceUnavailableException(String.format(NO_ROUTERS_AVAILABLE, routingTable.database().description()));
        this.lookupClusterComposition(routingTable, connectionPool, 0, 0L, result, bookmark, impersonatedUser, baseError);
        return result;
    }

    private void lookupClusterComposition(RoutingTable routingTable, ConnectionPool pool, int failures, long previousDelay, CompletableFuture<ClusterCompositionLookupResult> result, Bookmark bookmark, String impersonatedUser, Throwable baseError) {
        this.lookup(routingTable, pool, bookmark, impersonatedUser, baseError).whenComplete((compositionLookupResult, completionError) -> {
            Throwable error = Futures.completionExceptionCause(completionError);
            if (error != null) {
                result.completeExceptionally(error);
            } else if (compositionLookupResult != null) {
                result.complete((ClusterCompositionLookupResult)compositionLookupResult);
            } else {
                int newFailures = failures + 1;
                if (newFailures >= this.settings.maxRoutingFailures()) {
                    result.completeExceptionally(baseError);
                } else {
                    long nextDelay = Math.max(this.settings.retryTimeoutDelay(), previousDelay * 2L);
                    this.log.info("Unable to fetch new routing table, will try again in " + nextDelay + "ms", new Object[0]);
                    this.eventExecutorGroup.next().schedule(() -> this.lookupClusterComposition(routingTable, pool, newFailures, nextDelay, result, bookmark, impersonatedUser, baseError), nextDelay, TimeUnit.MILLISECONDS);
                }
            }
        });
    }

    private CompletionStage<ClusterCompositionLookupResult> lookup(RoutingTable routingTable, ConnectionPool connectionPool, Bookmark bookmark, String impersonatedUser, Throwable baseError) {
        CompletionStage<ClusterCompositionLookupResult> compositionStage = routingTable.preferInitialRouter() ? this.lookupOnInitialRouterThenOnKnownRouters(routingTable, connectionPool, bookmark, impersonatedUser, baseError) : this.lookupOnKnownRoutersThenOnInitialRouter(routingTable, connectionPool, bookmark, impersonatedUser, baseError);
        return compositionStage;
    }

    private CompletionStage<ClusterCompositionLookupResult> lookupOnKnownRoutersThenOnInitialRouter(RoutingTable routingTable, ConnectionPool connectionPool, Bookmark bookmark, String impersonatedUser, Throwable baseError) {
        HashSet<BoltServerAddress> seenServers = new HashSet<BoltServerAddress>();
        return this.lookupOnKnownRouters(routingTable, connectionPool, seenServers, bookmark, impersonatedUser, baseError).thenCompose(compositionLookupResult -> {
            if (compositionLookupResult != null) {
                return CompletableFuture.completedFuture(compositionLookupResult);
            }
            return this.lookupOnInitialRouter(routingTable, connectionPool, seenServers, bookmark, impersonatedUser, baseError);
        });
    }

    private CompletionStage<ClusterCompositionLookupResult> lookupOnInitialRouterThenOnKnownRouters(RoutingTable routingTable, ConnectionPool connectionPool, Bookmark bookmark, String impersonatedUser, Throwable baseError) {
        Set<BoltServerAddress> seenServers = Collections.emptySet();
        return this.lookupOnInitialRouter(routingTable, connectionPool, seenServers, bookmark, impersonatedUser, baseError).thenCompose(compositionLookupResult -> {
            if (compositionLookupResult != null) {
                return CompletableFuture.completedFuture(compositionLookupResult);
            }
            return this.lookupOnKnownRouters(routingTable, connectionPool, new HashSet<BoltServerAddress>(), bookmark, impersonatedUser, baseError);
        });
    }

    private CompletionStage<ClusterCompositionLookupResult> lookupOnKnownRouters(RoutingTable routingTable, ConnectionPool connectionPool, Set<BoltServerAddress> seenServers, Bookmark bookmark, String impersonatedUser, Throwable baseError) {
        BoltServerAddress[] addresses = routingTable.routers().toArray();
        CompletionStage result = Futures.completedWithNull();
        for (BoltServerAddress address : addresses) {
            result = result.thenCompose(composition -> {
                if (composition != null) {
                    return CompletableFuture.completedFuture(composition);
                }
                return this.lookupOnRouter(address, true, routingTable, connectionPool, seenServers, bookmark, impersonatedUser, baseError);
            });
        }
        return result.thenApply(composition -> composition != null ? new ClusterCompositionLookupResult((ClusterComposition)composition) : null);
    }

    private CompletionStage<ClusterCompositionLookupResult> lookupOnInitialRouter(RoutingTable routingTable, ConnectionPool connectionPool, Set<BoltServerAddress> seenServers, Bookmark bookmark, String impersonatedUser, Throwable baseError) {
        List<BoltServerAddress> resolvedRouters;
        try {
            resolvedRouters = this.resolve();
        }
        catch (Throwable error) {
            return Futures.failedFuture(error);
        }
        HashSet<BoltServerAddress> resolvedRouterSet = new HashSet<BoltServerAddress>(resolvedRouters);
        resolvedRouters.removeAll(seenServers);
        CompletionStage result = Futures.completedWithNull();
        for (BoltServerAddress address : resolvedRouters) {
            result = result.thenCompose(composition -> {
                if (composition != null) {
                    return CompletableFuture.completedFuture(composition);
                }
                return this.lookupOnRouter(address, false, routingTable, connectionPool, null, bookmark, impersonatedUser, baseError);
            });
        }
        return result.thenApply(composition -> composition != null ? new ClusterCompositionLookupResult((ClusterComposition)composition, (Set<BoltServerAddress>)resolvedRouterSet) : null);
    }

    private CompletionStage<ClusterComposition> lookupOnRouter(BoltServerAddress routerAddress, boolean resolveAddress, RoutingTable routingTable, ConnectionPool connectionPool, Set<BoltServerAddress> seenServers, Bookmark bookmark, String impersonatedUser, Throwable baseError) {
        CompletableFuture<BoltServerAddress> addressFuture = CompletableFuture.completedFuture(routerAddress);
        return ((CompletableFuture)((CompletableFuture)((CompletableFuture)((CompletableFuture)((CompletableFuture)addressFuture.thenApply(address -> resolveAddress ? this.resolveByDomainNameOrThrowCompletionException((BoltServerAddress)address, routingTable) : address)).thenApply(address -> this.addAndReturn(seenServers, address))).thenCompose(connectionPool::acquire)).thenApply(connection -> ImpersonationUtil.ensureImpersonationSupport(connection, impersonatedUser))).thenCompose(connection -> this.provider.getClusterComposition((Connection)connection, routingTable.database(), bookmark, impersonatedUser))).handle((response, error) -> {
            Throwable cause = Futures.completionExceptionCause(error);
            if (cause != null) {
                return this.handleRoutingProcedureError(cause, routingTable, routerAddress, baseError);
            }
            return response;
        });
    }

    private ClusterComposition handleRoutingProcedureError(Throwable error, RoutingTable routingTable, BoltServerAddress routerAddress, Throwable baseError) {
        if (error instanceof SecurityException || error instanceof FatalDiscoveryException || error instanceof IllegalStateException && "Pool closed".equals(error.getMessage())) {
            throw new CompletionException(error);
        }
        DiscoveryException discoveryError = new DiscoveryException(String.format(RECOVERABLE_ROUTING_ERROR, routerAddress), error);
        Futures.combineErrors(baseError, discoveryError);
        String warningMessage = String.format(RECOVERABLE_DISCOVERY_ERROR_WITH_SERVER, routerAddress);
        this.log.warn(warningMessage, new Object[0]);
        this.log.debug(warningMessage, discoveryError);
        routingTable.forget(routerAddress);
        return null;
    }

    @Override
    public List<BoltServerAddress> resolve() throws UnknownHostException {
        LinkedList<BoltServerAddress> resolvedAddresses = new LinkedList<BoltServerAddress>();
        UnknownHostException exception = null;
        for (ServerAddress serverAddress : this.resolver.resolve(this.initialRouter)) {
            try {
                this.resolveAllByDomainName(serverAddress).unicastStream().forEach(resolvedAddresses::add);
            }
            catch (UnknownHostException e) {
                if (exception == null) {
                    exception = e;
                    continue;
                }
                exception.addSuppressed(e);
            }
        }
        if (resolvedAddresses.isEmpty() && exception != null) {
            throw exception;
        }
        return resolvedAddresses;
    }

    private <T> T addAndReturn(Collection<T> collection, T element) {
        if (collection != null) {
            collection.add(element);
        }
        return element;
    }

    private BoltServerAddress resolveByDomainNameOrThrowCompletionException(BoltServerAddress address, RoutingTable routingTable) {
        try {
            ResolvedBoltServerAddress resolvedAddress = this.resolveAllByDomainName(address);
            routingTable.replaceRouterIfPresent(address, resolvedAddress);
            return resolvedAddress.unicastStream().findFirst().orElseThrow(() -> new IllegalStateException("Unexpected condition, the ResolvedBoltServerAddress must always have at least one unicast address"));
        }
        catch (Throwable e) {
            throw new CompletionException(e);
        }
    }

    private ResolvedBoltServerAddress resolveAllByDomainName(ServerAddress address) throws UnknownHostException {
        return new ResolvedBoltServerAddress(address.host(), address.port(), this.domainNameResolver.resolve(address.host()));
    }
}

