/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.cluster;

import io.netty.util.concurrent.EventExecutorGroup;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.TimeUnit;
import org.neo4j.driver.internal.async.AsyncConnection;
import org.neo4j.driver.internal.async.pool.AsyncConnectionPool;
import org.neo4j.driver.internal.cluster.ClusterComposition;
import org.neo4j.driver.internal.cluster.ClusterCompositionProvider;
import org.neo4j.driver.internal.cluster.ClusterCompositionResponse;
import org.neo4j.driver.internal.cluster.HostNameResolver;
import org.neo4j.driver.internal.cluster.RoutingSettings;
import org.neo4j.driver.internal.cluster.RoutingTable;
import org.neo4j.driver.internal.net.BoltServerAddress;
import org.neo4j.driver.internal.spi.ConnectionPool;
import org.neo4j.driver.internal.spi.PooledConnection;
import org.neo4j.driver.internal.util.Clock;
import org.neo4j.driver.v1.Logger;
import org.neo4j.driver.v1.exceptions.SecurityException;
import org.neo4j.driver.v1.exceptions.ServiceUnavailableException;

public class Rediscovery {
    private static final String NO_ROUTERS_AVAILABLE = "Could not perform discovery. No routing servers available.";
    private final BoltServerAddress initialRouter;
    private final RoutingSettings settings;
    private final Clock clock;
    private final Logger logger;
    private final ClusterCompositionProvider provider;
    private final HostNameResolver hostNameResolver;
    private final EventExecutorGroup eventExecutorGroup;
    private volatile boolean useInitialRouter;

    public Rediscovery(BoltServerAddress initialRouter, RoutingSettings settings, ClusterCompositionProvider provider, EventExecutorGroup eventExecutorGroup, HostNameResolver hostNameResolver, Clock clock, Logger logger) {
        this(initialRouter, settings, provider, hostNameResolver, eventExecutorGroup, clock, logger, false);
    }

    public Rediscovery(BoltServerAddress initialRouter, RoutingSettings settings, ClusterCompositionProvider provider, HostNameResolver hostNameResolver, EventExecutorGroup eventExecutorGroup, Clock clock, Logger logger, boolean useInitialRouter) {
        this.initialRouter = initialRouter;
        this.settings = settings;
        this.clock = clock;
        this.logger = logger;
        this.provider = provider;
        this.hostNameResolver = hostNameResolver;
        this.eventExecutorGroup = eventExecutorGroup;
        this.useInitialRouter = useInitialRouter;
    }

    public ClusterComposition lookupClusterComposition(RoutingTable routingTable, ConnectionPool connections) {
        int failures = 0;
        long start = this.clock.millis();
        long delay = 0L;
        while (true) {
            long waitTime = start + delay - this.clock.millis();
            this.sleep(waitTime);
            start = this.clock.millis();
            ClusterComposition composition = this.lookup(routingTable, connections);
            if (composition != null) {
                return composition;
            }
            if (++failures >= this.settings.maxRoutingFailures()) {
                throw new ServiceUnavailableException(NO_ROUTERS_AVAILABLE);
            }
            delay = Math.max(this.settings.retryTimeoutDelay(), delay * 2L);
        }
    }

    public CompletionStage<ClusterComposition> lookupClusterCompositionAsync(RoutingTable routingTable, AsyncConnectionPool connectionPool) {
        CompletableFuture<ClusterComposition> result = new CompletableFuture<ClusterComposition>();
        this.lookupClusterComposition(routingTable, connectionPool, 0, 0L, result);
        return result;
    }

    private void lookupClusterComposition(RoutingTable routingTable, AsyncConnectionPool pool, int failures, long previousDelay, CompletableFuture<ClusterComposition> result) {
        if (failures >= this.settings.maxRoutingFailures()) {
            result.completeExceptionally(new ServiceUnavailableException(NO_ROUTERS_AVAILABLE));
            return;
        }
        this.lookupAsync(routingTable, pool).whenComplete((composition, error) -> {
            if (error != null) {
                result.completeExceptionally((Throwable)error);
            } else if (composition != null) {
                result.complete((ClusterComposition)composition);
            } else {
                long nextDelay = Math.max(this.settings.retryTimeoutDelay(), previousDelay * 2L);
                this.logger.info("Unable to fetch new routing table, will try again in " + nextDelay + "ms", new Object[0]);
                this.eventExecutorGroup.next().schedule(() -> this.lookupClusterComposition(routingTable, pool, failures + 1, nextDelay, result), nextDelay, TimeUnit.MILLISECONDS);
            }
        });
    }

    private ClusterComposition lookup(RoutingTable routingTable, ConnectionPool connections) {
        ClusterComposition composition;
        if (this.useInitialRouter) {
            composition = this.lookupOnInitialRouterThenOnKnownRouters(routingTable, connections);
            this.useInitialRouter = false;
        } else {
            composition = this.lookupOnKnownRoutersThenOnInitialRouter(routingTable, connections);
        }
        if (composition != null && !composition.hasWriters()) {
            this.useInitialRouter = true;
        }
        return composition;
    }

    private CompletionStage<ClusterComposition> lookupAsync(RoutingTable routingTable, AsyncConnectionPool connectionPool) {
        CompletionStage<ClusterComposition> compositionStage;
        if (this.useInitialRouter) {
            compositionStage = this.lookupOnInitialRouterThenOnKnownRoutersAsync(routingTable, connectionPool);
            this.useInitialRouter = false;
        } else {
            compositionStage = this.lookupOnKnownRoutersThenOnInitialRouterAsync(routingTable, connectionPool);
        }
        return compositionStage.whenComplete((composition, error) -> {
            if (composition != null && !composition.hasWriters()) {
                this.useInitialRouter = true;
            }
        });
    }

    private ClusterComposition lookupOnKnownRoutersThenOnInitialRouter(RoutingTable routingTable, ConnectionPool connections) {
        HashSet<BoltServerAddress> seenServers = new HashSet<BoltServerAddress>();
        ClusterComposition composition = this.lookupOnKnownRouters(routingTable, connections, seenServers);
        if (composition == null) {
            return this.lookupOnInitialRouter(routingTable, connections, seenServers);
        }
        return composition;
    }

    private CompletionStage<ClusterComposition> lookupOnKnownRoutersThenOnInitialRouterAsync(RoutingTable routingTable, AsyncConnectionPool connectionPool) {
        HashSet<BoltServerAddress> seenServers = new HashSet<BoltServerAddress>();
        return this.lookupOnKnownRoutersAsync(routingTable, connectionPool, seenServers).thenCompose(composition -> {
            if (composition != null) {
                return CompletableFuture.completedFuture(composition);
            }
            return this.lookupOnInitialRouterAsync(routingTable, connectionPool, seenServers);
        });
    }

    private ClusterComposition lookupOnInitialRouterThenOnKnownRouters(RoutingTable routingTable, ConnectionPool connections) {
        Set<BoltServerAddress> seenServers = Collections.emptySet();
        ClusterComposition composition = this.lookupOnInitialRouter(routingTable, connections, seenServers);
        if (composition == null) {
            return this.lookupOnKnownRouters(routingTable, connections, new HashSet<BoltServerAddress>());
        }
        return composition;
    }

    private CompletionStage<ClusterComposition> lookupOnInitialRouterThenOnKnownRoutersAsync(RoutingTable routingTable, AsyncConnectionPool connectionPool) {
        Set<BoltServerAddress> seenServers = Collections.emptySet();
        return this.lookupOnInitialRouterAsync(routingTable, connectionPool, seenServers).thenCompose(composition -> {
            if (composition != null) {
                return CompletableFuture.completedFuture(composition);
            }
            return this.lookupOnKnownRoutersAsync(routingTable, connectionPool, new HashSet<BoltServerAddress>());
        });
    }

    private ClusterComposition lookupOnKnownRouters(RoutingTable routingTable, ConnectionPool connections, Set<BoltServerAddress> seenServers) {
        BoltServerAddress[] addresses;
        for (BoltServerAddress address : addresses = routingTable.routers().toArray()) {
            ClusterComposition composition = this.lookupOnRouter(address, routingTable, connections);
            if (composition != null) {
                return composition;
            }
            seenServers.add(address);
        }
        return null;
    }

    private CompletionStage<ClusterComposition> lookupOnKnownRoutersAsync(RoutingTable routingTable, AsyncConnectionPool connectionPool, Set<BoltServerAddress> seenServers) {
        BoltServerAddress[] addresses = routingTable.routers().toArray();
        CompletionStage<Object> result = CompletableFuture.completedFuture(null);
        for (BoltServerAddress address : addresses) {
            result = result.thenCompose(composition -> {
                if (composition != null) {
                    return CompletableFuture.completedFuture(composition);
                }
                return this.lookupOnRouterAsync(address, routingTable, connectionPool).whenComplete((ignore, error) -> seenServers.add(address));
            });
        }
        return result;
    }

    private ClusterComposition lookupOnInitialRouter(RoutingTable routingTable, ConnectionPool connections, Set<BoltServerAddress> seenServers) {
        Set<BoltServerAddress> ips = this.hostNameResolver.resolve(this.initialRouter);
        ips.removeAll(seenServers);
        for (BoltServerAddress address : ips) {
            ClusterComposition composition = this.lookupOnRouter(address, routingTable, connections);
            if (composition == null) continue;
            return composition;
        }
        return null;
    }

    private CompletionStage<ClusterComposition> lookupOnInitialRouterAsync(RoutingTable routingTable, AsyncConnectionPool connectionPool, Set<BoltServerAddress> seenServers) {
        Set<BoltServerAddress> addresses = this.hostNameResolver.resolve(this.initialRouter);
        addresses.removeAll(seenServers);
        CompletionStage<Object> result = CompletableFuture.completedFuture(null);
        for (BoltServerAddress address : addresses) {
            result = result.thenCompose(composition -> {
                if (composition != null) {
                    return CompletableFuture.completedFuture(composition);
                }
                return this.lookupOnRouterAsync(address, routingTable, connectionPool);
            });
        }
        return result;
    }

    private ClusterComposition lookupOnRouter(BoltServerAddress routerAddress, RoutingTable routingTable, ConnectionPool connections) {
        ClusterCompositionResponse response;
        try (PooledConnection connection = connections.acquire(routerAddress);){
            response = this.provider.getClusterComposition(connection);
        }
        catch (SecurityException e) {
            throw e;
        }
        catch (Throwable t) {
            this.logger.error(String.format("Failed to connect to routing server '%s'.", routerAddress), t);
            routingTable.forget(routerAddress);
            return null;
        }
        ClusterComposition cluster = response.clusterComposition();
        this.logger.info("Got cluster composition %s", cluster);
        return cluster;
    }

    private CompletionStage<ClusterComposition> lookupOnRouterAsync(BoltServerAddress routerAddress, RoutingTable routingTable, AsyncConnectionPool connectionPool) {
        CompletionStage<AsyncConnection> connectionStage = connectionPool.acquire(routerAddress);
        return this.provider.getClusterComposition(connectionStage).handle((response, error) -> {
            if (error != null) {
                return this.handleRoutingProcedureError((Throwable)error, routingTable, routerAddress);
            }
            ClusterComposition cluster = response.clusterComposition();
            this.logger.info("Got cluster composition %s", cluster);
            return cluster;
        });
    }

    private ClusterComposition handleRoutingProcedureError(Throwable error, RoutingTable routingTable, BoltServerAddress routerAddress) {
        if (error instanceof SecurityException) {
            throw new CompletionException(error);
        }
        this.logger.error(String.format("Failed to connect to routing server '%s'.", routerAddress), error);
        routingTable.forget(routerAddress);
        return null;
    }

    private void sleep(long millis) {
        if (millis > 0L) {
            try {
                this.clock.sleep(millis);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new ServiceUnavailableException("Thread was interrupted while performing discovery", e);
            }
        }
    }
}

