/* Copyright (c) 2023 LibJ
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * You should have received a copy of The MIT License (MIT) along with this
 * program. If not, see <http://opensource.org/licenses/MIT/>.
 */

package org.libj.util;

import java.util.Comparator;
import java.util.Set;
import java.util.SortedSet;

/**
 * A {@link DelegateSet} that provides callback methods to observe the retrieval, addition, and removal of elements, either due to
 * direct method invocation on the set instance itself, or via {@link #iterator()}, {@link #stream()}, and any other entrypoint that
 * facilitates modification of the elements in this set.
 *
 * @param <E> The type of elements in this set.
 * @see #afterGet(Object,RuntimeException)
 * @see #beforeAdd(Object,Object)
 * @see #afterAdd(Object,RuntimeException)
 * @see #beforeRemove(Object)
 * @see #afterRemove(Object,RuntimeException)
 */
public abstract class ObservableSortedSet<E> extends ObservableSet<E> implements SortedSet<E> {
  /**
   * Creates a new {@link ObservableSortedSet} with the specified target {@code set}.
   *
   * @param set The target {@link Set}.
   * @throws NullPointerException If {@code set} is null.
   */
  public ObservableSortedSet(final Set<E> set) {
    super(set);
  }

  @Override
  public Comparator<? super E> comparator() {
    return ((SortedSet<E>)target).comparator();
  }

  protected class SortedSubSet extends ObservableSortedSet<E> {
    protected SortedSubSet(final SortedSet<E> set) {
      super(set);
    }

    @Override
    protected E afterGet(final E value, final RuntimeException e) {
      return ObservableSortedSet.this.afterGet(value, e);
    }

    @Override
    protected Object beforeAdd(final E element, final Object preventDefault) {
      return ObservableSortedSet.this.beforeAdd(element, preventDefault);
    }

    @Override
    protected void afterAdd(final E element, final RuntimeException e) {
      ObservableSortedSet.this.afterAdd(element, e);
    }

    @Override
    protected boolean beforeRemove(final Object element) {
      return ObservableSortedSet.this.beforeRemove(element);
    }

    @Override
    protected void afterRemove(final Object element, final RuntimeException e) {
      ObservableSortedSet.this.afterRemove(element, e);
    }
  }

  @Override
  public SortedSet<E> subSet(final E fromElement, final E toElement) {
    return new SortedSubSet(((SortedSet<E>)target).subSet(fromElement, toElement));
  }

  @Override
  public SortedSet<E> headSet(final E toElement) {
    return new SortedSubSet(((SortedSet<E>)target).headSet(toElement));
  }

  @Override
  public SortedSet<E> tailSet(final E fromElement) {
    return new SortedSubSet(((SortedSet<E>)target).tailSet(fromElement));
  }

  @Override
  public E first() {
    return ((SortedSet<E>)target).first();
  }

  @Override
  public E last() {
    return ((SortedSet<E>)target).last();
  }
}