/* Copyright (c) 2023 LibJ
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * You should have received a copy of The MIT License (MIT) along with this
 * program. If not, see <http://opensource.org/licenses/MIT/>.
 */

package org.libj.util;

import java.util.Iterator;
import java.util.NavigableSet;
import java.util.Set;

/**
 * A {@link DelegateSet} that provides callback methods to observe the retrieval, addition, and removal of elements, either due to
 * direct method invocation on the set instance itself, or via {@link #iterator()}, {@link #stream()}, and any other entrypoint that
 * facilitates modification of the elements in this set.
 *
 * @param <E> The type of elements in this set.
 * @see #afterGet(Object,RuntimeException)
 * @see #beforeAdd(Object,Object)
 * @see #afterAdd(Object,RuntimeException)
 * @see #beforeRemove(Object)
 * @see #afterRemove(Object,RuntimeException)
 */
public abstract class ObservableNavigableSet<E> extends ObservableSortedSet<E> implements NavigableSet<E> {
  /**
   * Creates a new {@link ObservableNavigableSet} with the specified target {@code set}.
   *
   * @param set The target {@link Set}.
   * @throws NullPointerException If {@code set} is null.
   */
  public ObservableNavigableSet(final NavigableSet<E> set) {
    super(set);
  }

  @Override
  public E lower(final E e) {
    return ((NavigableSet<E>)target).lower(e);
  }

  @Override
  public E floor(final E e) {
    return ((NavigableSet<E>)target).floor(e);
  }

  @Override
  public E ceiling(final E e) {
    return ((NavigableSet<E>)target).ceiling(e);
  }

  @Override
  public E higher(final E e) {
    return ((NavigableSet<E>)target).higher(e);
  }

  @Override
  public E pollFirst() {
    return ((NavigableSet<E>)target).pollFirst();
  }

  @Override
  public E pollLast() {
    return ((NavigableSet<E>)target).pollLast();
  }

  protected class NavigableSubSet extends ObservableNavigableSet<E> {
    protected NavigableSubSet(final NavigableSet<E> set) {
      super(set);
    }

    @Override
    protected E afterGet(final E value, final RuntimeException e) {
      return ObservableNavigableSet.this.afterGet(value, e);
    }

    @Override
    protected Object beforeAdd(final E element, final Object preventDefault) {
      return ObservableNavigableSet.this.beforeAdd(element, preventDefault);
    }

    @Override
    protected void afterAdd(final E element, final RuntimeException e) {
      ObservableNavigableSet.this.afterAdd(element, e);
    }

    @Override
    protected boolean beforeRemove(final Object element) {
      return ObservableNavigableSet.this.beforeRemove(element);
    }

    @Override
    protected void afterRemove(final Object element, final RuntimeException e) {
      ObservableNavigableSet.this.afterRemove(element, e);
    }
  }

  @Override
  public NavigableSet<E> descendingSet() {
    return new NavigableSubSet(((NavigableSet<E>)target).descendingSet());
  }

  @Override
  public Iterator<E> descendingIterator() {
    return new ObservableIterator(((NavigableSet<E>)target).descendingIterator());
  }

  @Override
  public NavigableSet<E> subSet(final E fromElement, final boolean fromInclusive, final E toElement, final boolean toInclusive) {
    return new NavigableSubSet(((NavigableSet<E>)target).subSet(fromElement, fromInclusive, toElement, toInclusive));
  }

  @Override
  public NavigableSet<E> headSet(final E toElement, final boolean inclusive) {
    return new NavigableSubSet(((NavigableSet<E>)target).headSet(toElement, inclusive));
  }

  @Override
  public NavigableSet<E> tailSet(final E fromElement, final boolean inclusive) {
    return new NavigableSubSet(((NavigableSet<E>)target).tailSet(fromElement, inclusive));
  }
}