/* Copyright (c) 2023 LibJ
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * You should have received a copy of The MIT License (MIT) along with this
 * program. If not, see <http://opensource.org/licenses/MIT/>.
 */

package org.libj.util;

import java.util.Iterator;
import java.util.Map;
import java.util.NavigableMap;
import java.util.NavigableSet;

/**
 * A {@link DelegateMap} that provides callback methods to observe the retrieval, addition, and removal of elements, either due to
 * direct method invocation on the map instance itself, or via {@link #entrySet()}, {@link #values()},
 * {@link #forEach(java.util.function.BiConsumer)}, and any other entrypoint that facilitates modification of the elements in this
 * map.
 *
 * @param <K> The type of keys maintained by this map.
 * @param <V> The type of mapped values.
 * @see #beforeGet(Object)
 * @see #afterGet(Object,Object,RuntimeException)
 * @see #beforePut(Object,Object,Object,Object)
 * @see #afterPut(Object,Object,Object,RuntimeException)
 * @see #beforeRemove(Object,Object)
 * @see #afterRemove(Object,Object,RuntimeException)
 */
public abstract class ObservableNavigableMap<K,V> extends ObservableSortedMap<K,V> implements NavigableMap<K,V> {
  /**
   * Creates a new {@link ObservableNavigableMap} with the specified target {@link Map}.
   *
   * @param map The target {@link NavigableMap}.
   * @throws NullPointerException If {@code map} is null.
   */
  public ObservableNavigableMap(final NavigableMap<K,V> map) {
    super(map);
  }

  @Override
  public Map.Entry<K,V> lowerEntry(final K key) {
    return ((NavigableMap<K,V>)target).lowerEntry(key);
  }

  @Override
  public K lowerKey(final K key) {
    return ((NavigableMap<K,V>)target).lowerKey(key);
  }

  @Override
  public Map.Entry<K,V> floorEntry(final K key) {
    return ((NavigableMap<K,V>)target).floorEntry(key);
  }

  @Override
  public K floorKey(final K key) {
    return ((NavigableMap<K,V>)target).floorKey(key);
  }

  @Override
  public Map.Entry<K,V> ceilingEntry(final K key) {
    return ((NavigableMap<K,V>)target).ceilingEntry(key);
  }

  @Override
  public K ceilingKey(final K key) {
    return ((NavigableMap<K,V>)target).ceilingKey(key);
  }

  @Override
  public Map.Entry<K,V> higherEntry(final K key) {
    return ((NavigableMap<K,V>)target).higherEntry(key);
  }

  @Override
  public K higherKey(final K key) {
    return ((NavigableMap<K,V>)target).higherKey(key);
  }

  @Override
  public Map.Entry<K,V> firstEntry() {
    return ((NavigableMap<K,V>)target).firstEntry();
  }

  @Override
  public Map.Entry<K,V> lastEntry() {
    return ((NavigableMap<K,V>)target).lastEntry();
  }

  @Override
  public Map.Entry<K,V> pollFirstEntry() {
    return ((NavigableMap<K,V>)target).pollFirstEntry();
  }

  @Override
  public Map.Entry<K,V> pollLastEntry() {
    return ((NavigableMap<K,V>)target).pollLastEntry();
  }

  protected class ObservableNavigableSubMap extends ObservableNavigableMap<K,V> {
    protected ObservableNavigableSubMap(final NavigableMap<K,V> map) {
      super(map);
    }

    @Override
    protected boolean beforeContainsKey(final Object key) {
      return ObservableNavigableMap.this.beforeContainsKey(key);
    }

    @Override
    protected void afterContainsKey(final Object key, final boolean result, final RuntimeException e) {
      ObservableNavigableMap.this.afterContainsKey(key, result, e);
    }

    @Override
    protected void beforeContainsValue(final Object value) {
      ObservableNavigableMap.this.beforeContainsValue(value);
    }

    @Override
    protected void afterContainsValue(final Object value, final boolean result, final RuntimeException e) {
      ObservableNavigableMap.this.afterContainsValue(value, result, e);
    }

    @Override
    protected Object beforeGet(final Object key) {
      return ObservableNavigableMap.this.beforeGet(key);
    }

    @Override
    protected V afterGet(final Object key, final V value, final RuntimeException e) {
      return ObservableNavigableMap.this.afterGet(key, value, e);
    }

    @Override
    protected Object beforePut(final K key, final V oldValue, final V newValue, final Object preventDefault) {
      return ObservableNavigableMap.this.beforePut(key, oldValue, newValue, preventDefault);
    }

    @Override
    protected void afterPut(final K key, final V oldValue, final V newValue, final RuntimeException e) {
      ObservableNavigableMap.this.afterPut(key, oldValue, newValue, e);
    }

    @Override
    protected boolean beforeRemove(final Object key, final V value) {
      return ObservableNavigableMap.this.beforeRemove(key, value);
    }

    @Override
    protected void afterRemove(final Object key, final V value, final RuntimeException e) {
      ObservableNavigableMap.this.afterRemove(key, value, e);
    }
  }

  @Override
  public NavigableMap<K,V> descendingMap() {
    return new ObservableNavigableSubMap(((NavigableMap<K,V>)target).descendingMap());
  }

  /**
   * An {@link NavigableSet NavigableSet&lt;K&gt;} that delegates callback methods to the parent {@link NavigableMap} instance for the
   * retrieval and removal of keys.
   */
  protected class ObservableNavigableKeySet extends ObservableNavigableSet<K> {
    /**
     * Creates a new {@link ObservableNavigableKeySet} for the specified {@link NavigableSet NavigableSet&lt;K&gt;}.
     *
     * @param set The {@link Iterator}.
     * @throws NullPointerException If the specified {@link NavigableSet NavigableSet&lt;K&gt;} is null.
     */
    protected ObservableNavigableKeySet(final NavigableSet<K> set) {
      super(set);
    }

    private final ThreadLocal<Object> localKey = new ThreadLocal<>();
    private final ThreadLocal<V> localNewValue = new ThreadLocal<>();

    @Override
    protected K beforeAdd(final K element, final Object preventDefault) {
      throw new UnsupportedOperationException();
    }

    @Override
    protected boolean beforeRemove(final Object element) {
      localKey.set(element);
      final V value = ObservableNavigableMap.this.get(element);
      localNewValue.set(value);
      return ObservableNavigableMap.this.beforeRemove(element, value);
    }

    @Override
    protected void afterRemove(final Object element, final RuntimeException e) {
      ObservableNavigableMap.this.afterRemove(localKey.get(), localNewValue.get(), e);
    }
  }

  @Override
  public NavigableSet<K> navigableKeySet() {
    return new ObservableNavigableKeySet(((NavigableMap<K,V>)target).navigableKeySet());
  }

  @Override
  public NavigableSet<K> descendingKeySet() {
    return new ObservableNavigableKeySet(((NavigableMap<K,V>)target).descendingKeySet());
  }

  @Override
  public NavigableMap<K,V> subMap(final K fromKey, final boolean fromInclusive, final K toKey, final boolean toInclusive) {
    return new ObservableNavigableSubMap(((NavigableMap<K,V>)target).subMap(fromKey, fromInclusive, toKey, toInclusive));
  }

  @Override
  public NavigableMap<K,V> headMap(final K toKey, final boolean inclusive) {
    return new ObservableNavigableSubMap(((NavigableMap<K,V>)target).headMap(toKey, inclusive));
  }

  @Override
  public NavigableMap<K,V> tailMap(final K fromKey, final boolean inclusive) {
    return new ObservableNavigableSubMap(((NavigableMap<K,V>)target).tailMap(fromKey, inclusive));
  }
}