/* Copyright (c) 2023 LibJ
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * You should have received a copy of The MIT License (MIT) along with this
 * program. If not, see <http://opensource.org/licenses/MIT/>.
 */

package org.libj.util;

import java.util.AbstractMap;
import java.util.Map;
import java.util.NavigableMap;
import java.util.NavigableSet;

/**
 * A {@link DelegateNavigableMap} contains some other {@link Map}, to which it delegates its method calls, possibly transforming the
 * data along the way or providing additional functionality. The class {@link DelegateNavigableMap} itself simply overrides all
 * methods of {@link AbstractMap} with versions that pass all requests to the target {@link Map}. Subclasses of
 * {@link DelegateNavigableMap} may further override some of these methods and may also provide additional methods and fields.
 *
 * @param <K> The type of keys maintained by this map.
 * @param <V> The type of mapped values.
 */
public abstract class DelegateNavigableMap<K,V> extends DelegateSortedMap<K,V> implements NavigableMap<K,V> {
  /**
   * Creates a new {@link DelegateNavigableMap} with the specified target {@link Map}.
   *
   * @param target The target {@link NavigableMap}.
   * @throws NullPointerException If the target {@link NavigableMap} is null.
   */
  public DelegateNavigableMap(final NavigableMap<K,V> target) {
    super(target);
  }

  /**
   * Creates a new {@link DelegateNavigableMap} with a null target.
   */
  protected DelegateNavigableMap() {
  }

  @Override
  public Map.Entry<K,V> lowerEntry(final K key) {
    return ((NavigableMap<K,V>)target).lowerEntry(key);
  }

  @Override
  public K lowerKey(final K key) {
    return ((NavigableMap<K,V>)target).lowerKey(key);
  }

  @Override
  public Map.Entry<K,V> floorEntry(final K key) {
    return ((NavigableMap<K,V>)target).floorEntry(key);
  }

  @Override
  public K floorKey(final K key) {
    return ((NavigableMap<K,V>)target).floorKey(key);
  }

  @Override
  public Map.Entry<K,V> ceilingEntry(final K key) {
    return ((NavigableMap<K,V>)target).ceilingEntry(key);
  }

  @Override
  public K ceilingKey(final K key) {
    return ((NavigableMap<K,V>)target).ceilingKey(key);
  }

  @Override
  public Map.Entry<K,V> higherEntry(final K key) {
    return ((NavigableMap<K,V>)target).higherEntry(key);
  }

  @Override
  public K higherKey(final K key) {
    return ((NavigableMap<K,V>)target).higherKey(key);
  }

  @Override
  public Map.Entry<K,V> firstEntry() {
    return ((NavigableMap<K,V>)target).firstEntry();
  }

  @Override
  public Map.Entry<K,V> lastEntry() {
    return ((NavigableMap<K,V>)target).lastEntry();
  }

  @Override
  public Map.Entry<K,V> pollFirstEntry() {
    return ((NavigableMap<K,V>)target).pollFirstEntry();
  }

  @Override
  public Map.Entry<K,V> pollLastEntry() {
    return ((NavigableMap<K,V>)target).pollLastEntry();
  }

  @Override
  public NavigableMap<K,V> descendingMap() {
    return ((NavigableMap<K,V>)target).descendingMap();
  }

  @Override
  public NavigableSet<K> navigableKeySet() {
    return ((NavigableMap<K,V>)target).navigableKeySet();
  }

  @Override
  public NavigableSet<K> descendingKeySet() {
    return ((NavigableMap<K,V>)target).descendingKeySet();
  }

  @Override
  public NavigableMap<K,V> subMap(final K fromKey, final boolean fromInclusive, final K toKey, final boolean toInclusive) {
    return ((NavigableMap<K,V>)target).subMap(fromKey, fromInclusive, toKey, toInclusive);
  }

  @Override
  public NavigableMap<K,V> headMap(final K toKey, final boolean inclusive) {
    return ((NavigableMap<K,V>)target).headMap(toKey, inclusive);
  }

  @Override
  public NavigableMap<K,V> tailMap(final K fromKey, final boolean inclusive) {
    return ((NavigableMap<K,V>)target).tailMap(fromKey, inclusive);
  }
}