/*
 * Decompiled with CFR 0.152.
 */
package org.kaazing.gateway.util.asn1;

import java.nio.ByteBuffer;
import java.text.SimpleDateFormat;
import java.util.BitSet;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.kaazing.gateway.util.der.DerId;
import org.kaazing.gateway.util.der.DerUtils;

public final class Asn1Utils {
    private static final int ASN1_INTEGER_TAG_NUM = 2;
    private static final int ASN1_BIT_STRING_TAG_NUM = 3;
    private static final int ASN1_OCTET_STRING_TAG_NUM = 4;
    private static final int ASN1_SEQUENCE_TAG_NUM = 16;
    private static final int ASN1_GENERALIZED_TIME_TAG_NUM = 24;
    private static final int ASN1_IA5STRING_TAG_NUM = 27;
    private static final String GENERALIZED_TIME_REGEX = "(\\d{4})(\\d{2})(\\d{2})(\\d{2})(\\d{2})(\\d{2})(\\.\\d+)?(Z|[+-]\\d{4})?";
    private static final Pattern GENERALIZED_TIME_PATTERN = Pattern.compile("(\\d{4})(\\d{2})(\\d{2})(\\d{2})(\\d{2})(\\d{2})(\\.\\d+)?(Z|[+-]\\d{4})?");
    private static final String GENERALIZED_TIME_FORMAT = "yyyyMMddHHmmss'Z'";
    private static final int[] BIT_STRING_MASK = new int[]{128, 64, 32, 16, 8, 4, 2, 1};

    public static BitSet decodeBitString(ByteBuffer buf) {
        DerId id = DerId.decode(buf);
        if (!id.matches(DerId.TagClass.UNIVERSAL, DerId.EncodingType.PRIMITIVE, 3) && !id.matches(DerId.TagClass.UNIVERSAL, DerId.EncodingType.CONSTRUCTED, 3)) {
            throw new IllegalArgumentException("Expected BIT STRING identifier, received " + id);
        }
        int len = DerUtils.decodeLength(buf);
        if (buf.remaining() < len) {
            throw new IllegalArgumentException("Insufficient content for BIT STRING");
        }
        if (id.getEncodingType() == DerId.EncodingType.PRIMITIVE) {
            buf.get();
            --len;
        }
        int nbits = len * 8;
        BitSet bits = new BitSet(nbits);
        for (int i = 0; i < len; ++i) {
            int bitIndex = (i + 1) * 8 - 1;
            for (short next = (short)(0xFF & buf.get()); next != 0; next = (short)(next >>> 1)) {
                if ((next & 1) == 1) {
                    bits.set(bitIndex);
                }
                --bitIndex;
            }
        }
        return bits;
    }

    public static Date decodeGeneralizedTime(ByteBuffer buf) {
        Calendar cal;
        DerId id = DerId.decode(buf);
        if (!id.matches(DerId.TagClass.UNIVERSAL, DerId.EncodingType.PRIMITIVE, 24)) {
            throw new IllegalArgumentException("Expected GeneralizedTime identifier, received " + id);
        }
        int len = DerUtils.decodeLength(buf);
        if (buf.remaining() < len) {
            throw new IllegalArgumentException("Insufficient content for GeneralizedTime");
        }
        byte[] dst = new byte[len];
        buf.get(dst);
        String iso8601DateString = new String(dst);
        Matcher matcher = GENERALIZED_TIME_PATTERN.matcher(iso8601DateString);
        if (matcher.matches()) {
            String tzStr;
            cal = Calendar.getInstance();
            cal.clear();
            cal.set(1, Integer.parseInt(matcher.group(1)));
            cal.set(2, Integer.parseInt(matcher.group(2)) - 1);
            cal.set(5, Integer.parseInt(matcher.group(3)));
            cal.set(11, Integer.parseInt(matcher.group(4)));
            cal.set(12, Integer.parseInt(matcher.group(5)));
            cal.set(13, Integer.parseInt(matcher.group(6)));
            String fracSecStr = matcher.group(7);
            if (fracSecStr != null) {
                cal.set(14, (int)(Float.parseFloat(fracSecStr) * 1000.0f));
            }
            if ((tzStr = matcher.group(8)) != null) {
                cal.setTimeZone(TimeZone.getTimeZone("Z".equals(tzStr) ? "GMT" : "GMT" + tzStr));
            }
        } else {
            throw new IllegalArgumentException("Malformed GeneralizedTime " + iso8601DateString);
        }
        Date date = cal.getTime();
        return date;
    }

    public static String decodeIA5String(ByteBuffer buf) {
        DerId id = DerId.decode(buf);
        if (!id.matches(DerId.TagClass.UNIVERSAL, DerId.EncodingType.PRIMITIVE, 27)) {
            throw new IllegalArgumentException("Expected IA5String identifier, received " + id);
        }
        int len = DerUtils.decodeLength(buf);
        if (buf.remaining() < len) {
            throw new IllegalArgumentException("Insufficient content for IA5String");
        }
        byte[] dst = new byte[len];
        buf.get(dst);
        return new String(dst);
    }

    public static int decodeInteger(ByteBuffer buf) {
        DerId id = DerId.decode(buf);
        if (!id.matches(DerId.TagClass.UNIVERSAL, DerId.EncodingType.PRIMITIVE, 2)) {
            throw new IllegalArgumentException("Expected INTEGER identifier, received " + id);
        }
        int len = DerUtils.decodeLength(buf);
        if (buf.remaining() < len) {
            throw new IllegalArgumentException("Insufficient content for INTEGER");
        }
        int value = 0;
        for (int i = 0; i < len; ++i) {
            value = (value << 8) + (0xFF & buf.get());
        }
        return value;
    }

    public static short[] decodeOctetString(ByteBuffer buf) {
        DerId id = DerId.decode(buf);
        if (!id.matches(DerId.TagClass.UNIVERSAL, 4)) {
            throw new IllegalArgumentException("Expected OCTET STRING identifier, received " + id);
        }
        int len = DerUtils.decodeLength(buf);
        if (buf.remaining() < len) {
            throw new IllegalArgumentException("Insufficient content for OCTET STRING");
        }
        short[] dst = new short[len];
        for (int i = 0; i < len; ++i) {
            dst[i] = (short)(0xFF & buf.get());
        }
        return dst;
    }

    public static int decodeSequence(ByteBuffer buf) {
        DerId id = DerId.decode(buf);
        if (!id.matches(DerId.TagClass.UNIVERSAL, DerId.EncodingType.CONSTRUCTED, 16)) {
            throw new IllegalArgumentException("Expected SEQUENCE identifier, received " + id);
        }
        int len = DerUtils.decodeLength(buf);
        if (buf.remaining() < len) {
            throw new IllegalArgumentException("Insufficient content for SEQUENCE");
        }
        return len;
    }

    public static int encodeBitString(BitSet value, int nbits, ByteBuffer buf) {
        int contentLength;
        if (value == null || nbits < value.length()) {
            throw new IllegalArgumentException();
        }
        int pos = buf.position();
        for (int i = contentLength = (int)Math.ceil((double)nbits / 8.0); i > 0; --i) {
            byte octet = 0;
            for (int j = (i - 1) * 8; j < i * 8; ++j) {
                if (!value.get(j)) continue;
                octet = (byte)(octet | BIT_STRING_MASK[j % 8]);
            }
            buf.put(--pos, octet);
        }
        buf.put(--pos, (byte)0);
        buf.position(buf.position() - ++contentLength);
        int headerLength = DerUtils.encodeIdAndLength(DerId.TagClass.UNIVERSAL, DerId.EncodingType.PRIMITIVE, 3, contentLength, buf);
        return headerLength + contentLength;
    }

    public static int encodeGeneralizedTime(Date date, ByteBuffer buf) {
        if (date == null) {
            throw new IllegalArgumentException();
        }
        int pos = buf.position();
        SimpleDateFormat format = new SimpleDateFormat(GENERALIZED_TIME_FORMAT);
        format.setTimeZone(TimeZone.getTimeZone("GMT"));
        String value = format.format(date);
        byte[] data = value.getBytes();
        for (int i = data.length - 1; i >= 0; --i) {
            buf.put(--pos, data[i]);
        }
        buf.position(buf.position() - data.length);
        int headerLength = DerUtils.encodeIdAndLength(DerId.TagClass.UNIVERSAL, DerId.EncodingType.PRIMITIVE, 24, data.length, buf);
        return headerLength + data.length;
    }

    public static int encodeIA5String(String value, ByteBuffer buf) {
        int pos = buf.position();
        byte[] data = value == null ? new byte[]{} : value.getBytes();
        for (int i = data.length - 1; i >= 0; --i) {
            buf.put(--pos, data[i]);
        }
        buf.position(buf.position() - data.length);
        int headerLength = DerUtils.encodeIdAndLength(DerId.TagClass.UNIVERSAL, DerId.EncodingType.PRIMITIVE, 27, data.length, buf);
        return headerLength + data.length;
    }

    public static int encodeInteger(int value, ByteBuffer buf) {
        int pos = buf.position();
        int contentLength = 0;
        do {
            buf.put(--pos, (byte)(value & 0xFF));
            ++contentLength;
        } while ((value >>>= 8) != 0);
        buf.position(buf.position() - contentLength);
        int headerLen = DerUtils.encodeIdAndLength(DerId.TagClass.UNIVERSAL, DerId.EncodingType.PRIMITIVE, 2, contentLength, buf);
        return headerLen + contentLength;
    }

    public static int encodeOctetString(short[] octets, ByteBuffer buf) {
        if (octets == null) {
            octets = new short[]{};
        }
        int pos = buf.position();
        for (int i = octets.length - 1; i >= 0; --i) {
            buf.put(--pos, (byte)octets[i]);
        }
        buf.position(buf.position() - octets.length);
        int headerLength = DerUtils.encodeIdAndLength(DerId.TagClass.UNIVERSAL, DerId.EncodingType.PRIMITIVE, 4, octets.length, buf);
        return headerLength + octets.length;
    }

    public static int encodeSequence(int contentLength, ByteBuffer buf) {
        int headerLength = DerUtils.encodeIdAndLength(DerId.TagClass.UNIVERSAL, DerId.EncodingType.CONSTRUCTED, 16, contentLength, buf);
        return headerLength + contentLength;
    }

    public static int sizeOfBitString(BitSet value, int nbits) {
        return DerUtils.sizeOf(3, (int)Math.ceil((double)nbits / 8.0) + 1);
    }

    public static int sizeOfGeneralizedTime(Date date) {
        return DerUtils.sizeOf(24, GENERALIZED_TIME_FORMAT.length() - 2);
    }

    public static int sizeOfIA5String(String value) {
        return DerUtils.sizeOf(27, value == null ? 0 : value.getBytes().length);
    }

    public static int sizeOfInteger(int value) {
        int contentLength = 0;
        do {
            ++contentLength;
        } while ((value >>>= 8) != 0);
        return DerUtils.sizeOf(2, contentLength);
    }

    public static int sizeOfOctetString(short[] octets) {
        return DerUtils.sizeOf(4, octets == null ? 0 : octets.length);
    }

    public static int sizeOfSequence(int length) {
        return DerUtils.sizeOf(16, length);
    }

    private Asn1Utils() {
    }
}

