/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.cli.patching;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.infinispan.cli.logging.Messages;
import org.infinispan.cli.patching.PatchInfo;
import org.infinispan.cli.patching.PatchOperation;
import org.infinispan.cli.patching.ServerFile;
import org.infinispan.cli.util.Utils;
import org.infinispan.commons.dataconversion.internal.Json;
import org.infinispan.commons.util.Version;

public class PatchTool {
    public static final String PATCHES_DIR = ".patches";
    public static final String PATCHES_FILE = "patches.json";
    private final PrintStream out;
    private final PrintStream err;

    public PatchTool(PrintStream out, PrintStream err) {
        this.out = out;
        this.err = err;
    }

    public void createPatch(String qualifier, Path patch, Path target, Path ... sources) throws IOException {
        Version targetVersion = this.getVersion(target);
        Map<Path, ServerFile> targetFiles = this.getServerFiles(target);
        try (FileSystem zipfs = this.getPatchFile(patch, true);){
            for (Path source : sources) {
                this.createSinglePatch(qualifier, source, target, targetVersion, targetFiles, zipfs);
            }
        }
    }

    public void describePatch(Path patch, boolean verbose) throws IOException {
        try (FileSystem zipfs = this.getPatchFile(patch);){
            this.getPatchInfos(zipfs).forEach(patchInfo -> {
                this.out.println(patchInfo);
                if (verbose) {
                    patchInfo.getOperations().forEach(op -> this.out.println("  " + String.valueOf(op)));
                }
            });
        }
    }

    public void listPatches(Path target, boolean verbose) {
        List<PatchInfo> installedPatches = this.getInstalledPatches(target);
        if (installedPatches.isEmpty()) {
            this.out.println(Messages.MSG.patchNoPatchesInstalled());
        } else {
            for (PatchInfo patchInfo : installedPatches) {
                this.out.println(Messages.MSG.patchInfo(patchInfo));
                if (!verbose) continue;
                patchInfo.getOperations().forEach(op -> this.out.println("  " + String.valueOf(op)));
            }
        }
    }

    public void installPatch(Path patch, Path target, boolean dryRun) throws IOException {
        Version targetVersion = this.getVersion(target);
        String version = targetVersion.brandVersion();
        String brandName = targetVersion.brandName();
        List<PatchInfo> installedPatches = this.getInstalledPatches(target);
        try (FileSystem zipfs = this.getPatchFile(patch);){
            Path file;
            String sha256;
            String sha2562;
            PatchInfo patchInfo = this.getPatchInfos(zipfs).stream().filter(info -> brandName.equals(info.getBrandName()) && version.equals(info.getSourceVersion())).findFirst().orElseThrow(() -> {
                throw Messages.MSG.patchCannotApply(brandName, version);
            });
            List<PatchOperation> operations = patchInfo.getOperations();
            ArrayList<String> errors = new ArrayList<String>();
            for (PatchOperation operation : operations) {
                switch (operation.getAction()) {
                    case ADD: 
                    case SOFT_REPLACE: 
                    case HARD_REPLACE: 
                    case UPGRADE: {
                        sha2562 = Utils.sha256(zipfs.getPath(operation.getNewPath().toString(), new String[0]));
                        if (sha2562 != null && sha2562.equals(operation.getNewDigest())) break;
                        errors.add(Messages.MSG.patchCorruptArchive(operation));
                    }
                }
            }
            if (!errors.isEmpty()) {
                throw Messages.MSG.patchValidationErrors(String.join((CharSequence)"\n", errors));
            }
            for (PatchOperation operation : operations) {
                switch (operation.getAction()) {
                    case ADD: 
                    case SOFT_REPLACE: {
                        break;
                    }
                    case HARD_REPLACE: 
                    case UPGRADE: 
                    case REMOVE: {
                        sha2562 = Utils.sha256(target.resolve(operation.getPath()));
                        if (sha2562 != null && sha2562.equals(operation.getDigest())) break;
                        errors.add(Messages.MSG.patchShaMismatch(operation.getPath(), operation.getDigest(), sha2562));
                    }
                }
            }
            if (!errors.isEmpty()) {
                throw Messages.MSG.patchValidationErrors(String.join((CharSequence)"\n", errors));
            }
            Path backup = this.getBackupPath(target, patchInfo);
            Files.createDirectories(backup, new FileAttribute[0]);
            for (PatchOperation operation : operations) {
                switch (operation.getAction()) {
                    case ADD: {
                        break;
                    }
                    case SOFT_REPLACE: {
                        sha256 = Utils.sha256(target.resolve(operation.getPath()));
                        if (sha256 == null || !sha256.equals(operation.getDigest())) break;
                    }
                    case HARD_REPLACE: 
                    case UPGRADE: 
                    case REMOVE: {
                        file = backup.resolve(operation.getPath());
                        this.println(dryRun, Messages.MSG.patchBackup(target.resolve(operation.getPath()), file));
                        if (dryRun) break;
                        Files.createDirectories(file.getParent(), new FileAttribute[0]);
                        Files.move(target.resolve(operation.getPath()), file, new CopyOption[0]);
                    }
                }
            }
            for (PatchOperation operation : operations) {
                switch (operation.getAction()) {
                    case REMOVE: {
                        break;
                    }
                    case SOFT_REPLACE: {
                        sha256 = Utils.sha256(target.resolve(operation.getPath()));
                        if (sha256 == null || sha256.equals(operation.getDigest())) {
                            if (dryRun) break;
                            file = Files.copy(zipfs.getPath(operation.getNewPath().toString(), new String[0]), target.resolve(operation.getNewPath()), new CopyOption[0]);
                            Files.setPosixFilePermissions(file, PosixFilePermissions.fromString(operation.getNewPermissions()));
                            break;
                        }
                        if (dryRun) break;
                        file = target.resolve(operation.getNewPath());
                        file = file.getParent().resolve(file.getFileName().toString() + "-" + patchInfo.getTargetVersion());
                        Files.copy(zipfs.getPath(operation.getNewPath().toString(), new String[0]), file, new CopyOption[0]);
                        Files.setPosixFilePermissions(file, PosixFilePermissions.fromString(operation.getNewPermissions()));
                        break;
                    }
                    case ADD: 
                    case HARD_REPLACE: 
                    case UPGRADE: {
                        if (dryRun) break;
                        file = target.resolve(operation.getNewPath());
                        if (file.getParent() != null) {
                            Files.createDirectories(file.getParent(), new FileAttribute[0]);
                        }
                        Files.copy(zipfs.getPath(operation.getNewPath().toString(), new String[0]), file, StandardCopyOption.REPLACE_EXISTING);
                        Files.setPosixFilePermissions(file, PosixFilePermissions.fromString(operation.getNewPermissions()));
                    }
                }
            }
            patchInfo.setInstallationDate(new Date());
            if (!dryRun) {
                installedPatches.add(patchInfo);
                this.writeInstalledPatches(target, installedPatches);
            }
            this.println(dryRun, Messages.MSG.patchInfo(patchInfo));
        }
    }

    public void rollbackPatch(Path target, boolean dryRun) throws IOException {
        List<PatchInfo> installedPatches = this.getInstalledPatches(target);
        if (installedPatches.isEmpty()) {
            throw Messages.MSG.patchNoPatchesInstalledToRollback();
        }
        PatchInfo patchInfo = installedPatches.remove(installedPatches.size() - 1);
        Path backup = this.getBackupPath(target, patchInfo);
        block6: for (PatchOperation operation : patchInfo.getOperations()) {
            switch (operation.getAction()) {
                case ADD: {
                    Path file = target.resolve(operation.getNewPath());
                    this.println(dryRun, Messages.MSG.patchRollbackFile(file));
                    if (dryRun) continue block6;
                    Files.delete(file);
                    break;
                }
                case SOFT_REPLACE: {
                    Path file = target.resolve(operation.getPath());
                    String sha256 = Utils.sha256(file);
                    if (sha256 != null && sha256.equals(operation.getNewDigest())) {
                        this.println(dryRun, Messages.MSG.patchRollbackFile(file));
                        if (!dryRun) {
                            Files.move(backup.resolve(operation.getPath()), target.resolve(operation.getPath()), StandardCopyOption.REPLACE_EXISTING);
                        }
                    }
                    file = target.resolve(operation.getNewPath());
                    if (!Files.exists(file = file.getParent().resolve(file.getFileName().toString() + "-" + patchInfo.getTargetVersion()), new LinkOption[0])) break;
                    this.println(dryRun, Messages.MSG.patchRollbackFile(file));
                    if (dryRun) break;
                    Files.delete(file);
                    break;
                }
                case UPGRADE: {
                    Path file = target.resolve(operation.getNewPath());
                    if (!dryRun) {
                        Files.delete(file);
                    }
                }
                case HARD_REPLACE: 
                case REMOVE: {
                    Path file = target.resolve(operation.getPath());
                    this.println(dryRun, Messages.MSG.patchRollbackFile(file));
                    if (dryRun) break;
                    Files.move(backup.resolve(operation.getPath()), target.resolve(operation.getPath()), StandardCopyOption.REPLACE_EXISTING);
                }
            }
        }
        if (!dryRun) {
            this.writeInstalledPatches(target, installedPatches);
        }
        this.println(dryRun, Messages.MSG.patchRollback(patchInfo));
    }

    private void println(boolean dryRun, String msg) {
        if (dryRun) {
            this.out.print(Messages.MSG.patchDryRun());
        }
        this.out.println(msg);
    }

    private Path getBackupPath(Path target, PatchInfo patchInfo) {
        return target.resolve(PATCHES_DIR).resolve(patchInfo.getSourceVersion() + "_" + patchInfo.getTargetVersion());
    }

    private List<PatchInfo> getInstalledPatches(Path target) {
        Path patchesFile = target.resolve(PATCHES_DIR).resolve(PATCHES_FILE);
        try {
            Json read = Json.read((String)Files.readString(patchesFile));
            return read.asJsonList().stream().map(PatchInfo::fromJson).collect(Collectors.toList());
        }
        catch (NoSuchFileException e) {
            return new ArrayList<PatchInfo>();
        }
        catch (IOException e) {
            throw Messages.MSG.patchCannotRead(patchesFile, e);
        }
    }

    private void writeInstalledPatches(Path target, List<PatchInfo> patches) {
        try (OutputStream os = Files.newOutputStream(Files.createDirectories(target.resolve(PATCHES_DIR), new FileAttribute[0]).resolve(PATCHES_FILE), StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);){
            String json = Json.make(patches).toString();
            os.write(json.getBytes(StandardCharsets.UTF_8));
        }
        catch (IOException e) {
            throw Messages.MSG.patchCannotWritePatchesFile(e);
        }
    }

    private void createSinglePatch(String qualifier, Path source, Path target, Version targetVersion, Map<Path, ServerFile> targetFiles, FileSystem zipfs) throws IOException {
        String targetBrand;
        Version sourceVersion = this.getVersion(source);
        String sourceBrand = sourceVersion.brandName();
        if (!sourceBrand.equals(targetBrand = targetVersion.brandName())) {
            throw Messages.MSG.patchIncompatibleProduct(sourceBrand, targetBrand);
        }
        if (sourceVersion.brandVersion().equals(targetVersion.brandVersion())) {
            throw Messages.MSG.patchServerAndTargetMustBeDifferent(sourceVersion.brandVersion());
        }
        PatchInfo patchInfo = new PatchInfo(sourceBrand, sourceVersion.brandVersion(), targetVersion.brandVersion(), qualifier);
        Map<Path, ServerFile> v1Files = this.getServerFiles(source);
        List<PatchOperation> operations = patchInfo.getOperations();
        v1Files.forEach((k1, v1File) -> {
            if (!targetFiles.containsKey(k1)) {
                operations.add(PatchOperation.remove(v1File.getVersionedPath(), v1File.getDigest(), v1File.getPermissions()));
            } else {
                ServerFile targetFile = (ServerFile)targetFiles.get(k1);
                if (!v1File.getFilename().equals(targetFile.getFilename())) {
                    operations.add(PatchOperation.upgrade(v1File.getVersionedPath(), v1File.getDigest(), v1File.getPermissions(), targetFile.getVersionedPath(), targetFile.getDigest(), targetFile.getPermissions()));
                    this.addFileToZip(zipfs, target, targetFile);
                } else if (!v1File.getDigest().equals(targetFile.getDigest())) {
                    operations.add(PatchOperation.replace(targetFile.isSoft(), targetFile.getVersionedPath(), v1File.getDigest(), v1File.getPermissions(), targetFile.getDigest(), targetFile.getPermissions()));
                    this.addFileToZip(zipfs, target, targetFile);
                }
            }
        });
        targetFiles.forEach((k2, targetFile) -> {
            if (!v1Files.containsKey(k2)) {
                operations.add(PatchOperation.add(targetFile.getVersionedPath(), targetFile.getDigest(), targetFile.getPermissions()));
                this.addFileToZip(zipfs, target, (ServerFile)targetFile);
            }
        });
        Path patchPath = zipfs.getPath("patch-" + patchInfo.getSourceVersion() + "_" + patchInfo.getTargetVersion() + ".json", new String[0]);
        try (OutputStream os = Files.newOutputStream(patchPath, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);){
            String json = patchInfo.toJson().toPrettyString();
            os.write(json.getBytes(StandardCharsets.UTF_8));
        }
    }

    private Map<Path, ServerFile> getServerFiles(final Path base) throws IOException {
        final Pattern IGNORE = Pattern.compile("^(\\.patches/|server/data/|server/log/|server/lib/).*$");
        final Pattern SOFT_REPLACE_PATTERN = Pattern.compile("^server/conf/.*$");
        final TreeMap<Path, ServerFile> files = new TreeMap<Path, ServerFile>();
        Files.walkFileTree(base, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(this){
            final /* synthetic */ PatchTool this$0;
            {
                this.this$0 = this$0;
            }

            @Override
            public FileVisitResult visitFile(Path oPath, BasicFileAttributes attrs) {
                Path rPath = base.relativize(oPath);
                String rPathName = rPath.toString();
                if (!IGNORE.matcher(rPathName).matches()) {
                    ServerFile file = new ServerFile(rPath, Utils.sha256(oPath), PatchTool.getPermissions(oPath), SOFT_REPLACE_PATTERN.matcher(rPathName).matches());
                    files.put(file.getUnversionedPath(), file);
                }
                return FileVisitResult.CONTINUE;
            }
        });
        return files;
    }

    private static String getPermissions(Path path) {
        try {
            return PosixFilePermissions.toString(Files.getPosixFilePermissions(path, new LinkOption[0]));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private List<PatchInfo> getPatchInfos(FileSystem zipfs) throws IOException {
        List<Path> paths = Files.find(zipfs.getPath("/", new String[0]), 1, (p, a) -> a.isRegularFile() && p.getFileName().toString().startsWith("patch-") && p.getFileName().toString().endsWith(".json"), new FileVisitOption[0]).toList();
        ArrayList<PatchInfo> infos = new ArrayList<PatchInfo>(paths.size());
        for (Path path : paths) {
            Json json = Json.read((String)Files.readString(path));
            infos.add(PatchInfo.fromJson(json));
        }
        return infos;
    }

    private Version getVersion(Path base) throws IOException {
        Path lib = base.resolve("lib");
        File[] commons = lib.toFile().listFiles((dir, name) -> name.startsWith("infinispan-commons-") && name.endsWith(".jar"));
        if (commons == null || commons.length != 1) {
            throw Messages.MSG.patchCannotFindCommons(lib);
        }
        URI jarUri = URI.create("jar:" + String.valueOf(commons[0].toURI()));
        try (FileSystem zipfs = FileSystems.newFileSystem(jarUri, Collections.emptyMap());){
            Version version;
            block13: {
                InputStream in = Files.newInputStream(zipfs.getPath("META-INF", "infinispan-version.properties"), new OpenOption[0]);
                try {
                    version = Version.from((InputStream)in);
                    if (in == null) break block13;
                }
                catch (Throwable throwable) {
                    if (in != null) {
                        try {
                            in.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                in.close();
            }
            return version;
        }
    }

    private FileSystem getPatchFile(Path patch, boolean create) throws IOException {
        if (create && patch.toFile().exists()) {
            throw Messages.MSG.patchFileAlreadyExists(patch);
        }
        URI jarUri = URI.create("jar:" + String.valueOf(patch.toUri()));
        return FileSystems.newFileSystem(jarUri, create ? Collections.singletonMap("create", "true") : Collections.emptyMap());
    }

    private FileSystem getPatchFile(Path patch) throws IOException {
        return this.getPatchFile(patch, false);
    }

    private void addFileToZip(FileSystem zipfs, Path basePath, ServerFile file) {
        try {
            Path target = zipfs.getPath(file.getVersionedPath().toString(), new String[0]);
            this.out.println(Messages.MSG.patchCreateAdd(target));
            if (target.getParent() != null) {
                Files.createDirectories(target.getParent(), new FileAttribute[0]);
            }
            Files.copy(basePath.resolve(file.getVersionedPath()), target, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            throw Messages.MSG.patchCreateError(e);
        }
    }
}

