package org.immutables.value.processor.encode;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;

/**
 * EncElementMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.encode.EncodingMetadata.Element}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.encode.Mirrors.EncElement} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class EncElementMirror implements Mirrors.EncElement {
  public static final String QUALIFIED_NAME = "org.immutables.encode.EncodingMetadata.Element";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.encode.Mirrors.EncElement";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "Element";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code EncElementMirror}, present if this annotation found
   */
  public static Optional<EncElementMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code EncElementMirror}, present if this annotation found
   */
  public static Optional<EncElementMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new EncElementMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code EncElementMirror}s
   */
  public static ImmutableList<EncElementMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<EncElementMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new EncElementMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code EncElementMirror}
   */
  public static EncElementMirror from(TypeElement element) {
    return new EncElementMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code EncElementMirror}, present if mirror matched this annotation type
   */
  public static Optional<EncElementMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final String name;
  private final String type;
  private final String naming;
  private final String stdNaming;
  private final String[] tags;
  private final String[] typeParams;
  private final String[] params;
  private final String[] thrown;
  private final String[] annotations;
  private final String[] doc;
  private final String code;

  private EncElementMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    String name = null;
    String type = null;
    String naming = null;
    String stdNaming = null;
    String[] tags = null;
    String[] typeParams = null;
    String[] params = null;
    String[] thrown = null;
    String[] annotations = null;
    String[] doc = null;
    String code = null;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("name".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Element");
        }
        NameExtractor nameExtractor$ = new NameExtractor();
        annotationValue$.accept(nameExtractor$, null);

        name = nameExtractor$.get();
        continue;
      }
      if ("type".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Element");
        }
        TypeExtractor typeExtractor$ = new TypeExtractor();
        annotationValue$.accept(typeExtractor$, null);

        type = typeExtractor$.get();
        continue;
      }
      if ("naming".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Element");
        }
        NamingExtractor namingExtractor$ = new NamingExtractor();
        annotationValue$.accept(namingExtractor$, null);

        naming = namingExtractor$.get();
        continue;
      }
      if ("stdNaming".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Element");
        }
        StdNamingExtractor stdNamingExtractor$ = new StdNamingExtractor();
        annotationValue$.accept(stdNamingExtractor$, null);

        stdNaming = stdNamingExtractor$.get();
        continue;
      }
      if ("tags".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Element");
        }
        TagsExtractor tagsExtractor$ = new TagsExtractor();
        annotationValue$.accept(tagsExtractor$, null);

        tags = tagsExtractor$.get();
        continue;
      }
      if ("typeParams".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Element");
        }
        TypeParamsExtractor typeParamsExtractor$ = new TypeParamsExtractor();
        annotationValue$.accept(typeParamsExtractor$, null);

        typeParams = typeParamsExtractor$.get();
        continue;
      }
      if ("params".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Element");
        }
        ParamsExtractor paramsExtractor$ = new ParamsExtractor();
        annotationValue$.accept(paramsExtractor$, null);

        params = paramsExtractor$.get();
        continue;
      }
      if ("thrown".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Element");
        }
        ThrownExtractor thrownExtractor$ = new ThrownExtractor();
        annotationValue$.accept(thrownExtractor$, null);

        thrown = thrownExtractor$.get();
        continue;
      }
      if ("annotations".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Element");
        }
        AnnotationsExtractor annotationsExtractor$ = new AnnotationsExtractor();
        annotationValue$.accept(annotationsExtractor$, null);

        annotations = annotationsExtractor$.get();
        continue;
      }
      if ("doc".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Element");
        }
        DocExtractor docExtractor$ = new DocExtractor();
        annotationValue$.accept(docExtractor$, null);

        doc = docExtractor$.get();
        continue;
      }
      if ("code".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Element");
        }
        CodeExtractor codeExtractor$ = new CodeExtractor();
        annotationValue$.accept(codeExtractor$, null);

        code = codeExtractor$.get();
        continue;
      }
    }
    this.name = Preconditions.checkNotNull(name, "default attribute 'name'");
    this.type = Preconditions.checkNotNull(type, "default attribute 'type'");
    this.naming = Preconditions.checkNotNull(naming, "default attribute 'naming'");
    this.stdNaming = Preconditions.checkNotNull(stdNaming, "default attribute 'stdNaming'");
    this.tags = Preconditions.checkNotNull(tags, "default attribute 'tags'");
    this.typeParams = Preconditions.checkNotNull(typeParams, "default attribute 'typeParams'");
    this.params = Preconditions.checkNotNull(params, "default attribute 'params'");
    this.thrown = Preconditions.checkNotNull(thrown, "default attribute 'thrown'");
    this.annotations = Preconditions.checkNotNull(annotations, "default attribute 'annotations'");
    this.doc = Preconditions.checkNotNull(doc, "default attribute 'doc'");
    this.code = Preconditions.checkNotNull(code, "default attribute 'code'");
  }

  private EncElementMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    String name = null;
    String type = null;
    String naming = null;
    String stdNaming = null;
    String[] tags = null;
    String[] typeParams = null;
    String[] params = null;
    String[] thrown = null;
    String[] annotations = null;
    String[] doc = null;
    String code = null;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("name".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'name' attribute of @Element");
        }
        NameExtractor nameExtractor$ = new NameExtractor();
        annotationValue$.accept(nameExtractor$, null);

        name = nameExtractor$.get();
        continue;
      }
      if ("type".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'type' attribute of @Element");
        }
        TypeExtractor typeExtractor$ = new TypeExtractor();
        annotationValue$.accept(typeExtractor$, null);

        type = typeExtractor$.get();
        continue;
      }
      if ("naming".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'naming' attribute of @Element");
        }
        NamingExtractor namingExtractor$ = new NamingExtractor();
        annotationValue$.accept(namingExtractor$, null);

        naming = namingExtractor$.get();
        continue;
      }
      if ("stdNaming".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'stdNaming' attribute of @Element");
        }
        StdNamingExtractor stdNamingExtractor$ = new StdNamingExtractor();
        annotationValue$.accept(stdNamingExtractor$, null);

        stdNaming = stdNamingExtractor$.get();
        continue;
      }
      if ("tags".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'tags' attribute of @Element");
        }
        TagsExtractor tagsExtractor$ = new TagsExtractor();
        annotationValue$.accept(tagsExtractor$, null);

        tags = tagsExtractor$.get();
        continue;
      }
      if ("typeParams".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'typeParams' attribute of @Element");
        }
        TypeParamsExtractor typeParamsExtractor$ = new TypeParamsExtractor();
        annotationValue$.accept(typeParamsExtractor$, null);

        typeParams = typeParamsExtractor$.get();
        continue;
      }
      if ("params".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'params' attribute of @Element");
        }
        ParamsExtractor paramsExtractor$ = new ParamsExtractor();
        annotationValue$.accept(paramsExtractor$, null);

        params = paramsExtractor$.get();
        continue;
      }
      if ("thrown".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'thrown' attribute of @Element");
        }
        ThrownExtractor thrownExtractor$ = new ThrownExtractor();
        annotationValue$.accept(thrownExtractor$, null);

        thrown = thrownExtractor$.get();
        continue;
      }
      if ("annotations".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'annotations' attribute of @Element");
        }
        AnnotationsExtractor annotationsExtractor$ = new AnnotationsExtractor();
        annotationValue$.accept(annotationsExtractor$, null);

        annotations = annotationsExtractor$.get();
        continue;
      }
      if ("doc".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'doc' attribute of @Element");
        }
        DocExtractor docExtractor$ = new DocExtractor();
        annotationValue$.accept(docExtractor$, null);

        doc = docExtractor$.get();
        continue;
      }
      if ("code".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'code' attribute of @Element");
        }
        CodeExtractor codeExtractor$ = new CodeExtractor();
        annotationValue$.accept(codeExtractor$, null);

        code = codeExtractor$.get();
        continue;
      }
    }
    this.name = Preconditions.checkNotNull(name, "value for 'name'");
    this.type = Preconditions.checkNotNull(type, "value for 'type'");
    this.naming = Preconditions.checkNotNull(naming, "value for 'naming'");
    this.stdNaming = Preconditions.checkNotNull(stdNaming, "value for 'stdNaming'");
    this.tags = Preconditions.checkNotNull(tags, "value for 'tags'");
    this.typeParams = Preconditions.checkNotNull(typeParams, "value for 'typeParams'");
    this.params = Preconditions.checkNotNull(params, "value for 'params'");
    this.thrown = Preconditions.checkNotNull(thrown, "value for 'thrown'");
    this.annotations = Preconditions.checkNotNull(annotations, "value for 'annotations'");
    this.doc = Preconditions.checkNotNull(doc, "value for 'doc'");
    this.code = Preconditions.checkNotNull(code, "value for 'code'");
  }

  /**
   * @return value of attribute {@code name}
   */
  @Override
  public String name() {
    return name;
  }

  /**
   * @return value of attribute {@code type}
   */
  @Override
  public String type() {
    return type;
  }

  /**
   * @return value of attribute {@code naming}
   */
  @Override
  public String naming() {
    return naming;
  }

  /**
   * @return value of attribute {@code stdNaming}
   */
  @Override
  public String stdNaming() {
    return stdNaming;
  }

  /**
   * @return value of attribute {@code tags}
   */
  @Override
  public String[] tags() {
    return tags.clone();
  }

  /**
   * @return value of attribute {@code typeParams}
   */
  @Override
  public String[] typeParams() {
    return typeParams.clone();
  }

  /**
   * @return value of attribute {@code params}
   */
  @Override
  public String[] params() {
    return params.clone();
  }

  /**
   * @return value of attribute {@code thrown}
   */
  @Override
  public String[] thrown() {
    return thrown.clone();
  }

  /**
   * @return value of attribute {@code annotations}
   */
  @Override
  public String[] annotations() {
    return annotations.clone();
  }

  /**
   * @return value of attribute {@code doc}
   */
  @Override
  public String[] doc() {
    return doc.clone();
  }

  /**
   * @return value of attribute {@code code}
   */
  @Override
  public String code() {
    return code;
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code EncElement.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return Mirrors.EncElement.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "name".hashCode() ^ name.hashCode();
    h += 127 * "type".hashCode() ^ type.hashCode();
    h += 127 * "naming".hashCode() ^ naming.hashCode();
    h += 127 * "stdNaming".hashCode() ^ stdNaming.hashCode();
    h += 127 * "tags".hashCode() ^ Arrays.hashCode(tags);
    h += 127 * "typeParams".hashCode() ^ Arrays.hashCode(typeParams);
    h += 127 * "params".hashCode() ^ Arrays.hashCode(params);
    h += 127 * "thrown".hashCode() ^ Arrays.hashCode(thrown);
    h += 127 * "annotations".hashCode() ^ Arrays.hashCode(annotations);
    h += 127 * "doc".hashCode() ^ Arrays.hashCode(doc);
    h += 127 * "code".hashCode() ^ code.hashCode();
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof EncElementMirror) {
      EncElementMirror otherMirror = (EncElementMirror) other;
      return name.equals(otherMirror.name)
          && type.equals(otherMirror.type)
          && naming.equals(otherMirror.naming)
          && stdNaming.equals(otherMirror.stdNaming)
          && Arrays.equals(tags, otherMirror.tags)
          && Arrays.equals(typeParams, otherMirror.typeParams)
          && Arrays.equals(params, otherMirror.params)
          && Arrays.equals(thrown, otherMirror.thrown)
          && Arrays.equals(annotations, otherMirror.annotations)
          && Arrays.equals(doc, otherMirror.doc)
          && code.equals(otherMirror.code);
    }
    return false;
  }

  @Override
  public String toString() {
    return "EncElementMirror:" + annotationMirror;
  }

  private static class NameExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'name' in @" + QUALIFIED_NAME);
    }
  }

  private static class TypeExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'type' in @" + QUALIFIED_NAME);
    }
  }

  private static class NamingExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'naming' in @" + QUALIFIED_NAME);
    }
  }

  private static class StdNamingExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'stdNaming' in @" + QUALIFIED_NAME);
    }
  }

  private static class TagsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String[] values;
    int position;

    @Override
    public Void visitString(String value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new String[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    String[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'tags' in @" + QUALIFIED_NAME);
    }
  }

  private static class TypeParamsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String[] values;
    int position;

    @Override
    public Void visitString(String value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new String[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    String[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'typeParams' in @" + QUALIFIED_NAME);
    }
  }

  private static class ParamsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String[] values;
    int position;

    @Override
    public Void visitString(String value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new String[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    String[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'params' in @" + QUALIFIED_NAME);
    }
  }

  private static class ThrownExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String[] values;
    int position;

    @Override
    public Void visitString(String value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new String[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    String[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'thrown' in @" + QUALIFIED_NAME);
    }
  }

  private static class AnnotationsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String[] values;
    int position;

    @Override
    public Void visitString(String value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new String[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    String[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'annotations' in @" + QUALIFIED_NAME);
    }
  }

  private static class DocExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String[] values;
    int position;

    @Override
    public Void visitString(String value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new String[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    String[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'doc' in @" + QUALIFIED_NAME);
    }
  }

  private static class CodeExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'code' in @" + QUALIFIED_NAME);
    }
  }
}
