/*
 * Decompiled with CFR 0.152.
 */
package org.jivesoftware.openfire.audit.spi;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.TimeZone;
import java.util.TimerTask;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import org.dom4j.DocumentFactory;
import org.dom4j.Element;
import org.jivesoftware.openfire.audit.AuditManager;
import org.jivesoftware.openfire.audit.Auditor;
import org.jivesoftware.openfire.session.Session;
import org.jivesoftware.util.FastDateFormat;
import org.jivesoftware.util.JiveGlobals;
import org.jivesoftware.util.LocaleUtils;
import org.jivesoftware.util.StringUtils;
import org.jivesoftware.util.TaskEngine;
import org.jivesoftware.util.XMLWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xmpp.packet.IQ;
import org.xmpp.packet.Message;
import org.xmpp.packet.Packet;
import org.xmpp.packet.Presence;

public class AuditorImpl
implements Auditor {
    private static final Logger Log = LoggerFactory.getLogger(AuditorImpl.class);
    private AuditManager auditManager;
    private File currentAuditFile;
    private Writer writer;
    private XMLWriter xmlWriter;
    private Date currentDateLimit;
    private long maxTotalSize;
    private long maxFileSize;
    private int maxDays;
    private boolean closed = false;
    private String logDir;
    private File baseFolder;
    private BlockingQueue<AuditPacket> logQueue = new LinkedBlockingQueue<AuditPacket>();
    private final int maxTotalFilesDay = 1000;
    private int filesIndex = 0;
    private SaveQueuedPacketsTask saveQueuedPacketsTask;
    private FastDateFormat dateFormat;
    private static FastDateFormat auditFormat;

    public AuditorImpl(AuditManager manager) {
        this.auditManager = manager;
        this.dateFormat = FastDateFormat.getInstance("yyyyMMdd", TimeZone.getTimeZone("UTC"));
        auditFormat = FastDateFormat.getInstance("MMM dd, yyyy hh:mm:ss:SSS a", JiveGlobals.getLocale());
    }

    protected void setMaxValues(int totalSize, int fileSize, int days) {
        this.maxTotalSize = (long)totalSize * 1024L * 1024L;
        this.maxFileSize = (long)fileSize * 1024L * 1024L;
        this.maxDays = days;
    }

    public void setLogTimeout(int logTimeout) {
        if (this.saveQueuedPacketsTask != null) {
            this.saveQueuedPacketsTask.cancel();
        }
        this.saveQueuedPacketsTask = new SaveQueuedPacketsTask();
        TaskEngine.getInstance().schedule((TimerTask)this.saveQueuedPacketsTask, logTimeout, (long)logTimeout);
    }

    public void setLogDir(String logDir) {
        this.logDir = logDir;
        this.baseFolder = new File(logDir);
        if (!this.baseFolder.exists() && !this.baseFolder.mkdir()) {
            Log.error("Unable to create log directory: {}", (Object)this.baseFolder);
        }
    }

    @Override
    public int getQueuedPacketsNumber() {
        return this.logQueue.size();
    }

    @Override
    public void audit(Packet packet, Session session) {
        if (this.auditManager.isEnabled()) {
            if (packet instanceof Message) {
                if (this.auditManager.isAuditMessage()) {
                    this.writePacket(packet, session);
                }
            } else if (packet instanceof Presence) {
                if (this.auditManager.isAuditPresence()) {
                    this.writePacket(packet, session);
                }
            } else if (packet instanceof IQ && this.auditManager.isAuditIQ()) {
                this.writePacket(packet, session);
            }
        }
    }

    private void writePacket(Packet packet, Session session) {
        if (!this.closed) {
            this.logQueue.add(new AuditPacket(packet.createCopy(), session));
        }
    }

    @Override
    public void stop() {
        this.closed = true;
        this.saveQueuedPackets();
        this.close();
    }

    private void close() {
        if (this.xmlWriter != null) {
            try {
                this.xmlWriter.flush();
                this.writer.write("</jive>");
                this.xmlWriter.close();
                this.writer = null;
                this.xmlWriter = null;
            }
            catch (Exception e) {
                Log.error(LocaleUtils.getLocalizedString("admin.error"), (Throwable)e);
            }
        }
    }

    private void prepareAuditFile(Date auditDate) throws IOException {
        this.ensureMaxTotalSize();
        if (this.currentAuditFile == null || this.currentAuditFile.length() > this.maxFileSize || this.xmlWriter == null || this.currentDateLimit == null || auditDate.after(this.currentDateLimit)) {
            this.createAuditFile(auditDate);
        }
    }

    private void ensureMaxTotalSize() {
        FilenameFilter filter = new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.startsWith("jive.audit-") && name.endsWith(".log");
            }
        };
        File[] files = this.baseFolder.listFiles(filter);
        if (files == null) {
            Log.debug("Path '{}' does not denote a directory, or an IO exception occured while trying to list its content.", (Object)this.baseFolder);
            return;
        }
        long totalLength = 0L;
        for (File file : files) {
            totalLength += file.length();
        }
        if (totalLength > this.maxTotalSize) {
            ArrayList<File> sortedFiles = new ArrayList<File>(Arrays.asList(files));
            Collections.sort(sortedFiles, new Comparator<File>(){

                @Override
                public int compare(File o1, File o2) {
                    return o1.getName().compareTo(o2.getName());
                }
            });
            while (totalLength > this.maxTotalSize && !sortedFiles.isEmpty()) {
                File fileToDelete = (File)sortedFiles.remove(0);
                totalLength -= fileToDelete.length();
                if (fileToDelete.equals(this.currentAuditFile)) {
                    this.close();
                }
                if (fileToDelete.delete()) continue;
                Log.warn("Unable to delete file '{}' as part of regular log rotation based on size of files (Openfire failed to clean up after itself)!", (Object)fileToDelete);
            }
        }
    }

    private void ensureMaxDays() {
        File[] files;
        if (this.maxDays == -1) {
            return;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.add(5, this.maxDays * -1);
        final String oldestFile = "jive.audit-" + this.dateFormat.format(calendar.getTime()) + "-000.log";
        FilenameFilter filter = new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.startsWith("jive.audit-") && name.endsWith(".log") && name.compareTo(oldestFile) < 0;
            }
        };
        for (File fileToDelete : files = this.baseFolder.listFiles(filter)) {
            if (fileToDelete.equals(this.currentAuditFile)) {
                this.close();
            }
            if (fileToDelete.delete()) continue;
            Log.warn("Unable to delete file '{}' as part of regular log rotation based on age of file. (Openfire failed to clean up after itself)!", (Object)fileToDelete);
        }
    }

    private void createAuditFile(Date auditDate) throws IOException {
        final String filePrefix = "jive.audit-" + this.dateFormat.format(auditDate) + "-";
        if (this.currentDateLimit == null || auditDate.after(this.currentDateLimit)) {
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(auditDate);
            calendar.set(11, 23);
            calendar.set(12, 59);
            calendar.set(13, 59);
            calendar.set(14, 999);
            this.currentDateLimit = calendar.getTime();
            this.filesIndex = 0;
        }
        FilenameFilter filter = new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.startsWith(filePrefix) && name.endsWith(".log");
            }
        };
        File[] files = this.baseFolder.listFiles(filter);
        this.filesIndex = Math.max(files.length, this.filesIndex);
        if (this.filesIndex >= 1000) {
            return;
        }
        File tmpAuditFile = new File(this.logDir, filePrefix + StringUtils.zeroPadString(Integer.toString(this.filesIndex), 3) + ".log");
        if (this.filesIndex == 999 && !tmpAuditFile.exists()) {
            Log.warn("Creating last audit file for this date: " + this.dateFormat.format(auditDate));
        }
        while (this.filesIndex < 999 && tmpAuditFile.exists()) {
            Log.debug("Audit file '" + tmpAuditFile.getName() + "' does already exist.");
            ++this.filesIndex;
            tmpAuditFile = new File(this.logDir, filePrefix + StringUtils.zeroPadString(Integer.toString(this.filesIndex), 3) + ".log");
        }
        this.currentAuditFile = tmpAuditFile;
        this.close();
        this.writer = new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(this.currentAuditFile, true), StandardCharsets.UTF_8));
        this.writer.write("<jive xmlns=\"http://www.jivesoftware.org\">");
        this.xmlWriter = new XMLWriter(this.writer);
    }

    private void saveQueuedPackets() {
        ArrayList packets = new ArrayList(this.logQueue.size());
        this.logQueue.drainTo(packets);
        for (AuditPacket auditPacket : packets) {
            try {
                this.prepareAuditFile(auditPacket.getCreationDate());
                Element element = auditPacket.getElement();
                if (element == null) continue;
                this.xmlWriter.write(element);
            }
            catch (IOException e) {
                Log.error(LocaleUtils.getLocalizedString("admin.error"), (Throwable)e);
                if (this.xmlWriter == null) continue;
                this.logQueue.add(auditPacket);
            }
        }
        try {
            if (this.xmlWriter != null) {
                this.xmlWriter.flush();
            }
        }
        catch (IOException ioe) {
            Log.error(ioe.getMessage(), (Throwable)ioe);
        }
    }

    private static class AuditPacket {
        private static DocumentFactory docFactory = DocumentFactory.getInstance();
        private Element element = docFactory.createElement("packet", "http://www.jivesoftware.org");
        private Date creationDate = new Date();

        public AuditPacket(Packet packet, Session session) {
            if (session != null && session.getStreamID() != null) {
                this.element.addAttribute("streamID", session.getStreamID().toString());
            }
            switch (session == null ? 0 : session.getStatus()) {
                case 3: {
                    this.element.addAttribute("status", "auth");
                    break;
                }
                case -1: {
                    this.element.addAttribute("status", "closed");
                    break;
                }
                case 1: {
                    this.element.addAttribute("status", "connected");
                    packet.setFrom((String)null);
                    break;
                }
                default: {
                    this.element.addAttribute("status", "unknown");
                }
            }
            this.element.addAttribute("timestamp", auditFormat.format(this.creationDate));
            this.element.add(packet.getElement());
        }

        public Element getElement() {
            return this.element;
        }

        public Date getCreationDate() {
            return this.creationDate;
        }
    }

    private class SaveQueuedPacketsTask
    extends TimerTask {
        private SaveQueuedPacketsTask() {
        }

        @Override
        public void run() {
            try {
                AuditorImpl.this.ensureMaxDays();
                AuditorImpl.this.saveQueuedPackets();
            }
            catch (Throwable e) {
                Log.error(LocaleUtils.getLocalizedString("admin.error"), e);
            }
        }
    }
}

