/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.core.genscavenge.remset;

import com.oracle.svm.core.SubstrateUtil;
import com.oracle.svm.core.UnmanagedMemoryUtil;
import com.oracle.svm.core.config.ConfigurationValues;
import com.oracle.svm.core.hub.LayoutEncoding;
import com.oracle.svm.core.log.Log;
import com.oracle.svm.core.util.UnsignedUtils;
import org.graalvm.word.Pointer;
import org.graalvm.word.UnsignedWord;
import org.graalvm.word.WordBase;
import org.graalvm.word.WordFactory;

final class FirstObjectTable {
    private static final int BYTES_COVERED_BY_ENTRY = 512;
    private static final int ENTRY_SIZE_BYTES = 1;
    private static final int ENTRY_MIN = -128;
    private static final int ENTRY_MAX = 127;
    private static final int MEMORY_OFFSET_MIN = -128;
    private static final int MEMORY_OFFSET_MAX = 0;
    private static final int LINEAR_OFFSET_MIN = 1;
    private static final int LINEAR_OFFSET_MAX = 63;
    private static final int EXPONENT_MIN = 6;
    private static final int EXPONENT_MAX = 55;
    private static final int EXPONENT_BIAS = 58;
    private static final int UNINITIALIZED_ENTRY = 127;

    private FirstObjectTable() {
    }

    public static void initializeTable(Pointer table, UnsignedWord size) {
        if (SubstrateUtil.HOSTED) {
            FirstObjectTable.doInitializeTable(table, size);
        } else assert (FirstObjectTable.doInitializeTable(table, size));
    }

    private static boolean doInitializeTable(Pointer table, UnsignedWord size) {
        UnmanagedMemoryUtil.fill(table, size, (byte)127);
        return true;
    }

    public static void setTableForObject(Pointer table, UnsignedWord startOffset, UnsignedWord endOffset) {
        assert (startOffset.belowThan(endOffset));
        UnsignedWord startIndex = FirstObjectTable.memoryOffsetToIndex(startOffset);
        UnsignedWord endIndex = FirstObjectTable.memoryOffsetToIndex(endOffset.subtract(1));
        boolean startsAtCardBoundary = startOffset.unsignedRemainder(512).equal(0);
        if (startIndex.equal(endIndex) && !startsAtCardBoundary) {
            return;
        }
        if (startsAtCardBoundary) {
            FirstObjectTable.setEntryAtIndex(table, startIndex, 0);
        } else {
            startIndex = startIndex.add(1);
            UnsignedWord memoryIndexOffset = FirstObjectTable.indexToMemoryOffset(startIndex);
            int entry = FirstObjectTable.memoryOffsetToEntry(memoryIndexOffset.subtract(startOffset));
            FirstObjectTable.setEntryAtIndex(table, startIndex, entry);
        }
        UnsignedWord linearIndexMax = UnsignedUtils.min(endIndex, startIndex.add(63));
        UnsignedWord entryIndex = startIndex.add(1);
        int entry = 1;
        while (entryIndex.belowOrEqual(linearIndexMax)) {
            FirstObjectTable.setEntryAtIndex(table, entryIndex, entry);
            entryIndex = entryIndex.add(1);
            ++entry;
        }
        int unbiasedExponent = 6;
        while (entryIndex.belowOrEqual(endIndex)) {
            for (int count = 0; count < 1 << unbiasedExponent; ++count) {
                int biasedEntry = FirstObjectTable.biasExponent(unbiasedExponent);
                FirstObjectTable.setEntryAtIndex(table, entryIndex, biasedEntry);
                entryIndex = entryIndex.add(1);
                if (entryIndex.aboveThan(endIndex)) break;
            }
            ++unbiasedExponent;
        }
    }

    public static Pointer getFirstObjectImprecise(Pointer tableStart, Pointer objectsStart, Pointer objectsLimit, UnsignedWord index) {
        Pointer result;
        Pointer indexedMemoryStart;
        Pointer firstObject = FirstObjectTable.getFirstObject(tableStart, objectsStart, objectsLimit, index);
        if (firstObject.belowThan((UnsignedWord)(indexedMemoryStart = objectsStart.add(FirstObjectTable.indexToMemoryOffset(index))))) {
            Object crossingObject = firstObject.toObject();
            result = LayoutEncoding.getObjectEnd(crossingObject);
        } else {
            assert (firstObject.equal((UnsignedWord)indexedMemoryStart)) : "preciseFirstPointer.equal(indexedMemoryStart)";
            result = indexedMemoryStart;
        }
        assert (objectsStart.belowOrEqual((UnsignedWord)result)) : "memoryStart.belowOrEqual(result)";
        assert (result.belowOrEqual((UnsignedWord)objectsLimit)) : "result.belowOrEqual(memoryLimit)";
        return result;
    }

    private static Pointer getFirstObject(Pointer tableStart, Pointer objectsStart, Pointer objectsLimit, UnsignedWord index) {
        UnsignedWord currentIndex = index;
        int currentEntry = FirstObjectTable.getEntryAtIndex(tableStart, currentIndex);
        assert (currentEntry != 127) : "uninitialized first object table entry";
        if (currentEntry > 0) {
            while (currentEntry > 63) {
                int exponent = FirstObjectTable.unbiasExponent(currentEntry);
                UnsignedWord deltaIndex = FirstObjectTable.exponentToOffset(exponent);
                assert (deltaIndex.belowOrEqual(currentIndex)) : "Delta out of bounds.";
                currentIndex = currentIndex.subtract(deltaIndex);
                currentEntry = FirstObjectTable.getEntryAtIndex(tableStart, currentIndex);
            }
            if (currentEntry > 0) {
                currentIndex = currentIndex.subtract(currentEntry);
                currentEntry = FirstObjectTable.getEntryAtIndex(tableStart, currentIndex);
            }
        }
        UnsignedWord memoryOffset = FirstObjectTable.entryToMemoryOffset(currentIndex, currentEntry);
        Pointer result = objectsStart.add(memoryOffset);
        assert (objectsStart.belowOrEqual((UnsignedWord)result)) : "chunkStart.belowOrEqual(result)";
        assert (result.belowThan((UnsignedWord)objectsLimit)) : "result.belowThan(memoryLimit)";
        return result;
    }

    private static UnsignedWord entryToMemoryOffset(UnsignedWord index, int entry) {
        assert (FirstObjectTable.isMemoryOffsetEntry(entry)) : "Entry out of bounds.";
        UnsignedWord entryOffset = WordFactory.unsigned((int)(-entry)).multiply(FirstObjectTable.memoryOffsetScale());
        assert (entryOffset.belowThan(512)) : "Entry out of bounds.";
        UnsignedWord indexOffset = FirstObjectTable.indexToMemoryOffset(index);
        return indexOffset.subtract(entryOffset);
    }

    public static boolean verify(Pointer tableStart, Pointer objectsStart, Pointer objectsLimit) {
        UnsignedWord indexLimit = FirstObjectTable.getTableSizeForMemoryRange(objectsStart, objectsLimit);
        UnsignedWord index = WordFactory.unsigned((int)0);
        while (index.belowThan(indexLimit)) {
            Pointer objStart = FirstObjectTable.getFirstObject(tableStart, objectsStart, objectsLimit, index);
            if (objStart.belowThan((UnsignedWord)objectsStart) || objectsLimit.belowOrEqual((UnsignedWord)objStart)) {
                Log.log().string("The first object table entry at index ").unsigned((WordBase)index).string(" points to an object that is outside of the current chunk:  obj: ").zhex((WordBase)objStart).string(", chunk: ").zhex((WordBase)objectsStart).string(" - ").zhex((WordBase)objectsLimit).newline();
                return false;
            }
            Pointer entryStart = objectsStart.add(FirstObjectTable.indexToMemoryOffset(index));
            if (!objStart.belowOrEqual((UnsignedWord)entryStart)) {
                Log.log().string("The first object table entry at index ").unsigned((WordBase)index).string(" points to an object is not crossing nor starting at a card boundary:  obj: ").zhex((WordBase)objStart).string(", chunk: ").zhex((WordBase)objectsStart).string(" - ").zhex((WordBase)objectsLimit).newline();
                return false;
            }
            Object obj = objStart.toObject();
            Pointer objEnd = LayoutEncoding.getObjectEnd(obj);
            if (!entryStart.belowThan((UnsignedWord)objEnd)) {
                Log.log().string("The first object table entry at index ").unsigned((WordBase)index).string(" points to an object is not crossing nor starting at a card boundary:  obj: ").zhex((WordBase)objStart).string(" - ").zhex((WordBase)objEnd).string(", chunk: ").zhex((WordBase)objectsStart).string(" - ").zhex((WordBase)objectsLimit).newline();
                return false;
            }
            index = index.add(1);
        }
        return true;
    }

    private static UnsignedWord getTableSizeForMemoryRange(Pointer memoryStart, Pointer memoryLimit) {
        assert (memoryStart.belowOrEqual((UnsignedWord)memoryLimit)) : "Pointers out of order";
        Pointer memorySize = memoryLimit.subtract((UnsignedWord)memoryStart);
        UnsignedWord roundedMemory = UnsignedUtils.roundUp((UnsignedWord)memorySize, WordFactory.unsigned((int)512));
        UnsignedWord index = FirstObjectTable.memoryOffsetToIndex(roundedMemory);
        return index.multiply(1);
    }

    private static int memoryOffsetScale() {
        return ConfigurationValues.getObjectLayout().getAlignment();
    }

    private static int getEntryAtIndex(Pointer table, UnsignedWord index) {
        return table.readByte((WordBase)FirstObjectTable.indexToTableOffset(index));
    }

    private static void setEntryAtIndex(Pointer table, UnsignedWord index, int value) {
        assert (FirstObjectTable.isValidEntry(value)) : "Invalid entry";
        assert (FirstObjectTable.isUninitializedIndex(table, index) || FirstObjectTable.getEntryAtIndex(table, index) == value) : "Overwriting!";
        table.writeByte((WordBase)FirstObjectTable.indexToTableOffset(index), (byte)value);
    }

    private static boolean isUninitializedIndex(Pointer table, UnsignedWord index) {
        int entry = FirstObjectTable.getEntryAtIndex(table, index);
        return FirstObjectTable.isUninitializedEntry(entry);
    }

    private static boolean isValidEntry(int entry) {
        return -128 <= entry && entry <= 127;
    }

    private static boolean isUninitializedEntry(int entry) {
        assert (FirstObjectTable.isValidEntry(entry)) : "Invalid entry";
        return entry == 127;
    }

    private static boolean isMemoryOffsetEntry(int entry) {
        assert (FirstObjectTable.isValidEntry(entry)) : "Invalid entry";
        return -128 <= entry && entry <= 0;
    }

    private static int biasExponent(int exponent) {
        assert (6 <= exponent && exponent <= 55) : "Exponent out of bounds.";
        return exponent + 58;
    }

    private static int unbiasExponent(int entry) {
        int exponent = entry - 58;
        assert (6 <= exponent && exponent <= 55) : "Exponent out of bounds.";
        return exponent;
    }

    private static UnsignedWord exponentToOffset(int n) {
        assert (0 <= n && n <= 63) : "Exponent out of bounds.";
        return WordFactory.unsigned((long)(1L << n));
    }

    private static UnsignedWord indexToTableOffset(UnsignedWord index) {
        return index.multiply(1);
    }

    private static UnsignedWord indexToMemoryOffset(UnsignedWord index) {
        return index.multiply(512);
    }

    private static UnsignedWord memoryOffsetToIndex(UnsignedWord offset) {
        return offset.unsignedDivide(512);
    }

    private static int memoryOffsetToEntry(UnsignedWord memoryOffset) {
        assert (memoryOffset.belowThan(512)) : "Offset out of bounds.";
        UnsignedWord scaledOffset = memoryOffset.unsignedDivide(FirstObjectTable.memoryOffsetScale());
        assert (scaledOffset.multiply(FirstObjectTable.memoryOffsetScale()).equal(memoryOffset)) : "Not a multiple.";
        long result = -scaledOffset.rawValue();
        assert (-128L <= result && result <= 0L) : "Scaled offset out of bounds.";
        return (int)result;
    }
}

